#! /usr/local/bin/scotty -inf
##
## monitordaemon utility for the [TK]INED editor.
##
## Copyright (c) 1993 St. Schoek
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

LoadDefaults netguard

IpInit Netguard-Admin

##
## These are the global parameters. The port number and the monitor_name are
## used to connect to our netguard-server.  
##

if {[info exists default(port)]} {
    set port $default(port)
} else {
    set port 6341
}
if {[info exists default(server)]} {
    set monitor_name $default(server)
} else {
    set monitor_name sol.ibr.cs.tu-bs.de
}


##
## global variables used by the procedures 
##

set menus            ""
set to_socket        ""
set sema             ""
set socket           ""

##
## writes errormsg to stderr
##

proc scottyerror { msg } {
    global errorInfo

    catch {puts stderr "$msg\n$errorInfo"; flush stderr}
}

##
## convert datestring in readable string
##

proc date_convert {date} {
    
    ## date == year.month.day in "day month year"
    set date [split $date .]
    return "[lindex $date 2] [lindex $date 1] [lindex $date 0]"  
}

##
## the following procedures are used to build up the connection
## to the monitorserver
##
## proc tries to connect automatically the monitoring server
## when starting tkined
## deletes the menu if connect failed 10 times
##

proc auto_connect {} {
    global port
    global socket
    global monitor_name
    global sema

    set socket ""
    set count  0

    while {$socket == ""} {
	if {$count >= 3} {
	    set answer [ined request \
			"Connect to $monitor_name $port failed $count times." \
			[list [list "Host:" $monitor_name] \
                              [list "Port:" $port] ] \
			[list cancel connect] ]
	    if {[lindex $answer 0] == "cancel"} {
		"Delete Netguard-Admin" ""
		set socket ""
		set count 3
	    } else {
		set monitor_name [lindex $answer 1]
		set port  [lindex $answer 2]
		if {[catch {tcp connect $monitor_name $port} socket]} {
		    ined acknowledge "Connect to $monitor_name $port failed."
		    "Delete Netguard-Admin" ""
		    set socket ""
		    set count 3
		} else {
		    break
		}
	    }
	}
	if {[catch {tcp connect $monitor_name $port} socket]} {
	    catch {writeln "$socket"}
	    set socket ""
	    sleep 5
	}
	incr count
    }
    ## create a job that read the monitor socket and
    ## checks answer from monitor_name

    addinput -read $socket read_socket

    ## create the new menu
    ##

    menu_close

    # here we must set accept_job to 1 because we do not write anything
    # with write_socket until we have got something from monitor
    set sema 1
}

##
## give message about the connect command to monitor
##

proc accept_message {} {
    global to_socket

    ## tell the monitor your name
    if {[catch {exec whoami} name]} {
        set name ""
    }
    lappend to_socket "name $name"
    ## create a job that writes all messages to the monitor socket
    job create write_socket 1000
}

##
## proc asks the monitor after the connection is created about
## the current variables and interfaces measured by the measurement-agents
## of the monitor
##

proc insert_data_now {var_arg} {
    global data_info_now
    
    if {$var_arg == ""} {
	writeln "server do not give result"
	return
    }
    set data_info_now $var_arg
}

##
## Become a slave. Connect to a server
##

proc "Try connect" {list} {
    global monitor_name
    global port
    global socket
    global sema

    if {$socket != ""} {
	# connection already exists
	writeln "Connection to monitor $monitor_name is already open"
	return 
    }
    set result [ined request "Choose hostname and port number of server:" \
		[list [list "Host:" $monitor_name] \
		      [list "Port:" $port] ] \
		[list cancel connect] ]
    if {[lindex $result 0] == "cancel"} return
    set monitor_name [lindex $result 1]
    set port         [lindex $result 2]
    if {[catch {tcp connect $monitor_name $port} socket]} {
	writeln "Can not connect to $monitor_name: $socket"
	set socket ""
	return
    }

    addinput -read $socket read_socket

    menu_close

    # here we must set accept_job to 1 because we do not write anything
    # with write_socket until we have got something from monitor
    set sema 1
}

##
## menu close the connection to the monitor
##

proc "Close connection" {list} {
    global to_socket
    global socket
    
    if {$socket == ""} { 
	ined acknowledge "No connection to close"
	return
    }
    # send close_command to monitor
    set to_socket [linsert $to_socket 0 "close_connect slave"]
}

##
## close the connection to the monitor
##

proc close_connection {} { 
    global data_info_now
    global var_oid_list
    global var_request_list
    global monitor_name
    global socket
    global port
    global to_socket
    global menus

    if {[info exists var_request_list]} {
	catch {puts $socket "data_reset_tag"; flush $socket}
	catch {puts $socket "limit_delete_req all"; flush $socket}
	catch {unset var_oid_list}
	catch {unset data_info_now}
	catch {unset var_request_list}
    }
    catch {removeinput $socket}
    catch {tcp shutdown $socket}
    catch {tcp close $socket}
    set socket ""
    set to_socket ""
    writeln "Connection closed by user at [exec date]."
    job create dummy 10000
    
    menu_connect
}

##
## let the interpreter stay alive
##

proc dummy {} {
    global socket
    
    if {$socket == ""} { return }
    job_kill [job current]
}

##
## the next procedures are used to get information about the status and
## configuration of the monitor server
## e.g. connections, data request, measured variables
##

##
## proc request monitor for a list of all states of the 
## measurementagents served by monitor
## asks for the variables measured by the agents
##

proc "Info agents" {list} {
    global socket
    global to_socket
    global monitor_name
    
    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }  
    lappend to_socket "info tools"
}

##
## proc request monitor for the current states of some global variables
##

proc "Info server" {list} {
    global socket
    global to_socket
    global monitor_name
    
    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }  
    lappend to_socket "info monitor"
}

##
## proc writes the answers in a log window from tkined
##

proc show_info {answer_arg}  {
    global to_socket
    global monitor_name

    if {$answer_arg == ""} {
	writeln "server do not give result"
	return
    } else { 
	foreach elem $answer_arg {
	    writeln $elem
	}
    }
}

##
## the following proc can be used to close the connection and delete
## the menu entry in tkined
##
##

##
## Delete the menu and this interpreter
##

proc "Delete Netguard-Admin" {list} {
    global control_socket
    global menus
    global socket
    global var_request_list
    
    if {[info exists var_request_list]} {
	ined acknowledge "Please reset all current data request, before\
                          deleting the tool."
	return
    }
    # send close_command to monitor
    catch {puts $socket "close_connect slave"; flush $socket}
    catch {removeinput $socket}
    catch {tcp shutdown $socket all}
    catch {tcp close $socket}
    foreach id $menus { ined delete $id }
    exit
}

##
##
##  here starts the section with the limit procedures
##  
##
##
## give infos about limit values and alarm requests
##

proc "Limit Info" {list} {
    global to_socket
    global socket
    global monitor_name
    global limit_info_button
    global data_info_now

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists limit_info_button]} {
	set limit_info_button info
    }

    set request [ined request "Request for variables with limit control." \
		 [list [list "Information about:" $limit_info_button \
			radio info overflow] ] [list cancel ok] ]

    if {[lindex $request 0] == "cancel"} return

    set limit_info_button [lindex $request 1]

    if {$limit_info_button == "info"} {
	lappend to_socket "limit_info info"
	return
    }
    if {$limit_info_button == "overflow"} {
	lappend to_socket "limit_info all"
	return
    }
    if {$limit_info_button == "some-variables"} {
	return	
    }
}

##
## here are procedures for doing the communication with the monitorserver
##
##
## wait for answers from monitor
##
##

proc read_socket {} {
    global socket
    global monitor_name
    global port
    global to_socket 
    global sema

    if {[catch {gets $socket} answer]} {
	writeln "fatal error: can not read from monitorsocket"
	close_connection
	return
    }
    if {[eof $socket]} {
	writeln "error: read_socket connection $monitor_name closed"
	close_connection
	return
    }
    set answer [string trimright $answer "\r\n"]
    if {[llength [split $answer]] < "2"} {
	writeln "error: answer $answer format from monitor is wrong"
	set sema ""
	lappend to_socket "error tkined wrong_format"
	return
    }
    set answer_action [lindex $answer 0]
    set answer_cmd    [lindex $answer 1]
    set answer_arg    [lindex $answer 2]
    process $answer_action $answer_cmd $answer_arg $answer
    set sema ""
}

##
## write to monitorsocket
##

proc write_socket {} {
    global monitor_name
    global port
    global to_socket
    global sema
    global socket

    if {$socket == ""} { 
	job_kill [job current]
	return
    }
    if {$to_socket == ""} { 
	return
    }
    set message [lindex $to_socket 0]
    if {$sema != ""} { 
	return
    }
    set to_socket [ldelete to_socket $message]
    if {[catch {puts $socket $message; flush $socket}]} {
	writeln "Write to monitorsocket $monitor_name $port failed."
	job_kill [job current]
	close_connection
	return
    }
}

##
## proc handle incoming errormsg from server
##

proc error_handle {cmd message} {
    global old_data_sema

    case $cmd in {
	{data_old_tag} { 
	    ined acknowledge "Request of old data failed."
	    catch {unset old_data_sema} 
	    return
	}
	{close} {
	    if {$message == "no agents"} {
		ined acknowledge "No agents to close."
		return
	    } else {
		ined  acknowledge "Wrong message format."
		return
	    }
	}
	{limit_request} {
	    case [lindex [split $message] 0] in {
		{insert} {
		    writeln "Sorry, alarm request failed."
		    return
		}
		{delete} {
		    writeln "Sorry, limit control delete failed."
		    return
		}
		{set_warn} {
		    writeln "Sorry, alarm suspend failed."
		    return
		}
		{default} {
		    writeln $message
		    return
		}
	    }
	}
	{default} {
	    ined  acknowledge "Server returns error. See Netguard-Admin\
                               Report."
	    writeln $message
	}
    }
}

##
## proc request monitor for a list of some commands
##

proc show_help {answer_arg} {
    
    if {$answer_arg == ""} {
	writeln "Monitor gives no help ???"
	return
    } else {
	writeln "List of commands for interactive use"
	foreach elem $answer_arg {
	    writeln $elem
	}
    }
}

##
## proc deletes jobs created by scotty`s job command
##

proc job_kill {job_id} {
    
    if {$job_id == ""} { return }
    after 100 "job kill $job_id"
}

##
## parser for answer from monitor
##

proc process {action cmd answer_arg answer} {
    global to_socket
    global monitor_name
    global port
    global old_data_sema

    if {$action == "error"} {
	error_handle $cmd $answer_arg
        return
    }
    if {$action == "result"} {
	case $cmd in {
	    {message} {
		if {$answer_arg != ""} {
		    ined acknowledge "message from monitor"
		    writeln $answer_arg
		}
		return
	    }
	    {secret_message} {
		writeln "server writes to history: $answer_arg"
		return
	    }
	    {info} {
		show_info $answer_arg
		return
	    }
	    {help} {
		show_help $answer_arg
		return
	    }
	    {data_info_now} {
		insert_data_now $answer_arg
		return
	    }
	    {data_tag_now} {
		show_actual_data $answer_arg
		return
	    }
	    {data_info_old} {
		request_data_old $answer_arg
		return
	    }
	    {data_tag_old} {
		control_old_data $answer_arg
		return		
	    }
	    {limit_info_now limit_info_req limit_info_all} {
		show_info $answer_arg
		return 
	    } 
	    {limit_info_tag} {
		writeln $answer_arg
		return
	    }
	    {limit_request_now} {
		alarm_warning $answer_arg
		return		
	    }
	    {limit_request} {
		return
	    }
	    {default} {
		writeln "monitor returns result $cmd"
		return
	    }
	}
    }
    if {$action == "ok"} {
	case $cmd in {
	    {accepted} {
		accept_message
		return
	    }
	    {name} {
		lappend to_socket "data_info now"
		return
	    }
	    {yau} {
		show_info $answer_arg
		return
	    }
	    {close_tools} {
		writeln "All clients of server $monitor_name $port are\
                         closed." 
		return
	    }
	    {close} {
		close_connection
		return
	    }
	    {kill} {
		kill_message
		return
	    }
	    {data_request} {
		return
	    }
	    {limit_request} {
		return
	    }
	    {default} {
		writeln "monitor accepts $cmd."
		return
	    }
	}
    }
    writeln "Unknown action $answer"
}

##
##
## here starts the section with the master commands
##
##
##
##
## menu send monitor exit message
##

proc "Kill server" {list} {
    global monitor_name
    global socket
    global port
    global to_socket

    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {$port == "6341"} {
	if {[ined confirm "Really quit monitor ?" [list no yes]] != "yes"} {
	    return
	}
    }
    if {[ined confirm "Really quit monitor ?" [list no yes]] != "yes"} {
	return
    }
    set to_socket [linsert $to_socket 0 "kill_monitor"]
}

##
## react to monitor exit message
##

proc kill_message {} {
    global var_oid_list
    global var_request_list
    global monitor_name
    global socket
    global to_socket
    
    close_connection
   
    writeln "Connection to monitor closed and monitor killed."
}

proc "Kill agents" {list} {
    global to_socket
    global socket
    global monitor_name

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {[ined confirm "Really kill running agents?" [list no yes]] != "yes"} {
	return
    }
    lappend to_socket "close_connect tools"
}

##
## proc inserts an entry for controlling its overflow values
##

proc "Limit control insert" {list} {
    global to_socket
    global socket
    global monitor_name
    global limit_new_choose
    global data_info_now

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }

    if {![info exists data_info_now]} {
	writeln "server has not sent variable information"
	return
    }
    if {![info exists limit_new_choose]} {
	set host *
	set variables *
    } else { 
	set host [list [lindex $limit_new_choose 0]]
	set variables [list [lindex $limit_new_choose 1]]
    }
    set request [ined request "Choose host and variable to insert into\
                               overflow control:" \
		 [list [list "Host:" $host] \
                       [list "Variables:" $variables] ] \
		 [list cancel ok] ]
  
    if {[lindex $request 0] == "cancel"} return

    set limit_new_choose [list [lindex $request 1] [lindex $request 2]]
    set host     [lindex $limit_new_choose 0]
    set variables [lindex $limit_new_choose 1]

    foreach req_host $host {
	foreach elem $data_info_now {
	    if {![string match $req_host [lindex $elem 0]]} { continue }
	    foreach req_var $variables {
		foreach var_elem [lindex $elem 1] {
		    if {![string match $req_var [lindex $var_elem 0]]} { 
			continue 
		    }
		    lappend req_result [format "%10s %15s %3s"  \
				       [lindex $elem 0] [lindex $var_elem 0] \
				       [lindex $var_elem 1]]
		}		
	    }
	}
    }
    if {![info exists req_result]} { 
	ined acknowledge "No match for \"$host\" and \"$variables\"."
	return
    }
    set header "Select hosts and variables for inserting into overflow\
                control."
    set res 1
    set request_limit ""
    while {$res != ""} {

	set res [ined list $header $req_result [list cancel ok]]
	if {[lsearch $request_limit [lindex $res 1]] == -1} {
	    if {[lindex $res 0] != "cancel"} {
		lappend request_limit [lindex $res 1]
	    }
	}
    }
    if {$request_limit == ""} { return }
    set i 0
    set res ""
    set value1 0
    foreach elem $request_limit {
	set elem [lindex $elem 0]
	set host [lindex $elem 0]
	set var [lindex $elem 1]
	foreach limit_elem $data_info_now { 
	    if {!([lindex $limit_elem 0] == $host)} { continue }
	    foreach var_limit_elem [lindex $limit_elem 1]  {
		if {!([lindex $var_limit_elem 0] == $var)} { continue }
		set value [ined request "Enter limit control values for $host\
                            $var:" \
			   [list [list "limit       value:" $value1] \
                                 [list "hysterese   value:" $value1] \
			         [list "alarm-count value:" $value1] ] \
			   [list cancel ok] ]
		if {[lindex $value 0] == "cancel"} continue
		lappend res "[lindex $var_limit_elem 2] [lrange $value 1 end]"
	    }
	}
    }
    if {$res == ""} { return }
    lappend to_socket [list limit_set $res]
}

##
## proc deletes an entry from controlling its overflow values
##

proc "Limit control delete" {list} {
    global to_socket
    global socket
    global monitor_name
    global limit_new_choose
    global data_info_now

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists data_info_now]} {
	writeln "server has not sent variable information."
	return
    }
    if {![info exists limit_new_choose]} {
	set host *
	set variables *
    } else { 
	set host [list [lindex $limit_new_choose 0]]
	set variables [list [lindex $limit_new_choose 1]]
    }
    set request [ined request "Choose host and variable for deleting overflow\
                               request:" \
		 [list [list "Host:" $host] \
                       [list "Variables:" $variables] ] \
		 [list cancel ok] ]
    
    if {[lindex $request 0] == "cancel"} return
    
    set limit_new_choose [list [lindex $request 1] [lindex $request 2]]
    set host     [lindex $limit_new_choose 0]
    set variables [lindex $limit_new_choose 1]
    
    foreach req_host $host {
	foreach elem $data_info_now {
	    if {![string match $req_host [lindex $elem 0]]} { continue }
	    foreach req_var $variables {
		foreach var_elem [lindex $elem 1] {
		    if {![string match $req_var [lindex $var_elem 0]]} { 
			continue 
		    }
		    lappend req_result [format "%10s %15s %3s"  \
				       [lindex $elem 0] [lindex $var_elem 0] \
                                       [lindex $var_elem 1]]
		}		
	    }
	}
    }
    if {![info exists req_result]} { 
	ined acknowledge "No match for \"$host\" and \"$variables\"."
	return
    }
    set header "Select hosts and variables for deleting overflow request." 
    set res 1
    set request_limit ""
    while {$res != ""} {
	set res [ined list $header $req_result [list cancel ok]]
	if {[lsearch $request_limit [lindex $res 1]] == -1} {
	    if {[lindex $res 0] != "cancel"} {
		lappend request_limit [lindex $res 1]
	    }
	}
    }
    if {$request_limit == ""} { return }
    set i 0
    set var_ids ""
    foreach elem $request_limit {
	set elem [lindex $elem 0]
	set host [lindex $elem 0]
	set var [lindex $elem 1]
	foreach limit_elem $data_info_now { 
	    if {!([lindex $limit_elem 0] == $host)} { continue }
	    foreach var_limit_elem [lindex $limit_elem 1]  {
		if {!([lindex $var_limit_elem 0] == $var)} { continue }
		lappend var_ids [lindex $var_limit_elem 2]
	    }
	}
    }
    lappend to_socket [list limit_delete_tag $var_ids]
}

##
## proc deletes all entries from controlling its overflow values
##

proc "Limit control reset" {list} {
    global to_socket
    global socket
    global monitor_name
    global limit_new_choose
    global data_info_now
    
    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists data_info_now]} {
	writeln "server has not sent variable information."
	return
    }
    lappend to_socket [list limit_delete_tag all]  
}


##
## proc write a list with all jobs 
##

##
## Display the jobs currently running.
##

proc "job info" {list} {

    set jobs [job list]

    if {$jobs == ""} {
	ined acknowledge "Sorry, no jobs available."
	return
    }
    
    set result ""
    set len 0
    foreach j $jobs {

	set jobid   [lindex $j 0]
	set jobcmd  [lindex $j 1]
	set jobitv  [expr {[lindex $j 2] / 1000.0}]
	set jobrem  [expr {[lindex $j 3] / 1000.0}]
	set jobcnt  [lindex $j 4]
	set jobstat [lindex $j 5]

	# Convert the id's to hostnames for readability.
	
	set hosts ""
	foreach id [lindex $jobcmd 1] {
	    lappend hosts [lindex [ined name $id] 0]
	}
	if {[llength $hosts] > 3} {
	    set hosts "[lrange $hosts 0 2] ..."
	}

	set line \
	     [format "%s %6.1f %6.1f %3d %8s %s %s" \
	      $jobid $jobitv $jobrem $jobcnt $jobstat \
	      [lindex $jobcmd 0] $hosts ]

	if {[string length $line] > $len} {
	    set len [string length $line]
	}

	lappend result $line
    }

    set header " ID    INTV    REM CNT  STATUS    CMD"

    for {set i [string length $header]} {$i < $len} {incr i} {
	append header " "
    }

    ined browse $header $result
}

##
## list the list to_socket
##

proc "Request queue info" {list} {
    global to_socket

    if {$to_socket != ""} {
	writeln "Requests to send to monitordaemon:"
	foreach elem $to_socket {
	    writeln $elem
	}
    } else {
	writeln "Sorry, no request in list."
    } 
}

##
## Display some help about this tool.
##

proc "Help Netguard-Admin" {list} {
    ined browse "Help about Netguard-Admin" { 
	"The Netguard-Admin is a tkined-front-end for a Monitoring-Server" 
	"which collects current data of monitoring agents and stores the data" 
        "in files. This client retrieves data from the server and displays it" 
	"in stripcharts." 
	"" 
	"Info agents:" 
	"    Ask Monitor-Server about the state of its agents" 
	"    and list the variables measured by the agents." 
	"" 
	"Info server:" 
	"    Ask Monitor-Server about its state and active connections." 
	"" 
	"Limit Info:" 
	"    Ask Monitor-Server about which variables of hosts are under"  
	"    Limit-control. Show the actual state of each controlled" 
	"    variable." 
	"" 
	"Limit control insert:" 
	"    Insert or change existing limit-values in the limit control list" 
	"    of the Monitor-Server." 
	"" 
	"Limit control delete:" 
	"    Delete existing limit-values from limit control list  " 
	"    of the Monitor-Server." 
	"" 
	"Limit control reset:" 
	"    Reset (Delete) the whole limit control list." 
    }
}

##
## proc creates a menu with the posibility to close a connertion
##

proc menu_close {} {
    global menus
    
    foreach id $menus { ined delete $id }

    set menus [ ined create MENU "Netguard-Admin" \
	       "Close connection" "" \
		   "Kill agents" "Kill server" "" \
		   "Info agents" "Info server" "" \
		   "Limit Info" \
		   "Limit control insert" "Limit control delete" \
		   "Limit control reset" "" \
		   "job info" "" \
		   "Help Netguard-Admin" \
		   "Delete Netguard-Admin"]
}

##
## proc creates a menu with the posibility to open a connertion
##

proc menu_connect {} {
    global menus

    foreach id $menus { ined delete $id }

    set menus [ ined create MENU "Netguard-Admin" \
	       "Try connect" "" \
		   "Kill agents" "Kill server" "" \
		   "Info agents" "Info server" "" \
		   "Limit Info" \
		   "Limit control insert" "Limit control delete" \
		   "Limit control reset" "" \
		   "job info" "" \
		   "Help Netguard-Admin" \
		   "Delete Netguard-Admin"] 
}

menu_connect

auto_connect
