/*
 * ined.c
 *
 * Extend a tcl command interpreter with an ined command. See the
 * documentation of tkined for more info about the ined command.
 *
 * Copyright (c) 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <errno.h>

#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"
#include "xread.h"

#define BUF_INCR 1024

/*
 * The list that is used to queue received messages.
 */

struct q_elem {
    char   *msg;
    struct q_elem *next;
};
typedef struct q_elem q_elem;

static q_elem *queue = NULL;

/*
 * Forward declarations for procedures defined later in this file:
 */

static int 
string_to_type		_ANSI_ARGS_((char *str));

static void 
ined_fatal		_ANSI_ARGS_((void));

static void
ined_queue		_ANSI_ARGS_((void));

static void 
ined_append		_ANSI_ARGS_((char *msg));

static char*
ined_gets		_ANSI_ARGS_((void));

static int 
ined_comp_cmd		_ANSI_ARGS_((char *cmd, Tcl_Interp *interp, 
				     int argc, char **argv));
static void
inedReceive		_ANSI_ARGS_((ClientData clientData, int mask));

/*
 * The following definitions and the string_to_type() function
 * are taken from the tkined source. These may need some update
 * if the tkined supports new object types.
 */

#define TKINED_NONE         0
#define TKINED_ALL          1
#define TKINED_NODE         2
#define TKINED_GROUP        3
#define TKINED_NETWORK      4
#define TKINED_LINK         5
#define TKINED_TEXT         6
#define TKINED_IMAGE        7
#define TKINED_INTERPRETER  8
#define TKINED_MENU         9
#define TKINED_LOG         10
#define TKINED_REFERENCE   11
#define TKINED_STRIPCHART  12
#define TKINED_BARCHART    13
#define TKINED_GRAPH	   14

static int 
string_to_type (str)
    char *str;
{
    int type = TKINED_NONE;

    if (str != NULL) {
	if      (strcmp(str, "NODE") == 0)        type = TKINED_NODE;
	else if (strcmp(str, "GROUP") == 0)       type = TKINED_GROUP;
	else if (strcmp(str, "NETWORK") == 0)     type = TKINED_NETWORK;
	else if (strcmp(str, "LINK") == 0)        type = TKINED_LINK;
	else if (strcmp(str, "TEXT") == 0)        type = TKINED_TEXT;
	else if (strcmp(str, "IMAGE") == 0)       type = TKINED_IMAGE;
	else if (strcmp(str, "INTERPRETER") == 0) type = TKINED_INTERPRETER;
	else if (strcmp(str, "MENU") == 0)        type = TKINED_MENU;
	else if (strcmp(str, "LOG") == 0)         type = TKINED_LOG;
	else if (strcmp(str, "REFERENCE") == 0)   type = TKINED_REFERENCE;
	else if (strcmp(str, "STRIPCHART") == 0)  type = TKINED_STRIPCHART;
	else if (strcmp(str, "BARCHART") == 0)    type = TKINED_BARCHART;
	else if (strcmp(str, "GRAPH") == 0)       type = TKINED_GRAPH;
   }

    return type;
}

/*
 * Handle an error when talking to the tkined editor.
 */

static void
ined_fatal()
{
    fputs ("lost connection -- shuting down interpreter\n", stderr);
    exit (1);
}

/*
 * Write a queue message to the tkined editor. No acknowledge is
 * expected. a queue message is used to let tkined know how busy
 * we are.
 */

static void
ined_queue ()
{
    int len = 0;
    q_elem *p;
    char buf[256];

    for (p = queue; p != NULL; p = p->next) len++;

    sprintf (buf, "ined queue %d\n", len);
    
    len = strlen (buf);
    if (xwrite (fileno(stdout), buf, len) < 0) {
	fprintf (stderr, "ignoring ined queue message\n");
    }
}

/*
 * Process the commands stored in the queue.
 */

void
inedFlush (interp)
     Tcl_Interp *interp;
{
    q_elem *p;

    if (queue == NULL) return;

    ined_queue ();
    while ((p = queue) != NULL) {
	queue = queue->next;

	if (Tcl_GlobalEval (interp, p->msg) != TCL_OK) {
	    Tcl_DString ds;
	    Tcl_DStringInit (&ds);
	    Tcl_DStringAppendElement (&ds, "scottyerror");
	    Tcl_DStringAppendElement (&ds, interp->result);
	    Tcl_GlobalEval (interp, Tcl_DStringValue (&ds));
	    Tcl_DStringFree (&ds);
	}

	free (p->msg);
	free ((char *) p);
    }

    ined_queue ();
}

/*
 * Append a message to the queue.
 */

static void
ined_append (msg)
    char *msg;
{
    q_elem *np;
    q_elem *p;

    if (msg == (char *) NULL) return;

    np = (q_elem *) xmalloc (sizeof(q_elem));
    np->msg = msg;
    np->next = (q_elem *) NULL;

    if (queue == (q_elem *) NULL) {
        queue = np;
        return;
    }

    for (p = queue; (p->next != (q_elem *) NULL); p = p->next) ;
    p->next = np;

    ined_queue ();
}

/*
 * Read a message from the tkined editor. Returns the complete message
 * in a malloced buffer. The caller must free this buffer.
 * because chunks are read, anything behind a newline is stored and 
 * returned with the next call, only this routine must read from stdin.
 */

static char*
ined_gets ()
{
    char *p;
    char ch = '\0';
    int buffer_size = BUF_INCR;
    char *buffer = xmalloc (buffer_size);
#define IN_SIZ	256
    static char in_buf [IN_SIZ];
    static char *in_ptr = in_buf;
    static int in_len = 0;

    buffer[0] = ch;
    p = buffer;

    for (;;) {
	  if (in_len <= 0) {
	      /* read next chunk: */
	      in_len = xread (fileno(stdin), in_ptr = in_buf, IN_SIZ);
	      if (in_len <= 0) {
		  free (buffer);
		  return (char *) NULL;
	      }
	  }

	  while (in_len-- > 0 && (ch = *in_ptr++) != '\n') {
	      *p = ch;
	      p++;
	      if (p-buffer == buffer_size) {
		  int i = p-buffer;
		  buffer_size += BUF_INCR;
		  buffer = xrealloc (buffer, buffer_size);
		  p = buffer + i;
	      }
	  }
	  if (ch == '\n') {
	      *p = '\0';
	      break;
	  }
    }
    
    if (ch != '\n') {
        free (buffer);
        return (char *) NULL;
    }

    return buffer;
}

/*
 * Check if we can evaluate cmd on the external object 
 * representation given in largv.
 */

static int
ined_comp_cmd (cmd, interp, argc, argv)
    char *cmd;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int type = string_to_type (argv[0]);
    if ((type == TKINED_NONE) || (type == TKINED_ALL)) return TCL_ERROR;

    if ((strcmp (cmd, "type") == 0) && (argc > 0)) {
	Tcl_SetResult (interp, argv[0], TCL_VOLATILE);
	return TCL_OK;

    } else if ((strcmp (cmd, "id") == 0) && (argc > 1)) {
	Tcl_SetResult (interp, argv[1], TCL_VOLATILE);
	return TCL_OK;

    } else if ((strcmp (cmd, "name") == 0) && (argc > 2)) {
        if ((type == TKINED_NODE) || (type == TKINED_NETWORK)
	    || (type == TKINED_BARCHART) || (type == TKINED_STRIPCHART)
	    || (type == TKINED_GROUP) || (type == TKINED_REFERENCE)
	    || (type == TKINED_MENU) || (type == TKINED_LOG) 
	    || (type == TKINED_GRAPH))
	    Tcl_SetResult (interp, argv[2], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp (cmd, "address") == 0) && (argc > 3)) {
        if ((type == TKINED_NODE) || (type == TKINED_NETWORK)
	    || (type == TKINED_BARCHART) || (type == TKINED_STRIPCHART)
	    || (type == TKINED_REFERENCE) || (type == TKINED_GRAPH))
	    Tcl_SetResult (interp, argv[3], TCL_VOLATILE);
        return TCL_OK;

    } else if (strcmp (cmd, "oid") == 0) {
        if ((type == TKINED_GROUP) && (argc > 3)) {
	    Tcl_SetResult (interp, argv[3], TCL_VOLATILE);
	}
        if ((type == TKINED_NODE || type == TKINED_NETWORK) && (argc > 4)) {
	    Tcl_SetResult (interp, argv[4], TCL_VOLATILE);
	}
	return TCL_OK;

    } else if ((strcmp (cmd, "links") == 0) && (argc > 5)) {
        if ((type == TKINED_NODE) || (type == TKINED_NETWORK))
	    Tcl_SetResult (interp, argv[5], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp (cmd, "member") == 0) && (argc > 4)) {
        if (type == TKINED_GROUP)
	    Tcl_SetResult (interp, argv[4], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp (cmd, "src") == 0) && (argc > 2)) {
        if (type == TKINED_LINK)
	    Tcl_SetResult (interp, argv[2], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp (cmd, "dst") == 0) && (argc > 3)) {
        if (type == TKINED_LINK)
	    Tcl_SetResult (interp, argv[3], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp (cmd, "text") == 0) && (argc > 2)) {
        if (type == TKINED_LINK)
	    Tcl_SetResult (interp, argv[2], TCL_VOLATILE);
        return TCL_OK;

    }

    return TCL_ERROR;
}

/*
 * This is the ined command as described in the tkined documentation.
 * It just sends the command to the tkined editor and waits for an
 * acknowledge containing the answer or the error description.
 *
 * Everything received while waiting for the acknowledge is queued
 * for later execution.
 */

int
inedCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int i;
    char *p;
    static int initialized = 0;

    if (! initialized) {

	initialized = 1;

	/*
	 * Make stdin unbuffered and register a file handler to receive
	 * commands from the tkined editor. Make sure that an already
	 * existing file handler is removed first (make wish happy).
	 */

	setbuf (stdin, (char *) 0);
	Tk_DeleteFileHandler (0);
	Tk_CreateFileHandler (0, TK_READABLE, inedReceive, interp);
	inedFlush (interp);

	/* 
	 * Adjust the auto_path to take care of $HOME/.tkined 
	 * and TKINED_PATH.
	 */

	Tcl_Eval (interp, 
		  "set auto_path \"/usr/local/lib/tkined [info library]\"");
	Tcl_VarEval (interp, "set auto_path \"", TKINEDLIB, " $auto_path\"",
		     (char *) NULL);
	Tcl_VarEval (interp, "set auto_path \"", TKINEDLIB, 
		     "/apps $auto_path\"", (char *) NULL);
	Tcl_VarEval (interp, "if [info exists env(HOME)] { ",
		     "set auto_path \"$env(HOME)/.tkined $auto_path\"}",
		     (char *) NULL);
	Tcl_VarEval (interp, "if [info exists env(TKINED_PATH)] {",
		     "set auto_path \"$env(TKINED_PATH) $auto_path\"}",
		     (char *) NULL);
    }

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " command ?arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /* the loop command is used by programs that do not wish */
    /* to use their own main or event loop mechanism */

    if ((argc == 2) && (strcmp (argv[1], "loop") == 0)) {
	char *line;
	inedFlush (interp);
	while ((line = ined_gets()) != (char *) NULL) {
	    ined_append (line);
	    inedFlush (interp);
	}
	return TCL_OK;
    }

    /* check for commands that act on external object representations */

    if (argc == 3) {
        int largc;
	char **largv;
        int rc = Tcl_SplitList (interp, argv[2], &largc, &largv);

	if (rc == TCL_OK) {
	    if (ined_comp_cmd (argv[1], interp, largc, largv) == TCL_OK) {
		free (largv);
		return TCL_OK;
	    }
 	    free (largv);
	}
    }

    for (i = 0; i < argc; i++) {
	if (fputc ('{', stdout) == EOF) ined_fatal ();
        for (p = argv[i]; *p; p++) {
	    if (*p != '\n') {
	        if (fputc (*p, stdout) == EOF) ined_fatal ();
	    } else {
	        if (fputs ("\\n", stdout) == EOF) ined_fatal ();
	    }
	}
        if (fputs ("} ", stdout) == EOF) ined_fatal ();
    }
    if (fputc ('\n', stdout) == EOF) ined_fatal ();
    fflush (stdout);

    while ((p = ined_gets ()) != (char *) NULL) {
        if (*p == '\0') continue;
	if (strncmp (p, "ined ok", 7) == 0) {
	    char *r = p+7;
	    while (*r && isspace(*r)) r++;
	    Tcl_SetResult(interp, r, TCL_VOLATILE);
	    free (p);
	    return TCL_OK;
	} else if (strncmp (p, "ined error", 10) == 0) {
	    char *r = p+10;
	    while (*r && isspace(*r)) r++;
	    Tcl_SetResult(interp, r, TCL_VOLATILE);
	    free (p);
	    return TCL_ERROR;
	} else {
	    ined_append (p);
	}
    }

    ined_fatal();

    return TCL_ERROR;
}

/*
 * inedReceive() is the called from the event handler whenever
 * a command can be read from stdin.
 */

static void
inedReceive (clientData, mask)
    ClientData clientData;
    int mask;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    char *cmd = ined_gets();

    if (cmd != NULL) {
        ined_append (cmd);
        inedFlush (interp);
    } else {
	Tk_DeleteFileHandler (0);
    }
}
