/*
 * event.c
 *
 * This is the implementation of a simple event loop. It supports three
 * wellkown tk commands, namely after, addinput removeinput. The 
 * implementation is totally different from tk. We use the tclXselect
 * command already available for scotty. This might be a bit slower but
 * makes this module easier to implement.
 *
 * The command syntax and this implementation is inspired by the addinput
 * tk extension written by Mark Diekhans (markd@grizzly.com).
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <sys/time.h>
#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"

typedef struct TimerEvent {
    double time;                /* The time value in seconds. */
    Tcl_Interp *interp;         /* The interpreter used to eval the */
    char* cmd;                  /* command itself. */
    Tk_TimerProc *proc;         /* The function to call. */
    ClientData clientData;      /* The client data for the call. */
    int id;                     /* The Timer Token. */

    struct TimerEvent *nextPtr; /* Next event in queue, or NULL for
                                 * end of queue. */
} TimerEvent;

static TimerEvent *timerEventList = NULL;

typedef struct FileEvent {
    char *fileId;               /* The TCL file id. */
    Tcl_Interp *interp;         /* The interpreter used to eval the */
    char *cmd;                  /* command itself. */
    int mask;                   /* The type of the handler, either
				   TK_READABLE, TK_WRITABLE or 
				   TK_EXCEPTION */
    Tk_FileProc *proc;          /* The function to call. */
    ClientData clientData;      /* The client data for the call. */

    struct FileEvent *nextPtr;  /* Next in list of all files we
                                 * care about (NULL for end of list). */
} FileEvent;

static FileEvent *fileEventList = NULL;

/*
 * Forward declarations for procedures defined later in this file:
 */

static double
getTime		_ANSI_ARGS_((void));

void
bgerror		_ANSI_ARGS_((Tcl_Interp *interp));

static void
doOneTimerEvent	_ANSI_ARGS_((Tcl_Interp *interp));

static void
DoFileEvent	_ANSI_ARGS_((Tcl_Interp *interp, char *fileId));

static void
doOneFileEvent	_ANSI_ARGS_((Tcl_Interp *interp));


/*
 * Get the current time as a double in seconds.
 */

static double
getTime()
{
    struct timeval curTime;
    double time;

    (void) gettimeofday(&curTime, (struct timezone *) NULL);
    time = curTime.tv_sec + curTime.tv_usec / 1000000.0;
	
    return time;
}

/*
 * Process an error that occured while evaluating a command that 
 * is bound to an event loop. We call the procedure scottyerror 
 * in global context with an argument containing the error message 
 * that is stored in interp.
 */

void
bgerror (interp)
    Tcl_Interp* interp;
{
    Tcl_DString ds;
    Tcl_DStringInit (&ds);
    Tcl_DStringAppendElement (&ds, "scottyerror");
    Tcl_DStringAppendElement (&ds, interp->result);
    Tcl_GlobalEval (interp, Tcl_DStringValue (&ds));
    Tcl_DStringFree (&ds);
}

/*
 * Implementation of the after command. Just add a new TimerEvent
 * to the event list. The event loop will eval the command stored 
 * in the TimerEvent using the given interpreter if the time
 * has elapsed. No care is taken if the interpreter does not exist
 * anymore. No problem for scotty.
 */

int
afterCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
    int secs;
    double time;
    TimerEvent *te;
    TimerEvent *p;
    TimerEvent *q;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " milliseconds ?command? ?arg arg ...?\"",
			 (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_GetInt(interp, argv[1], &secs) != TCL_OK) {
        return TCL_ERROR;
    }

    time = (secs < 0) ? 0 : secs / 1000.0;

    if (argc == 2) {
	secs = (int) (time + 0.5);
	sleep (secs);
	return TCL_OK;
    }

    time += getTime();

    te = (TimerEvent *) xmalloc(sizeof(TimerEvent));
    te->time = time;
    te->interp = interp;
    te->cmd = Tcl_Concat(argc-2, argv+2);
    te->proc = NULL;
    te->clientData = NULL;
    te->id = 0;
    
    for (p = timerEventList, q = NULL; p != NULL; q = p, p = p->nextPtr) {
	if (p->time > time) break;
    }

    if (q == NULL) {
	te->nextPtr = timerEventList;
	timerEventList = te;
    } else {
	te->nextPtr = p;
	q->nextPtr = te;	
    }
    
    return TCL_OK;
}

/*
 * Create a new FileEvent and put it on the fileEventList.
 * Make sure that the given file is not already on the list.
 */

int
addInputCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    FileEvent* fe;
    FILE *filePtr;
    int permissions;
    int i, mask = 0;
    int found = 0;

    if ((argc < 3) || (argc > 6)) {
	goto invalidArgs;
    }

    for (i = 1; (i < argc) && (argv[i][0] == '-'); i++) {
	if (strcmp (argv[i], "-read") == 0) {
	    if (mask & TK_READABLE)
	      goto invalidArgs;
	    mask |= TK_READABLE;
	    continue;
	}
	if (strcmp (argv[i], "-write") == 0) {
	    if (mask & TK_WRITABLE)
	      goto invalidArgs;
	    mask |= TK_WRITABLE;
	    continue;
	}
        if (strcmp (argv[i], "-exception") == 0) {
	    if (mask & TK_EXCEPTION)
	      goto invalidArgs;
	    mask |= TK_EXCEPTION;
	    continue;
	}
 	Tcl_AppendResult(interp, "invalid option \"", argv[i],
		 "\" expected one of \"-read\", \"-write\", or \"-exception\"",
                 (char *) NULL);
 	return TCL_ERROR;
	
    }

    if (mask == 0) mask = TK_READABLE;
    
    if (argc - i != 2) {
	goto invalidArgs;
    }
    
    if (Tcl_GetOpenFile(interp, argv[i], 0, 0, &filePtr) != TCL_OK) {
 	return TCL_ERROR;
    }
    permissions = Tcl_FilePermissions (filePtr);
 
    if ((mask & TK_READABLE) && ((permissions & TCL_FILE_READABLE) == 0)) {
 	Tcl_AppendResult(interp, "\"", argv[i],
			 "\" wasn't opened for reading", (char *) NULL);
 	return TCL_ERROR;
    }
    if ((mask & TK_WRITABLE) && ((permissions & TCL_FILE_WRITABLE) == 0)) {
 	Tcl_AppendResult(interp, "\"", argv[i],
			 "\" wasn't opened for writing", (char *) NULL);
 	return TCL_ERROR;
    }
    
    for (fe = fileEventList; fe != NULL; fe = fe->nextPtr) {
	if (strcmp(argv[i], fe->fileId) == 0) {
	    found++;
	    break;
	}
    }

    if (found) {
	free (fe->cmd);
	fe->cmd = Tcl_Concat(argc-i-1, argv+i+1);
    } else {
	fe = (FileEvent *) xmalloc(sizeof(FileEvent));
	if (strcmp (argv[i], "stdin") == 0) {
	    fe->fileId = xstrdup("file0");
	} else if (strcmp (argv[i], "stdout") == 0) {
            fe->fileId = xstrdup("file1");
        } else if (strcmp (argv[i], "stderr") == 0) {
            fe->fileId = xstrdup("file2");
	} else {
	    fe->fileId = xstrdup(argv[i]);
	}
	fe->proc = NULL;
	fe->clientData = NULL;
	fe->interp = interp;
	fe->cmd = Tcl_Concat(argc-i-1, argv+i+1);
	fe->mask = mask;
	fe->nextPtr = fileEventList;
	fileEventList = fe;
    }
    
    return TCL_OK;
    
 invalidArgs:
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ?-read? ?-write? ?-exception? fileid cmd\"",
		     (char *) NULL);
    return TCL_ERROR;     
}

/*
 * Remove a FileEvent for the given file id from our fileEventList.
 */

int
remInputCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    FILE *filePtr;
    int found = 0;
    FileEvent *p;
    FileEvent *q;
    char *fixedFileId;
 
    if (argc != 2) {
 	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], " fileid\"", (char *) NULL);
 	return TCL_ERROR;
    }
    
    if (Tcl_GetOpenFile (interp, argv[1], 0, 0, &filePtr) != TCL_OK) {
 	return TCL_ERROR;
    }

#if 0
{
    int i;
    fputs ("removeinput ", stderr);
    for (i = 0; i < argc; i++) {
	fputs (argv[i], stderr);
    }
    fputs ("\n", stderr);
}
#endif

    if (strcmp (argv[1], "stdin") == 0) {
	fixedFileId = "file0";
    } else if (strcmp (argv[1], "stdout") == 0) {
	fixedFileId = "file1";
    } else if (strcmp (argv[1], "stderr") == 0) {
	fixedFileId = "file2";
    } else {
	fixedFileId = argv[1];
    }

    for (q = NULL, p = fileEventList; 
	 p != NULL; q = p, p = p->nextPtr) {
	if (strcmp(fixedFileId, p->fileId) == 0) {
	    found++;
	    break;
	}
    }

    if (!found) {
 	Tcl_AppendResult (interp, "no callback associated with \"", argv[1],
			  "\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (q == NULL) {
	fileEventList = p->nextPtr;
    } else {
	q->nextPtr = p->nextPtr;
    }

    free (p->cmd);
    free (p->fileId);
    free (p);

    return TCL_OK;
}

/*
 * Do one Timer Event. Return if there is nothing to do now.
 */

static void 
doOneTimerEvent (interp)
     Tcl_Interp *interp;
{
    double time;
    TimerEvent *p;
    int id;

    if (timerEventList == NULL) return;
	    
    time = getTime();
    if (timerEventList->time < time) {
	p = timerEventList;
	id = p->id;

	/*
	 * Handle file events for C functions here.
	 */
	
	if (p->proc != NULL) {
	    (*p->proc)(p->clientData);
	} else {
	    if (Tcl_GlobalEval (p->interp, p->cmd) != TCL_OK) 
		    bgerror (p->interp);
	    Tcl_ResetResult (p->interp);
	}

	Tk_DeleteTimerHandler ((Tk_TimerToken) id);
    }
}

/*
 * Do the command associated with the file given by fileId.
 */

static void
DoFileEvent (interp, fileId)
    Tcl_Interp *interp;
    char *fileId;
{
    FileEvent *fe;
    FILE *filePtr;
    Tcl_DString command;
    char *startPtr, *scanPtr;
    Tcl_Interp *finterp;

    for (fe = fileEventList; fe != NULL; fe = fe->nextPtr) {
	if (strcmp (fileId, fe->fileId) == 0) break;
    }

    if (fe == NULL) return;

    /*
     * Handle file events for C functions here.
     */

    if (fe->proc != NULL) {
	(*fe->proc)(fe->clientData, fe->mask);
	return;
    }

    /*
     * Check that the file is still valid.
     */

    if (Tcl_GetOpenFile(interp, fileId, 0, 0, &filePtr) != TCL_OK) {
	bgerror (interp);
 	return;
    }

    /*
     * Do substitution on command to evaluate.
     */

    Tcl_DStringInit (&command);
    startPtr = fe->cmd;
    for (scanPtr = startPtr; *scanPtr != '\0'; scanPtr++) {
	if (*scanPtr != '%')
		continue;
	Tcl_DStringAppend (&command, startPtr, scanPtr - startPtr);
	scanPtr++;
	startPtr = scanPtr + 1;
	switch (*scanPtr) {
    case 'F':
	    Tcl_DStringAppend (&command, fileId, -1);
	    break;
    case 'E':
	    Tcl_DStringAppend (&command, "{", -1);
	    if (fe->mask & TK_READABLE)
		    Tcl_DStringAppend (&command, "READ ", -1);
	    if (fe->mask & TK_WRITABLE)
		    Tcl_DStringAppend (&command, "WRITE ", -1);
	    if (fe->mask & TK_EXCEPTION)
		    Tcl_DStringAppend (&command, "EXCEPTION ", -1);
	    command.string [command.length - 1] = '}';
	    break;
    case '%':
	    Tcl_DStringAppend (&command, "%", -1);
	    break;
    default:
	    Tcl_AppendResult (interp,
			      "invalid character following % in ",
			      " command \"", fe->cmd, "\"",
			      (char *) NULL);
	    Tcl_DStringFree (&command);
	    return;
	}
    }
    Tcl_DStringAppend (&command, startPtr, scanPtr - startPtr);

    finterp = fe->interp;
    if (Tcl_GlobalEval (finterp, Tcl_DStringValue(&command)) != TCL_OK) {
	bgerror (finterp);
    }

    Tcl_DStringFree (&command);
}

/*
 * Do one file event. Make a select call and wait if one of
 * the file handles needs attention. This is not very efficient,
 * since we put together a string, call the select cmd, decompose
 * the result and finally call DoFileEvent to evaluate the command
 * belonging to this event.
 */

static void 
doOneFileEvent (interp)
     Tcl_Interp *interp;
{

    Tcl_DString selectArgs;
    FileEvent *e;
    int largc;
    char **largv;
    int res;

    Tcl_DStringInit (&selectArgs);
    Tcl_DStringAppend (&selectArgs, "select ", 7);

    Tcl_DStringStartSublist (&selectArgs);
    for (e = fileEventList; e != NULL; e = e->nextPtr) {
	if (e->mask & TK_READABLE) 
		Tcl_DStringAppendElement (&selectArgs, e->fileId);
    }
    Tcl_DStringEndSublist (&selectArgs);

    Tcl_DStringStartSublist (&selectArgs);
    for (e = fileEventList; e != NULL; e = e->nextPtr) {
	if (e->mask & TK_WRITABLE) 
		Tcl_DStringAppendElement (&selectArgs, e->fileId);
    }
    Tcl_DStringEndSublist (&selectArgs);

    Tcl_DStringStartSublist (&selectArgs);
    for (e = fileEventList; e != NULL; e = e->nextPtr) {
	if (e->mask & TK_EXCEPTION) 
		Tcl_DStringAppendElement (&selectArgs, e->fileId);
    }
    Tcl_DStringEndSublist (&selectArgs);

    if (timerEventList != NULL) {
	double time;
	char buffer[20];
	time = timerEventList->time - getTime();
	if (time < 0) time = 0;
	sprintf (buffer, " %f", time);
	Tcl_DStringAppend (&selectArgs, buffer, -1);
    }

    Tcl_SplitList (interp, Tcl_DStringValue (&selectArgs), &largc, &largv);

    Tcl_DStringFree (&selectArgs);

    Tcl_ResetResult (interp);
    res = Tcl_SelectCmd (NULL, interp, largc, largv);

    if (res == TCL_OK) {
	char *tmp;
	int i;
	free (largv);

	tmp = xstrdup(interp->result);
	Tcl_SplitList (interp, tmp, &largc, &largv);
	free (tmp);

	tmp = Tcl_Concat (largc, largv);
	free (largv);

	Tcl_SplitList (interp, tmp, &largc, &largv);
	free (tmp);

	for (i = 0; i < largc; i++) {
	    DoFileEvent (interp, largv[i]);
	    Tcl_ResetResult (interp);
	}
    }

    Tcl_ResetResult (interp);

    free (largv);
}

#if 0
static void
dump_events()
{
    FileEvent *e;
    TimerEvent *t;

    fputs ("\n", stderr);
    for (e = fileEventList; e != NULL; e = e->nextPtr) {
	fprintf (stderr, "** FileEvent %s cmd %s\n", e->fileId, e->cmd);
    }

    if (timerEventList == NULL) return;
	    
    for (t = timerEventList; t != NULL; t = t->nextPtr) {
	fprintf (stderr, "** TimerEvent %f cmd %s\n", t->time, t->cmd);
    }
}
#endif

void
eventLoop (interp)
     Tcl_Interp *interp;
{

    while (1) {

#if 0
	dump_events();
#endif
	if ((fileEventList == NULL) && (timerEventList == NULL)) break;
	doOneTimerEvent (interp);
	    

        /* process all queued ined commands before waiting
           for the next event */

        inedFlush (interp);

	if ((fileEventList == NULL) && (timerEventList == NULL)) break;
	doOneFileEvent (interp);
    }
}

/*
 * An somewhat ugly interface to the TK interface for the event handler.
 * This will change when TK 4.0 comes out.
 */

extern Tcl_Interp *sc_interp;

Tk_TimerToken
Tk_CreateTimerHandler (milliseconds, proc, clientData)
    int milliseconds;
    Tk_TimerProc *proc;
    ClientData clientData;
{
    Tcl_Interp *interp = sc_interp;
    double time = milliseconds / 1000.0;
    TimerEvent *te, *p, *q;
    static int lastid = 1;

    time += getTime();

    te = (TimerEvent *) xmalloc(sizeof(TimerEvent));
    te->time = time;
    te->interp = interp;
    te->cmd = xstrdup("");
    te->proc = proc;
    te->clientData = clientData;
    te->id = lastid++;
    
    for (p = timerEventList, q = NULL; p != NULL; q = p, p = p->nextPtr) {
	if (p->time > time) break;
    }
    
    if (q == NULL) {
	te->nextPtr = timerEventList;
	timerEventList = te;
    } else {
	te->nextPtr = p;
	q->nextPtr = te;	
    }

    return (Tk_TimerToken) te->id;
}

void
Tk_DeleteTimerHandler (token)
    Tk_TimerToken token;
{
    TimerEvent *p, *q;
    int found = 0;

    for (q = NULL, p = timerEventList; 
	 p != NULL; q = p, p = p->nextPtr) {

	if (p->id == (int) token) {
	    found++;
	    break;
	}
    }

    if (found) {

	if (q == NULL) {
	    timerEventList = p->nextPtr;
	} else {
	    q->nextPtr = p->nextPtr;
	}
	
	free (p->cmd);
	free (p);
    }
}

void
Tk_CreateFileHandler (fd, mask, proc, clientData)
    int fd;
    int mask;
    Tk_FileProc *proc;
    ClientData clientData;
{
    Tcl_Interp *interp = sc_interp;
    FILE *f = fdopen (fd, "r+");
    FileEvent *fe;

    Tcl_EnterFile (interp, f, TCL_FILE_READABLE | TCL_FILE_WRITABLE);

    fe = (FileEvent *) xmalloc(sizeof(FileEvent));
    if (strcmp (interp->result, "stdin") == 0) {
	fe->fileId = xstrdup("file0");
    } else if (strcmp (interp->result, "stdout") == 0) {
	fe->fileId = xstrdup("file1");
    } else if (strcmp (interp->result, "stderr") == 0) {
	fe->fileId = xstrdup("file2");
    } else {
	fe->fileId = xstrdup(interp->result);
    }
    fe->proc = proc;
    fe->clientData = clientData;
    fe->interp = interp;
    fe->cmd = xstrdup("");
    fe->mask = mask;
    fe->nextPtr = fileEventList;
    fileEventList = fe;

    Tcl_ResetResult (interp);
}

void
Tk_DeleteFileHandler (fd)
    int fd;
{
    Tcl_Interp *interp = sc_interp;
    FileEvent *p, *q;
    FILE *filePtr;
    int found = 0;

    for (q = NULL, p = fileEventList; 
	 p != NULL; q = p, p = p->nextPtr) {
	
	if (Tcl_GetOpenFile (interp, p->fileId, 0, 0, &filePtr) != TCL_OK) 
		continue;	
	if (fileno(filePtr) == fd) {
	    found++;
	    break;
	}
    }

    if (found) {

/*	fclose (filePtr); */

	if (q == NULL) {
	    fileEventList = p->nextPtr;
	} else {
	    q->nextPtr = p->nextPtr;
	}
	
	free (p->cmd);
	free (p->fileId);
	free (p);
    }
}
