/*
 * clock.c
 *
 * This file includes the getclock, getdate, sleep and nslook command.
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifndef DBMALLOC
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif /* ! DBMALLOC */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <time.h>
#include <sys/time.h>
#include <netdb.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef __osf__
#include <machine/endian.h>
#endif

#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"

/*
 * Get the current time.
 */

int
getclockCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    time_t clock;

    if (argc > 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", 
			  argv[0], " ?date?\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if (argc == 2) {
	
	struct tm stm;
	char *day, *month;
	int res;

	static char *months[] = {
	    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
	    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
	};
	
	day = xstrdup (argv[1]);
	month = xstrdup (argv[1]);
	
	res = sscanf (argv[1], "%s%s%2d%2d:%2d:%2d%4d", 
		      day, month, &stm.tm_mday, 
		      &stm.tm_hour, &stm.tm_min, &stm.tm_sec, &stm.tm_year);

	if (res != 7) {
	    free (day);
	    free (month);
	    Tcl_AppendResult (interp, "illegal date \"",
			      argv[1], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	
	stm.tm_year -= 1900;

	for (stm.tm_mon = 0; stm.tm_mon < 12; stm.tm_mon++) {
	    if (strcmp(months[stm.tm_mon], month) == 0) break;
	}
	if (stm.tm_mon == 12) {
	    Tcl_AppendResult (interp, "unknown month \"",
			      month, "\"", (char *) NULL);
	    free (day);
	    free (month);
	    return TCL_ERROR;
	}
	
	stm.tm_wday = 0;
	stm.tm_yday = 0;
	stm.tm_isdst = -1;
	
	free (day);
	free (month);	
	
	clock = mktime (&stm);

    } else {
        clock = time ((time_t *) NULL);
    }

    sprintf (interp->result, "%ld", clock);
    return TCL_OK;
}

/*
 * Get the current date.
 */

int
getdateCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    time_t clock;
    char *p;

    if (argc > 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", 
			  argv[0], " ?clock?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (argc == 2) {
	if (Tcl_GetInt (interp, argv[1], (int *) &clock) != TCL_OK)
		return TCL_ERROR;
    } else {
	clock = time ((time_t *) NULL);
    }
    
    interp->result = ctime (&clock);
    for (p = interp->result; *p != '\0'; p++) if (*p == '\n') *p = '\0';

    return TCL_OK;
}

/*
 * sleep -- what else?
 */

int
sleepCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int time;

    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " time\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if (Tcl_GetInt (interp, argv[1], &time) != TCL_OK)
        return TCL_ERROR;

    sleep (time);
    
    return TCL_OK;
}

/*
 * The nslook command is just a wrapper around gethostbyname() and 
 * gethostbyaddr() to do quick name or address lookups. It was motivated
 * by nslookup written by Juergen Nickelsen <nickel@cs.tu-berlin.de>.
 */

int
nslookCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    struct hostent *host;
    unsigned long addr;
    struct in_addr *paddr;
    char buffer[20];
    
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " host-or-address\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if ((addr = inet_addr(argv[1])) != -1) {
	if (! (host = gethostbyaddr ((char *) &addr, 4, AF_INET))) {
	    Tcl_AppendResult (interp, "can not lookup \"", 
			      argv[1], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_AppendElement (interp, (char *) host->h_name);
	while (*host->h_aliases) {
	    Tcl_AppendElement (interp, *host->h_aliases++);
	}
    } else {
	if ((host = gethostbyname (argv[1])) == NULL) {
	    Tcl_AppendResult (interp, "can not lookup \"", 
			      argv[1], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	while (*host->h_addr_list) {
	    paddr = (struct in_addr *) *host->h_addr_list++;
            addr = ntohl (paddr->s_addr);
            sprintf (buffer, "%lu.%lu.%lu.%lu",
                     (addr >> 24) & 0xff, (addr >> 16) & 0xff,
                     (addr >> 8) & 0xff, addr & 0xff);
	    Tcl_AppendElement (interp, buffer);
	}
    }
    
    return TCL_OK;
}
