/*
  tkUEvent.c: user event handling for Tcl/Tk
  version 0.95
  
  Michael Halle
  MIT Media Laboratory
  mhalle@media.mit.edu


  * Permission to use, copy, modify, and distribute this
  * software and its documentation for any purpose and without
  * fee is hereby granted, provided that the above copyright
  * notice appear in all copies.  The Massachusetts Institute
  * of Technology makes no representations about the suitability
  * of this software for any purpose.
  * It is provided "as is" without  express or implied warranty.
  *
*/


/* 
  set id [userevent bind -includeevent {a b c d} command]
  userevent raise -async {a b c d} args...

  userevent destroy  $id
  userevent enable   $id ?new_state?
  userevent getcommand $id
*/

#include "tk.h"
static Tcl_HashTable idTable;

typedef struct _TkUE_Node TkUE_Node;
typedef struct _TkUE_Leaf TkUE_Leaf;


struct _TkUE_Leaf {
  int id;
  int active;
  int sendEvent;
  char *command;
  TkUE_Leaf *nextSibling, *prevSibling;
  TkUE_Node *parent;
};

struct _TkUE_Node {
  Tk_Uid name;
  TkUE_Node *parent;
  TkUE_Node *children;
  TkUE_Node *nextSibling, *prevSibling;
  TkUE_Node *question;
  TkUE_Node *asterisk;
  TkUE_Leaf *commands;
};


static void FreeLeaf _ANSI_ARGS_((TkUE_Leaf *l));
static void FreeNode _ANSI_ARGS_((TkUE_Node *n));
static TkUE_Leaf *AllocLeaf _ANSI_ARGS_((void));
static TkUE_Node *AllocNode _ANSI_ARGS_((TkUE_Node *parent));
static int UniqueId _ANSI_ARGS_((void));
static void InitRoot _ANSI_ARGS_((void));
static int AddEvent _ANSI_ARGS_((int argc, char **argv, 
				 char *command, int sendEvent));
static int CommandEval _ANSI_ARGS_((Tcl_Interp *interp, char *command, 
				    char *event, char *args));
static int MatchAndEval _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv,
				     char *event, char *commandArgs));

static int NodeMatchAndEval _ANSI_ARGS_((Tcl_Interp *interp, TkUE_Node *node,
					 int argc, char **argv,
					 char *event, char *commandArgs));


static int IdGetState _ANSI_ARGS_((int id));
static int IdSetState _ANSI_ARGS_((int id, int state));
static int IdDelete _ANSI_ARGS_((int id));

static char *IdGetCommand _ANSI_ARGS_((id));
static int IdSetCommand _ANSI_ARGS_((int id, char *command));

int TkUE_UserEventCmd _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
				   int argc, char **argv));

int TkUserEvent_Init _ANSI_ARGS_((Tcl_Interp *interp));

struct EvalArg {
  Tcl_Interp *interp;
  char *cmd;
};

static void AsyncCallback _ANSI_ARGS_((ClientData clientData));
static int Tk_GlobalEvalAsync _ANSI_ARGS_((Tcl_Interp *interp, char *command));
static void FreeEvalArg _ANSI_ARGS_((void));
static struct EvalArg *CreateEvalArg _ANSI_ARGS_((Tcl_Interp *interp, 
						  char *command));

int TkUserEvent_Init(interp)
     Tcl_Interp *interp;
{
  Tcl_CreateCommand(interp, "userevent", TkUE_UserEventCmd, 
		    (ClientData)Tk_MainWindow(interp),
		    (void (*)()) NULL);
  
  return(TCL_OK);
}

int
TkUE_UserEventCmd(clientData, interp, argc, argv)
     ClientData clientData;	/* Main window associated with
				 * interpreter. */
     Tcl_Interp *interp;	/* Current interpreter. */
     int argc;			/* Number of arguments. */
     char **argv;		/* Argument strings. */
{
  int id;
  
  if(argc == 1){
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " command ?options?\"", (char *) NULL);
    return TCL_ERROR;
  }


  /* id functions */
  if(argv[1][0] == 'e' && strcmp(argv[1], "enable") == 0){
    int state;

    if(argc < 3 || Tcl_GetInt(interp, argv[2], &id) == TCL_ERROR) {
      Tcl_AppendResult(interp, "bad args: should be \"",
		       argv[0], " ", argv[1], " id ?new_state?", (char *)NULL);
      return(TCL_ERROR);
    }

    if(argc > 3){
      if(Tcl_GetBoolean(interp, argv[3], &state) == TCL_ERROR){
	return(TCL_ERROR);
      }
      if(IdSetState(id, state)){
	Tcl_AppendResult(interp, argv[0], ": no such id \"", 
			 argv[2], "\"", (char *)NULL);
	return(TCL_ERROR);
      }
    }
    else {
      if((state = IdGetState(id)) == -1){
	Tcl_AppendResult(interp, argv[0], ": no such id \"", 
			 argv[2], "\"", (char *)NULL);
	return(TCL_ERROR);
      }
      interp->result = state ? "1" : "0";
    }
    return(TCL_OK);
  }

  else if(argv[1][0] == 'd' && strcmp(argv[1], "destroy") == 0){

    if(argc < 3 || Tcl_GetInt(interp, argv[2], &id) == TCL_ERROR) {
      Tcl_AppendResult(interp, "bad args: should be \"",
		       argv[0], " ", argv[1], " id", (char *)NULL);
      return(TCL_ERROR);
    }

    if(IdDelete(id)){
      Tcl_AppendResult(interp, argv[0], ": no such id \"", 
		       argv[2], "\"", (char *)NULL);
      return(TCL_ERROR);
    }

    return(TCL_OK);
  }

  else if(argv[1][0] == 'g' && strcmp(argv[1], "getcommand") == 0){
    char *cmd;

    if(argc < 3 || Tcl_GetInt(interp, argv[2], &id) == TCL_ERROR) {
      Tcl_AppendResult(interp, "bad args: should be \"",
		       argv[0], " ", argv[1], " id", (char *)NULL);
      return(TCL_ERROR);
    }

    if((cmd = IdGetCommand(id)) == (char *)NULL){
      Tcl_AppendResult(interp, argv[0], ": no such id \"", 
		       argv[2], "\"", (char *)NULL);
      return(TCL_ERROR);
    }

    Tcl_SetResult(interp, cmd, TCL_STATIC);
    return(TCL_OK);
  }


  else if(argv[1][0] == 'b' && strcmp(argv[1], "bind") == 0){
    int ec;
    int sendEvent = 0;
    char **ev;
    char *evstr, *cmdstr;
    static char idstr[64];


    if(argc != 4 && argc != 5){
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " bind ?-includeevent? eventlist command\"", 
		       (char *) NULL);
      return TCL_ERROR;
    }      

    if(argc == 5 && argv[2][0] == '-' && strcmp(argv[2], "-includeevent")==0){
      sendEvent = 1;
      evstr  = argv[3];
      cmdstr = argv[4];
    }
    else {
      evstr = argv[2];
      cmdstr = argv[3];
    }

    if(Tcl_SplitList(interp, evstr, &ec, &ev) != TCL_OK)
      return(TCL_ERROR);


    id = AddEvent(ec, ev, cmdstr, sendEvent);
    sprintf(idstr, "%d", id);
    Tcl_SetResult(interp, idstr, TCL_STATIC);

    free((char *)ev);
    return(TCL_OK);
  }
  else if(argv[1][0] == 'r' && strcmp(argv[1], "raise") == 0){
    char *s;
    int ec;
    char **ev;
    int ret;
    int argind, evind;
    int async;

    if(argc < 3){
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " raise ?-async? eventlist ?args?\"", 
		       (char *) NULL);
      return TCL_ERROR;
    }      

    if(argv[2][0] == '-' && strcmp(argv[2], "-async") == 0){
      async = 1;
      evind  = 3;
      argind = 4;
      if(argc < 4){
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " raise ?-async? eventlist ?args?\"", 
			 (char *) NULL);
	return TCL_ERROR;
      }      
    }
    else {
      async = 0;
      evind = 2;
      argind = 3;
    }

    if(Tcl_SplitList(interp, argv[evind], &ec, &ev) != TCL_OK)
      return(TCL_ERROR);

    if(argc > argind)
      s = Tcl_Merge(argc - argind, argv + argind);
    else
      s = "";

    ret = MatchAndEval(interp, ec, ev, argv[evind], s, async);

    free((char *)ev);

    if(argc > argind)
      free((char *)s);

    return(ret);
  }
  else {
    Tcl_AppendResult(interp, argv[1], 
		     ": unknown arg: should be one of: ",
		     "bind, raise, enable, destroy, getcommand",
		     (char *) NULL);
    return TCL_ERROR;
  }      
}


static int UniqueId()
{
  static int id = 0;
  return(id++);
}

static TkUE_Node *rootNode = (TkUE_Node *)NULL;

static TkUE_Node *AllocNode(parent)
     TkUE_Node *parent;
{
  TkUE_Node *n;

  n = (TkUE_Node *)ckalloc((unsigned) sizeof(TkUE_Node));

  n->name = (Tk_Uid)NULL;
  n->parent      = parent;
  n->children    = (TkUE_Node *)NULL;
  n->nextSibling = (TkUE_Node *)NULL;
  n->prevSibling = (TkUE_Node *)NULL;

  n->question = (TkUE_Node *)NULL;  
  n->asterisk = (TkUE_Node *)NULL;

  n->commands  = (TkUE_Leaf *)NULL;
  return(n);
}
  
static void FreeNode(n)
     TkUE_Node *n;
{
  TkUE_Leaf *t;

  if(n->parent == (TkUE_Node *)NULL) /* don't free root */
    return;

  while(n->commands != (TkUE_Leaf *)NULL){
    t = n->commands->nextSibling;
    FreeLeaf(n->commands);
    n->commands = t;
  }

  if(n->nextSibling)
    n->nextSibling->prevSibling = n->prevSibling;
  
  if(n->prevSibling){
    n->prevSibling->nextSibling = n->nextSibling;
  }
  
  else{
    n->parent->children = n->nextSibling;
    
    /* look up the tree for empty nodes */
    if(n->parent->commands == (TkUE_Leaf *)NULL && 
       n->parent->children == (TkUE_Node *)NULL){
      FreeNode(n->parent);
    }
  }

  ckfree((char *)n);
}

static TkUE_Leaf *AllocLeaf()
{
  TkUE_Leaf *l;
  
  l = (TkUE_Leaf *)ckalloc((unsigned) sizeof(TkUE_Leaf));
  
  l->nextSibling = l->prevSibling = (TkUE_Leaf *)NULL;
  l->command = (char *)NULL;
  l->active = 1;
  l->sendEvent = 1;
  l->id = -1;

  return(l);
}
  
static void FreeLeaf(l)
     TkUE_Leaf *l;
{
  if(l->command != (char *)NULL)
    ckfree((char *)l->command);

  { 
    /* unhash */
    Tcl_HashEntry *ent;
    ent = Tcl_FindHashEntry(&idTable, (char *)l->id);
    if(ent)
      Tcl_DeleteHashEntry(ent);
  }

  if(l->nextSibling)
    l->nextSibling->prevSibling = l->prevSibling;

  if(l->prevSibling)
    l->prevSibling->nextSibling = l->nextSibling;
  else 
    l->parent->commands = l->nextSibling;

  ckfree((char *)l);
}


static void InitRoot()
{
  if(rootNode == (TkUE_Node *)NULL){
    rootNode = AllocNode((TkUE_Node *)NULL);
    Tcl_InitHashTable(&idTable, TCL_ONE_WORD_KEYS);
  }
}

static int IdDelete(id)
     int id;
{
  Tcl_HashEntry *ent;

  InitRoot();
  ent = Tcl_FindHashEntry(&idTable, (char *)id);

  if(ent == (Tcl_HashEntry *)NULL)
    return(1);

  
  FreeLeaf((TkUE_Leaf *)Tcl_GetHashValue(ent));
  return(0);
}

static int IdSetState(id, state)
     int id;
     int state;
{
  Tcl_HashEntry *ent;
  TkUE_Leaf *l;

  InitRoot();
  ent = Tcl_FindHashEntry(&idTable, (char *)id);

  if(ent == (Tcl_HashEntry *)NULL)
    return(1);

  l = (TkUE_Leaf *)Tcl_GetHashValue(ent);
  l->active = state ? 1 : 0;

  return(0);
}

static int IdGetState(id)
     int id;
{
  Tcl_HashEntry *ent;
  TkUE_Leaf *l;

  InitRoot();
  ent = Tcl_FindHashEntry(&idTable, (char *)id);

  if(ent == (Tcl_HashEntry *)NULL)
    return(-1);

  l = (TkUE_Leaf *)Tcl_GetHashValue(ent);

  return(l->active);
}

static char *IdGetCommand(id)
     int id;
{
  Tcl_HashEntry *ent;
  TkUE_Leaf *l;

  InitRoot();
  ent = Tcl_FindHashEntry(&idTable, (char *)id);

  if(ent == (Tcl_HashEntry *)NULL)
    return((char *)NULL);

  l = (TkUE_Leaf *)Tcl_GetHashValue(ent);

  return(l->command);
}

static int IdSetCommand(id, command)
     int id;
     char *command;
{
  Tcl_HashEntry *ent;
  TkUE_Leaf *l;

  InitRoot();
  ent = Tcl_FindHashEntry(&idTable, (char *)id);

  if(ent == (Tcl_HashEntry *)NULL)
    return(1);

  l = (TkUE_Leaf *)Tcl_GetHashValue(ent);

  if(l->command)
    ckfree(l->command);

  l->command = (char *)ckalloc(strlen(command) + 3);
  strcpy(l->command, command);

  return(0);
}


static int AddEvent(argc, argv, command, sendEvent)
     int argc;
     char **argv;
     char *command;
     int sendEvent;
{
  TkUE_Node *t, *p, *c;
  TkUE_Leaf *l;
  int i;
  Tk_Uid asteriskUid, questionUid, curUid;
  InitRoot();
  
  asteriskUid = Tk_GetUid("*");
  questionUid = Tk_GetUid("?");

  /* build the tree */

  p = rootNode;

  for(i = 0; i < argc; i++){
    curUid = Tk_GetUid(argv[i]);
    
    if(curUid == questionUid){
      if(p->question == (TkUE_Node *)NULL){
	p->question = AllocNode(p);
	p->question->name = questionUid;
      }
      p = p->question;
    }
    else if(curUid == asteriskUid){
      if(p->asterisk == (TkUE_Node *)NULL){
	p->asterisk = AllocNode(p);
	p->asterisk->name = asteriskUid;
      }
      p = p->asterisk;
    }
    else { /* general case */
      /* look for match in kids */
      if(p->children == (TkUE_Node *)NULL){
	p->children = AllocNode(p);
	p->children->name = curUid;
      }
      c = p->children;

      for(;;){
	if(curUid == c->name){
	  p = c;
	  break;
	}

	if(c->nextSibling == (TkUE_Node *)NULL){
	  c->nextSibling = AllocNode(c->parent);
	  c->nextSibling->prevSibling = c;
	  c->nextSibling->name = curUid;
	}

	c = c->nextSibling;
      }
    }
  }      
  
  /* here, we have traversed the tree, and p points to the node to get
   a command */
  
  l = AllocLeaf();
  l->nextSibling = p->commands;
  p->commands = l;
  if(l->nextSibling)
    l->nextSibling->prevSibling = l;
  l->parent = p;

  l->command = (char *)ckalloc(strlen(command) + 3);

  strcpy(l->command, command);
  l->id = UniqueId();
  l->sendEvent = sendEvent;

  {
    /* hash id to leaf */

    int isnew;
    Tcl_HashEntry *ent;
    ent = Tcl_CreateHashEntry(&idTable, (char *)l->id, &isnew);
    Tcl_SetHashValue(ent, (ClientData)l);
  }
  return(l->id);
}

static int NodeMatchAndEval(interp, p, argc, argv, event, commandArgs, async)
     Tcl_Interp *interp;
     TkUE_Node *p;
     int argc;
     char **argv;
     char *event;
     char *commandArgs;
     int async;
{
  TkUE_Node *t, *c;
  TkUE_Leaf *l;
  int i;
  int atleaf = 0;
  Tk_Uid curUid;

  if(p == (TkUE_Node *)NULL)
    return;

  if(argc == 1)
      atleaf = 1;

  curUid = Tk_GetUid(argv[0]);
  
  /* decend "exact match" tree */
  for(c = p->children; c != (TkUE_Node *)NULL; c = c->nextSibling){
    if(c->name == curUid){
      if(!atleaf){

	/* bring node to front */
	if(c->prevSibling != (TkUE_Node *)NULL) {
	  c->prevSibling->nextSibling = c->nextSibling;
	  
	  if(c->nextSibling != (TkUE_Node *)NULL)
	    c->nextSibling->prevSibling = c->prevSibling;
	  
	  c->prevSibling = (TkUE_Node *)NULL;
	  c->nextSibling = c->parent->children;
	  c->parent->children->prevSibling = c;
	  c->parent->children = c;
	}

	if(NodeMatchAndEval(interp, c, argc-1, argv+1,
			    event, commandArgs, async) == TCL_ERROR)
	  return(TCL_ERROR);
      }	  
      else {
	for(l = c->commands; l != (TkUE_Leaf *)NULL; l = l->nextSibling){
	  if(l->active)
	    if(CommandEval(interp, async, l->command, 
			   l->sendEvent ? event : (char *)NULL, 
			   commandArgs) == TCL_ERROR)
	      return(TCL_ERROR);
	}
      }
    }
  }

  c = p->question;
  if(c != (TkUE_Node *)NULL){
    if(!atleaf){
      if(NodeMatchAndEval(interp, c, argc-1, argv+1, 
			  event, commandArgs, async) == TCL_ERROR)
	return(TCL_ERROR);
    }	  
    else {
      for(l = c->commands; l != (TkUE_Leaf *)NULL; l = l->nextSibling){
	if(l->active)
	  if(CommandEval(interp, async, l->command, 
			 l->sendEvent ? event : (char *)NULL, 
			 commandArgs) == TCL_ERROR)
	    return(TCL_ERROR);
      }
    }
  }
  
  c = p->asterisk;
  if(c != (TkUE_Node *)NULL){
    for(l = c->commands; l != (TkUE_Leaf *)NULL; l = l->nextSibling){
      if(l->active)
	if(CommandEval(interp, async, l->command, 
		       l->sendEvent ? event : (char *)NULL, 
		       commandArgs) == TCL_ERROR)
	  return(TCL_ERROR);
    }
  }
  
  return(TCL_OK);
}

static int MatchAndEval(interp, argc, argv, event, commandArgs, async)
     Tcl_Interp *interp;
     int argc;
     char **argv;
     char *event;
     char *commandArgs;
     int async;
{
  InitRoot();
  
  return(NodeMatchAndEval(interp, rootNode, argc, argv, event, 
			  commandArgs, async));
}

static int CommandEval(interp, async, command, event, args)
     Tcl_Interp *interp;
     int async;
     char *command;
     char *event;

     char *args;
{
  int len, ret;
  char *ncmd;
#define FIXED_SPACE 200
  char fixed[FIXED_SPACE];

  if(event == (char *)NULL)
    event = "";
  
  len = strlen(command) + strlen(args) + strlen(event) + 10;

  if(len < FIXED_SPACE)
    ncmd = fixed;
  else
    ncmd = (char *)ckalloc((unsigned) len);

  sprintf(ncmd, *event == '\0' ? "%s %s%s" : "%s [list %s] %s", 
	  command, event, args);

  if(async)
    ret = Tk_GlobalEvalAsync(interp, ncmd);
  else
    ret = Tcl_GlobalEval(interp, ncmd);

  if(ncmd != fixed)
    ckfree((char *)ncmd);

  return(ret);
}

static struct EvalArg *CreateEvalArg(interp, cmd) 
     Tcl_Interp *interp;
     char *cmd;
{
  struct EvalArg *t;
  t = (struct EvalArg *)ckalloc(sizeof(struct EvalArg));
  t->interp = interp;
  t->cmd = (char *)ckalloc((unsigned) (strlen(cmd) + 1));
  strcpy(t->cmd, cmd);
  return(t);
}

static void FreeEvalArg(arg)
     struct EvalArg *arg;

{
  struct EvalArg *t;
  if(t == (struct EvalArg *)NULL)
    return;
  if(t->cmd)
    ckfree((char *)t->cmd);
  ckfree((char *)t);
}

static int Tk_GlobalEvalAsync(interp, cmd)
     Tcl_Interp *interp;
     char *cmd;
{
  struct  EvalArg *arg;

  arg = CreateEvalArg(interp, cmd);
  Tk_DoWhenIdle(AsyncCallback, (ClientData)arg);
  return(TCL_OK);
}

static void AsyncCallback(cd)
     ClientData cd;
{
  struct EvalArg *arg = (struct EvalArg *)cd;

  Tcl_GlobalEval(arg->interp, arg->cmd);
  FreeEvalArg(arg);
}


