# text.tcl --
#
# This file contains Tcl procedures used to manage Tk text widget
# bindings.  The procedures are based on the standard Tk procedures
# supplied in the Tk distribution.
#
# The procedure below is invoked when dragging one end of the selection.
# The arguments are the text window name and the index of the character
# that is to be the new end of the selection.

proc Text_SelectTo {w index} {
    global tk_priv
    set modify_select 1

    case $tk_priv(selectMode) {
	char {
	    if [$w compare $index < anchor] {
		set first $index
		set last anchor
	    } else {
                if [$w compare $index > anchor] {
		  set first anchor
		  set last [$w index $index]
                } else {
                  set modify_select 0
                }
	    }
	}
	word {
	    if [$w compare $index < anchor] {
		set first [$w index "$index wordstart"]
		set last [$w index "anchor wordend"]
	    } else {
		set first [$w index "anchor wordstart"]
		set last [$w index "$index wordend"]
	    }
	}
	line {
	    if [$w compare $index < anchor] {
		set first [$w index "$index linestart"]
		set last [$w index "anchor lineend + 1c"]
	    } else {
		set first [$w index "anchor linestart"]
		set last [$w index "$index lineend + 1c"]
	    }
	}
    }
    if $modify_select {
      $w tag remove sel 0.0 $first
      $w tag add sel $first $last
      $w tag remove sel $last end
    }
}

# The procedure below is invoked to backspace over one character in
# a text widget.  The name of the widget is passed as argument.

proc Text_Backspace w {
    $w delete insert-1c insert
}

# The procedure below compares three indices, a, b, and c.  Index b must
# be less than c.  The procedure returns 1 if a is closer to b than to c,
# and 0 otherwise.  The "w" argument is the name of the text widget in
# which to do the comparison.

proc Text_IndexCloser {w a b c} {
    set a [$w index $a]
    set b [$w index $b]
    set c [$w index $c]
    if [$w compare $a <= $b] {
	return 1
    }
    if [$w compare $a >= $c] {
	return 0
    }
    scan $a "%d.%d" lineA chA
    scan $b "%d.%d" lineB chB
    scan $c "%d.%d" lineC chC
    if {$chC == 0} {
	incr lineC -1
	set chC [string length [$w get $lineC.0 $lineC.end]]
    }
    if {$lineB != $lineC} {
	return [expr {($lineA-$lineB) < ($lineC-$lineA)}]
    }
    return [expr {($chA-$chB) < ($chC-$chA)}]
}

# The procedure below is called to reset the selection anchor to
# whichever end is FARTHEST from the index argument.

proc Text_ResetAnchor {w index} {
    global tk_priv
    if {[$w tag ranges sel] == ""} {
	set tk_priv(selectMode) char
	$w mark set anchor $index
	return
    }
    if [Text_IndexCloser $w $index sel.first sel.last] {
	if {$tk_priv(selectMode) == "char"} {
	    $w mark set anchor sel.last
	} else {
	    $w mark set anchor sel.last-1c
	}
    } else {
	$w mark set anchor sel.first
    }
}

#
# The following procedures implement basic actions for text widgets
#
# Copyright Paul Alexander, All rights reserved
#


# Text_DeleteSelection :::: delete the current text selection
# 
#  Parameters:
#             1   window name for the text widget
#
proc Text_DeleteSelection { w } {
  catch {$w delete sel.first sel.last}
}


# Text_InsertSelection :::: insert the current text selection
# 
#  Parameters:
#             1   window name for the text widget
#
proc Text_InsertSelection { w } {
  catch {$w insert insert [selection get]}
  $w yview -pickplace insert
}

# Text_MoveLeft :::: move insertion left one character
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveLeft { w } {
  $w mark set insert {insert - 1 chars}
  $w yview -pickplace insert
}


# Text_MoveRight :::: move insertion right one character
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveRight { w } {
  $w mark set insert {insert + 1 chars}
  $w yview -pickplace insert
}


# Text_MoveUp :::: move insertion up one line
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveUp { w } {
  $w mark set insert {insert - 1 lines}
  $w yview -pickplace insert
}

# Text_MoveDown :::: move insertion down one line
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveDown { w } {
  $w mark set insert {insert + 1 lines}
  $w yview -pickplace insert
}


# Text_MoveTop :::: move insertion cursor to top of document
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveTop { w } {
  $w mark set insert 0.0
  $w yview -pickplace insert
}


# Text_MoveBottom :::: move insertion cursor to bottom of document
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveBottom { w } {
  $w mark set insert end
  $w yview -pickplace insert
}


# Text_MoveStart :::: move insertion cursor to start of line
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveStart { w } {
  $w mark set insert "insert linestart"
}


# Text_MoveEnd :::: move insertion cursor to end of line
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveEnd { w } {
  $w mark set insert "insert lineend"
}


# Text_MoveDownpage :::: move insertion cursor down one page
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveDownpage { w } {
  $w mark set insert "insert +  20 line"
  $w yview -pickplace insert
}


# Text_MoveUppage :::: move insertion cursor up one page
#  Parameters: 
#             1    window name for the text widget
proc Text_MoveUppage { w } {
  $w mark set insert "insert - 20 line"
  $w yview -pickplace insert
}



# Text_InsertChar :::: insert a character
#  Parameters: 
#             1    window name for text widget
#             2    character to inset
proc Text_InsertChar { w char } {
  if {"$char" != ""} {
    $w insert insert $char
    $w yview -pickplace insert
  }
}



# Text_InsertCursor :::: locate insertion cursor
#  Parameters: 
#             1    window name for text widget
#             2    index in text for insertion
proc Text_InsertCursor { w index } {
  global tk_priv
  set tk_priv(selectMode) char
  $w mark set insert $index
  $w mark set anchor insert
  if {[lindex [$w config -state] 4] == "normal"} {focus $w}
}



# Text_InsertNewline :::: insert a newline character
#  Parameters: 
#             1    window name for text widget
proc Text_InsertNewline { w } {
  $w insert insert \n
  $w yview -pickplace insert
}



# Text_SelectWord :::: locate insertion cursor and select word
#  Parameters: 
#             1    window name for text widget
#             2    index in text for insertion
proc Text_SelectWord { w index } {
  global tk_priv
  set tk_priv(selectMode) word
  $w mark set insert "$index wordstart"
  Text_SelectTo $w insert
}


# Text_SelectLine :::: locate insertion cursor and select line
#  Parameters: 
#             1    window name for text widget
#             2    index in text for insertion
proc Text_SelectLine { w index } {
  global tk_priv
  set tk_priv(selectMode) line
  $w mark set insert "$index linestart"
  Text_SelectTo $w insert
}

# Text_DeleteChar :::: delete character to left of insertion cursor
#  Parameters:
#             1 window name for text widget
proc Text_DeleteChar { w } {
  Text_Backspace $w
  $w yview -pickplace insert
}

# Text_DeleteRChar :::: delete character to right of insertion cursor
#  Parameters:
#             1 window name for text widget
proc Text_DeleteRChar { w } {
  $w delete insert insert+1c
  $w yview -pickplace insert
}

# Text_WordMark :::: insert word mark at insertion cursor
#  Parameters:
#             1 window name for text window
proc Text_WordMark { w } {
   $w mark set word_mark "insert wordstart"
}

# Text_LineMark :::: insert line mark at insertion cursor
#  Parameters:
#             1 window name for text window
proc Text_LineMark { w } {
   $w mark set line_mark "insert linestart"
}

# Text_WordMark :::: return currently marked word
#  Parameters:
#             1 window name for text window
proc Text_WordGet { w } {
   return [$w get [$w index "word_mark wordstart"] \
                  [$w index "word_mark wordend"]]
}

# Text_WordReplace :::: replace currently marked word with new text
#  Parameters:
#             1 window name for text window
#             2 replacement word
proc Text_WordReplace { w word } {
   set i1 [$w index "word_mark wordstart"]
   set i2 [$w index "word_mark wordend"]
   $w delete $i1 $i2
   $w insert $i1 $word
   $w mark set word_mark $i1
}

# Text_WordDelete :::: delete currently marked word
#  Parameters:
#             1 window name for text window
proc Text_WordDelete { w } {
   set i1 [$w index "word_mark wordstart"]
   set i2 [$w index "word_mark wordend"]
   $w delete "$i1 - 1 c" $i2
   $w mark set word_mark "$i1 - 1 c wordstart"
   if ![Text_ctype [Text_WordGet $w]] then {
     Text_WordPrevious $w
   }
}

# Text_LineGet :::: return currently marked line
#  Parameters:
#             1 window name for text window
proc Text_LineGet { w } {
   return [$w get [$w index "line_mark linestart"] \
                  [$w index "line_mark lineend"]]
}

# Text_LineReplace :::: replace currently marked line with new text
#  Parameters:
#             1 window name for text window
#             2 replacement text
proc Text_LineReplace { w line } {
   set i1 [$w index "line_mark linestart"]
   set i2 [$w index "line_mark lineend"]
   $w delete $i1 $i2
   $w insert $i1 $line
   $w mark set line_mark $i1
}

# Text_LineDelete :::: delete marked line
#  Parameters:
#             1 window name for text window
proc Text_LineDelete { w } {
   set i1 [$w index "line_mark linestart"]
   set i2 [$w index "line_mark lineend"]
   $w delete "$i1 - 1 c" $i2
   $w mark set line_mark $i1
}

# Text_WordNext :::: move the mark to the next word
#  Parameters:
#             1 window name for text window
proc Text_WordNext { w } {
   set status_ok 1
   for { } {1} { } {
     $w mark set word_mark [$w index "word_mark wordend"]
     if [$w compare end == word_mark] then { 
        set status_ok 0 ; break
     }
     if [Text_ctype [Text_WordGet $w]] then {
        set status_ok 1 ; break
     }
   }      
   return $status_ok
}

# Text_LineNext :::: move the mark to the next line
#  Parameters:
#             1 window name for text window
proc Text_LineNext { w } {
   set status_ok 1
   $w mark set line_mark {line_mark + 1 lines}
   if [$w compare end == line_mark] then { 
        set status_ok 0 
   } else {
        set status_ok 1
   }
   return $status_ok
}

# Text_WordPrevious :::: move the mark to the previous word
#  Parameters:
#             1 window name for text window
proc Text_WordPrevious { w } {
   set status_ok 1
   for { } {1} { } {
     $w mark set word_mark [$w index "word_mark wordstart - 1 chars"]
     if [$w compare 1.0 == word_mark] then { 
        set status_ok 0 ; break
     }
     if [Text_ctype [Text_WordGet $w]] then {
        set status_ok 1 ; break
     }
   }      
   return $status_ok
}

# Text_LinePrevious :::: move the mark to the previous line
#  Parameters:
#             1 window name for text window
proc Text_LinePrevious { w } {
   set status_ok 1
   $w mark set line_mark {line_mark - 1 lines}
   if [$w compare 1.0 == line_mark] then { 
        set status_ok 0 
   } else {
        set status_ok 1
   }
   return $status_ok
}

# Text_WordTag :::: tag the currently marked word
#  Parameters:
#             1 window name for text window
#  Options:
#   All options have their normal meanings
#             -background -foreground -underline 
#             -relief -borderwidth -font
proc Text_WordTag { w args } {
   set relief raised
   set underline 0
   set foreground black
   set background lightskyblue
   set borderwidth 1
   set do_font 0
   for {set i 0} {$i < [llength $args]} {incr i} {
     set word [lindex $args $i]
     case $word in {
        {-relief}      {incr i ; set relief [lindex $args $i]}
        {-foreground}  {incr i ; set foreground [lindex $args $i]}
        {-background}  {incr i ; set background [lindex $args $i]}
        {-underline}   {incr i ; set underline [lindex $args $i]}
        {-borderwidth} {incr i ; set borderwidth [lindex $args $i]}
        {-font}        {incr i ; set font [lindex $args $i] ; set do_font 1}
     }
   }
   catch "$w tag delete word_tag"
   $w tag add word_tag [$w index "word_mark wordstart"] \
                       [$w index "word_mark wordend"]
   $w tag configure word_tag -background $background -relief $relief \
                             -foreground $foreground -underline $underline\
                             -borderwidth $borderwidth
   if $do_font then {
     $w tag configure word_tag -font $font
   }
   $w yview -pickplace word_mark
}

# Text_LineTag :::: tag the currently marked line
#  Parameters:
#             1 window name for text window
#  Options:
#   All options have their normal meanings
#             -background -foreground -underline 
#             -relief -borderwidth -font
proc Text_LineTag { w args } {
   set relief raised
   set underline 0
   set foreground black
   set background lightskyblue
   set borderwidth 1
   set do_font 0
   for {set i 0} {$i < [llength $args]} {incr i} {
     set word [lindex $args $i]
     case $word in {
        {-relief}     {incr i ; set relief [lindex $args $i]}
        {-foreground} {incr i ; set foreground [lindex $args $i]}
        {-background} {incr i ; set background [lindex $args $i]}
        {-underline}  {incr i ; set underline [lindex $args $i]}
        {-borderwidth} {incr i ; set borderwidth [lindex $args $i]}
        {-font}        {incr i ; set font [lindex $args $i] ; set do_font 1}
     }
   }
   catch "$w tag delete line_tag"
   $w tag add line_tag [$w index "line_mark linestart"] \
                       [$w index "line_mark lineend"]
   $w tag configure line_tag -background $background -relief $relief \
                             -foreground $foreground -underline $underline\
                             -borderwidth $borderwidth
   if $do_font then {
      $w tag configure line_tag -font $font
   }
   $w yview -pickplace line_mark
}

# Text_WordUnTag :::: remove tag from current word
#  Parameters:
#             1  window name for text window
proc Text_WordUnTag { w } {
   catch "$w tag delete word_tag"
}

# Text_LineUnTag :::: remove tag from current line
#  Parameters:
#             1  window name for text window
proc Text_LineUnTag { w } {
   catch "$w tag delete line_tag"
}

proc Text_ctype { text } {
  if [regexp {[\*`~!@#$%^&()_+=\{\}"';:,./?1234567890 ]} $text] then {
    return 0
  } else {
    return [regexp {[a-z]} $text]
  }
}


# Text_CharWidth :::: return the width of the text window
#  Parameters:
#             1  window name for text window
proc Text_CharWidth { w } {
   set saveSetGrid [lindex [$w configure -setgrid] 4]
   $w configure -setgrid 1
   set width [lindex [split [wm geom [winfo top $w]] +x] 0]
   $w configure -setgrid $saveSetGrid
   return $width
}

# Text_CharHeight :::: return the height of the text window
#  Parameters:
#             1  window name for text window
proc Text_CharHeight { w } {
   set saveSetGrid [lindex [$w configure -setgrid] 4]
   $w configure -setgrid 1
   set length [lindex [split [wm geom [winfo top $w]] +x] 1]
   $w configure -setgrid $saveSetGrid
   return $length
}


