
// Objectify -- turn C++ classes into Tcl objects.
//
// Wayne Christopher, faustus@cs.berkeley.edu
// Version 0.1, May 1, 1993
//
// Copyright (c) 1993 Wayne Christopher.  Permission is granted
// to copy, modify and distribute, provided this notice is
// preserved.  No warranty is provided or implied, etc, etc.

// This routine calls the appropriate configure routine, but has to flag
// those options that are eaten up.  The eaten vector is 0 if the flag
// hasdn't been used yet and 1 if it has.  This routine will set the
// appropriate flag if the arg is used.

//extern "C" {
#define class _____class
#include <stdlib.h>
#include "tcl.h"
#include "tk.h"
#undef class
//}

#include "objectify.h"

static int
ConfigureObject(Tcl_Interp *interp, Tk_ConfigSpec *specs,
		    int argc, char **argv, char* widgRec, int flags);

static int
ConfigureInfo(Tcl_Interp *interp, Tk_ConfigSpec *specs,
		  char* widgRec, char* argvName, int flags);

// Which is 1 for creation, 0 for configure.

int
Objectify_Configure(Tcl_Interp* interp, Tk_Window tkwin, Tk_ConfigSpec* specs,
		    int ac, char** av, char* object, int flags, int* eaten,
		    int which)
{
    // We have to muck around here a bit since the result will get overwritten
    // by Configure*.
    int reslen = (interp->result ? strlen(interp->result) : 0);
    char* save_result = (reslen ? new char[reslen + 1] : NULL);
    if (save_result)
	strcpy(save_result, interp->result);
    Tcl_ResetResult(interp);
    
    int res = TCL_OK;
    
    if (!ac && !which) {
	if (tkwin) {
	    res = Tk_ConfigureInfo(interp, tkwin, specs, object, NULL, 0);
	} else {
	    res = ConfigureInfo(interp, specs, object, NULL, 0);
	}
    } else {
	int nac = 0;
	char** nav = new char* [ac + 1];
	
	for (int i = 0; i < ac; i += 2) {
	    char* opt = av[i];
	    if (eaten[i / 2]) continue;
	    for (int j = 0; specs[j].type != TK_CONFIG_END; j++)
		if (specs[j].argvName &&
		    !strcmp(specs[j].argvName, opt))
		    break;
	    if (specs[j].type != TK_CONFIG_END) {
		nav[nac++] = av[i];
		nav[nac++] = av[i + 1];
		eaten[i / 2] = 1;
	    }
	}
	if (nac || which) {
	    if (!nav[nac - 1]) nac--;
	    nav[nac] = NULL;
	    
	    if (which && (nac == 1)) {
		Tcl_SetResult(interp, "wrong number of args for creation", 0);
		res = TCL_ERROR;
		
	    } else if (tkwin) {
		if (!which && (nac == 1))
		    res = Tk_ConfigureInfo(interp, tkwin, specs, object,
					   nav[0], 0);
		else
		    res = Tk_ConfigureWidget(interp, tkwin, specs, nac, nav,
					     object, flags);
	    		
	    } else {
		if (nac == 1)
		    res = ConfigureInfo(interp, specs, object, nav[0], 0);
		else
		    res = ConfigureObject(interp, specs, nac, nav,
					      object, flags);
	    }
	}
	delete nav;
    }
    
    // We don't care about the ordering.
    if (save_result) {
	if (res == TCL_OK) {
	    Tcl_AppendResult(interp, " ", 0);
	    Tcl_AppendResult(interp, save_result, 0);
	}
	delete save_result;
    }
    
    return (res);
}

int
Objectify_Get(Tcl_Interp* interp, Tk_Window tkwin, Tk_ConfigSpec* specs,
	      char* name, char* object, int* got)
{
    char c1 = name[1];
    for (int j = 0; specs[j].type != TK_CONFIG_END; j++) {
	char* an = specs[j].argvName;
	if (an && (c1 == an[1]) && !strcmp(an, name))
	    break;
    }
    if (specs[j].type == TK_CONFIG_END)
	return (TCL_OK);

    if (tkwin) {
	// Yuck.
	Tk_ConfigSpec tempspecs[2];
	tempspecs[0] = specs[j];
	tempspecs[1].type = TK_CONFIG_END;
	
	if (Tk_ConfigureInfo(interp, tkwin, tempspecs, object, name, 0) != TCL_OK)
	    return (TCL_ERROR);
	int ac;
	char** av;
	if (Tcl_SplitList(interp, interp->result, &ac, &av) != TCL_OK)
	    return (TCL_ERROR);
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, av[4], 0);
	free((char *) av);
	*got = 1;
	return (TCL_OK);
    } else {
	// This is the same thing as in objConfig.c
	char* val;
	char buffer[200];
	Tk_ConfigSpec *spec = &specs[j];
	Tcl_FreeProc *freeProc = NULL;
	char* ptr = object + spec->offset;
	switch (spec->type) {
	  case TK_CONFIG_BOOLEAN:
	    if (*((int *) ptr) == 0) {
		val = "0";
	    } else {
		val = "1";
	    }
	    break;
	  case TK_CONFIG_INT:
	    sprintf(buffer, "%d", *((int *) ptr));
	    val = buffer;
	    break;
	  case TK_CONFIG_DOUBLE:
	    sprintf(buffer, "%g", *((double *) ptr));
	    val = buffer;
	    break;
	  case TK_CONFIG_STRING:
	    val = (*(char **) ptr);
	    break;
	  case TK_CONFIG_CUSTOM:
	    val = (*spec->customPtr->printProc)(spec->customPtr->clientData, NULL,
						object, spec->offset, &freeProc);
	    break;
	  default: 
	    val = "?? unknown type ??";
	}
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, val, 0);
	*got = 1;
	return (TCL_OK);
    }
}

// This code originally came from tkConfig.c.

/*
 * Values for "flags" field of Obj_ConfigSpec structures.  Be sure
 * to coordinate these values with those defined in tk.h
 * (TK_CONFIG_COLOR_ONLY, etc.).  There must not be overlap!
 *
 * INIT -		Non-zero means (char *) things have been
 *			converted to Tk_Uid's.
 */

#define INIT		0x20

/*
 * Forward declarations for procedures defined later in this file:
 */

static int		DoConfig _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_ConfigSpec *specPtr,
			    Tk_Uid value, int valueIsUid, char *widgRec));
static Tk_ConfigSpec *	FindConfigSpec _ANSI_ARGS_ ((Tcl_Interp *interp,
			    Tk_ConfigSpec *specs, char *argvName,
			    int needFlags, int hateFlags));
static char *		FormatConfigInfo _ANSI_ARGS_ ((Tk_ConfigSpec *specPtr,
						       char *widgRec));

/*
 *--------------------------------------------------------------
 *
 * ConfigureObject --
 *
 *	Process command-line options and database options to
 *	fill in fields of a widget record with resources and
 *	other parameters.
 *
 * Results:
 *	A standard Tcl return value.  In case of an error,
 *	interp->result will hold an error message.
 *
 * Side effects:
 *	The fields of widgRec get filled in with information
 *	from argc/argv and the option database.  Old information
 *	in widgRec's fields gets recycled.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureObject(Tcl_Interp *interp, Tk_ConfigSpec *specs, int argc, char **argv,
		char *widgRec, int flags)
{
    Tk_ConfigSpec *specPtr;
    int needFlags;		/* Specs must contain this set of flags
				 * or else they are not considered. */
    int hateFlags;		/* If a spec contains any bits here, it's
				 * not considered. */
    char* value;

    needFlags = flags & ~(TK_CONFIG_USER_BIT - 1);
    hateFlags = 0;

    /*
     * Pass one:  scan through all the option specs, clearing
     * the TK_CONFIG_OPTION_SPECIFIED flags.
     */

    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	specPtr->specFlags = (specPtr->specFlags & ~TK_CONFIG_OPTION_SPECIFIED)
		| INIT;
    }

    /*
     * Pass two:  scan through all of the arguments, processing those
     * that match entries in the specs.
     */

    for ( ; argc > 0; argc -= 2, argv += 2) {
	specPtr = FindConfigSpec(interp, specs, *argv, needFlags, hateFlags);
	if (specPtr == NULL) {
	    return TCL_ERROR;
	}

	/*
	 * Process the entry.
	 */

	if (argc < 2) {
	    Tcl_AppendResult(interp, "value for \"", *argv,
		    "\" missing", (char *) NULL);
	    return TCL_ERROR;
	}
	if (DoConfig(interp, specPtr, argv[1], 0, widgRec) != TCL_OK) {
	    char msg[100];

	    sprintf(msg, "\n    (processing \"%.40s\" option)",
		    specPtr->argvName);
	    Tcl_AddErrorInfo(interp, msg);
	    return TCL_ERROR;
	}
	specPtr->specFlags |= TK_CONFIG_OPTION_SPECIFIED;
    }

    /*
     * Pass three:  scan through all of the specs again;  if no
     * command-line argument matched a spec, then check for info
     * in the option database.  If there was nothing in the
     * database, then use the default.
     */

    if (!(flags & TK_CONFIG_ARGV_ONLY)) {
	for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	    if ((specPtr->specFlags & TK_CONFIG_OPTION_SPECIFIED)
		    || (specPtr->argvName == NULL)
		    || (specPtr->type == TK_CONFIG_SYNONYM)) {
		continue;
	    }
	    if (((specPtr->specFlags & needFlags) != needFlags)
		    || (specPtr->specFlags & hateFlags)) {
		continue;
	    }
	    
	    value = specPtr->defValue;
	    if ((value != NULL) && !(specPtr->specFlags
				     & TK_CONFIG_DONT_SET_DEFAULT)) {
		if (DoConfig(interp, specPtr, value, 1, widgRec) != TCL_OK) {
		    char msg[200];
		    
		    sprintf(msg,
			    "\n    (%s \"%.50s\" in widget)",
			    "default value for", specPtr->dbName);
		    Tcl_AddErrorInfo(interp, msg);
		    return TCL_ERROR;
		}
	    }
	}
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * FindConfigSpec --
 *
 *	Search through a table of configuration specs, looking for
 *	one that matches a given argvName.
 *
 * Results:
 *	The return value is a pointer to the matching entry, or NULL
 *	if nothing matched.  In that case an error message is left
 *	in interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static Tk_ConfigSpec *
FindConfigSpec(Tcl_Interp *interp, Tk_ConfigSpec *specs, char *argvName,
	       int needFlags, int hateFlags)
{
    Tk_ConfigSpec *specPtr;
    char c;		/* First character of current argument. */
    Tk_ConfigSpec *matchPtr;	/* Matching spec, or NULL. */
    int length;

    c = argvName[1];
    length = strlen(argvName);
    matchPtr = NULL;
    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	if (specPtr->argvName == NULL) {
	    continue;
	}
	if ((specPtr->argvName[1] != c)
		|| (strcmp(specPtr->argvName, argvName) != 0)) {
	    continue;
	}
	if (((specPtr->specFlags & needFlags) != needFlags)
		|| (specPtr->specFlags & hateFlags)) {
	    continue;
	}
	if (!specPtr->argvName[length]) {
	    matchPtr = specPtr;
	    goto gotMatch;
	}
	if (matchPtr != NULL) {
	    Tcl_AppendResult(interp, "ambiguous option \"", argvName,
		    "\"", (char *) NULL);
	    return (Tk_ConfigSpec *) NULL;
	}
	matchPtr = specPtr;
    }

    if (matchPtr == NULL) {
	Tcl_AppendResult(interp, "unknown option \"", argvName,
		"\"", (char *) NULL);
	return (Tk_ConfigSpec *) NULL;
    }

    /*
     * Found a matching entry.  If it's a synonym, then find the
     * entry that it's a synonym for.
     */

    gotMatch:
    specPtr = matchPtr;
    if (specPtr->type == TK_CONFIG_SYNONYM) {
	for (specPtr = specs; ; specPtr++) {
	    if (specPtr->type == TK_CONFIG_END) {
		Tcl_AppendResult(interp,
			"couldn't find synonym for option \"",
			argvName, "\"", (char *) NULL);
		return (Tk_ConfigSpec *) NULL;
	    }
	    if ((specPtr->dbName == matchPtr->dbName) 
		    && (specPtr->type != TK_CONFIG_SYNONYM)
		    && ((specPtr->specFlags & needFlags) == needFlags)
		    && !(specPtr->specFlags & hateFlags)) {
		break;
	    }
	}
    }
    return specPtr;
}

/*
 *--------------------------------------------------------------
 *
 * DoConfig --
 *
 *	This procedure applies a single configuration option
 *	to a widget record.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	WidgRec is modified as indicated by specPtr and value.
 *	The old value is recycled, if that is appropriate for
 *	the value type.
 *
 *--------------------------------------------------------------
 */

static int
DoConfig(Tcl_Interp *interp, Tk_ConfigSpec *specPtr, char *value,
	 int valueIsUid, char *widgRec)
{
    char *ptr;
    int nullValue;

    nullValue = 0;
    if (!*value && (specPtr->specFlags & TK_CONFIG_NULL_OK)) {
	nullValue = 1;
    }

    do {
	ptr = widgRec + specPtr->offset;
	switch (specPtr->type) {
	    case TK_CONFIG_BOOLEAN:
		if (Tcl_GetBoolean(interp, value, (int *) ptr) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;
	    case TK_CONFIG_INT:
		if (!*value) {
		    *((int *) ptr) = 0;
		} else if (Tcl_GetInt(interp, value, (int *) ptr) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;
	    case TK_CONFIG_DOUBLE:
		if (!*value) {
		    *((double *) ptr) = 0;
		} else if (Tcl_GetDouble(interp, value, (double *) ptr) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;
	    case TK_CONFIG_STRING: {
		char *old, *nstr;

		if (nullValue) {
		    nstr = NULL;
		} else {
		    nstr = (char *) ckalloc((unsigned) (strlen(value) + 1));
		    strcpy(nstr, value);
		}
		old = *((char **) ptr);
		if (old != NULL) {
		    /* printf("free with %s\n", specPtr->argvName); */
		    ckfree(old);
		}
		*((char **) ptr) = nstr;
		break;
	    }

	    case TK_CONFIG_CUSTOM:
		if ((*specPtr->customPtr->parseProc)(
			specPtr->customPtr->clientData, interp, NULL,
			value, widgRec, specPtr->offset) != TCL_OK) {
		    return TCL_ERROR;
		}
		break;
	    default: {
		sprintf(interp->result, "bad config table: unknown type %d",
			specPtr->type);
		return TCL_ERROR;
	    }
	}
	specPtr++;
    } while ((specPtr->argvName == NULL) && (specPtr->type != TK_CONFIG_END));
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tk_ConfigureInfo --
 *
 *	Return information about the configuration options
 *	for a window, and their current values.
 *
 * Results:
 *	Always returns TCL_OK.  Interp->result will be modified
 *	hold a description of either a single configuration option
 *	available for "widgRec" via "specs", or all the configuration
 *	options available.  In the "all" case, the result will
 *	available for "widgRec" via "specs".  The result will
 *	be a list, each of whose entries describes one option.
 *	Each entry will itself be a list containing the option's
 *	name for use on command lines, database name, database
 *	class, default value, and current value (empty string
 *	if none).  For options that are synonyms, the list will
 *	contain only two values:  name and synonym name.  If the
 *	"name" argument is non-NULL, then the only information
 *	returned is that for the named argument (i.e. the corresponding
 *	entry in the overall list is returned).
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureInfo(Tcl_Interp *interp, Tk_ConfigSpec *specs, char *widgRec,
	      char *argvName, int flags)
{
    Tk_ConfigSpec *specPtr;
    int needFlags, hateFlags;
    char *list;
    char *leader = "{";

    needFlags = flags & ~(TK_CONFIG_USER_BIT - 1);
    hateFlags = 0;
    
    /*
     * If information is only wanted for a single configuration
     * spec, then handle that one spec specially.
     */

    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
    if (argvName != NULL) {
	specPtr = FindConfigSpec(interp, specs, argvName, needFlags,
		hateFlags);
	if (specPtr == NULL) {
	    return TCL_ERROR;
	}
	interp->result = FormatConfigInfo(specPtr, widgRec);
	interp->freeProc = TCL_DYNAMIC;
	return TCL_OK;
    }

    /*
     * Loop through all the specs, creating a big list with all
     * their information.
     */

    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	if ((argvName != NULL) && (specPtr->argvName != argvName)) {
	    continue;
	}
	if (((specPtr->specFlags & needFlags) != needFlags)
		|| (specPtr->specFlags & hateFlags)) {
	    continue;
	}
	if (specPtr->argvName == NULL) {
	    continue;
	}
	list = FormatConfigInfo(specPtr, widgRec);
	Tcl_AppendResult(interp, leader, list, "}", (char *) NULL);
	ckfree(list);
	leader = " {";
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * FormatConfigInfo --
 *
 *	Create a valid Tcl list holding the configuration information
 *	for a single configuration option.
 *
 * Results:
 *	A Tcl list, dynamically allocated.  The caller is expected to
 *	arrange for this list to be freed eventually.
 *
 * Side effects:
 *	Memory is allocated.
 *
 *--------------------------------------------------------------
 */

static char *
FormatConfigInfo(Tk_ConfigSpec *specPtr, char *widgRec)
{
    char *argv[6], *ptr, *result;
    char buffer[200];
    Tcl_FreeProc *freeProc = (Tcl_FreeProc *) NULL;

    argv[0] = specPtr->argvName;
    argv[1] = specPtr->dbName;
    argv[2] = specPtr->dbClass;
    argv[3] = specPtr->defValue;
    if (specPtr->type == TK_CONFIG_SYNONYM) {
	return Tcl_Merge(2, argv);
    }
    ptr = widgRec + specPtr->offset;
    argv[4] = "";
    switch (specPtr->type) {
	case TK_CONFIG_BOOLEAN:
	    if (*((int *) ptr) == 0) {
		argv[4] = "0";
	    } else {
		argv[4] = "1";
	    }
	    break;
	case TK_CONFIG_INT:
	    sprintf(buffer, "%d", *((int *) ptr));
	    argv[4] = buffer;
	    break;
	case TK_CONFIG_DOUBLE:
	    sprintf(buffer, "%g", *((double *) ptr));
	    argv[4] = buffer;
	    break;
	case TK_CONFIG_STRING:
	    argv[4] = (*(char **) ptr);
	    break;

	case TK_CONFIG_CUSTOM:
	    argv[4] = (*specPtr->customPtr->printProc)(
		    specPtr->customPtr->clientData, NULL, widgRec,
		    specPtr->offset, &freeProc);
	    break;
	default: 
	    argv[4] = "?? unknown type ??";
    }
    if (argv[1] == NULL) {
	argv[1] = "";
    }
    if (argv[2] == NULL) {
	argv[2] = "";
    }
    if (argv[3] == NULL) {
	argv[3] = "";
    }
    if (argv[4] == NULL) {
	argv[4] = "";
    }
    result = Tcl_Merge(5, argv);
    if (freeProc != NULL) {
	if (freeProc == (Tcl_FreeProc *) free) {
	    ckfree(argv[4]);
	} else {
	    (*freeProc)(argv[4]);
	}
    }
    return result;
}

// -------------------------

int
Objectify_DeleteObject(Tcl_Interp* interp, void* ob)
{
    char* name = Objectify_LookupPtr(ob);
    if (!name) {
	Tcl_AppendResult(interp, "name not found for object", 0);
	return (TCL_ERROR);
    }
    char* buf = new char[strlen(name) + 16];
    sprintf(buf, "%s delete", name);
    return (Tcl_Eval(interp, buf));
}

int
Objectify_PointerParse(ClientData cld, Tcl_Interp* interp, Tk_Window win,
		       char* value, char* record, int offset)
{
    void** c = (void **) (record + offset);
    
    if (!strcmp(value, "NULL")) {
	*c = NULL;
	return (TCL_OK);
    }
    
    void* v = Objectify_LookupName(value);
    if (!v) {
	Tcl_AppendResult(interp, "can't find ", (char *) cld,
			 " value for \"", value, "\"", 0);
	return (TCL_ERROR);
    }
    *c = v;
    return (TCL_OK);
}

char*
Objectify_PointerPrint(ClientData cld, Tk_Window win, char* record, int offset,
		       Tcl_FreeProc **freep)
{
    void* v = * (void **) (record + offset);
    if (!v)
	return ("NULL");
    char* s = Objectify_LookupPtr(v);
    if (!s)
	return ("unknown object");
    else
	return (s);
}
