/*
 *-----------------------------------------------------------------------------
 *  Copyright (c) 1993-1995 European Synchrotron Radiation Facility
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the "Terms and Conditions of Distribution", given in the documentation
 *  to this software, are applicable and the above copyright message appears
 *  in each copy.
 *
 *-----------------------------------------------------------------------------
 *
 *
 *  CTAXT:      Combine Tcl/Tk with arbitrary X Toolkits 
 *              (into a single application)
 *
 *  Module:	ctaxt.c
 *
 *  Purpose:    All functions to export to applications
 *
 *  20.10.1993
 * -25.10.1993	ds/hp	first version
 *  26.10.1993  hp      extending to TclX
 *  03.11.1993  hp      since Tk considers its main-window to be the
 *                      application-main-window we have to compute the absolute
 *                      x- and y-coordinates of the window
 *  04.11.1993  hp      Positioning and Resizing is now done by the Generic-
 *                      Event-Handler, so there's no more need for explicit
 *                      Call-Back-Functions for Resizing and therefore no more
 *                      need for using the Motif-DrawingArea as the placeholder
 *                      for the Tk-Main-Window.
 *                      Functions for Explicit-Event-Handling
 *  08.11.1993  hp      Hiding the Tcl-related stuff
 *                      Preparing the Interface for handling multiple Tk-Main-
 *                      Windows and multiple Tcl-Interpreters
 *  09.11.1993  hp      Improving the command registering
 *  10.11.1993  hp      To process the first incoming X-Events (mainly exposure
 *                      events for windows created by the application)
 *                      install a slower first version of CTAXT_EventHandler()
 *                      before creating Tk-Main-Windows
 *  15.11.1993  hp      Mechanism for adding and freeing extensions to a Tcl 
 *                      interpreter
 *  29.11.1993  hp      changed CTAXT_PutTclTkInWindow() to work with Tcl7.3 
 *                      and Tk3.6
 *  09.12.1993  hp      pass an additional argument to CTAXT_PutTclTkInWindow()
 *                      to name the created interpreter. This name is needed
 *                      for command sending.
 *  10.12.1993  hp      changes to keep track of the focus in 
 *                      CTAXT_PutTclTkInWindow(). Register an event handler
 *                      for each Tk-Main-Window which masks for EnterNotify
 *                      and LeaveNotify.
 *                      changed CTAXT_MainLoop() to process the last events
 *                      for an application after it has deleted all its Tk-
 *                      Main-Windows.
 *  21.12.1993  hp      Changed the result protocol; functions now return int
 *                      values CTAXT_OK, CTAXT_WARNING, CTAXT_ERROR or 
 *                      CTAXT_FATAL_ERROR and leave the error message in a
 *                      dynamic Tcl string maintained by the module.
 *                      Wrote functions to get and check software versions
 *  14.01.1994  hp      Fixed some bugs of incorrect casts like
 *                        (Tk_EventProc) proc --> (Tk_EventProc *) proc
 *  19.01.1994  hp      Defined own realloc, the C library's doesn't seem to 
 *                      work properly on all systems
 *
 */




#include "ctaxtInt.h"

#ifdef CTAXTX
#include <tclExtend.h>
#endif




/*
 *  In the TclX distribution there is a reference to "tcl_RcFileName" in the
 *  module, where TclX_Init() is declared. The variable itself is declared in
 *  "tkXshell.c" as a pointer to character and initialized to NULL. If we don't
 *  declare the variable in our modules, the linker will bring in "tkXshell.o".
 *  Within that module, there is a reference to Tcl_AppInit(), which is not
 *  part of the tkx- nor the tclx-library. There are two ways to avoid an
 *  "unsatisfied symbol linker error": one is to declare a dummy Tcl_AppInit()
 *  procedure, the other to declare the "tcl_RcFileName" variable. We chose the
 *  second way.
 */

#ifdef CTAXTX
char *tcl_RcFileName = NULL;
#endif




/*
 *  Protoypes for Forward-References
 */

Tk_Window
CTAXT_CreateMainWindow _ANSI_ARGS_ ((Tcl_Interp *interp,
				     Display *display, 
				     Window window, 
				     char *baseName,
				     char *className));




/*  Declaration of the Pointer to the Generic-Event-Handler-Functions  */

extern Tk_GenericProc CTAXT_FirstEventHandler;
extern Tk_GenericProc CTAXT_EventHandler;
extern Tk_EventProc CTAXT_KeepTrackOfFocus;




/*
 *  Declaration of module global Variables
 */

static int CTAXT_IsInit = 0;       /*  Needed so that the generic event handler
				       is just installed once. This variable is
				       set to one as soon as the handler is 
				       installed  */
static int CTAXT_FirstHandlerInstalled = 0;
static int CTAXT_TclExtended = 0;
static CTAXT_GlobalData CTAXT_Global;
                                   /*  All information about CTAXT-Main-Windows
				       is stored in this variable  */

/*  Declare the global array for registered application-commands  */

static int CTAXT_AppCmds = 0;
static CTAXT_ApplicationCmds *CTAXT_AppCmd = NULL;

/*  Dynamic Tcl string for error messages  */

static char staticBuf[100];
static int CTAXT_ErrorInit = 0;
static Tcl_DString CTAXT_ErrorMsg;




/*
 *  void *CTAXT_realloc (void *ptr,unsigned size);
 *
 *  Allocates size bytes memory, copies the contents of the memory at ptr to
 *  the new allocated block, frees the old memory and returns the pointer to
 *  the new allocated. If called with NULL for ptr, it allocates size bytes
 *  via malloc() and returns the address. If called with ptr inequal NULL and
 *  size equal zero, it frees the memory to which ptr points and returns NULL.
 */

void *
CTAXT_realloc (ptr,size)
void     *ptr;
size_t   size;
{
  void  *p;


  if (ptr == NULL) {
    if (size == 0) {
      return (ptr);
    }
    return (malloc (size));
  }
  if (size == 0) {
    free (ptr);
    return (NULL);
  }
  if ((p = malloc (size)) == NULL) {
    return (NULL);
  }
  memcpy (p,ptr,size);
  free (ptr);
  return (p);
}


/*
 *  int CTAXT_AddApplicationCommand (char              *name,
 *                                   Tcl_CmdProc       *cmdProc,
 *                                   ClientData        clientData,
 *                                   Tcl_CmdDeleteProc *delProc,
 *                                   Window            window,
 *                                   Display           *display);
 *
 *  This function adds an application-specific-command to the module-global
 *  array CTAXT_AppCmd. This command will be created in the Tcl-Interpreter
 *  associated with window and display when CTAXT_PutTclTkInWindow() is called.
 *
 *  For parameter description see the TCL-Draft Part III Chapter 29. If 
 *  clientData is NULL, CTAXT will pass the Tk window token of the window in
 *  which the command is created instead (useful for new Tk-widgets)
 *
 *  I   window      Window to register the command to
 *  I   display     Display where window is located
 *  O   Proper CTAXT result
 */

int 
CTAXT_AddApplicationCommand (name,cmdProc,clientData,delProc,window,display)
char              *name;
Tcl_CmdProc       *cmdProc;
ClientData        clientData;
Tcl_CmdDeleteProc *delProc;
Window            window;
Display           *display;
{
  static cache = -1;
  CTAXT_ApplicationCmds *p1;
  CTAXT_ApplicationCmd *p2;
  int i,found;


  /*  
   *  initialize the dynamic Tcl string for holding the error messages, if 
   *  not already done
   */

  if (! CTAXT_ErrorInit) {
    CTAXT_ErrorInit = 1;
    Tcl_DStringInit (&CTAXT_ErrorMsg);
  }

  /*  check if there are already commands registered for this window  */

  found = -1;
  if (cache != -1) {
    if ((CTAXT_AppCmd[cache].window == window) &&
	(CTAXT_AppCmd[cache].display == display)) {
      found = cache;
    }
  }
  if (found == -1) {
    for (i = 0;i < CTAXT_AppCmds;i++) {
      if ((CTAXT_AppCmd[i].window == window) &&
	  (CTAXT_AppCmd[i].display == display)) {
	found = i;
	cache = found;
	break;
      }
    }
  }
  if (found == -1) {

    /*  no commands registered to this window, create a new list  */

    p1 = CTAXT_AppCmd;
    if ((p1 = realloc (p1,
		       (CTAXT_AppCmds+1)*
		       sizeof(CTAXT_ApplicationCmds))) == NULL) {
      Tcl_DStringFree (&CTAXT_ErrorMsg);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 "Could not add function ",
			 -1);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 name,
			 -1);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 " in CTAXT_AddApplicationCommand(). Out of memory!",
			 -1);
      return (CTAXT_ERROR);
    }
    p1[CTAXT_AppCmds].window = window;
    p1[CTAXT_AppCmds].display = display;
    p1[CTAXT_AppCmds].cmdList = NULL;
    p1[CTAXT_AppCmds].commands = 0;
    CTAXT_AppCmd = p1;
    found = CTAXT_AppCmds;
    cache = found;
    CTAXT_AppCmds++;
  }
  p2 = CTAXT_AppCmd[found].cmdList;
  if ((p2 = realloc (p2,
		     (CTAXT_AppCmd[found].commands+1)*
		     sizeof(CTAXT_ApplicationCmd))) == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Could not add function ",
		       -1);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       name,
		       -1);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       " in CTAXT_AddApplicationCommand(). Out of memory!",
		       -1);
    return (CTAXT_ERROR);
  }
  p2[CTAXT_AppCmd[found].commands].name = name;
  p2[CTAXT_AppCmd[found].commands].cmdProc = cmdProc;
  p2[CTAXT_AppCmd[found].commands].clientData = clientData;
  p2[CTAXT_AppCmd[found].commands].delProc = delProc;
  CTAXT_AppCmd[found].cmdList = p2;
  (CTAXT_AppCmd[found].commands)++;
  return (CTAXT_OK);
}


/*
 *  int CTAXT_AddTclExtension 
 *        (CTAXT_TclExtProc *tclExtension,
 *         CTAXT_FreeTclExtProc *freeTclExtension,
 *         ClientData clientData);
 *
 *  Adds a function to the interface which is called whenever a Tcl interpreter
 *  is created and a function which is called whenever a Tcl interpreter will
 *  be deleted.
 *  
 *  I   tclExtension      Pointer to a function which will add new commands to
 *                        Tcl
 *  I   freeTclExtension  Pointer to a function which will free up memory
 *                        allocated for the Tcl extension
 *  I   clientData        Argument to pass to the functions. This can either
 *                        be an integer or a pointer.
 *  O   Proper CTAXT result
 */

int
CTAXT_AddTclExtension (tclExtension,freeTclExtension,clientData)
CTAXT_TclExtProc *tclExtension;
CTAXT_FreeTclExtProc *freeTclExtension;
ClientData clientData;
{
  CTAXT_TclExtension *p;


  /*  
   *  initialize the dynamic Tcl string for holding the error messages, if 
   *  not already done
   */

  if (! CTAXT_ErrorInit) {
    CTAXT_ErrorInit = 1;
    Tcl_DStringInit (&CTAXT_ErrorMsg);
  }

  if (CTAXT_IsInit) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "CTAXT_AddTclExtension() has to be called before any "
		       "call to CTAXT_PutTclTkInWindow() or "
		       "CTAXT_RegisterExplicitEventHandler(). Ignoring the "
		       "extension(s)!",
		       -1);
    return (CTAXT_ERROR);
  }
  if (! CTAXT_TclExtended) {
    CTAXT_TclExtended = 1;
    CTAXT_Global.extensions = 0;
    CTAXT_Global.extension = NULL;
  }
  if ((p = realloc (CTAXT_Global.extension,
		    CTAXT_Global.extensions+1*
		    sizeof(CTAXT_TclExtension))) == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Could not add Tcl-Extension in "
		       "CTAXT_AddTclExtension(). Out of memory!",
		       -1);
    return (CTAXT_ERROR);
  }
  CTAXT_Global.extension = p;
  CTAXT_Global.extension[CTAXT_Global.extensions].tclExtension = tclExtension;
  CTAXT_Global.extension[CTAXT_Global.extensions].freeTclExtension = 
                                                  freeTclExtension;
  CTAXT_Global.extension[CTAXT_Global.extensions].clientData = clientData;
  CTAXT_Global.extensions++;
  return (CTAXT_OK);
}


/*
 *  int CTAXT_RegisterExplicitEventHandler 
 *        (CTAXT_ExplEventProc *handleEventExplicit);
 *
 *  This function should be called to inform the CTAXT-Interface about the
 *  existence of an explicit event handler. After the handler is registered
 *  it is called with a pointer to the x-event-structure whenever an event for
 *  the application is processed. You have register an explicit event handler
 *  to cause your application to respond to X-Events (one of the first ones
 *  is the Expose-Event, without responding to this event your appl. even will
 *  not appear on the screen!). If you don't need to handle special events
 *  in a different way simply call your favorite event dispatcher within that
 *  function and return.
 *
 *  I   handleEventExplicit    Pointer to the function which should from now
 *                             on be called for every event
 *  O   Proper CTAXT result
 */

int
CTAXT_RegisterExplicitEventHandler (handleEventExplicit)
CTAXT_ExplEventProc *handleEventExplicit;
{
  /*  
   *  initialize the dynamic Tcl string for holding the error messages, if 
   *  not already done
   */

  if (! CTAXT_ErrorInit) {
    CTAXT_ErrorInit = 1;
    Tcl_DStringInit (&CTAXT_ErrorMsg);
  }

  if (! CTAXT_IsInit) {
    CTAXT_IsInit = 1;
    CTAXT_Global.winList = NULL;
    CTAXT_Global.windows = 0;
    CTAXT_Global.cache = 0;
    CTAXT_Global.handleEventExplicit = NULL;
    if (! CTAXT_TclExtended) {
      CTAXT_Global.extension = NULL;
      CTAXT_Global.extensions = 0;
    }
  }
  if (CTAXT_Global.handleEventExplicit != NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Overwriting previous installed explicit event handler "
		       "in CTAXT_RegisterExplicitEventHandler()!",
		       -1);
    return (CTAXT_WARNING);
  }
  CTAXT_Global.handleEventExplicit = handleEventExplicit;
  return (CTAXT_OK);
}


/*
 *  int CTAXTPutTclTkInWindow (Window mainWin,Window win,Display *disp,
 *                             char *interpName,char *script,
 *                             int argc,char *argv[]);
 *
 *  This function creates a Tcl-Interpreter, creates a Tk-MainWindow and maps 
 *  it to the already existing X-Window (winid, dispid) and evaluates the 
 *  Tcl-Script-File given by script.
 *
 *  I    mainWin      X-Window-ID of the application-main- or toplevel-window
 *  I    win          X-Window-ID of the window to be used by Tcl/Tk
 *  I    disp         X-Display-Pointer to the display, where winid is
 *                    positioned
 *  I    interpName   Name under which the interpreter should be known for
 *                    command sending
 *  I    script       Pointer to the File-Name of the Script to execute
 *  I    argc,argv    Arguments to create in the Tcl-Interpreter
 *  O    Proper CTAXT Result
 *       If an error occured before any Tcl script is evaluated, no Tk-Window 
 *       is created. Otherwise if the error occured during script-evaluation, 
 *       the Tk-Window is created.
 */

int
CTAXT_PutTclTkInWindow (mainWin,win,disp,interpName,script,argc,argv)
  Window  mainWin;
  Window  win;
  Display *disp;
  char    *interpName;
  char    *script;
  int     argc;
  char    *argv[];
{
  CTAXT_MainWindow      *w;
  Tcl_Interp            *interp;
  Tk_Window             tkwin;
  Tcl_DString           libDir;
  Window                root,mainWinFrame,*children;
  char                  *args,buf[20],*value;
  int                   i,nchildren,found,sameMainWin;


#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nCTAXT_PutTclTkInWindow() called for X-Window 0x%lX, "
	            "child of 0x%lX on Display %p",win,mainWin,disp);
#endif

  /*  
   *  initialize the dynamic Tcl string for holding the error messages, if 
   *  not already done
   */

  if (! CTAXT_ErrorInit) {
    CTAXT_ErrorInit = 1;
    Tcl_DStringInit (&CTAXT_ErrorMsg);
  }

  if (! CTAXT_IsInit) {
    CTAXT_IsInit = 1;
    CTAXT_Global.winList = NULL;
    CTAXT_Global.windows = 0;
    CTAXT_Global.cache = 0;
    CTAXT_Global.handleEventExplicit = NULL;
    if (! CTAXT_TclExtended) {
      CTAXT_Global.extension = NULL;
      CTAXT_Global.extensions = 0;
    }
  }

  /*  check the window and display ids, don't accept NULL, 
      this will crash X  */

  if ((mainWin == 0L) || (win == 0L) || (disp == NULL)) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Trying to map illegal window id ",
		       -1);
    sprintf (staticBuf,"0x%lX",win);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       staticBuf,
		       -1);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       ", child of ",
		       -1);
    sprintf (staticBuf,"0x%lX",mainWin);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       staticBuf,
		       -1);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       ", on display ",
		       -1);
    sprintf (staticBuf,"0x%p",disp);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       staticBuf,
		       -1);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       " in CTAXT_PutTclTkInWindow()!",
		       -1);
    return (CTAXT_ERROR);
  }

  /*  check, if window is already mapped to a Tk-Window  */

  for (i = 0;i < CTAXT_Global.windows;i++) {
    if ((CTAXT_Global.winList[i].window == win) && 
	(CTAXT_Global.winList[i].display == disp)) {
      Tcl_DStringFree (&CTAXT_ErrorMsg);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 "Trying to map already mapped Tk window ",
			 -1);
      sprintf (staticBuf,"0x%lX",win);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 staticBuf,
			 -1);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 " on display ",
			 -1);
      sprintf (staticBuf,"0x%p",disp);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 staticBuf,
			 -1);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 " in CTAXT_PutTclTkInWindow()!",
			 -1);
      return (CTAXT_WARNING);
    }
  }

  /*
   *  Create a new Tcl-Interpreter
   */

  interp = Tcl_CreateInterp();
  if (interp == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Could not create a Tcl interpreter in "
		       "CTAXT_PutTclTkInWindow()!",
		       -1);
    return (CTAXT_ERROR);
  }
#if CTAXT_DEBUG_LEVEL > 1
  fprintf (stderr,"\nCreated Tcl-Interpreter 0x%p",interp);
#endif


  if (! CTAXT_FirstHandlerInstalled) {

    /*  
     *  Create a first Event-Handler to process XEvents for the Application
     *  This Handler has to be installed before(!) the Tk-Main-Window is 
     *  created, otherwise there is no chance to handle the first exposure 
     *  events for the application windows. I'm not sure where in the Tk-Stuff
     *  a Tk_HandleEvent() is invoked during Main-Window-Creation but it seems
     *  that it gets invoked somewhere so this is the safer way!
     */

#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nInstalling first generic event handler");
#endif
#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nBefore Tk_CreateGenericHandler()\n");
#endif
    Tk_CreateGenericHandler ((Tk_GenericProc *) CTAXT_FirstEventHandler,
			     (ClientData) &CTAXT_Global);
#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nAfter Tk_CreateGenericHandler()\n");
#endif
    CTAXT_FirstHandlerInstalled = 1;
  }


  /*  
   *  Call CTAXT_CreateMainWindow to initialize all the Tk-Window-Structures
   *  and register the Tk-Commands to the Tcl-Interpreter
   */

  tkwin = CTAXT_CreateMainWindow (interp,disp,win,interpName,"Tk");
  if (tkwin == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       &(interp->result[1]),
		       -1);
    switch (interp->result[0]) {
      case 'W':
        return (CTAXT_WARNING);
      case 'E':
	return (CTAXT_ERROR);
      case 'F':
	return (CTAXT_FATAL_ERROR);
    }
    return (CTAXT_ERROR);
  }


  /*
   *  Install the event handler to keep track of the focus
   */

  Tk_CreateEventHandler (tkwin,
			 EnterWindowMask | LeaveWindowMask,
			 (Tk_EventProc *) CTAXT_KeepTrackOfFocus,
			 (ClientData) &CTAXT_Global);


  /*  
   *  Get the Frame of the Application-Main- or -Toplevel-Window. This infor-
   *  mation is used by the Generic-Event-Handler to mask Configure-Events for
   *  the Application-Main-Window and inform Tk about the new geometry of the 
   *  window so that the Tk-Main-Window fits the situation.
   *  Since these Configure-Events are generated not for the Window created
   *  as the AMW by the application but for the frame of the AMW (created by 
   *  the Window-Manager) we have to query X for the parent of the AMW
   */

  if (XQueryTree (disp,
		  mainWin,
		  &root,
		  &mainWinFrame,
		  &children,
		  &nchildren) == 0) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "XQueryTree() complained about window ",
		       -1);
    sprintf (staticBuf,"0x%lX",mainWin);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       staticBuf,
		       -1);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       " on display ",
		       -1);
    sprintf (staticBuf,"0x%p",disp);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       staticBuf,
		       -1);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       " in CTAXT_PutTclTkInWindow()!",
		       -1);
    return (CTAXT_ERROR);
  }
#if CTAXT_DEBUG_LEVEL > 1
    fprintf (stderr,"\nCTAXT_PutTclTkInWindow() called XQuery() for X-Window "
	            "0x%lX on Display %p"
	            "\nResult: Root is 0x%lX, Parent is 0x%lX",
	     mainWin,disp,root,mainWinFrame);
#endif


  /*
   * Make command-line arguments available in the Tcl variables "argc"
   * and "argv".  Also set the "geometry" variable from the geometry
   * specified on the command line.
   */

  args = Tcl_Merge(argc-1, argv+1);
  Tcl_SetVar(interp,"argv",args,TCL_GLOBAL_ONLY);
  ckfree(args);
  sprintf(buf, "%d", argc-1);
  Tcl_SetVar(interp,"argc",buf,TCL_GLOBAL_ONLY);
  Tcl_SetVar(interp,"argv0",script,TCL_GLOBAL_ONLY);
  Tcl_SetVar(interp,"tcl_interactive","0",TCL_GLOBAL_ONLY);


#ifdef CTAXTX

  /*  Initialize TclX and TkX  */

  if (TclX_Init (interp) == TCL_ERROR) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       interp->result,
		       -1);
    return (CTAXT_ERROR);
  }
  if (TkX_Init (interp) == TCL_ERROR) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       interp->result,
		       -1);
    return (CTAXT_ERROR);
  }
  tclSignalBackgroundError = Tk_BackgroundError;
#endif

  
#ifdef CTAXT

  /*  Initialize Tcl and Tk  */

  if (Tcl_Init (interp) == TCL_ERROR) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       interp->result,
		       -1);
    return (CTAXT_ERROR);
  }
  if (Tk_Init (interp) == TCL_ERROR) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       interp->result,
		       -1);
    return (CTAXT_ERROR);
  }
#endif


  /*
   *  Ok, time now for creating a new entry to our dynamic array containing
   *  the main-windows created by CTAXT, ordered by mainWin and display
   */

  if ((w = (CTAXT_MainWindow *) realloc (
	      CTAXT_Global.winList,
	      (CTAXT_Global.windows+1)*sizeof (CTAXT_MainWindow))) == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Out of memory in CTAXT_PutTclTkInWindow()!",
		       -1);
    return (CTAXT_ERROR);
  }
  CTAXT_Global.winList = w;

  /*  check if there is already an entry with the same mainWin and display  */

  found = CTAXT_Global.windows;
  sameMainWin = 0;
  i = 0;
  while (i < CTAXT_Global.windows) {
    if ((CTAXT_Global.winList[i].mainWin == mainWin) &&
	(CTAXT_Global.winList[i].display == disp)) {
      found = i;
      sameMainWin = CTAXT_Global.winList[found].sameMainWin;
      break;
    }
    i += CTAXT_Global.winList[i].sameMainWin;
  }
  for (i = CTAXT_Global.windows;i > found;i--) {
    memcpy (&(CTAXT_Global.winList[i]),
	    &(CTAXT_Global.winList[i-1]),
	    sizeof (CTAXT_MainWindow));
  }
  CTAXT_Global.winList[found].interp = interp;
  CTAXT_Global.winList[found].tkWin = tkwin;
  CTAXT_Global.winList[found].mainWinFrame = mainWinFrame;
  CTAXT_Global.winList[found].display = disp;
  CTAXT_Global.winList[found].mainWin = mainWin;
  CTAXT_Global.winList[found].window = win;
  CTAXT_Global.winList[found].sameMainWin = sameMainWin+1;
  (CTAXT_Global.windows)++;


  /*  Call the functions for Tcl extensions  */

  for (i = 0;i < CTAXT_Global.extensions;i++) {
    if (CTAXT_Global.extension[i].tclExtension != NULL) {
      CTAXT_Global.extension[i].tclExtension (
        interp,
	tkwin,
	CTAXT_Global.extension[i].clientData);
    }
  }


  /*  Register all application-specific commands  */

  found = -1;
  for (i = 0;i < CTAXT_AppCmds;i++) {
    if ((CTAXT_AppCmd[i].window == win) &&
	(CTAXT_AppCmd[i].display == disp)) {
      found = i;
      break;
    }
  }
  if (found != -1) {
    for (i = 0;i < CTAXT_AppCmd[found].commands;i++) {
      if (CTAXT_AppCmd[found].cmdList[i].clientData == NULL) {
	Tcl_CreateCommand (interp,
			   CTAXT_AppCmd[found].cmdList[i].name,
			   CTAXT_AppCmd[found].cmdList[i].cmdProc,
			   (ClientData) tkwin,
			   CTAXT_AppCmd[found].cmdList[i].delProc);
      } else {
	  Tcl_CreateCommand (interp,
			     CTAXT_AppCmd[found].cmdList[i].name,
			     CTAXT_AppCmd[found].cmdList[i].cmdProc,
			     CTAXT_AppCmd[found].cmdList[i].clientData,
			     CTAXT_AppCmd[found].cmdList[i].delProc);
	}
    }
  }

  if ((script != NULL) && (*script)) {

    /*  Evaluate the Tcl-Script  */

    Tcl_VarEval (interp,"source ",script,(char *) NULL);
    if(*(interp->result) != 0) {
      Tcl_DStringFree (&CTAXT_ErrorMsg);
      Tcl_DStringAppend (&CTAXT_ErrorMsg,
			 interp->result,
			 -1);
      return (CTAXT_ERROR);
    }
    Tcl_Eval (interp,"update");
  }
  if(*(interp->result) != 0) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       interp->result,
		       -1);
    return (CTAXT_ERROR);
  }

  return (CTAXT_OK);
}


/*
 *  int CTAXT_MainLoop (void);
 *
 *  This function is mostly a similar to the Tk-Function Tk_MainLoop(). In
 *  addition it first of all removes the previous installed generic event
 *  handler, installs a more complex one before entering the infinite Main-
 *  Loop and deletes it as soon as there do no (CTAXT created) Tk-Main-Windows
 *  exist anymore.
 */

int
CTAXT_MainLoop ()
{
#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nReplacing previous generic event handler in "
	          "CTAXT_MainLoop ()");
#endif

  /*  First of all delete the previous installed generic event handler  */

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nBefore Tk_DeleteGenericHandler()\n");
#endif

  Tk_DeleteGenericHandler ((Tk_GenericProc *) CTAXT_FirstEventHandler,
			   (ClientData) &CTAXT_Global);

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nAfter Tk_DeleteGenericHandler()\n");
#endif

  CTAXT_FirstHandlerInstalled = 0;


  /*  
   *  Create the Generic-Event-Handler to process XEvents for the 
   *  application and to mask certain events.
   */

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nBefore Tk_CreateGenericHandler()\n");
#endif

  Tk_CreateGenericHandler ((Tk_GenericProc *) CTAXT_EventHandler,
			   (ClientData) &CTAXT_Global);

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nAfter Tk_CreateGenericHandler()\n");
#endif


  /*  Enter the infinite Main-Loop  */

  while (CTAXT_Global.windows > 0) {
    Tk_DoOneEvent (0);
  }


  /*
   *  Delete the Generic-Event-Handler, it is no longer needed since no 
   *  Tk-Windows exist anymore. From now on the application has itself to care
   *  for event handling
   */

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nBefore Tk_DeleteGenericHandler()\n");
#endif

  Tk_DeleteGenericHandler ((Tk_GenericProc *) CTAXT_EventHandler,
			   (ClientData) &CTAXT_Global);

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nAfter Tk_DeleteGenericHandler()");
#endif


  /*
   *  To process the last calls to CTAXT_DeleteInterpreter, handle all
   *  pending events for the application. As soon as the event queue for
   *  the application is empty, all Tcl interpreters are deleted and no
   *  idle callbacks remain, exit the loop.
   */

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nProcessing the last events for the application");
#endif

  while (Tk_DoOneEvent (TK_DONT_WAIT) != 0) {
  }

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,"\nReturning from CTAXT_MainLoop()\n\n");
#endif


  /*  make the compiler happy  */

  return (CTAXT_OK);


  /*
   *  If the application still exists, it has to enter its own event loop,
   *  otherwise it will not be able to respond to user interactions.
   */
}


/*
 *  Tk_Window CTAXT_GetWindow (Window window,Display *display);
 *
 *  Returns a pointer to the Tk-Token for the window. This is needed e.g. if
 *  you want to create new widget-commands in the attached interpreter.
 *
 *  I   window    Window to register the command to
 *  I   display   Display where window is located
 *  O   A Tk-Token for the window or NULL for window not found
 */

Tk_Window
CTAXT_GetWindow (window,display)
Window            window;
Display           *display;
{
  int i;


  /*  check, if window is known as a Tk-Window  */

  for (i = 0;i < CTAXT_Global.windows;i++) {
    if ((CTAXT_Global.winList[i].window == window) && 
	(CTAXT_Global.winList[i].display == display)) {
      return (CTAXT_Global.winList[i].tkWin);
    }
  }
  return ((Tk_Window) NULL);
}


/*
 *  Tcl_Interp *CTAXT_GetInterpreter (Window window,Display *display);
 *
 *  Returns a pointer to the Tcl-Interpreter attached to the window
 *
 *  I   window    Window to register the command to
 *  I   display   Display where window is located
 *  O   Pointer to the Tcl-Interpreter or NULL for Window not found
 */

Tcl_Interp *
CTAXT_GetInterpreter (window,display)
Window            window;
Display           *display;
{
  int i;


  /*  check, if window is known as a Tk-Window  */

  for (i = 0;i < CTAXT_Global.windows;i++) {
    if ((CTAXT_Global.winList[i].window == window) && 
	(CTAXT_Global.winList[i].display == display)) {
      return (CTAXT_Global.winList[i].interp);
    }
  }
  return ((Tcl_Interp *) NULL);
}


/*
 *  char *CTAXT_GetErrorMessage (void);
 *
 *  Returns a pointer to a string containing a message which describes the last
 *  error.
 *
 *  O   Pointer to the error message
 */

char *
CTAXT_GetErrorMessage ()
{
  /*  
   *  initialize the dynamic Tcl string for holding the error messages, if 
   *  not already done
   */

  if (! CTAXT_ErrorInit) {
    CTAXT_ErrorInit = 1;
    Tcl_DStringInit (&CTAXT_ErrorMsg);
  }

  return (CTAXT_ErrorMsg.string);
}


/*
 *  int CTAXT_CheckVersions (void);
 *
 *  Checks, if the Tcl- and Tk versions compiled into the CTAXT library and
 *  into the Tcl- and Tk library are the same. Shouldn't be called before
 *  at least one Tk main window is created by a call to 
 *  CTAXT_PutTclTkInWindow().
 *
 *  O   Proper CTAXT result. If the Tcl-Version is less than 7.3 or the Tk-
 *      Version is less than 3.6 than CTAXT_FATAL_ERROR is returned.
 */

int
CTAXT_CheckVersions ()
{
  Tcl_Interp *interp;
  Tk_Window tkwin;
  char *tkVersion,*tclMajor,*tclMinor,*tkMajor,*tkMinor,result[41];
  int tclVersion;


  /*  
   *  initialize the dynamic Tcl string for holding the error messages, if 
   *  not already done
   */

  if (! CTAXT_ErrorInit) {
    CTAXT_ErrorInit = 1;
    Tcl_DStringInit (&CTAXT_ErrorMsg);
  }

  /*
   *  There should at least be one Tk main window created by 
   *  CTAXT_PutTclTkInWindow()
   */

  if ((! CTAXT_IsInit) || (CTAXT_Global.windows == 0)) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Cannot check the Tcl/Tk versions before a Tk main "
		       "window is created by a call to "
		       "CTAXT_PutTclTkInWindow() in CTAXT_CheckVersions()!",
		       -1);
    return (CTAXT_ERROR);
  }

  /*  Create a Tcl interpreter (needed for reading the Tcl version)  */

  interp = Tcl_CreateInterp();
  if (interp == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Could not create a Tcl interpreter in "
		       "CTAXT_CheckVersions()!",
		       -1);
    return (CTAXT_ERROR);
  }

  /*  get the version compiled into the Tcl library  */

  if (Tcl_GlobalEval (interp,"info tclversion") != TCL_OK) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Could not read the Tcl version compiled into the Tcl "
		       "library in CTAXT_CheckVersions()!",
		       -1);
    Tcl_DeleteInterp (interp);
    return (CTAXT_ERROR);
  }
  tclVersion = strcmp (interp->result,TCL_VERSION);
  strncpy (result,interp->result,40);
  result[40] = 0;

  /*  check if the Tcl library is too old  */

  tclMajor = strtok (result,".");
  if ((tclMajor == NULL) || (atoi (tclMajor) < TCL_MAJOR_VERSION)) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "The Tcl version compiled into the Tcl library is less"
		       " than 7.3 in CTAXT_CheckVersions()!",
		       -1);
    Tcl_DeleteInterp (interp);
    return (CTAXT_FATAL_ERROR);
  } else {
      tclMinor = strtok (NULL,".");
      if ((atoi (tclMajor) == TCL_MAJOR_VERSION) &&
	  ((tclMinor == NULL) || (atoi (tclMinor) < TCL_MINOR_VERSION))) {
	Tcl_DStringFree (&CTAXT_ErrorMsg);
	Tcl_DStringAppend (&CTAXT_ErrorMsg,
			   "The Tcl version compiled into the Tcl library is"
			   " less than 7.3 in CTAXT_CheckVersions()!",
			   -1);
	Tcl_DeleteInterp (interp);
	return (CTAXT_FATAL_ERROR);
      }
    }

  /*  Create a Tk main window (otherwise there's no tkVersion in interp)  */

  tkwin = Tk_CreateMainWindow (interp,NULL,"","Tk");
  if (tkwin == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Could not create a Tk main window in "
		       "CTAXT_CheckVersions()!",
		       -1);
    Tcl_DeleteInterp (interp);
    return (CTAXT_ERROR);
  }

  /*  get the version compiled into the Tk library  */

  if ((tkVersion = Tcl_GetVar (interp,"tkVersion",TCL_GLOBAL_ONLY)) == NULL) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "Could not read the global variable \"tkVersion\" in "
		       "CTAXT_CheckVersions()!",
		       -1);
    Tk_DestroyWindow (tkwin);
    Tcl_DeleteInterp (interp);
    return (CTAXT_ERROR);
  }
  strncpy (result,tkVersion,40);
  result[40] = 0;

  /*  check if the Tk library is too old  */

  tkMajor = strtok (result,".");
  if ((tkMajor == NULL) || (atoi (tkMajor) < TK_MAJOR_VERSION)) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "The Tk version compiled into the Tk library is less"
		       " than 3.6 in CTAXT_CheckVersions()!",
		       -1);
    Tk_DestroyWindow (tkwin);
    Tcl_DeleteInterp (interp);
    return (CTAXT_FATAL_ERROR);
  } else {
      tkMinor = strtok (NULL,".");
      if ((atoi (tkMajor) == TK_MAJOR_VERSION) &&
	  ((tkMinor == NULL) || (atoi (tkMinor) < TK_MINOR_VERSION))) {
	Tcl_DStringFree (&CTAXT_ErrorMsg);
	Tcl_DStringAppend (&CTAXT_ErrorMsg,
			   "The Tk version compiled into the Tk library is"
			   " less than 3.6 in CTAXT_CheckVersions()!",
			   -1);
        Tk_DestroyWindow (tkwin);
	Tcl_DeleteInterp (interp);
	return (CTAXT_FATAL_ERROR);
      }
    }

  /*  
   *  check if the Tcl version compiled into the CTAXT library differs from
   *  the one compiled into the Tcl library
   */

  if (tclVersion != 0) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "The Tcl version compiled into the CTAXT library "
		       "differs from the one compiled into the Tcl "
		       "library in CTAXT_CheckVersions()!",
		       -1);
    Tk_DestroyWindow (tkwin);
    Tcl_DeleteInterp (interp);
    return (CTAXT_WARNING);
  }

  /*  
   *  check if the Tk version compiled into the CTAXT library differs from
   *  the one compiled into the Tk library
   */

  if (strcmp (tkVersion,TK_VERSION) != 0) {
    Tcl_DStringFree (&CTAXT_ErrorMsg);
    Tcl_DStringAppend (&CTAXT_ErrorMsg,
		       "The Tk version compiled into the CTAXT library "
		       "differs from the one compiled into the Tk "
		       "library in CTAXT_CheckVersions()!",
		       -1);
    Tk_DestroyWindow (tkwin);
    Tcl_DeleteInterp (interp);
    return (CTAXT_WARNING);
  }

  Tk_DestroyWindow (tkwin);
  Tcl_DeleteInterp (interp);
  return (CTAXT_OK);
}


/*
 *  char *CTAXT_GetVersion (void);
 *
 *  Returns a pointer to a string containing the version number of the CTAXT
 *  library.
 *
 *  O   Pointer to the version number
 */

char *
CTAXT_GetVersion ()
{
  return (CTAXT_VERSION);
}
