# sedit
#
# A simple editor for composing mail messages.
# See also the Text and Entry bindings in seditBind.tcl
#
# Copyright (c) 1993 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

proc SeditHelp {} {
    if [Exwin_Toplevel .sedithelp "Sedit help" Help] {
	set t .sedithelp
	Widget_Message $t msg -text \
"
Sedit is a simple editor built into exmh.
The editor works on draft messages that are kept in your
MH draft folder.  When you do Compose, Reply, or Forward,
a new draft is created for you.

Here is what the buttons do:

Abort - remove the draft message from the drafts folder.
    The editor window is removed.
Save&Quit - saves the message in the draft folder but does not send it.
Help - displays this window.
Insert @ - Insert a copy of the current message (replies only)
Send - saves and posts the message.  By default it is removed from
    the draft folder when you do this, unless you select Keep on Send.
Sign - append your .signature file to the message.
    If you have autoSign set, or multiple files match .signature*
    then this button is replaced with the following menu.
Sign... - Select the signature file for the message.  It is added
    when you select it, or when you Send the message if you
    have the autoSign Preferences item enabled.

More... - Bring up a menu with the following options:
Keep on Send - if this button is highlighted, then Send will not
    remove the editor window and the draft will remain in the drafts
    folder after you send it.
Format mail - this breaks long lines (> 79 characters) at word boundaries.
    Note that the auto-wrapping done by the TK text widget does not insert
    newline characters.  This will be done by the editor when you save
    the message and the Format mail option is selected.
Attempt mhn - run the message body through mhn before sending.  If you know
    the # directives of mhn, you can use this quick hack to compose MIME
    messages.  A GUI alternative will appear someday.
Quoted Printable Text - make the content transfer encoding of regular
	text messages quoted printable to protect 8-bit characters.
Insert File... - Insert a file directly into the message.
Insert Part... - Insert a file as a part into a multipart message.
Whom - displays the list of recipients for the message.
Spell... - run UNIX spell over the message body.
Find SEL - search the message body for the selected string.
Save Buffer - write out the editor buffer to disk.
	This is only for paranoid folks.
	The Save&Quit button or Keep on Send options actually result in
	the draft remaining.  This Save button just writes the buffer to
	disk but does not change the default behavior of renaming drafts
	after you send them.
Mime Preview - Display the message as it will be seen by an exmh user.
"
    }
}
proc SeditId { draft } {
    global mhProfile
    if [regsub ^$mhProfile(path)/$mhProfile(draft-folder)/ $draft {} newdraft] {
	return $newdraft
    } else {
	set newdraft $draft	;# TCL 7.0 bug
	regsub ^$mhProfile(path)/ $draft {} newdraft
	regsub -all {\.} $newdraft _ newdraft
	return $newdraft
    }
}
proc SeditSigfileDefault {} {
    global sedit
    if {[string length $sedit(sigfileDefault)] == 0} {
	set sedit(sigfileDefault) ~/.signature
    }
    if ![regexp {^[/~]} $sedit(sigfileDefault)] {
	set sedit(sigfileDefault) ~/$sedit(sigfileDefault)
    }
    return [glob -nocomplain $sedit(sigfileDefault)]
}
proc Sedit_Start { draft } {
    global sedit
    if ![info exists sedit(init)] {
	Sedit_Init
    }
    set id [SeditId $draft]
    set b .sedit${id}.but
    if {[Exwin_Toplevel .sedit$id $draft Sedit] == 0} {
	# Reuse existing window
	set t $sedit($id,text)
	SeditMsg $t $draft
	$t delete 1.0 end
	.sedit$id.but.send config -state normal
	set sedit($t,sigfile) [SeditSigfileDefault]
    } else {
	wm iconname .sedit$id draft/$id
	set f [Widget_Frame .sedit$id f Frame {top expand fill}]
	set t [Widget_Text $f $sedit(height) -cursor xterm -setgrid true]
	set sedit($t,status) [Widget_Entry .sedit${id} status {top fill} -relief raised]

	# Nuke the Dismiss button because the Abort, Send, and Save&Quit
	# buttons pretty much cover the gamut
	set cmd [option get .sedit$id.but.quit command {}]
	if {[string length $cmd] == 0} {
	    set cmd [list SeditQuit $draft $t]
	} else {
	    set cmd [eval list $cmd]
	}
	destroy $b.quit
	wm protocol .sedit$id WM_DELETE_WINDOW $cmd

	# Send has command defined by app-defaults, but we
	# need to fix it up with an eval here
	Widget_AddButDef $b send
	Widget_ReEvalCmd $b.send	;# expand variables now

	if [catch {glob ~/.signature*} sigFiles1] {
	    set sigFiles1 [glob ~]/.signature
	}
	set sigFiles {}
	foreach sig $sigFiles1 {
	    if {! [string match *~ $sig]} {
		lappend sigFiles $sig
	    }
	}
	set sedit($t,sigfile) [SeditSigfileDefault]
	set sigFiles [lsort $sigFiles]
	if {([llength $sigFiles] <= 1) && !$sedit(autoSign)} {
	    Widget_AddButDef $b sign
	    Widget_ReEvalCmd $b.sign
	    # Fix up third argument to SeditSign
	    if {[string length $sigFiles] != 0} {
		set cmd [lindex [$b.sign config -command] 4]
		lappend cmd $sigFiles
		$b.sign config -command $cmd
	    }
	} else {
	    set menu [Widget_AddMenuBDef $b sign {right padx 1}]
	    set cmd [option get $b.sign command {}]
	    set txt [option get $b.sign text {}]
	    if ![string match *... $txt] {
		$b.sign config -text $txt...
	    }
	    if {$sedit(autoSign)} {
		Widget_RadioMenuItem $menu "(none)" { } sedit($t,sigfile) -value {}
		$menu add separator
		set i 1
	    } else {
		set i -1
	    }
	    foreach file $sigFiles {
		if {$sedit(autoSign)} {
		    incr i
		    Widget_RadioMenuItem $menu [file tail $file] { } sedit($t,sigfile) -value $file
#		    if {[string compare [file tail $file] [file tail $sedit(sigfileDefault)]] == 0} {
#			$menu invoke $i
#		    }
		} else {
		    # The eval-hairyness causes the variable references
		    # in $cmd to be expanded at this point.
			Widget_AddMenuItem $menu [file tail $file] \
			[eval list $cmd $file]
		}
	    }
	}
	foreach but [concat [option get $b buttonlist {}] \
		      [option get $b ubuttonlist {}]] {
	    if [regexp (abort|save) $but] {
		Widget_AddButDef $b $but {left padx 5}
	    } else {
		Widget_AddButDef $b $but {right padx 1}
	    }
	    Widget_ReEvalCmd $b.$but	;# expand variables now
	}

	foreach M [concat [option get .sedit$id.but menulist {}] \
			  [option get .sedit$id.but umenulist {}]] {
	    set menu [Widget_AddMenuBDef .sedit$id.but $M {right padx 1}]
	    #
	    # Here is another way to set context for the menu commands.
	    # Jam the draft and text widget name into a global that
	    # can be accessed by the menu-invoked commands.
	    #
	    $menu config -postcommand [list SeditSetContext $draft $t]
	    ButtonMenuInner $menu
	}

	SeditMsg $t $draft

	# Define a bunch of maps among things
	set sedit($t,toplevel) .sedit$id
	set sedit($id,text) $t
	set sedit($t,id) $id
	lappend sedit(allids) .sedit$id
	set sedit(.sedit$id,draft) $draft
	set sedit(.sedit$id,id) $id
    }
    Exwin_ToplevelFocus .sedit$id $t

    SeditTextBindings $draft $t		;# set up sendMsg binding
    if [file readable "@"] {
	$b.repl configure -state normal -command \
		[list SeditInsertFile $draft $t "@"]
    } else {
	$b.repl configure -state disabled
    }
    set sedit($t,keep) $sedit(keepDefault)
    set sedit($t,format) $sedit(formatDefault)
    set sedit($t,mhn) $sedit(mhnDefault)
    switch -- $sedit(quoteDefault) {
	always	{ set sedit($t,quote) 1 }
	never	{ set sedit($t,quote) 0 }
	default { set sedit($t,quote) -1 }
    }
    set sedit($t,8bit) 0
    set sedit($t,sent) 0
    set sedit($t,dirty) 0
    set sedit($t,encoding) {}
    set sedit(t) $t	;# ugly state hack
    global exmh
    if {! [info exists exmh($id,action)]} {
	# If someone cares to figure out how this happens, that would be nice.
	# It might happen after a send error.
	Exmh_Debug "Set action for $id"
	set exmh($id,action) {}
    }
    SeditMimeReset $t
    if [catch {open $draft r} in] {
	$t insert 1.0 "Cannot open $draft"
    } else {
	$t insert 1.0 [read $in]
	close $in
	SeditPositionCursor $t
    }
    focus $t
    SeditMimeParse $t
    if {$sedit(iso)} {
	SeditInitMimeType $draft $t
    }
    foreach cmd [info commands Hook_SeditInit*] {
	if [catch {$cmd $draft $t} err] {
	    SeditMsg $t "$cmd $err"
	}
    }
}
proc SeditSetContext { draft t } {
    # Called when menus are posted to set the context for some commands
    global sedit
    set sedit(draft) $draft
    set sedit(t) $t
    Exmh_Status "Sedit $t [file tail $draft]"
}
proc SeditPositionCursor { t } {
    global sedit
    # Position cursor when the draft is first open.
    # Either on the first blank header line, or the first line of the message.
    # Body tag is assigned to the body and is used later when/if
    # composing MIME multipart messages.
    set l 1
    set insert 0	;# insert mark set
    set header 0	;# header insert mark set (new headers go here)
    set hlimit 0	;# header limit mark set (cannto do richtext here)
    set sedit($t,dash) 0
    for {set l 1} {1} {incr l} {
	if {[$t compare $l.0 > end]} {
	    if {! $insert} {
		$t mark set insert end
	    }
	    if {! $header} {
		$t mark set hlimit $l.end
		incr l -1
		$t mark set header $l.end
	    }
	    $t tag add Body "header +1c" end
	    return
	}
	set line [$t get $l.0 $l.end]
	if [regexp {^[^ X].*: *$} $line] {
	    if {! $insert} {
		$t mark set insert $l.end
		set insert 1
	    }
	}
	if {[regexp {^--} $line]} {
	    set sedit($t,dash) 1
	    set line {}
	}
	if {[string length $line] == 0} {
	    # hlimit is used for <Tab> control
	    # header is used to insert new header information
	    $t mark set hlimit $l.end
	    incr l -1
	    $t mark set header $l.end
	    if {! $insert} {
		incr l 2
		$t mark set insert $l.0
	    }
	    $t tag add Body "header +1c" end
	    return
	}
    }
}

proc SeditQuit { draft t } {
    global sedit
    if [SeditIsDirty $t] {
	catch {destroy $t.seditDirty}
	set f [frame $t.seditDirty -bd 4 -relief raised]
	Widget_Message $f msg  -aspect 1000 -text "
$draft
has not been saved or sent.
Do you want to abort (destroy) it,
send it now,
save it for later editting,
or do nothing?"
	Widget_Frame $f f Dialog
	$f.f configure -bd 10
	Widget_AddBut $f.f ok "Abort" [list SeditAbortDirect $draft $t]
	Widget_AddBut $f.f send "Send" [list SeditSend $draft $t]
	Widget_AddBut $f.f save "Save" \
		[list SeditSave $draft $t SeditNuke]
	Widget_AddBut $f.f no "Do nothing" [list destroy $f]
	Widget_PlaceDialog $t $f
    } else {
	SeditNuke $draft $t
    }
}
proc SeditAbortDirect { draft t } {
    global mhProfile
    set id [SeditId $draft]
    if [regexp -- $mhProfile(draft-folder)/\[0-9\]+$ $draft] {
	Edit_Done abort $id	;# Nuke (rm) draft message
    }
    SeditNuke $draft $t
}
proc SeditAbort { draft t } {
    global sedit
    if ![SeditIsDirty $t] {
	SeditAbortDirect $draft $t
	return
    }
    if [catch {frame $t.abort -bd 4 -relief ridge} f] {
	# dialog already up
	catch {destroy $t.abort}
	SeditAbortDirect $draft $t
	return
    }
    Widget_Message $f msg -aspect 1000 -text "
Really ABORT?
Draft will be destroyed."
    pack $f.msg -padx 10 -pady 10
    frame $f.but -bd 10 -relief flat
    pack $f.but -expand true -fill both
    set sedit($t,abort) 1
    Widget_AddBut $f.but ok "Abort" [list SeditAbortConfirm $f $t 1] left
    Widget_AddBut $f.but send "Do Nothing" [list SeditAbortConfirm $f $t 0] right
    Widget_PlaceDialog $t $f
    tkwait window $f
    if {$sedit($t,abort)} {
	SeditAbortDirect $draft $t
    }
}
proc SeditAbortConfirm { f t yes } {
    global sedit
    set sedit($t,abort) $yes
    destroy $f
}
proc SeditNuke { draft t } {
    global sedit
    SeditMarkClean $t
    catch {destroy .seditUnsent}
    catch {destroy $t.seditDirty}
    catch {destroy $sedit($t,toplevel).whom}
    catch {destroy $sedit($t,toplevel).spell}
    Exwin_Dismiss $sedit($t,toplevel)
}
proc SeditMsg { t text } {
    # Status line message output
    global sedit
    $sedit($t,status) configure -state normal
    $sedit($t,status) delete 0 end
    $sedit($t,status) insert 0 $text
    $sedit($t,status) configure -state disabled
    update idletasks
}

proc SeditSend { draft t } {
    global sedit exmh
    set id [SeditId $draft]
    Exmh_Debug SeditSend id=$id action=$exmh($id,action)
    if {$sedit(autoSign) && ($sedit($t,sigfile) != "") &&
	([string compare $exmh($id,action) "dist"] != 0)} {
	set b .sedit${id}.but
	set cmd [option get $b.sign command {}]
	if {[string length $cmd] == 0} {
	    Exmh_Debug SeditSend null cmd for $b.sign"
	    set cmd {SeditSign $draft $t}
	}
	eval $cmd $sedit($t,sigfile)
    }
    foreach cmd [info commands Hook_SeditSave*] {
	if [catch {$cmd $draft $t} err] {
	    SeditMsg $t "$cmd $err"
	}
    }
    if {$sedit($t,mhn)} {
	SeditFixupMhn $draft $t
    }
    if {$sedit(iso)} {
	SeditFixupCharset $draft $t
    }
    if [SeditSave $draft $t] {
	global env sedit
	$sedit($t,toplevel).but.send config -state disabled
	# Decide if this file needs to go through mhn
	if {$sedit($t,mhn) && ![catch {exec grep -l ^# $draft}]} {
	    set env(mhdraft) $draft
	    SeditMsg $t "Running mhn..."
	    if [catch {exec mhn $draft} err] {
		SeditMsg $t $err
		$sedit($t,toplevel).but.send config -state normal
		return
	    }
	}
	if {$sedit($t,8bit)} {
	    # Turn on automatic quoting if we've entered 8-bit characters.
	    if {$sedit($t,quote) < 0} {
		set sedit($t,quote) 1
	    }
	}
	if {$sedit($t,8bit) || ($sedit($t,quote) > 0)} {
	    # Insert content-transfer-encoding headers
	    SeditFixupEncoding $draft $t [expr ($sedit($t,quote) > 0)]
	}
	foreach cmd [info commands Hook_SeditSend*] {
	    if [catch {$cmd $draft $t} err] {
		SeditMsg $t "$cmd $err"
		$sedit($t,toplevel).but.send config -state normal
		return
	    }
	}
	# Keep on send hack
	global mhProfile
	set async $mhProfile(sendasync)
	if {$sedit($t,keep)} {
	    if {$async} {
		set mhProfile(sendasync) 0
	    }
	}
	SeditMsg $t "Sending message..."
	SeditMarkSent $t
	Edit_Done send $id
	SeditMsg $t "Message sent"
	global sedit
	if {! $sedit($t,keep)} {
	    SeditNuke $draft $t
	} else {
	    SeditSave $draft $t
	    set mhProfile(sendasync) $async
	    $sedit($t,toplevel).but.send config -state normal
	}
    }
}

proc SeditSave { draft t {hook {}} } {
    global sedit mhProfile
    if [catch {
	SeditMsg $t "Saving message..."
	set out [open $draft w]
	if {$sedit($t,format)} {
	    SeditFormatMail $t $out
	} else {
	    puts $out [$t get 1.0 end]
	}
	close $out
	SeditMsg $t "Message saved"
	if ![regexp -- $mhProfile(draft-folder)/\[0-9\]+$ $draft] {
	    # Not from the drafts folder - see if we need to update
	    # the main display.
	    Msg_Redisplay $draft
	}
	if {$hook != {}} {
	    after 1 [list $hook $draft $t]
	}
    } err] {
	global errorInfo
	error "SeditSave $draft: $err" $errorInfo
	return 0
    }
    SeditMarkClean $t
    return 1
}
proc SeditSaveBody { t outfile } {
    set out [open $outfile w]
    puts $out [$t get [$t index "header + 1 line"] end]
    close $out
}

proc SeditMarkSent { t } {
    global sedit
    set sedit($t,sent) 1
}
proc SeditNotSent { t } {
    global sedit
    return [expr {! $sedit($t,sent)}]
}

proc SeditTextBindings { draft t } {
    global sedit
    SeditBind $t sendMsg [list SeditSend $draft $t]
    SeditBind Entry sendMsg { }
}

proc Sedit_CheckPoint {} {
    global sedit
    foreach top $sedit(allids) {
	if [info exists sedit($top,id)] {
	    set draft $sedit($top,draft)
	    set id $sedit($top,id)
	    set t $sedit($id,text)
	    if [SeditIsDirty $t] {
		Exmh_Status "Saving draft $id"
		SeditSave $draft $t
	    }
	}
    }
}
proc SeditFixupMhn { draft t } {
    global sedit
    set state header
    set mhn 0
    set lines {}
    for {set i 1} {[$t compare $i.0 < end]} {incr i} {
	set line [$t get $i.0 $i.end]
	set len [string length $line]
	if {$state == "header"} {
	    if [regexp -nocase {^(content-type|mime-version|content-transfer-encoding):} $line match type] {
		lappend lines $i
	    }
	    if [regexp {^(--+.*--+)?$} $line] {
		set state body
	    }
	} else {
	    if [regexp ^# $line] {
		set mhn 1
	    }
	}
    }
    if {$mhn} {
	if [llength $lines] {
	    SeditMsg $t "Cleaning up for MHN"
	}
	foreach i [lsort -decreasing $lines] {
	    $t delete $i.0 "$i.end +1 char"
	}
	set sedit($t,8bit) 0	;# Let MHN do quote-printable
	set sedit($t,quote) 0
    }
}
