# jprefs.tcl - utilities for user preferences and configuration
#
######################################################################
# Copyright 1992-1996 by Jay Sekora.  This file may be freely        #
# distributed, modified or unmodified, for any purpose, provided     #
# that this copyright notice is retained verbatim in all copies and  #
# no attempt is made to obscure the authorship of this file.  If you #
# distribute any modified versions, I ask, but do not require, that  #
# you clearly mark any changes you make as such and that you provide #
# your users with instructions for getting the original sources.     #
######################################################################

### TO DO

######################################################################
# global variables:
#
global J_PREFS env
j:default J_PREFS(autoposition) 0
j:default J_PREFS(confirm) 1
#
######################################################################


######################################################################
# j:source_config ?options? file - read user configuration from a file
#   option is -directory
# file is assumed to be in env(HOME)/.tk unless dir is specified
# NOTE: this can also be used just to source an arbitrary Tcl file
######################################################################

proc j:source_config { args } {
  j:parse_args { {directory {} } }

  set file [lindex $args 0]
  global env

  if {$directory == {}} then {
    set directory $env(HOME)/.tk
  }

  if {[file isfile "$directory/$file"]} then {
    uplevel 1 "source $directory/$file"
  }
}

######################################################################
# j:read_prefs ?options? defaults - read preferences from file, set array
# options are:
#   -file (default defaults)
#   -directory (default ~/.tk)
#   -array (default J_PREFS)
#   -prefix (default "")
# <defaults> is a list of two-element sublists.  the first element of
#   each sublist is the name of the default (in the file and in the
#   $array array); the second is the value to use if no such default
#   exists (ie, the hardwired application default)
# If a _default_ is "tk_strictMotif", it sets the element of $array,
#   but also the global tk_strictMotif variable
# If -prefix is non-null, it (plus a comma) is prepended to each 
#   preference name, so that for instance you can set -prefix to
#   "friend" and access preferences (and array indices) like
#   "friend,name", "friend,age", etc.
######################################################################

proc j:read_prefs { args } {
  j:parse_args {
    {array J_PREFS}
    {prefix {}}
    {directory {} }
    {file defaults}
  }
  set defaults [lindex $args 0]
  
  set file $file.tcl
  
  global env tk_strictMotif $array
  
  if {"x$directory" == "x"} {
    set directory $env(HOME)/.tk	;# NOTE: created if necessary!
  }
  
  if {"x$prefix" != "x"} {		;# if prefix is non-null...
    set prefix "$prefix,"		;# ...add a comma to it
  }
  
  set [format {%s(0)} $array] 1		;# dummy to make sure it's an array

  foreach pair $defaults {
    set pref_name [lindex $pair 0]
    set hard_default [lindex $pair 1]
    
    set [format %s($prefix$pref_name) $array] $hard_default
  }
  
  j:source_config -directory $directory $file

  foreach pair $defaults {
    set pref_name [lindex $pair 0]
    set hard_default [lindex $pair 1]
    
    if {"x$pref_name" == "xtk_strictMotif"} {
      set tk_strictMotif [set [format %s($prefix$pref_name) $array]]
    }
  }
}

######################################################################
# j:read_global_prefs - read common jstools preferences from ~/.tk/defaults
######################################################################

proc j:read_global_prefs {} {
  global J_PREFS
  
  j:read_prefs {
    {language en}
    {autoposition 0}
    {bindings basic}
    {typeover 1}
    {confirm 1}
    {visiblebell 1}
    {audiblebell 1}
    {printer lp}
    {scrollbarside right}
    {j_fs_fast 0}
    {tk_strictMotif 0}
    {web_browser mosaic}
    {man_viewer builtin}
  }
}

# alias for backwards-compatibility:
proc j:read_standard_prefs {} [info body j:read_global_prefs]

######################################################################
# j:write_prefs ?options? - write X defaults to file from array
# options are:
#   -file (default defaults)
#   -directory (default ~/.tk)
#   -array (default J_PREFS)
#   -prefix (default "")
# writes all elements of array $array
# If -prefix is null, writes all elements of array $array which
#   don't have a comma in their names.
# If -prefix is non-null, writes all elements of array $array whose
#   names start with "$prefix,"
# For instance you can set -prefix to "friend" and access preferences
#   (and array indices) like "friend,name", "friend,age", etc.
######################################################################

proc j:write_prefs { args } {
  j:parse_args {
    {array J_PREFS}
    {prefix ""}
    {directory {} }
    {file defaults}
  }
  global env $array
  
  set file $file.tcl
  
  if {"x$directory" == "x"} then {
    set directory $env(HOME)/.tk	;# NOTE: created if necessary!
  }

  if {! [file isdirectory $directory]} {;# make sure directory exists
    exec mkdir -p $directory
  }
  set f [open $directory/$file {w}]
  
  if {"x$prefix" == "x"} {		;# just names with no comma
    foreach pref_name [lsort [array names $array]] {
      if {[string first , $pref_name] == -1} {
        set value [set [format {%s(%s)} $array $pref_name]]
        puts $f [list set ${array}(${pref_name}) ${value}]
      }
    }
  } else {
    foreach pref_name [lsort [array names $array]] {
      if [string match "$prefix,*" $pref_name] {
        set value [set [format {%s(%s)} $array $pref_name]]
        puts $f [list set ${array}(${pref_name}) ${value}]
      }
    }
  }
  
  close $f
  return 0
}

######################################################################
# j:pref:module - create new module, creating it if necessary
######################################################################

proc j:pref:module { module args } {
  j:parse_args {
    {label {Preferences}}
    {panel main}
    {array {J_PREFS}}
  }
  global JPREF_MODULES			;# list of defined modules
  global JPREF_INFO			;# various preference info
  upvar #0 $array global		;# this is the variable for this panel
  set global(0) 1			;# make sure it exists and is an array
  
  append JPREF_MODULES($panel) ""	;# make sure it exists
  if {[lsearch -exact $JPREF_MODULES($panel) $module] == -1} {
    lappend JPREF_MODULES($panel) $module 	;# record existence of module
  }
  set JPREF_INFO(module_name,$module) [j:ldb $label]
  set JPREF_INFO(panel,$module) $panel
  set JPREF_INFO(array,$module) $array
}

######################################################################
# j:pref:create_pref - define new preference
######################################################################

proc j:pref:create_pref { global module args } {
  j:parse_args {
    {prompt {}}
    {type string}
    {link {}}
    {text {}}
    {values {{{} nothing} {something something}}}
    {default {}}
  }
  global JPREF_MODULES			;# list of defined modules
  global JPREF_PREFS			;# list of prefs in each module
  global JPREF_INFO			;# characteristics of each pref
  
  set array $JPREF_INFO(array,$module)	;# name of global array for module
  set element [set array]($global)	;# particular element this pref sets
  upvar #0 $element variable		;# alias "variable" to real global
  j:default variable $default		;# set if undefined
  
  # append JPREF_MODULES($panel) ""		;# make sure it exists
  # if {[lsearch -exact $JPREF_MODULES($panel) $module] == -1} {
  #   lappend JPREF_MODULES($panel) $module	;# record name of preference
  # }
  append JPREF_PREFS($module) ""		;# make sure it exists
  if {[lsearch -exact $JPREF_PREFS($module) $global] == -1} {
    lappend JPREF_PREFS($module) $global	;# record name of preference
  }
  append JPREF_INFO(module_name,$module) ""	;# make sure it exists
  set JPREF_INFO(type,$global) $type		;# string, boolean, etc.
  set JPREF_INFO(prompt,$global) $prompt	;# localised when used !!!
  set JPREF_INFO(module,$global) $module	;# not sure this is needed
  set JPREF_INFO(link,$global) $link		;# NOT YET USED
  set JPREF_INFO(text,$global) [j:ldb $text]	;# NOT YET USED
  set JPREF_INFO(values,$global) $values	;# {{en English} {fr French}}
  set JPREF_INFO(default,$global) $default	;# for eg "factory settings"
}

######################################################################
# j:pref:create_standard_prefs - create standard prefs for main panel
######################################################################

proc j:pref:create_standard_prefs {} {
  j:pref:module general \
    -label General \
    -panel .global_prefs \
    -array J_PREFS
  
  j:pref:create_pref language general \
    -prompt prefs:language \
    -default en
  j:pref:create_pref autoposition general \
    -type boolean \
    -default 0 \
    -prompt prefs:autoposition
  j:pref:create_pref confirm general \
    -type boolean \
    -default 1 \
    -prompt prefs:confirm
  j:pref:create_pref j_fs_fast general \
    -type boolean \
    -default 0 \
    -prompt prefs:j_fs_fast
  j:pref:create_pref tk_strictMotif general \
    -type boolean \
    -default 0 \
    -prompt prefs:tk_strictMotif
  j:pref:create_pref scrollbarside general \
    -type radio \
    -default right \
    -values {{left left} {right right}} \
    -prompt prefs:scrollbarside
  j:pref:create_pref visiblebell general \
    -type boolean \
    -default 1 \
    -prompt prefs:audiblebell
  j:pref:create_pref visiblebell general \
    -type boolean \
    -default 1 \
    -prompt prefs:visiblebell
  j:pref:create_pref printer general \
    -default lp \
    -prompt prefs:printer
  
  j:pref:module keyboard \
    -label "Keyboard" \
    -panel .global_prefs \
    -array J_PREFS
  
  j:pref:create_pref bindings keyboard \
    -type radio \
    -default basic \
    -values {
        {basic basic}
        {vi vi}
        {edt edt}
        {emacs emacs}
      } \
    -prompt {Keyboard Bindings:}
  j:pref:create_pref typeover keyboard \
    -type boolean \
    -default 1 \
    -prompt prefs:typeover
  
  j:pref:module helpers \
    -label "Helper Apps" \
    -panel .global_prefs \
    -array J_PREFS
  j:pref:create_pref web_browser helpers \
    -type radio \
    -values {
        {mosaic mosaic}
        {netscape netscape}
        {arena arena}
        {lynx lynx}
      } \
    -prompt {Web Browser:}
  
  j:pref:create_pref man_viewer helpers \
    -type radio \
    -values {
        {builtin builtin}
        {tkman tkman}
        {man man}
      } \
    -prompt {Man Page Viewer:}
  
  j:pref:module fonts \
    -label "Screen Fonts" \
    -panel .global_prefs \
    -array J_PREFS
  
  j:pref:create_pref screen_roman_font fonts \
    -prompt "Roman font:" \
    -default -adobe-helvetica-medium-r-normal--*-120-*-*-*-*-iso8859-1
  j:pref:create_pref screen_italic_font fonts \
    -prompt "Italic font:" \
    -default -adobe-helvetica-medium-o-normal--*-120-*-*-*-*-iso8859-1
  j:pref:create_pref screen_bold_font fonts \
    -prompt "Bold font:" \
    -default -adobe-helvetica-bold-r-normal--*-120-*-*-*-*-iso8859-1
  j:pref:create_pref screen_bolditalic_font fonts \
    -prompt "Bold Italic font:" \
    -default -adobe-helvetica-bold-o-normal--*-120-*-*-*-*-iso8859-1
  j:pref:create_pref screen_monospace_font fonts \
    -prompt "Monospaced font:" \
    -default -adobe-courier-medium-r-normal--*-120-*-*-*-*-iso8859-1
  
  j:pref:create_pref screen_heading0_font fonts \
    -prompt "Title font:" \
    -default -adobe-helvetica-bold-o-normal-*-*-240-*-*-*-*-*-*
  j:pref:create_pref screen_heading1_font fonts \
    -prompt "Heading 1 font:" \
    -default -adobe-helvetica-bold-o-normal-*-*-180-*-*-*-*-*-*
  j:pref:create_pref screen_heading2_font fonts \
    -prompt "Heading 2 font:" \
    -default -adobe-helvetica-bold-o-normal-*-*-140-*-*-*-*-*-*
  j:pref:create_pref screen_heading3_font fonts \
    -prompt "Heading 3 font:" \
    -default -adobe-helvetica-bold-o-normal-*-*-120-*-*-*-*-*-*
  j:pref:create_pref screen_heading4_font fonts \
    -prompt "Heading 4 font:" \
    -default -adobe-helvetica-bold-o-normal-*-*-100-*-*-*-*-*-*
  j:pref:create_pref screen_heading5_font fonts \
    -prompt "Heading 5 font:" \
    -default -adobe-helvetica-bold-o-normal-*-*-80-*-*-*-*-*-*
  
  j:pref:module ps \
    -label PostScript \
    -panel .global_prefs \
    -array J_PREFS
  
  j:pref:create_pref ps_roman_font ps \
    -prompt "Roman font:" \
    -default /Times-Roman
  j:pref:create_pref ps_italic_font ps \
    -prompt "Italic font:" \
    -default /Times-Italic
  j:pref:create_pref ps_bold_font ps \
    -prompt "Bold font:" \
    -default /Times-Bold
  j:pref:create_pref ps_bolditalic_font ps \
    -prompt "Bold Italic font:" \
    -default /Times-BoldItalic
  j:pref:create_pref ps_monospace_font ps \
    -prompt "Monospaced font:" \
    -default /Courier
  j:pref:create_pref ps_heading_font ps \
    -prompt "Heading font:" \
    -default /Helvetica
  j:pref:create_pref ps_normal_size ps \
    -prompt "Normal font size:" \
    -default 12
  j:pref:create_pref ps_monospace_size ps \
    -prompt "Monospaced font size:" \
    -default 11
  j:pref:create_pref ps_linespacing ps \
    -prompt "Line spacing:" \
    -default 14
}

