/*
 * $Id: netscape-remote.c,v 1.5 1996/05/21 21:12:58 kenh Exp $
 *
 * netscape-remote - a Tcl/Tk extension to talk the remote command protocol
 *		     that Netscape uses
 *
 * This extension speaks the remote protocol that is used by the Netscape
 * web browser.  This lets us control netscape remotely without having to
 * start up a new netscape process (despite what the people at Netscape
 * say, starting up a whole new copy of Netscape takes too long for my
 * tastes).
 *
 * We also cache the window id used for Netscape so we don't have to call
 * XQueryTree for every command.
 *
 * Documentation on the protocol netscape uses can be found at the following
 * URL: http://home.netscape.com/newsref/std/x-remote-proto.html
 *
 * By Ken Hornstein <kenh@cmf.nrl.navy.mil>
 *
 */

#ifndef LINT
static char rcsid[]=
	"$Id: netscape-remote.c,v 1.5 1996/05/21 21:12:58 kenh Exp $";
#endif

#include <tk.h>
#include <X11/Xatom.h>
#include <X11/Xproto.h>

/*
 * Just include a prototype for XmuClientWindow here, since we bring
 * along a copy of ClientWin.c with this distribution
 */

extern Window XmuClientWindow(Display *, Window);

/*
 * Names of some of the Netscape internal properties, and variables to
 * store them in.
 */

#define MOZILLA_VERSION_PROP	"_MOZILLA_VERSION"
#define MOZILLA_LOCK_PROP	"_MOZILLA_LOCK"
#define MOZILLA_COMMAND_PROP	"_MOZILLA_COMMAND"
#define MOZILLA_RESPONSE_PROP	"_MOZILLA_RESPONSE"

static Atom XA_MOZILLA_VERSION	= 0;
static Atom XA_MOZILLA_LOCK	= 0;
static Atom XA_MOZILLA_COMMAND	= 0;
static Atom XA_MOZILLA_RESPONSE	= 0;

/*
 * This is a structure that contains all the info about pending things
 * happening on Netscape windows.  We use this to communicate between
 * NetscapeEventProc and what's happening now
 */

typedef struct PendingCommand {
	int state;		/* Type of sub-event we're waiting for */
	Window win;		/* Window we're waiting for */
	Atom atom;		/* Atom for PropertyChange/Delete */
	int response;		/* Did we get a response? */
} PendingCmd;

#define PENDING_OK 1
#define PENDING_TIMEOUT 2

/*
 * Prototypes for internal functions
 */

static int Netscape_Remote_Cmd(ClientData, Tcl_Interp *, int, char *[]);
static Window GetWindow(Tcl_Interp *, Tk_Window);
static int CheckForNetscape(Display *, Window);
static int SendCommand(Tcl_Interp *, Tk_Window, Window, char *, PendingCmd *);
static int NetscapeEventHandler(ClientData, XEvent *);
static int GetLock(Tcl_Interp *, Display *, Window, PendingCmd *);
static Tk_RestrictAction NetscapeRestrict(ClientData, XEvent *);
static void LockTimeout(ClientData);
static void ReleaseLock(Display *, Window);

static Window CachedWindow = None;

#define TIMEOUT 5000

/*
 * Our package init routine.  Set things up for our new interpreter command.
 */

int
Netscape_remote_Init(Tcl_Interp *interp)
{
	Tk_Window main;
	Display *dpy;

	if ((main = Tk_MainWindow(interp)) == NULL) {
		Tcl_AppendResult(interp, "No main window associated with ",
				 "this interpreter!", (char *) NULL);
		return TCL_ERROR;
	}

	dpy = Tk_Display(main);

	/*
	 * Get the Atoms corresponding to these property names
	 */

	if (! XA_MOZILLA_VERSION)
		XA_MOZILLA_VERSION = XInternAtom(dpy, MOZILLA_VERSION_PROP,
						 False);
	if (! XA_MOZILLA_LOCK)
		XA_MOZILLA_LOCK = XInternAtom(dpy, MOZILLA_LOCK_PROP, False);

	if (! XA_MOZILLA_COMMAND)
		XA_MOZILLA_COMMAND = XInternAtom(dpy, MOZILLA_COMMAND_PROP,
						 False);
	if (! XA_MOZILLA_RESPONSE)
		XA_MOZILLA_RESPONSE = XInternAtom(dpy, MOZILLA_RESPONSE_PROP,
						   False);
	
	Tcl_CreateCommand(interp, "send-netscape", Netscape_Remote_Cmd,
			  (ClientData) main, (void (*)()) NULL);

	return TCL_OK;
}

static int
Netscape_Remote_Cmd(ClientData clientData, Tcl_Interp *interp, int argc,
	char *argv[])
{
	Tk_Window main = (Tk_Window) clientData;
	Tk_ErrorHandler error;
	Window w = None;
	PendingCmd pending;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
			argv[0], " netscapeCommand\"", (char *) NULL);
		return TCL_ERROR;
	}

	/*
	 * Figure out which window to use.  Check to see if we have
	 * a cached window - if so, use that one, rather than iterating
	 * through all of the windows on our display.
	 *
	 * We need to setup an error handler here, otherwise we will
	 * exit if the window doesn't exist.
	 */

	error = Tk_CreateErrorHandler(Tk_Display(main), BadWindow,
				      X_GetProperty, -1, NULL, NULL);

	if (CachedWindow != None) {
		if (CheckForNetscape(Tk_Display(main), CachedWindow)) {
			w = CachedWindow;
		} else {
			CachedWindow = None;
		}
	}

	if (w == None) {
		if ((w = GetWindow(interp, main)) == None) {
			return TCL_ERROR;
		}
		CachedWindow = w;
	}

	Tk_DeleteErrorHandler(error);

	return SendCommand(interp, main, w, argv[1], &pending);
}

/*
 * Find the window to use on the remote display.  Most of this code is
 * taken from Netscape reference implementation.  We don't do any version
 * checking right now.
 */

static Window
GetWindow(Tcl_Interp *interp, Tk_Window main)
{
	int i;
	Window root = RootWindowOfScreen(Tk_Screen(main));
	Window root2, parent, *kids;
	unsigned int nkids;
	Window result = None;

	if (! XQueryTree(Tk_Display(main), root, &root2, &parent, &kids,
			 &nkids)) {
		Tcl_AppendResult(interp, "XQueryTree failed", (char *) NULL);
		return None;
	}

	if (root != root2) {
		Tcl_AppendResult(interp, "Root windows didn't match!",
				 (char *) NULL);
		return None;
	}

	if (parent != None) {
		Tcl_AppendResult(interp, "We got a valid parent window, but",
				 " we shouldn't have!", (char *) NULL);
		return None;
	}

	if (! (kids && nkids)) {
		Tcl_AppendResult(interp, "No children found!", (char *) NULL);
		return None;
	}

	for (i = 0; i < nkids; i++) {
		Window w = XmuClientWindow(Tk_Display(main), kids[i]);
		if (CheckForNetscape(Tk_Display(main), w)) {
			result = w;
			break;
		}
	}

	if (result == None) {
		Tcl_AppendResult(interp, "Couldn't find a netscape window",
				 (char *) NULL);
	}

	return result;
}

/*
 * See if the given window is a Netscape window by looking for the
 * XA_MOZILLA_VERSION property
 */

static int
CheckForNetscape(Display *d, Window w)
{
	Atom type;
	int format;
	unsigned long nitems, bytesafter;
	unsigned char *version = NULL;
	int status = XGetWindowProperty(d, w, XA_MOZILLA_VERSION, 0,
					65536 / sizeof(long), False,
					XA_STRING, &type, &format,
					&nitems, &bytesafter, &version);

	if (status != Success || !version) {
		if (version)
			XFree(version);
		return 0;
	}

	/*
	 * We don't do anything with the version right now
	 */

	XFree(version);

	return 1;
}

/*
 * Send a command to the Netscape window we found previously
 */

static int
SendCommand(Tcl_Interp *interp, Tk_Window mainwin, Window win, char *command,
	    PendingCmd *pending)
{
	Tk_RestrictProc *prevRestrict;
	ClientData prevArgs;
	int result;
	Atom actual_type;
	int actual_format;
	unsigned long nitems, bytes_after;
	unsigned char *data;

	/*
	 * Select for PropertyChange events on the Netscape window
	 */
	
	XSelectInput(Tk_Display(mainwin), win, (PropertyChangeMask));

	/*
	 * Create a generic event handler to get events on that window
	 */

	pending->state = 0;
	pending->win = None;
	pending->response = 0;

	Tk_CreateGenericHandler(NetscapeEventHandler, (ClientData) pending);

	if (GetLock(interp, Tk_Display(mainwin), win, pending) == 0) {
		Tk_DeleteGenericHandler(NetscapeEventHandler,
					(ClientData) pending);
		XSelectInput(Tk_Display(mainwin), win, 0);
		return TCL_ERROR;
	}

	/*
	 * We've got a successful lock, so send the command to Netscape
	 */

	XChangeProperty(Tk_Display(mainwin), win, XA_MOZILLA_COMMAND,
			XA_STRING, 8, PropModeReplace,
			(unsigned char *) command, strlen(command));

	/*
	 * Netscape should delete the property containing the command
	 * Wait for this to happen.
	 */

	prevRestrict = Tk_RestrictEvents(NetscapeRestrict,
				(ClientData) pending,
				&prevArgs);
	pending->win = win;
	pending->state = PropertyDelete;
	pending->atom = XA_MOZILLA_COMMAND;
	pending->response = 0;

	Tcl_CreateModalTimeout(TIMEOUT, LockTimeout, (ClientData) pending);
	while (!pending->response) {
		Tcl_DoOneEvent(TCL_WINDOW_EVENTS);
	}
	Tcl_DeleteModalTimeout(LockTimeout, (ClientData)
			       pending);
	Tk_RestrictEvents(prevRestrict, prevArgs, &prevArgs);

	if (pending->response == PENDING_TIMEOUT) {
		Tcl_AppendResult(interp, "Timeout waiting for Netscape to ",
				 "acknowledge command", (char *) NULL);
		ReleaseLock(Tk_Display(mainwin), win);
		XSelectInput(Tk_Display(mainwin), win, 0);
		return TCL_ERROR;
	}

	/*
	 * Wait for a response.  Netscape will write it's response code
	 * in the XA_MOZILLA_RESPONSE property -- check that for the
	 * response code
	 */

	prevRestrict = Tk_RestrictEvents(NetscapeRestrict,
				(ClientData) pending,
				&prevArgs);
	pending->win = win;
	pending->state = PropertyNewValue;
	pending->atom = XA_MOZILLA_RESPONSE;
	pending->response = 0;

	Tcl_CreateModalTimeout(TIMEOUT, LockTimeout, (ClientData) pending);
	while (!pending->response) {
		Tcl_DoOneEvent(TCL_WINDOW_EVENTS);
	}
	Tcl_DeleteModalTimeout(LockTimeout, (ClientData)
			       pending);
	Tk_RestrictEvents(prevRestrict, prevArgs, &prevArgs);

	if (pending->response == PENDING_TIMEOUT) {
		Tcl_AppendResult(interp, "Timeout waiting for a response from",
				 "Netscape", (char *) NULL);
		XSelectInput(Tk_Display(mainwin), win, 0);
		ReleaseLock(Tk_Display(mainwin), win);
		return TCL_ERROR;
	}

	/*
	 * Get the response string from Netscape
	 */
	
	result = XGetWindowProperty(Tk_Display(mainwin), win,
				    XA_MOZILLA_RESPONSE, 0,
				    65536 / sizeof(long), True /* delete */,
				    XA_STRING, &actual_type, &actual_format,
				    &nitems, &bytes_after, &data);
	
	if (result != Success) {
		Tcl_AppendResult(interp, "Failed to read response from "
				 "Netscape", (char *) NULL);
		XSelectInput(Tk_Display(mainwin), win, 0);
		ReleaseLock(Tk_Display(mainwin), win);
		return TCL_ERROR;
	}

	if (! data) {
		Tcl_AppendResult(interp, "No data returned from Netscape",
				 (char *) NULL);
		XSelectInput(Tk_Display(mainwin), win, 0);
		ReleaseLock(Tk_Display(mainwin), win);
		return TCL_ERROR;
	}

	Tcl_AppendResult(interp, data, (char *) NULL);

	XFree(data);

	/*
	 * Remove the lock on Netscape.
	 */
	
	ReleaseLock(Tk_Display(mainwin), win);

	/*
	 * Don't select these events anymore
	 */

	XSelectInput(Tk_Display(mainwin), win, 0);

	return TCL_OK;
}

static int
NetscapeEventHandler(ClientData clientData, XEvent *event)
{
	PendingCmd *pending = (PendingCmd *) clientData;

	if (pending->win == None)
		return 0;
	
	if (event->type == PropertyNotify && event->xproperty.window ==
	    pending->win && event->xproperty.state == pending->state &&
	    event->xproperty.atom == pending->atom) {
		pending->response = PENDING_OK;
	}

	return 0;
}

/*
 * Participate in the Netscape locking protocol so our commands don't
 * collide
 */

static int
GetLock(Tcl_Interp *interp, Display *d, Window win, PendingCmd *pending)
{
	char lock_data[255];
	Bool locked = False;
	Tk_RestrictProc *prevRestrict;
	ClientData prevArgs;

	sprintf(lock_data, "TkApp-pid%d@", getpid());
	if (gethostname(lock_data + strlen(lock_data), 100) == -1) {
		Tcl_AppendResult(interp, "gethostname() returned an error",
				 (char *) NULL);
		return 0;
	}

	do {
		int result;
		Atom actual_type;
		int actual_format;
		unsigned long nitems, bytes_after;
		unsigned char *data = NULL;

		/*
		 * Grab the server so nobody else can do anything
		 */

		XGrabServer(d);

		/*
		 * See if it's locked
		 */

		result = XGetWindowProperty(d, win, XA_MOZILLA_LOCK,
					    0, (65536 / sizeof(long)),
					    False, XA_STRING,
					    &actual_type, &actual_format,
					    &nitems, &bytes_after,
					    &data);

		if (result != Success || actual_type == None) {
			/*
			 * It's not locked now, lock it!
			 */

			 XChangeProperty(d, win, XA_MOZILLA_LOCK, XA_STRING,
					 8, PropModeReplace,
					 (unsigned char *) lock_data,
					 strlen(lock_data));
			locked = True;
		}

		/*
		 * Release the server grab
		 */

		XUngrabServer(d);
		XSync(d, False);

		if (! locked) {
			/*
			 * There was already a lock in place.  Wait for
			 * a PropertyDelete event.  Use a RestrictProc
			 * to make sure we're synchronous
			 */

			prevRestrict = Tk_RestrictEvents(NetscapeRestrict,
						(ClientData) pending,
						&prevArgs);
			pending->win = win;
			pending->state = PropertyDelete;
			pending->atom = XA_MOZILLA_LOCK;
			pending->response = 0;
			Tcl_CreateModalTimeout(TIMEOUT, LockTimeout,
					       (ClientData) pending);
			while (!pending->response) {
				Tcl_DoOneEvent(TCL_WINDOW_EVENTS);
			}
			Tcl_DeleteModalTimeout(LockTimeout, (ClientData)
					       pending);
			Tk_RestrictEvents(prevRestrict, prevArgs, &prevArgs);

			if (pending->response == PENDING_TIMEOUT) {
				Tcl_AppendResult(interp, "Timeout waiting for "
						 "locked to be released",
						 (char *) NULL);
				if (data) {
					Tcl_AppendResult(interp, " by ",
							 data, (char *) NULL);
					XFree(data);
				}
				break;
			}
		}

		if (data)
			XFree(data);
	} while (! locked);

	return locked == True ? 1 : 0;
}

/*
 * Unlock our lock with Netscape.  We should check for errors, but this
 * routine doesn't
 */

static void
ReleaseLock(Display *d, Window win)
{
	Atom actual_type;
	int actual_format;
	unsigned long nitems, bytes_after;
	unsigned char *data;

	XGetWindowProperty(d, win, XA_MOZILLA_LOCK, 0,
			   65536 / sizeof(long), True /* delete */,
			   XA_STRING, &actual_type, &actual_format,
			   &nitems, &bytes_after, &data);
		
	if (data)
		XFree(data);
}

static Tk_RestrictAction
NetscapeRestrict(ClientData clientData, XEvent *event)
{
	PendingCmd *pending = (PendingCmd *) clientData;

	if (event->type != PropertyNotify || event->xproperty.window !=
	    pending->win || event->xproperty.atom != pending->atom) {
		return TK_DEFER_EVENT;
	}

	return TK_PROCESS_EVENT;
}

static void
LockTimeout(ClientData clientData)
{
	PendingCmd *pending = (PendingCmd *) clientData;

	pending->response = PENDING_TIMEOUT;
}
