#include <tcl.h>
#include <tclInt.h>
#include <tclPort.h>
#include <assert.h>
#include "tclcomp.h"
#include "tclcompint.h"
#include "tclcompproc.h"

/*
 * The structure below is an exact copy of struct Proc, defined
 * in tclInt.h, except that it has an additional field to hold
 * the compiled code of the proc. 
 *
 * IMPORTANT NOTE: If the Proc structure is modified in later versions
 * of Tcl (the one below was extracted from Tcl7.4b2), then CompProc
 * should be modified accordingly.
 */

typedef struct CompProc {
    struct Interp *iPtr;	/* Interpreter for which this command
				 * is defined. */
    int refCount;		/* Reference count:  1 if still present
				 * in command table plus 1 for each call
				 * to the procedure that is currently
				 * active.  This structure can be freed
				 * when refCount becomes zero. */
    char *command;		/* Command that constitutes the body of
				 * the procedure (dynamically allocated). */
    Arg *argPtr;		/* Pointer to first of procedure's formal
				 * arguments, or NULL if none. */
    CmdStruct *code;		/* Contains the compiled code for the 
				 * procedure */
} CompProc;


/*
 *----------------------------------------------------------------------
 *
 * CleanupCompProc --
 *
 *	This procedure does all the real work of freeing up a Proc
 *	structure.  It's called only when the structure's reference
 *	count becomes zero.
 *	This is an almost verbatim copy of CleanupProc, except that 
 *      the code field of 'proc' is also freed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory gets freed.
 *
 *----------------------------------------------------------------------
 */

static void
CleanupCompProc(register CompProc *procPtr)	/* Procedure to be deleted. */
{
    register Arg *argPtr;

    ckfree((char *) procPtr->command);
    for (argPtr = procPtr->argPtr; argPtr != NULL; ) {
	Arg *nextPtr = argPtr->nextPtr;

	ckfree((char *) argPtr);
	argPtr = nextPtr;
    }
    FreeCmd (procPtr->code);
    ckfree((char *) procPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ProcDeleteCompProc --
 *
 *	This procedure is invoked just before a command procedure is
 *	removed from an interpreter.  Its job is to release all the
 *	resources allocated to the procedure.
 * 	This is an almost exact copy of ProcDeleteProc (tclProc.c),
 *	except that it calls CleanupCompProc instead of CleanupProc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory gets freed, unless the procedure is actively being
 *	executed.  In this case the cleanup is delayed until the
 *	last call to the current procedure completes.
 *
 *----------------------------------------------------------------------
 */

static void
ProcDeleteCompProc(ClientData clientData)	/* Procedure to be deleted. */
{
    CompProc *procPtr = (CompProc *) clientData;

    procPtr->refCount--;
    if (procPtr->refCount <= 0) {
	CleanupCompProc(procPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ExecuteProc --
 *
 *	When a Compiled Tcl procedure gets invoked, this routine gets invoked
 *	to execute the procedure. This routine is an almost exact copy
 *      of InterpProc, with the difference that instead of calling
 *	TclEval to interpret the body of the proc, we call ExecuteCommand
 *      to execute the code. Also, if needed, CleanUpCompProc is called
 *	instead of CleanUpProc
 *
 * Results:
 *	A standard Tcl result value, usually TCL_OK.
 *
 * Side effects:
 *	Depends on the commands in the procedure.
 *
 *----------------------------------------------------------------------
 */

static int
ExecuteProc(ClientData clientData,	/* Record describing procedure to be
					 * interpreted. */
	    Tcl_Interp *interp,		/* Interpreter in which procedure was
					 * invoked. */
	    int argc,			/* Count of number of arguments to this
					 * procedure. */
	    char **argv)		/* Argument values. */
{
    register CompProc *procPtr = (CompProc *) clientData;
    register Arg *argPtr;
    register Interp *iPtr;
    char **args;
    CallFrame frame;
    char *value;
    int result;

    /*
     * Set up a call frame for the new procedure invocation.
     */

    iPtr = procPtr->iPtr;
    Tcl_InitHashTable(&frame.varTable, TCL_STRING_KEYS);
    if (iPtr->varFramePtr != NULL) {
	frame.level = iPtr->varFramePtr->level + 1;
    } else {
	frame.level = 1;
    }
    frame.argc = argc;
    frame.argv = argv;
    frame.callerPtr = iPtr->framePtr;
    frame.callerVarPtr = iPtr->varFramePtr;
    iPtr->framePtr = &frame;
    iPtr->varFramePtr = &frame;
    iPtr->returnCode = TCL_OK;

    /*
     * Match the actual arguments against the procedure's formal
     * parameters to compute local variables.
     */

    for (argPtr = procPtr->argPtr, args = argv+1, argc -= 1;
	    argPtr != NULL;
	    argPtr = argPtr->nextPtr, args++, argc--) {

	/*
	 * Handle the special case of the last formal being "args".  When
	 * it occurs, assign it a list consisting of all the remaining
	 * actual arguments.
	 */

	if ((argPtr->nextPtr == NULL)
		&& (strcmp(argPtr->name, "args") == 0)) {
	    if (argc < 0) {
		argc = 0;
	    }
	    value = Tcl_Merge(argc, args);
	    Tcl_SetVar(interp, argPtr->name, value, 0);
	    ckfree(value);
	    argc = 0;
	    break;
	} else if (argc > 0) {
	    value = *args;
	} else if (argPtr->defValue != NULL) {
	    value = argPtr->defValue;
	} else {
	    Tcl_AppendResult(interp, "no value given for parameter \"",
		    argPtr->name, "\" to \"", argv[0], "\"",
		    (char *) NULL);
	    result = TCL_ERROR;
	    goto procDone;
	}
	Tcl_SetVar(interp, argPtr->name, value, 0);
    }
    if (argc > 0) {
	Tcl_AppendResult(interp, "called \"", argv[0],
		"\" with too many arguments", (char *) NULL);
	result = TCL_ERROR;
	goto procDone;
    }

    /*
     * Invoke the commands in the procedure's body.
     */

    procPtr->refCount++;
    result = ExecCommand (interp, procPtr->code);
    procPtr->refCount--;
    if (procPtr->refCount <= 0) {
	CleanupCompProc(procPtr);
    }
    if (result == TCL_RETURN) {
	result = iPtr->returnCode;
	iPtr->returnCode = TCL_OK;
	if (result == TCL_ERROR) {
	    Tcl_SetVar2(interp, "errorCode", (char *) NULL,
		    (iPtr->errorCode != NULL) ? iPtr->errorCode : "NONE",
		    TCL_GLOBAL_ONLY);
	    iPtr->flags |= ERROR_CODE_SET;
	    if (iPtr->errorInfo != NULL) {
		Tcl_SetVar2(interp, "errorInfo", (char *) NULL,
			iPtr->errorInfo, TCL_GLOBAL_ONLY);
		iPtr->flags |= ERR_IN_PROGRESS;
	    }
	}
    } else if (result == TCL_ERROR) {
	char msg[100];

	/*
	 * Record information telling where the error occurred.
	 */

	sprintf(msg, "\n    (procedure \"%.50s\" line %d)", argv[0],
		iPtr->errorLine);
	Tcl_AddErrorInfo(interp, msg);
    } else if (result == TCL_BREAK) {
	iPtr->result = "invoked \"break\" outside of a loop";
	result = TCL_ERROR;
    } else if (result == TCL_CONTINUE) {
	iPtr->result = "invoked \"continue\" outside of a loop";
	result = TCL_ERROR;
    }

    /*
     * Delete the call frame for this procedure invocation (it's
     * important to remove the call frame from the interpreter
     * before deleting it, so that traces invoked during the
     * deletion don't see the partially-deleted frame).
     */

    procDone:
    iPtr->framePtr = frame.callerPtr;
    iPtr->varFramePtr = frame.callerVarPtr;

    /*
     * The check below is a hack.  The problem is that there could be
     * unset traces on the variables, which cause scripts to be evaluated.
     * This will clear the ERR_IN_PROGRESS flag, losing stack trace
     * information if the procedure was exiting with an error.  The
     * code below preserves the flag.  Unfortunately, that isn't
     * really enough:  we really should preserve the errorInfo variable
     * too (otherwise a nested error in the trace script will trash
     * errorInfo).  What's really needed is a general-purpose
     * mechanism for saving and restoring interpreter state.
     */

    if (iPtr->flags & ERR_IN_PROGRESS) {
	TclDeleteVars(iPtr, &frame.varTable);
	iPtr->flags |= ERR_IN_PROGRESS;
    } else {
	TclDeleteVars(iPtr, &frame.varTable);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_CompProcCmd --
 *
 *	This procedure is invoked to process the "proc" Tcl command 
 *      when called by compiled code (instead of Tcl_Eval).
 *	It is an almost literal copy of Tcl_CompProc (module
 *	tclProc.c), with the following exceptions:
 *
 *      * Instead of just storing a copy of its 4th argument into a
 *        Proc structure, that argument is "compiled" and the code
 *        is stored in a CompiledProc (a Proc with an additional field).
 *        
 *      * The command procedure for executing the proc is set to
 *	  ExecuteProc instead of InterpProc
 *
 *	* The procedure for deleting the command is set to
 *	  ProcDeleteCompProc instead of ProcDeleteProc
 *
 * Results:
 *	A standard Tcl result value.
 *
 * Side effects:
 *	A new procedure gets created.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_CompProcCmd (ClientData dummy,		/* Not used. */
		 Tcl_Interp *interp,		/* Current interpreter. */
		 int argc,			/* Number of arguments. */
		 char **argv)			/* Argument strings. */
{
    register Interp *iPtr = (Interp *) interp;
    register CompProc *procPtr;
    int result, argCount, i;
    char **argArray = NULL;
    Arg *lastArgPtr;
    register Arg *argPtr = NULL;	/* Initialization not needed, but
					 * prevents compiler warning. */

    if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" name args body\"", (char *) NULL);
	return TCL_ERROR;
    }

    procPtr = (CompProc *) ckalloc(sizeof(CompProc));
    procPtr->iPtr = iPtr;
    procPtr->refCount = 1;
    procPtr->command = (char *) ckalloc((unsigned) strlen(argv[3]) + 1);    
    strcpy(procPtr->command, argv[3]);
    procPtr->argPtr = NULL;

    /* 
     * Compile the body of the proc and store the code in procPtr
     */

    result = CompileCommand (interp, argv [3], &procPtr->code);
    if (result != TCL_OK) {
       goto procError;
    }
      
    /*
     * Break up the argument list into argument specifiers, then process
     * each argument specifier.
     */

    result = Tcl_SplitList(interp, argv[2], &argCount, &argArray);
    if (result != TCL_OK) {
	goto procError;
    }
    lastArgPtr = NULL;
    for (i = 0; i < argCount; i++) {
	int fieldCount, nameLength, valueLength;
	char **fieldValues;

	/*
	 * Now divide the specifier up into name and default.
	 */

	result = Tcl_SplitList(interp, argArray[i], &fieldCount,
		&fieldValues);
	if (result != TCL_OK) {
	    goto procError;
	}
	if (fieldCount > 2) {
	    ckfree((char *) fieldValues);
	    Tcl_AppendResult(interp,
		    "too many fields in argument specifier \"",
		    argArray[i], "\"", (char *) NULL);
	    result = TCL_ERROR;
	    goto procError;
	}
	if ((fieldCount == 0) || (*fieldValues[0] == 0)) {
	    ckfree((char *) fieldValues);
	    Tcl_AppendResult(interp, "procedure \"", argv[1],
		    "\" has argument with no name", (char *) NULL);
	    result = TCL_ERROR;
	    goto procError;
	}
	nameLength = strlen(fieldValues[0]) + 1;
	if (fieldCount == 2) {
	    valueLength = strlen(fieldValues[1]) + 1;
	} else {
	    valueLength = 0;
	}
	argPtr = (Arg *) ckalloc((unsigned)
		(sizeof(Arg) - sizeof(argPtr->name) + nameLength
		+ valueLength));
	if (lastArgPtr == NULL) {
	    procPtr->argPtr = argPtr;
	} else {
	    lastArgPtr->nextPtr = argPtr;
	}
	lastArgPtr = argPtr;
	argPtr->nextPtr = NULL;
	strcpy(argPtr->name, fieldValues[0]);
	if (fieldCount == 2) {
	    argPtr->defValue = argPtr->name + nameLength;
	    strcpy(argPtr->defValue, fieldValues[1]);
	} else {
	    argPtr->defValue = NULL;
	}
	ckfree((char *) fieldValues);
    }

    Tcl_CreateCommand(interp, argv[1], ExecuteProc, (ClientData) procPtr,
	    ProcDeleteCompProc);
    ckfree((char *) argArray);
    return TCL_OK;

    procError:
    ckfree(procPtr->command);
    while (procPtr->argPtr != NULL) {
	argPtr = procPtr->argPtr;
	procPtr->argPtr = argPtr->nextPtr;
	ckfree((char *) argPtr);
    }

    /* Be sure to delete the code too */
    if (procPtr->code != NULL) {
        FreeCmd (procPtr->code);
    }

    ckfree((char *) procPtr);
    if (argArray != NULL) {
	ckfree((char *) argArray);
    }
    return result;
}




/*
 *----------------------------------------------------------------------
 *
 * Tcl_CompSourceCmd --
 *
 *	This procedure is invoked to process the "source" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * 	This is an almost exact copy of Tcl_SourceCmd, except that
 *	Tcl_CompEvalFile is called instead of Tcl_EvalFile.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_CompSourceCmd (
    ClientData dummy,			/* Not used. */
    Tcl_Interp *interp,			/* Current interpreter. */
    int argc,				/* Number of arguments. */
    char **argv)			/* Argument strings. */
{
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" fileName\"", (char *) NULL);
	return TCL_ERROR;
    }
    return Tcl_CompEvalFile(interp, argv[1]);
}
