/*
 * tclcompint.h ---
 * 
 * Data Structures used internally by the tcl proto compiler
 */

#define PREALLOC_SIZE 10	/* Default # elts for var sized arrays */
#define PARSE_BUF_SIZE 200	/* Default # chars for a parse buffer */

/*
 * The structure below is similar to Tcl's ParseValue, except that
 * buffer reallocation is always done with ckrealloc
 */

typedef struct ParseBuf {
   char fixedstorage [PARSE_BUF_SIZE]; /* Init. space allocated in the stack */
   char *first;	     /* First char of buffer */
   char *next;	     /* Where next char will be stored */
   char	*end;	     /* Points to the char beyond the last char of buffer */   
} ParseBuf;

#define INITBUF(name) 					\
   (name).first = (name).fixedstorage; 		       	\
   (name).next = (name).fixedstorage;			\
   (name).end = (name).fixedstorage + PARSE_BUF_SIZE;  

#define ROOMFOR(nchars,parsebuf) 				\
   if ((parsebuf).end-(parsebuf).next<(nchars)) { 		\
      int size = ((parsebuf).end-(parsebuf).first);		\
      int newsize = size + 					\
	 (size + (parsebuf).end-(parsebuf).next<(nchars)) ? 	\
	    size + nchars :					\
	    size + size;   					\
      if ((parsebuf).first==(parsebuf).fixedstorage) {	    	\
	 (parsebuf).first = ckalloc (newsize);	       		\
         (parsebuf).next = (parsebuf).first + 			\
	    ((parsebuf).next-(parsebuf).fixedstorage);		\
         memcpy ((parsebuf).first, (parsebuf).fixedstorage,	\
                 size);						\
      }								\
      else {							\
	 char *tmp = (parsebuf).first;	 			\
	 (parsebuf).first = ckrealloc (tmp, newsize);		\
	 (parsebuf).next = (parsebuf).first +			\
	    ((parsebuf).next-tmp);				\
      }								\
      (parsebuf).end = (parsebuf).first + newsize; 		\
   }

#define FREEBUF(parsebuf) 				\
   if ((parsebuf).first != (parsebuf).fixedstorage) {	\
      ckfree ((parsebuf).first);			\
   }


typedef enum ValueType {
   NullValue,		/* No value at all */
   LiteralValue,	/* Constant string, e.g., string in braces {} */
   CommandValue,	/* String returned by command, i.e., [command] */
   VarValue		/* Value from variable, e.g., $var */
} ValueType;
   
typedef struct ValStruct { 
   ValueType type;
   struct ValStruct *next;
   union {
      char *literal;
      struct CmdStruct *cmd;
      struct {
	 char *name;
	 struct ValStruct *index;
      } varvalue;      
   } detail;
} ValStruct;

typedef enum CmdType {
   NullCmd,
   SetCmd,
   IncrCmd,
   EvalCmd,
   WhileCmd,
   ForCmd,
   IfCmd,
   ExprCmd,
   RuntimeCmd,
   OtherCmd
} CmdType;

typedef struct CmdStruct {
   CmdType type;
   struct CmdStruct *next;
   union {
      struct {
	 struct ValStruct *name;
	 struct ValStruct *index;
	 struct ValStruct *value;
      } setcmd;
      struct {
	 struct ValStruct *name;
	 struct ValStruct *index;
	 struct ValStruct *increment;
      } incrcmd;
      struct {
	 struct ValStruct **body;
      } evalcmd;
      struct {
	 struct ValStruct *cond;
	 struct CmdStruct *body;
      } whilecmd;
      struct {
	 struct CmdStruct *first;
	 struct ValStruct *cond;
	 struct CmdStruct *incr;
	 struct CmdStruct *body;
      } forcmd;
      struct {
	 int ncond;
	 struct ValStruct **cond;
	 struct CmdStruct **body;
	 struct CmdStruct *elsebody;
      } ifcmd;
      struct {
	 struct ValStruct *body;
      } exprcmd;
      struct {
	 struct ValStruct *body;
      } runtimecmd;
      struct {
	 Command *tclcmdptr;
	 int argc;
	 struct ValStruct **words;
      } othercmd;
   } detail;
} CmdStruct;

void FreeCmd (CmdStruct *cmd);
/*
 * Deallocates dynamic memory associated with cmd
 */

int CompileCommand (Tcl_Interp* interp, char * string, CmdStruct ** cmd); 
/*  --------------
 *
 * Breaks up 'string' into separate commands and compiles them. The 
 * compiled code is put into *cmd. If a compilation error occurs,
 * *cmd is set to NULL and TCL_ERROR is returned.
 */

int ExecCommand (Tcl_Interp * interp, CmdStruct * cmd);
/*
 * Executes command cmd.
 *
 * Returns (hopefully) whatever should be the result of evaluating
 * the string that was compiled into the 'cmd', including possible errors.
 */ 
