%  Copyright (c) 1992 by P. Klosowski at UMd/NIST.  All Rights Reserved 
%
%  This file contains parts of Swiss Army Fit (SAF). SAF is free
%  software; you can redistribute it and/or modify it under the terms
%  of the GNU General Public License as published by the Free Software
%  Foundation; either version 2 of the License, or (at your option)
%  any later version.
%
%  This software is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You may already have a copy of the GNU General Public License; if
%  not, write to the Free Software Foundation, Inc., 675 Mass Ave,
%  Cambridge, MA 02139, USA.
%
% This source file is best viewed in Emacs under both nuweb.el and 
% outline-minor modes; I autoload the following function:          
% (defun web-outline-mode () (nuweb-mode) (outline-minor-mode)     
% (make-local-variable 'outline-regexp)                            
% (setq outline-regexp "[@@][od]\\|\\\\chap\\|\\\\sec\\|\\\\sub") (hide-body))
% Nuweb was written by Preston Briggs and can be obtained at cs.rice.edu 
%   HISTORY
%     przemek - Dec 18, 1992: Created.
%     przemek - Nov 1993: hacked: Tcl, derivatives, testing, nuweb...

\documentstyle{article}
\setlength{\oddsidemargin}{-.2in}
\setlength{\evensidemargin}{0in}
\setlength{\topmargin}{0in}
\addtolength{\topmargin}{-\headheight}
\addtolength{\topmargin}{-\headsep}
\setlength{\textheight}{8.9in}
\setlength{\textwidth}{6.5in}
\setlength{\marginparwidth}{0.5in}

\begin{document}
\title{{\tt purchase} --- a spreadsheet for preparing purchase requisitions}

\author{Przemek K\l{}osowski\\
	przemek.klosowski@@nist.gov \\
	$( $Revision: 1.6 $ )$
}

\maketitle
\tableofcontents

\section{Introduction}

This program (\verb-purchase-) is a Tcl/Tk code for an application to 
prepare purchase orders at Reactor Division at NIST. I wrote it to 
reacquaint myself with Tcl/Tk after some time spent not using it.
It contains some neat tricks, such as the code to dump the text contents
of a Tk window in PostScript.

The program is written in \verb-nuweb-, creating the Tcl source and \LaTeX{}
documentation from the master source.

\section{The program}

\subsection{Main code}

This is the skeleton of our program. It begins with the procedure
definitions, to avoid the ``use-before-define'' problem. Command line
parameters are handled after the main window is created, because file
reading requires that appropriate variables and windows are already in
place.

@o purchase @{#!/usr/local/bin/wish -f
set Revision_string {$Revision: 1.6 $}

@<Procedures@>

@<Global setup@>
@<Create menu bar@>
@<Create main window@>
@<Handle command line parameters@>
@}

Make the window resizable by specifying initial size.  Declare some
useful fonts. Declare the state of the data as ``not dirty''.

@d Global setup @{
wm minsize . 200 300

set bigfont -Adobe-Times-Medium-R-Normal-*-180-*

set current_file_dirty 0

@<Set up default bindings@>
@| current_file_dirty @}

Set up some bindings that Tk fails to include (borrowed from xgpl
package by P. Alexander of MRAO at Cambridge, whose copyright I
acknowledge).

Set up arrow keys to move forth and back, and handle an Emacs-like selection.

@d Set up default bindings @{
   bind Entry <Key-Left>  {Entry_MoveCursor %W -1}
   bind Entry <Key-Right> {Entry_MoveCursor %W 1}
   bind Entry <Return>    {puts "\a"}
   bind Entry <Control-y> {catch {%W insert insert [selection get]} 
                           tk_entrySeeCaret %W }
   bind Entry <Control-w> {catch {%W delete sel.first sel.last}
                           tk_entrySeeCaret %W}
  bind Text <Left>           {Text_MoveLeft %W}
  bind Text <Up>             {Text_MoveUp %W}
  bind Text <Right>          {Text_MoveRight %W}
  bind Text <Down>           {Text_MoveDown %W}

# bindings affecting selection
  bind Text <2>              {Text_InsertSelection %W}
  bind Text <Control-y>      {Text_InsertSelection %W}
  bind Text <Control-w>      {Text_DeleteSelection %W}
@}

Bindings for entry widgets. 

@d Procedures @{
proc Entry_MoveCursor {w inc} {
       set x [expr "[$w index insert] + $inc"]
       if {$x == -1} then {set x 0}
       if {$x >= [$w index end]} then {set x end}
       $w icursor $x
}
@|  Entry_MoveCursor @}

Bindings for text widgets.

@d Procedures @{
proc Text_DeleteSelection { w } {
  catch {$w delete sel.first sel.last}
}
proc Text_InsertSelection { w } {
  catch {$w insert insert [selection get]}
  $w yview -pickplace insert
}

proc Text_MoveLeft { w } {
  $w mark set insert {insert - 1 chars}
  $w yview -pickplace insert
}
proc Text_MoveRight { w } {
  $w mark set insert {insert + 1 chars}
  $w yview -pickplace insert
}
proc Text_MoveUp { w } {
  $w mark set insert {insert - 1 lines}
  $w yview -pickplace insert
}
proc Text_MoveDown { w } {
  $w mark set insert {insert + 1 lines}
  $w yview -pickplace insert
}
@| Text_DeleteSelection Text_InsertSelection Text_MoveLeft Text_MoveRight Text_MoveUp Text_MoveDown @}

Process command line parameters: check that the file exists and read it in.

@d Handle command line parameters @{
  if {$argc >= 1} {
    if {$argc > 1} {
      tk_dialog .error {Command line error} "Too many arguments ($argc)" error 0 OK
    }
    set current_file [lindex $argv 0]
    if [file isfile $current_file] {
      source $current_file
    } else {
      tk_dialog .info {New file} "File $current_file doesn't exist; creating a new one" info 0 OK
    }
  }
@| current_file @}

Create menu bar and set up menu button callbacks.

@d Create menu bar @{
pack [frame .mbar -relief raised -bd 2] \
     [frame .sheet] -side top -fill x

set mf .mbar.file
set mo .mbar.options

menubutton $mf -menu $mf.menu -underline 0 -text File
menubutton $mo -menu $mo.menu -underline 0 -text Options 

menu $mf.menu
$mf.menu add command -label "About.." -underline 0 -command { about  }
$mf.menu add command -label "Read"    -underline 0 -command { getfile  }
$mf.menu add command -label "Print"   -underline 0 -command { printfile }
$mf.menu add command -label "Save"    -underline 0 -command { savefile }
$mf.menu add command -label "Exit"    -underline 1 -command { destroyP }

menu $mo.menu
$mo.menu add radiobutton -variable .sheet.ordtype \
	-label "PO"  -value "Procurement request"
$mo.menu add radiobutton -variable .sheet.ordtype \
	-label "BPA" -value "BPA & Credit Card Order Form"
if {[set .sheet.ordtype]==""} {
	set .sheet.ordtype "BPA & Credit Card Order Form"
}

pack $mf $mo -side left
tk_menuBar .mbar $mf $mo
focus .mbar
@}

Now construct the main window. Bottom line is packed first to leave the middle
for the left-to-right spreadsheet columns. The spreadsheet window is created
in a trace callback for setting the value of \verb=Numlines= variable. The 
\verb-prevNumlines- variable is set to an impossible value so that first redraw
is always executed (see procedure \verb-redrawSpreadsheet-).

@d Create main window @{
  @<Create miscellaneous data part@>
  @<Create the bottom line@>
  trace variable Numlines w redrawSpreadsheet
  set  prevNumlines -999 
  set     Numlines 10
@| Numlines @}

@d Create miscellaneous data part @{
pack [label .sheet.title -font $bigfont -textvariable .sheet.ordtype] -side top 
set Date [exec date {+%B %d, %y}]
pack [label.entry .sheet.date Date $Date] -anchor nw -fill x
pack [frame .sheet.who] -fill x
  pack [label.entry .sheet.req  Requisitioner "Przemek Klosowski"] \
	-in .sheet.who -side left	
  pack [label.entry .sheet.tel  Telephone "x6249"] \
	-in .sheet.who -side right
pack [label.entry .sheet.datr "Date required" ] -anchor nw
pack [frame .sheet.2] -fill x
  pack [label .sheet.vendl -text Vendor] \
	-in .sheet.2 -side left
  pack [label .sheet.shiptol -text "Ship to"] \
	-in .sheet.2 -side right
pack [frame .sheet.3] -fill x
  pack [text  .sheet.vendor -relief sunken  -bd 2 -height 4 -width 30] \
	-in .sheet.3 -side left -anchor nw
  pack [text  .sheet.shipto -relief sunken  -bd 2 -height 2 -width 20] \
	-in .sheet.3 -side right -anchor ne
pack [label .sheet.superil -text "Supervisor initials"] -anchor ne
@}

@d Create the bottom line @{
pack [frame .sheet.totalframe] -side bottom -fill x -anchor se -pady 11
  label .sheet.totalfig -textvariable TotalFigure -width 7 -relief raised
  pack .sheet.totalfig \
	-in .sheet.totalframe \
	-side right -anchor n
  pack [label .sheet.totalL -text TOTAL] \
	-in .sheet.totalframe \
	-side right -anchor n -padx 20

  pack [button .sheet.numlinesD -relief raised  -text <  -command { incr Numlines -1 }] \
	-in .sheet.totalframe   -side left -anchor n
  pack [entry .sheet.numlines   -relief raised  -textvariable Numlines -width 3] \
	-in .sheet.totalframe   -side left -anchor n
  pack [button .sheet.numlinesI -relief raised  -text >  -command {incr Numlines}] \
	-in .sheet.totalframe   -side left -anchor n
@}

Redraw the spreadsheet. Optimization: if the number of lines was just
increased or decreased, presumably via button presses, don't tear down
the whole window, but rather add or destroy appropriate windows and
variables and return.

@d Procedures @{
proc redrawSpreadsheet {name line op} {
  global Columns Colnames Colwidth Numlines prevNumlines total 
  if {$Numlines == $prevNumlines+1} {
     createLine $prevNumlines     
     set prevNumlines $Numlines
     return
  }
  if {$Numlines == $prevNumlines-1} {
     delLine $Numlines
     set prevNumlines $Numlines
     return
  }
  set prevNumlines $Numlines
  catch {destroy .sheet.spread}
  @<Create column headers@>
  @<Create columns@>
}
@| redrawSpreadsheet @}

Procedure to delete last line is a little more complicated, because
we need to take out the corresponding variable, to prevent the total
from including the non-displayed line.

Delete the columns of the line, including the \verb-total- field, and unset
the associated variables. The unset callback \verb-updateTotal- will be called
via the trace mechanism.

@d Procedures @{
proc delLine {line} {
   global Columns total
   foreach col $Columns {
      destroy .sheet.spread.$col.$line
      global $col
      unset ${col}($line)
   }
   destroy .sheet.spread.total.$line
   unset total($line)
}
@| delLine @}

First, set up column attributes

@d Global setup @{
set Columns  { item descr        qty  disct   upric       }
set Colnames { Item Description  Qty "%disc" "Unit Price" }
set Colwidth { 10   60           3    4       7           }
@| Columns Colnames Colwidth @}

@d Create column headers @{
pack [frame .sheet.spread] -side left -fill x -expand 1
set i 0
foreach col $Columns {
   pack [frame .sheet.spread.$col] -side left -fill x -expand 1
   pack [label .sheet.spread.$col.label \
	   -text [lindex $Colnames $i] \
   	]
   incr i
}
pack [frame .sheet.spread.total] -side left 
pack [label .sheet.spread.total.label -text Total]

@}

@d Create columns @{
for {set j 0} {$j<$Numlines} {incr j} {
      createLine $j
}
@}

@d Procedures @{
proc createLine {line} {
    global Columns Colwidth total

    for {set column 0} {$column<[llength $Columns]} {incr column} {
	set col [lindex $Columns $column]
        pack [entry .sheet.spread.$col.$line \
		-relief sunken\
		-width [lindex $Colwidth $column] \
		-textvariable ${col}($line) \
	     ] -fill x -expand 1
        global $col
	trace variable ${col}($line) w updateLineTotal
	bind .sheet.spread.$col.$line <Return>    "move_over $column+1 $line"
	bind .sheet.spread.$col.$line <Tab>       "move_over $column+1 $line"
	bind .sheet.spread.$col.$line <Shift-Tab> "move_over $column-1 $line"
	bind .sheet.spread.$col.$line <Up>        "move_over $column   $line-1"
	bind .sheet.spread.$col.$line <Down>      "move_over $column   $line+1"
    }
    pack [entry .sheet.spread.total.$line -relief sunken  -width 7 -textvariable total($line)]
    .sheet.spread.total.$line configure -state disabled
    trace variable total($line) wu updateTotal
}
@| createLine @}

This procedure does a keyboard traversal with appropriate boundary
conditions: torus-like , except when leaving a row to the left
(right), in which case we will end up in previous (or, respectively,
next) row.

@d Procedures @{
proc move_over {col row} {
    global Columns Numlines
    set col [expr $col]
    if {$col==[llength $Columns]} {incr row}
    if {$col==-1}                 {incr row -1}
    set row [expr ($row)%$Numlines]
    set col [expr ($col)%[llength $Columns]]
    focus .sheet.spread.[lindex $Columns $col].$row
}
@| move_over @}

\subsection{Postscript printing}

Procedure \verb-widgets2ps- renders the text in a given window and its
children in PostScript. If a second argument is present, it is the
file name to which the PostScript will be written; otherwise the
procedure returns the actual PostScript code.

@d Procedures @{
proc widgets2ps {w {filename ""}} {
   set Date [exec date]

   @<Calculate scaling@>
   @<Generate PostScript@>

   if {$filename!=""} { 
	set PSfile [open $filename w] 
	puts $PSfile $PS
	close $PSfile
   }
   return $PS
}
@| widgets2ps PSfile @}

The window is scaled to fit in an arbitrarily chosen PostScript bounding
box of (30,100; 600,700). Some slack (20 points) is allocated at the
right edge to allow for fonts that are wider on paper than on screen.
Conversion factor (\verb-Xcoef- and \verb-Ycoef-) is selected to scale
the image to fit within the bounding box. Note that \verb-Xcoef==Ycoef-,
so that the aspect ratio is not modified.

@d Calculate scaling @{
   set BBoxMinX  30 
   set BBoxMinY 100
   set BBoxMaxX 600
   set BBoxMaxY 700

   set Width  [winfo width  .]  
   set Height [winfo height .]
   # to get floating point division
   set BBoxWidth  [expr 1.0*$BBoxMaxX-$BBoxMinX-20.0]
   set BBoxHeight [expr 1.0*$BBoxMaxY-$BBoxMinY]

   # We set X/Ycoef and tighten up BBox
   if {$Width/$Height >  $BBoxWidth/$BBoxHeight} {
     # widget wider than BBox
     set Xcoef [expr $BBoxWidth/$Width]
     set Ycoef $Xcoef
     set BBoxMaxY [expr $BBoxMinY+$Height/$Ycoef]
   } else {
     # widget taller than BBox
     set Ycoef [expr $BBoxHeight/$Height]
     set Xcoef $Ycoefx
     set BBoxMaxX [expr $BBoxMinX+$Width/$Xcoef]
   }
@}

Collect the PostScript commands in the string variable \verb-PS-.
The \verb-translate- command set up PostScript coordinate system so
that (0,0) corresponds to top left corner of the widget being printed.
The scale needs to remain in X pixels rather than in PostScript units
because font sizes are expressed in pixels. The conversion factors
will be passed down by \verb-upvar-.  The calculations use 580 as the
right margin, rather than 600, to correct for differences in font
metrics.

@d Generate PostScript@{
   append PS "%!PS-Adobe-3.0 EPSF-3.0\n"
   append PS "%%Creator: Purchase, v. 1.0\n"
   append PS "%%Title: Purchase order printout\n"
   append PS "%%CreationDate: $Date\n"
   append PS "%%BoundingBox: $BBoxMinX $BBoxMinY $BBoxMaxX $BBoxMaxY\n"
   append PS "%%Pages: 1\n"
   append PS "%%DocumentData: Clean7Bit\n"
   append PS "%%Orientation: Portrait\n"
   append PS "%%EndComments\n"
   
   append PS "\n\n 30 700 translate\n"   
   
   printwidgets $w 0 0
   append PS "showpage\n"
@}

@d Procedures @{
proc printwidgets {w X Y} {
   upvar Xcoef Xcoef Ycoef Ycoef PS PS

   foreach widget [winfo children $w] {
	set class [winfo class $widget]
	set value ""
	set x [expr  $Xcoef*($X+[winfo x $widget])]
	set y [expr -$Ycoef*($Y+[winfo y $widget])]
#        outPSstring $x $y-10 [list Adobe Helvetica 50] \
#           [list - $w:$widget ($class) - $X $Y [winfo x $widget] [winfo y $widget] $x $y]
	if {$class=="Entry"} { 
          @<Handle Entry widgets@>
	}
	if {$class=="Label"} { 
	    set value [lindex [$widget configure -text] 4]
            outPSstring $x $y [getfont $widget] $value
	}
	if {$class=="Text"} { 
          @<Handle Text widgets@>          
	}
	if {$class=="Frame"} { 
	    printwidgets $widget [expr $X+[winfo x $widget]] [expr $Y+[winfo y $widget]]
	}
   }
}
@| printwidgets @}

Adjust $y$ position by the bottom margin within entry window (window
coordinates are expressed in pixels and text size are in pixels times
10). No attempt is made to clip the text that overflows the Entry widget
on screen.

@d Handle Entry widgets @{
	    set value [$widget get] 
	    set fontInfo [getfont $widget]
	    set txtSize [lindex $fontInfo 2]
	    set y [expr $y+$Ycoef*([winfo height $widget]-$txtSize/10.0)/2.0]
            outPSstring $x $y $fontInfo $value
@}

Text is handled in a very primitive way; no attempt is made to pick up
font changes within text. We do trim the text above and/or below the
visible Text window, but, again, no trimming on the left and right
sides is done.

@d Handle Text widgets @{
          set fontInfo [getfont $widget]
          set first [$widget index @@0,0]
          regexp {([0-9]+)\.([0-9]+)} $first foo firstline firstchar
          for {set i $firstline} { $i<[lindex [$widget configure -height] 3] } {incr i} {
            set line [$widget get $i.$firstchar "$i.$firstchar lineend"]
            outPSstring $x $y $fontInfo $line
            set y [expr $y-[lindex $fontInfo 2]/10*1.1]
          }
@}

Procedure to get font name and size associated with the widget. 
It returns foundry, font name and font size in tenths of a point.

@d Procedures @{
proc getfont {w} {
	set fname [lindex [$w configure -font] 4]
	regexp {[-]([^-]*)-([^-]*-[^-]*)-.*-([0-9]+)-.*} $fname foo \
	            foundry font             size
	return [list  $foundry $font $size]
}
@| getfont @}

Put a Postscript \verb-string- rendered in \verb-font- at \verb-(x,y)-.

There should be a way to override this mapping. One possibility is to
move \verb-FontMap- to 

This procedure outputs a string at position \verb-(x,y)-. The 
Y coordinate points to upper left corner of the string bounding
box, and so it has to be corrected.

@d Procedures @{
proc outPSstring {x y font string} {
	upvar PS PS 

	# Set up font mapping for some often-used fonts
	set FontMap(Times-Medium) Times-Roman
	set FontMap(Helvetica-Medium) Helvetica

	set fontname [lindex $font 1]
	if [info exists FontMap($fontname)] {set fontname $FontMap($fontname)}
        # need slightly smaller font size; PS font metrics seem to be a little wider
	set fontsize [expr [lindex $font 2]/10.0/1.1]
        set y [expr $y-$fontsize]
        append PS "/$fontname findfont $fontsize scalefont setfont\n"
        append PS "$x $y moveto ($string) show\n"
}
@| outPSstring @}

\subsection{Miscellaneous routines}

Code to put out a label/entry pair for data entry.

@d Procedures @{
proc label.entry  {wfield labeltext {initval ""} {width 20}} {
	global $wfield
	frame $wfield
	label $wfield.lab -text $labeltext -relief flat
	entry $wfield.val -relief sunken -bd 1 -width $width -textvariable $wfield
	set $wfield $initval
	pack $wfield.lab $wfield.val -side left -padx 5
	return $wfield
}
@| label.entry @}

@d Procedures @{
proc updateLineTotal {name line op} {
   global qty disct upric total current_file_dirty
   set total($line) [expr \
	[Flt $qty($line)]*(1.0-[Flt $disct($line)]/100.0)*[Flt $upric($line)]]
   set current_file_dirty 1
}
@| updateLineTotal current_file_dirty @}

We need the procedure \verb-Flt- to handle fields without a value
(empty string).  Otherwise we would be getting errors from
\verb-expr----and somehow when these errors occur within the trace
callback, I wasn't getting the error messages, which made for an
interesting debugging session.

@d Procedures @{
proc Flt {f} {if {$f==""} {return 0} ; return $f}
@| Flt @}

Update the total of all lines. This procedure is called whenever any
line total changes.

@d Procedures @{
proc updateTotal {name line op} {
   global qty disct upric total TotalFigure Numlines 
   set TotalFigure 0
   for {set i 0} {$i<$Numlines} {incr i} {
       if [info exists total($i)] { 
	   set TotalFigure [expr $TotalFigure+[Flt $total($i)]]}
   }
}
@| updateTotal @}

Procedure to save the PO data to a file. The file will be a regular
Tk code, to be sourced later by \verb-getfile-.

@d Procedures @{
proc savefile {} {
   global Numlines Columns item descr qty disct upric total current_file  current_file_dirty

  if {[info vars current_file]=={}} {set current_file "my.order"}
  set fname [pkl_OutFileSelectorBox .f {Save file selection} $current_file]
  if {$fname != ""} {

   set current_file $fname
   set file [open $fname w]

   foreach fld { ordtype date req tel datr } {
	global .sheet.$fld
        puts $file "set .sheet.$fld [list [set .sheet.$fld]]"
   }
   puts $file  ".sheet.vendor delete 1.0 end"
   puts $file  ".sheet.shipto delete 1.0 end"
   puts $file  ".sheet.vendor insert 1.0 [list [.sheet.vendor get 1.0 end]]"
   puts $file  ".sheet.shipto insert 1.0 [list [.sheet.shipto get 1.0 end]]"

   puts $file "set Numlines $Numlines"
   for {set j 0} {$j<$Numlines} {incr j} {
      foreach col $Columns {
	set val [list [set ${col}($j)]]
        puts $file "set ${col}($j) $val"
      }
      # I don't need totals, because they will be recomputed by traces
      #puts $file "set total($j) $total($j)"
   }
   close $file
  }
  set current_file_dirty 0
}
@| savefile @}

@d Procedures @{
proc pkl_OutFileSelectorBox {w title current_file} {
  global $w.t.labent

  catch {destroy $w};   toplevel $w
  wm title $w $title
  pack [frame $w.t -relief groove -bd 2] [frame $w.b -bd 2]
  pack [label.entry  $w.t.labent File $current_file]
  bind $w.t.labent.val <Return>  "destroy $w"
  bind $w.t.labent.val <KP_Enter>  "destroy $w"
  bind $w.t.labent.val <Double-Button-1>  "destroy $w"
  pack [frame $w.b.default -relief sunken -bd 1] -side left -expand 1 -padx 3m -pady 1m
  pack [button $w.b.default.ok -text OK -command "destroy $w"] -expand 1 -padx 1m -pady 1m
  pack [button $w.b.can -text Cancel -command "set $w.t.labent {}; destroy $w"] -side right -padx 3m

  set oldfocus [focus];   grab $w ;   focus $w.t.labent.val
  tkwait window $w ;                  focus $oldfocus
  return [set $w.t.labent]
}
@| pkl_OutFileSelectorBox@}

For loading data, we just read the file and execute the commands within.
The callback is simply Tcl's \verb-source-.

@d Procedures @{
proc getfile {} {
  global current_file
  set current_file [gp_SingleFileSelectorBox .f {Input file selection} *.order]
  if {$current_file != ""} {
     uplevel "source $current_file"
  }
}
@| getfile current_file @}

@d Procedures @{
proc printfile {} {
   widgets2ps .sheet dump.ps
   exec lpr -P5 dump.ps
   exec /bin/rm dump.ps
}
@| printfile  @}

@d Procedures @{
proc destroyP {} {
  global current_file_dirty

  if {$current_file_dirty} {
    set res [tk_dialog .error {File not saved} "Your work hasn't been saved" error 0 Save Exit Cancel]
    if {$res==0} {
        savefile
    } elseif {$res==2} {
        return
    }
  }
  destroy .
}
@| destroyP @}

@d Procedures @{
proc about {} {
  global Revision_string
  tk_dialog .about {About...} \
      "Purchase order spreadsheet\n$Revision_string\nPrzemek Klosowski\nprzemek@@nist.gov" info 0 OK
}
@| about @}

\twocolumn
\subsection*{File index}
\addcontentsline{toc}{section}{Indices}
@f
\subsection*{Code scrap index}
@m
\subsection*{Identifier index}
@u

\end{document}
