/* 
 * tkWinInit.c --
 *
 *	This file contains Windows-specific interpreter initialization
 *	functions.
 *
 * Copyright (c) 1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkWinInit.c 1.13 96/03/18 14:22:29
 */

#include "tkWinInt.h"

/*
 * The following string is the startup script executed in new
 * interpreters.  It looks on disk in several different directories
 * for a script "tk.tcl" that is compatible with this version
 * of Tk.  The tk.tcl script does all of the real work of
 * initialization.
 */

static char *initScript =
"proc init {} {\n\
    global tk_library tk_version tk_patchLevel env\n\
    rename init {}\n\
    set dirs {}\n\
    if [info exists env(TK_LIBRARY)] {\n\
	lappend dirs $env(TK_LIBRARY)\n\
    }\n\
    lappend dirs $tk_library\n\
    lappend dirs [file dirname [info library]]/lib/tk$tk_version\n\
    lappend dirs [file dirname [file dirname [info nameofexecutable]]]/lib/tk$tk_version\n\
    if [string match {*[ab]*} $tk_patchLevel] {\n\
	set lib tk$tk_patchLevel\n\
    } else {\n\
	set lib tk$tk_version\n\
    }\n\
    lappend dirs [file dirname [file dirname [pwd]]]/$lib/library\n\
    lappend dirs [file dirname [pwd]]/library\n\
    foreach i $dirs {\n\
	set tk_library $i\n\
	if ![catch {uplevel #0 source [list $i/tk.tcl]}] {\n\
	    return\n\
	}\n\
    }\n\
    set msg \"Can't find a usable tk.tcl in the following directories: \n\"\n\
    append msg \"    $dirs\n\"\n\
    append msg \"This probably means that Tk wasn't installed properly.\n\"\n\
    error $msg\n\
}\n\
init";

#ifndef TCL_ACTIVE
#define TCL_ACTIVE (1<<4)
#endif

static void		DisplayCheckProc _ANSI_ARGS_((ClientData clientData,
			    int flags));
static void		DisplaySetupProc _ANSI_ARGS_((ClientData clientData,
			    int flags));

/*
 *----------------------------------------------------------------------
 *
 * DisplaySetupProc --
 *
 *	This procedure is part of the event source for Mac displays.
 *	It is invoked by Tcl_DoOneEvent before it calls select to check
 *	for events on all displays. Because for the Mac this kind of
 *	events is handled already in Tcl, all that needs to be done is
 *	check if there are any windows open.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Tells the notifier whether the Tk event-handler is active or not.
 *
 *----------------------------------------------------------------------
 */

static void
DisplaySetupProc(clientData, flags)
    ClientData clientData;		/* Not used. */
    int flags;				/* Flags passed to Tk_DoOneEvent:
					 * if it doesn't include
					 * TCL_WINDOW_EVENTS then we do
					 * nothing. */
{
    if ((flags & TCL_WINDOW_EVENTS) && (Tk_GetNumMainWindows()>0)) {
	Tcl_WatchFile((Tcl_File) NULL, TCL_ACTIVE);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayCheckProc --
 *
 *	This procedure is just a dummy function, because Mac events
 *	are handled in Tcl.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
DisplayCheckProc(clientData, flags)
    ClientData clientData;		/* Not used. */
    int flags;				/* Not used. */
{
    return;
}

/*
 *----------------------------------------------------------------------
 *
 * TkPlatformInit --
 *
 *	Performs Windows-specific interpreter initialization related to the
 *      tk_library variable.
 *
 * Results:
 *	A standard Tcl completion code (TCL_OK or TCL_ERROR).  Also
 *	leaves information in interp->result.
 *
 * Side effects:
 *	Sets "tk_library" Tcl variable, runs "tk.tcl" script.
 *
 *----------------------------------------------------------------------
 */

static char **initScripts = NULL;

int
TkPlatformInit(interp)
    Tcl_Interp *interp;
{
    char *libDir;
    static int initialized = 0;

    if (!initialized) {
	Tcl_CreateEventSource(DisplaySetupProc, DisplayCheckProc,
		(ClientData) NULL);
	initialized = 1;
    }

    libDir = Tcl_GetVar(interp, "tk_library", TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
	Tcl_SetVar(interp, "tk_library", ".", TCL_GLOBAL_ONLY);
    }

    if (initScripts != NULL) {
	char **p = initScripts;
	Tcl_DString data;

	Tcl_SetVar(interp, "tk_library", "", TCL_GLOBAL_ONLY);
	Tcl_DStringInit(&data);
	while(*p) {
	    /* Copy the constant into a dynamic string. This */
	    /* is necessary because Tcl7.5 doesn't accept    */
	    /* constants as an argument to Tcl_Eval()        */
	    Tcl_DStringSetLength(&data,0);
	    Tcl_DStringAppend(&data,*p++,-1);
	    if(Tcl_Eval(interp,Tcl_DStringValue(&data)) == TCL_ERROR) {
		Tcl_DStringFree(&data);
		return TCL_ERROR;
	    }
	}
	Tcl_DStringFree(&data);
	return TCL_OK;
    }
    return Tcl_Eval(interp, initScript);
}

/*
 *----------------------------------------------------------------------
 *
 * TkSetInitScript --
 *
 *	This procedure sets the initialization script used in Tk_Init().
 *	It will be used instead of the file "tk.tcl" in all future calls
 *	to Tk_Init. If the argument is NULL, tk.tcl will be used again.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

void
TkSetInitScript(script)
    char  **script;		/* Script to be executed. */
{
    initScripts = script;
}
