/*
 * tkImgGIF.c --
 *
 * A photo image file handler for GIF files. Reads 87a and 89a GIF files.
 * At present THERE IS WRITE function for 87a GIF.
 *
 * Derived from the giftoppm code found in the pbmplus package
 * the LUG package develop by Raul Rivero, the Xiaolin Wu quantize function 
 * and tkImgFmtPPM.c in the tk4.0b2 distribution by -
 *
 * Reed Wade (wade@cs.utk.edu), University of Tennessee
 *
 * Copyright (c) 1995-1996 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * This file also contains code from the giftoppm and the ppmtogif programs, 
 * which is copyrighted as follows:
 *
 * +-------------------------------------------------------------------+
 * | Copyright 1990, David Koblas.                                     |
 * |   Permission to use, copy, modify, and distribute this software   |
 * |   and its documentation for any purpose and without fee is hereby |
 * |   granted, provided that the above copyright notice appear in all |
 * |   copies and that both that copyright notice and this permission  |
 * |   notice appear in supporting documentation.  This software is    |
 * |   provided "as is" without express or implied warranty.           |
 * +-------------------------------------------------------------------+
 *
 * SCCS: @(#) tkImgGIF.c 1.7 96/04/09 17:11:46
 */

#include "tkInt.h"
#include "tkPort.h"

/*
 * The format record for the GIF file format:
 */

static int      FileMatchGIF _ANSI_ARGS_((FILE *f, char *fileName,
		    char *formatString, int *widthPtr, int *heightPtr));
static int      FileReadGIF  _ANSI_ARGS_((Tcl_Interp *interp,
		    FILE *f, char *fileName, char *formatString,
		    Tk_PhotoHandle imageHandle, int destX, int destY,
		    int width, int height, int srcX, int srcY));
static int 	FileWriteGIF _ANSI_ARGS_(( Tcl_Interp *interp,  
				char *filename, char *formatString, 
				Tk_PhotoImageBlock *blockPtr));

Tk_PhotoImageFormat tkImgFmtGIF = {
	"GIF",			/* name */
	FileMatchGIF,   /* fileMatchProc */
	NULL,           /* stringMatchProc */
	FileReadGIF,    /* fileReadProc */
	NULL,           /* stringReadProc */
	FileWriteGIF,           /* fileWriteProc */
	NULL,           /* stringWriteProc */
};

#define INTERLACE		0x40
#define LOCALCOLORMAP		0x80
#define BitSet(byte, bit)	(((byte) & (bit)) == (bit))
#define MAXCOLORMAPSIZE		256
#define CM_RED			0
#define CM_GREEN		1
#define CM_BLUE			2
#define MAX_LWZ_BITS		12
#define LM_to_uint(a,b)         (((b)<<8)|(a))
#define ReadOK(file,buffer,len)	(fread(buffer, len, 1, file) != 0)

/*
 * Prototypes for local procedures defined in this file:
 */

static int		DoExtension _ANSI_ARGS_((FILE *fd, int label,
			    int *transparent));
static int		GetCode _ANSI_ARGS_((FILE *fd, int code_size,
			    int flag));
static int		GetDataBlock _ANSI_ARGS_((FILE *fd,
			    unsigned char *buf));
static int		LWZReadByte _ANSI_ARGS_((FILE *fd, int flag,
			    int input_code_size));
static int		ReadColorMap _ANSI_ARGS_((FILE *fd, int number,
			    unsigned char buffer[3][MAXCOLORMAPSIZE]));
static int		ReadGIFHeader _ANSI_ARGS_((FILE *f, int *widthPtr,
			    int *heightPtr));
static int		ReadImage _ANSI_ARGS_((Tcl_Interp *interp,
			    char *imagePtr, FILE *fd, int len, int height,
			    unsigned char cmap[3][MAXCOLORMAPSIZE],
			    int interlace, int transparent));

/*
 *----------------------------------------------------------------------
 *
 * FileMatchGIF --
 *
 *  This procedure is invoked by the photo image type to see if
 *  a file contains image data in GIF format.
 *
 * Results:
 *  The return value is 1 if the first characters in file f look
 *  like GIF data, and 0 otherwise.
 *
 * Side effects:
 *  The access position in f may change.
 *
 *----------------------------------------------------------------------
 */

static int
FileMatchGIF(f, fileName, formatString, widthPtr, heightPtr)
    FILE *f;			/* The image file, open for reading. */
    char *fileName;		/* The name of the image file. */
    char *formatString;		/* User-specified format string, or NULL. */
    int *widthPtr, *heightPtr;	/* The dimensions of the image are
				 * returned here if the file is a valid
				 * raw GIF file. */
{
	return ReadGIFHeader(f, widthPtr, heightPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * FileReadGIF --
 *
 *	This procedure is called by the photo image type to read
 *	GIF format data from a file and write it into a given
 *	photo image.
 *
 * Results:
 *	A standard TCL completion code.  If TCL_ERROR is returned
 *	then an error message is left in interp->result.
 *
 * Side effects:
 *	The access position in file f is changed, and new data is
 *	added to the image given by imageHandle.
 *
 *----------------------------------------------------------------------
 */

static int
FileReadGIF(interp, f, fileName, formatString, imageHandle, destX, destY,
	width, height, srcX, srcY)
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
    FILE *f;			/* The image file, open for reading. */
    char *fileName;		/* The name of the image file. */
    char *formatString;		/* User-specified format string, or NULL. */
    Tk_PhotoHandle imageHandle;	/* The photo image to write into. */
    int destX, destY;		/* Coordinates of top-left pixel in
				 * photo image to be written to. */
    int width, height;		/* Dimensions of block of photo image to
				 * be written to. */
    int srcX, srcY;		/* Coordinates of top-left pixel to be used
				 * in image being read. */
{
    int fileWidth, fileHeight;
    int nBytes;
    Tk_PhotoImageBlock block;
    unsigned char buf[100];
    int bitPixel;
    unsigned int colorResolution;
    unsigned int background;
    unsigned int aspectRatio;
    unsigned char localColorMap[3][MAXCOLORMAPSIZE];
    unsigned char colorMap[3][MAXCOLORMAPSIZE];
    int useGlobalColormap;
    int transparent = -1;

    if (!ReadGIFHeader(f, &fileWidth, &fileHeight)) {
	Tcl_AppendResult(interp, "couldn't read GIF header from file \"",
		fileName, "\"", NULL);
	return TCL_ERROR;
    }
    if ((fileWidth <= 0) || (fileHeight <= 0)) {
	Tcl_AppendResult(interp, "GIF image file \"", fileName,
		"\" has dimension(s) <= 0", (char *) NULL);
	return TCL_ERROR;
    }

    if (fread(buf, 1, 3, f) != 3) {
	return TCL_OK;
    }
    bitPixel = 2<<(buf[0]&0x07);
    colorResolution = (((buf[0]&0x70)>>3)+1);
    background = buf[1];
    aspectRatio = buf[2];

    if (BitSet(buf[0], LOCALCOLORMAP)) {    /* Global Colormap */
	if (!ReadColorMap(f, bitPixel, colorMap)) {
	    Tcl_AppendResult(interp, "error reading color map",
		    (char *) NULL);
	    return TCL_ERROR;
	}
    }

    if ((srcX + width) > fileWidth) {
	width = fileWidth - srcX;
    }
    if ((srcY + height) > fileHeight) {
	height = fileHeight - srcY;
    }
    if ((width <= 0) || (height <= 0)
	    || (srcX >= fileWidth) || (srcY >= fileHeight)) {
	return TCL_OK;
    }

    Tk_PhotoExpand(imageHandle, destX + width, destY + height);

    block.width = fileWidth;
    block.height = fileHeight;
    block.pixelSize = 3;
    block.pitch = 3 * fileWidth;
    block.offset[0] = 0;
    block.offset[1] = 1;
    block.offset[2] = 2;
    nBytes = fileHeight * block.pitch;
    block.pixelPtr = (unsigned char *) ckalloc((unsigned) nBytes);

    while (1) {
	if (fread(buf, 1, 1, f) != 1) {
	    /*
	     * Premature end of image.  We should really notify
	     * the user, but for now just show garbage.
	     */

	    break;
	}

	if (buf[0] == ';') {
	    /*
	     * GIF terminator.
	     */

	    break;
	}

	if (buf[0] == '!') {
	    /*
	     * This is a GIF extension.
	     */

	    if (fread(buf, 1, 1, f) != 1) {
		interp->result =
			"error reading extension function code in GIF image";
		goto error;
	    }
	    if (DoExtension(f, buf[0], &transparent) < 0) {
		interp->result = "error reading extension in GIF image";
		goto error;
	    }
	    continue;
	}

	if (buf[0] != ',') {
	    /*
	     * Not a valid start character; ignore it.
	     */
	    continue;
	}

	if (fread(buf, 1, 9, f) != 9) {
	    interp->result = "couldn't read left/top/width/height in GIF image";
	    goto error;
	}

	useGlobalColormap = ! BitSet(buf[8], LOCALCOLORMAP);

	bitPixel = 1<<((buf[8]&0x07)+1);

	if (!useGlobalColormap) {
	    if (!ReadColorMap(f, bitPixel, localColorMap)) {
		    Tcl_AppendResult(interp, "error reading color map", 
			    (char *) NULL);
		    goto error;
	    }
	    if (ReadImage(interp, (char *) block.pixelPtr, f, fileWidth,
		    fileHeight, localColorMap, BitSet(buf[8], INTERLACE),
		    transparent) != TCL_OK) {
		goto error;
	    }
	} else {
	    if (ReadImage(interp, (char *) block.pixelPtr, f, fileWidth,
		    fileHeight, colorMap, BitSet(buf[8], INTERLACE),
		    transparent) != TCL_OK) {
		goto error;
	    }
	}

    }

    Tk_PhotoPutBlock(imageHandle, &block, destX, destY, fileWidth, fileHeight);
    ckfree((char *) block.pixelPtr);
    return TCL_OK;

    error:
    ckfree((char *) block.pixelPtr);
    return TCL_ERROR;

}

/*
 *----------------------------------------------------------------------
 *
 * ReadGIFHeader --
 *
 *	This procedure reads the GIF header from the beginning of a
 *	GIF file and returns the dimensions of the image.
 *
 * Results:
 *	The return value is 1 if file "f" appears to start with
 *	a valid GIF header, 0 otherwise.  If the header is valid,
 *	then *widthPtr and *heightPtr are modified to hold the
 *	dimensions of the image.
 *
 * Side effects:
 *	The access position in f advances.
 *
 *----------------------------------------------------------------------
 */

static int
ReadGIFHeader(f, widthPtr, heightPtr)
    FILE *f;			/* Image file to read the header from */
    int *widthPtr, *heightPtr;	/* The dimensions of the image are
				 * returned here. */
{
    unsigned char buf[7];

    if ((fread(buf, 1, 6, f) != 6)
	    || ((strncmp("GIF87a", (char *) buf, 6) != 0)
	    && (strncmp("GIF89a", (char *) buf, 6) != 0))) {
	return 0;
    }

    if (fread(buf, 1, 4, f) != 4) {
	return 0;
    }

    *widthPtr = LM_to_uint(buf[0],buf[1]);
    *heightPtr = LM_to_uint(buf[2],buf[3]);
    return 1;
}

/*
 *-----------------------------------------------------------------
 * The code below is copied from the giftoppm program and modified
 * just slightly.
 *-----------------------------------------------------------------
 */

static int
ReadColorMap(fd,number,buffer)
FILE        *fd;
int     number;
unsigned char   buffer[3][MAXCOLORMAPSIZE];
{
	int     i;
	unsigned char   rgb[3];

	for (i = 0; i < number; ++i) {
		if (! ReadOK(fd, rgb, sizeof(rgb)))
			return 0;

		buffer[CM_RED][i] = rgb[0] ;
		buffer[CM_GREEN][i] = rgb[1] ;
		buffer[CM_BLUE][i] = rgb[2] ;
	}
	return 1;
}



static int
DoExtension(fd, label, transparent)
FILE    *fd;
int label;
int	*transparent;
{
	static unsigned char buf[256];
	int count = 0;

	switch (label) {
		case 0x01:      /* Plain Text Extension */
			break;

		case 0xff:      /* Application Extension */
			break;

		case 0xfe:      /* Comment Extension */
			do {
				count = GetDataBlock(fd, (unsigned char*) buf);
			} while (count > 0);
			return count;

		case 0xf9:      /* Graphic Control Extension */
			count = GetDataBlock(fd, (unsigned char*) buf);
			if (count < 0) {
				return 1;
			}
			if ((buf[0] & 0x1) != 0) {
				*transparent = buf[3];
			}

			do {
			    count = GetDataBlock(fd, (unsigned char*) buf);
			} while (count > 0);
			return count;
	}

	do {
	    count = GetDataBlock(fd, (unsigned char*) buf);
	} while (count > 0);
	return count;
}

static int ZeroDataBlock = 0;

static int
GetDataBlock(fd, buf)
FILE        *fd;
unsigned char   *buf;
{
	unsigned char   count;

	if (! ReadOK(fd,&count,1)) {
		return -1;
	}

	ZeroDataBlock = count == 0;

	if ((count != 0) && (! ReadOK(fd, buf, count))) {
		return -1;
	}

	return count;
}


static int
ReadImage(interp, imagePtr, fd, len, height, cmap, interlace, transparent)
Tcl_Interp *interp;
char 	*imagePtr;
FILE    *fd;
int len, height;
unsigned char   cmap[3][MAXCOLORMAPSIZE];
int interlace;
int transparent;
{
	unsigned char   c;
	int     v;
	int     xpos = 0, ypos = 0, pass = 0;
	char 	*colStr;


	/*
	 *  Initialize the Compression routines
	 */
	if (! ReadOK(fd,&c,1))  {
	    Tcl_AppendResult(interp, "error reading GIF image: ",
		    Tcl_PosixError(interp), (char *) NULL);
	    return TCL_ERROR;
	}

	if (LWZReadByte(fd, 1, c) < 0) {
	    interp->result = "format error in GIF image";
	    return TCL_ERROR;
	}

	if (transparent!=-1 && 
		(colStr = Tcl_GetVar(interp, "TRANSPARENT_GIF_COLOR", 0L))) {
		XColor *colorPtr;
		colorPtr = Tk_GetColor(interp, Tk_MainWindow(interp), 
							  Tk_GetUid(colStr));
		if (colorPtr) {
/*
			printf("color is %d %d %d\n", 
					colorPtr->red >> 8, 
					colorPtr->green >> 8, 
					colorPtr->blue >> 8);
*/
			cmap[CM_RED][transparent] = colorPtr->red >> 8;
			cmap[CM_GREEN][transparent] = colorPtr->green >> 8;
			cmap[CM_BLUE][transparent] = colorPtr->blue >> 8;
			Tk_FreeColor(colorPtr);
		}
	}

	while ((v = LWZReadByte(fd,0,c)) >= 0 ) {

		imagePtr[ (xpos*3)  +  (ypos *len*3)] = cmap[CM_RED][v];
		imagePtr[ (xpos*3)  +  (ypos *len*3) +1] = cmap[CM_GREEN][v];
		imagePtr[ (xpos*3)  +  (ypos *len*3) +2] = cmap[CM_BLUE][v];

		++xpos;
		if (xpos == len) {
			xpos = 0;
			if (interlace) {
				switch (pass) {
					case 0:
					case 1:
						ypos += 8; break;
					case 2:
						ypos += 4; break;
					case 3:
						ypos += 2; break;
				}

				if (ypos >= height) {
					++pass;
					switch (pass) {
						case 1:
							ypos = 4; break;
						case 2:
							ypos = 2; break;
						case 3:
							ypos = 1; break;
						default:
							return TCL_OK;
					}
				}
			} else {
				++ypos;
			}
		}
		if (ypos >= height)
			break;
	}
	return TCL_OK;
}

static int
LWZReadByte(fd, flag, input_code_size)
FILE    *fd;
int flag;
int input_code_size;
{
	static int  fresh = 0;
	int     code, incode;
	static int  code_size, set_code_size;
	static int  max_code, max_code_size;
	static int  firstcode, oldcode;
	static int  clear_code, end_code;
	static int  table[2][(1<< MAX_LWZ_BITS)];
	static int  stack[(1<<(MAX_LWZ_BITS))*2], *sp;
	register int    i;


	if (flag) {

		set_code_size = input_code_size;
		code_size = set_code_size+1;
		clear_code = 1 << set_code_size ;
		end_code = clear_code + 1;
		max_code_size = 2*clear_code;
		max_code = clear_code+2;

		GetCode(fd, 0, 1);

		fresh = 1;

		for (i = 0; i < clear_code; ++i) {
			table[0][i] = 0;
			table[1][i] = i;
		}
		for (; i < (1<<MAX_LWZ_BITS); ++i) {
			table[0][i] = table[1][0] = 0;
		}

		sp = stack;

		return 0;

	} else if (fresh) {

		fresh = 0;
		do {
			firstcode = oldcode = GetCode(fd, code_size, 0);
		} while (firstcode == clear_code);
		return firstcode;
	}

	if (sp > stack)
		return *--sp;

	while ((code = GetCode(fd, code_size, 0)) >= 0) {
		if (code == clear_code) {
			for (i = 0; i < clear_code; ++i) {
				table[0][i] = 0;
				table[1][i] = i;
			}

			for (; i < (1<<MAX_LWZ_BITS); ++i) {
				table[0][i] = table[1][i] = 0;
			}

			code_size = set_code_size+1;
			max_code_size = 2*clear_code;
			max_code = clear_code+2;
			sp = stack;
			firstcode = oldcode = GetCode(fd, code_size, 0);
			return firstcode;

	} else if (code == end_code) {
		int     count;
		unsigned char   buf[260];

		if (ZeroDataBlock)
			return -2;

		while ((count = GetDataBlock(fd, buf)) > 0)
			;

		if (count != 0)
			return -2;
	}

	incode = code;

	if (code >= max_code) {
		*sp++ = firstcode;
		code = oldcode;
	}

	while (code >= clear_code) {
		*sp++ = table[1][code];
		if (code == table[0][code]) {
			return -2;

			/*
			 * Used to be this instead, Steve Ball suggested
			 * the change to just return.

			printf("circular table entry BIG ERROR\n");
			*/
		}
		code = table[0][code];
	}

	*sp++ = firstcode = table[1][code];

	if ((code = max_code) <(1<<MAX_LWZ_BITS)) {

		table[0][code] = oldcode;
		table[1][code] = firstcode;
		++max_code;
		if ((max_code>=max_code_size) && (max_code_size < (1<<MAX_LWZ_BITS))) {
			max_code_size *= 2;
			++code_size;
		}
	}

	oldcode = incode;

	if (sp > stack)
		return *--sp;
	}
	return code;
}

static int
GetCode(fd, code_size, flag)
FILE    *fd;
int code_size;
int flag;
{
	static unsigned char    buf[280];
	static int      curbit, lastbit, done, last_byte;
	int         i, j, ret;
	unsigned char       count;

	if (flag) {
		curbit = 0;
		lastbit = 0;
		done = 0;
		return 0;
	}


	if ( (curbit+code_size) >= lastbit) {
		if (done) {
			/* ran off the end of my bits */
			return -1;
		}
		buf[0] = buf[last_byte-2];
		buf[1] = buf[last_byte-1];

		if ((count = GetDataBlock(fd, &buf[2])) == 0)
			done = 1;

		last_byte = 2 + count;
		curbit = (curbit - lastbit) + 16;
		lastbit = (2+count)*8 ;
	}

	ret = 0;
	for (i = curbit, j = 0; j < code_size; ++i, ++j)
		ret |= ((buf[ i / 8 ] & (1 << (i % 8))) != 0) << j;


	curbit += code_size;

	return ret;
}

/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */



/*
 * FileWriteGIF - writes a image in GIF format.
 *-------------------------------------------------------------------------
 * Author:          		Lolo
 *                              Engeneering Projects Area 
 *	            		Department of Mining 
 *                  		University of Oviedo
 * e-mail			zz11425958@zeus.etsimo.uniovi.es
 * Date:            		Fri Oct 20 1995
 *----------------------------------------------------------------------
 * FileWriteGIF-
 *
 *    This procedure is called by the photo image type to write
 *    GIF format data from a photo image into a given file 
 *
 * Results:
 *	A standard TCL completion code.  If TCL_ERROR is returned
 *	then an error message is left in interp->result.
 *
 *----------------------------------------------------------------------
 */

 /*
  *  Types, defines and variables needed to write and compress a GIF.
  *  Tipos, definiciones y variables necesarios para escribir y
  *  comprimir los gifs
  */


typedef int (* ifunptr)();	

#define LSB(a)                  ((byte) (((short)(a)) & 0x00FF))
#define MSB(a)                  ((byte) (((short)(a)) >> 8))

#define count_int                int 
#define code_int                 long int 
#define byte                    unsigned char
#define GIFBITS 12
#define HSIZE  5003            /* 80% occupancy */
#define LUGUSED                 12345

long int ssize;
unsigned char *pixelo;
long int  num;
unsigned char mapa[3][256];

/*
 * Uso un formato intermedio   ( un simple bitmap ) con 
 * este formato ...
 *
 * I use a intermediate format ( a simple bitmap ) with
 * this format ...
 */

typedef struct {
        int xsize, ysize;       /* sizes */
        int depth;              /* # of colors */
        int colors;             /* # of colors */
        byte *r, *g, *b;        /* components or bitmap (planes < 8) */
        byte *cmap;             /* cmap if planes < 8 */
        int magic;              /* used ? */
} bitmap_hdr;

/*
 *	Definition of new functions to write GIFs
 *
 *	Definicion de funciones necesarias para escribir los GIF
 */

static int      FileMatchGIF _ANSI_ARGS_((FILE *f, char *fileName,
		    char *formatString, int *widthPtr, int *heightPtr));
 int color _ANSI_ARGS_((int red,int green, int blue));
 int nuevo _ANSI_ARGS_((int red, int green ,int blue));
 int savemap _ANSI_ARGS_((Tk_PhotoImageBlock *blockPtr));
 int  ReadValue _ANSI_ARGS_(());
 int  compress _ANSI_ARGS_(( int init_bits,FILE *outfile, ifunptr ReadValue));
 int  no_bits _ANSI_ARGS_((int colors));
 int read_dib _ANSI_ARGS_((bitmap_hdr *image, int xsize, int ysize));
 int write_dib _ANSI_ARGS_((bitmap_hdr *image)); 
char *Malloc _ANSI_ARGS_((int));

static int 	FileWriteGIF (interp, filename, formatString, blockPtr)
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
  char	     *filename;
  char       *formatString;
  Tk_PhotoImageBlock *blockPtr;
{
   FILE *fp;
   int	B;
   int  resolution;
   long int  numcolormap;

  long int  width,height,num,x;
   char s[10];
    byte c;
    unsigned int top,left;
    char *cadena;
    int ReadValue();
    bitmap_hdr in,new;

	top = 0;
	left = 0;
	fp=fopen(filename,"w");
	fwrite("GIF87a",1,6,fp);

	for(x=0;x<256;x++)
	{
		mapa[0][x] = -1;
		mapa[1][x] = -1;
		mapa[2][x] = -1;
	}

	
	fflush(fp);
	width=blockPtr->width;
	height=blockPtr->height;
	pixelo=blockPtr->pixelPtr;

	if ((num=savemap(blockPtr))<0)
	{
		read_dib( &in,width,height);
		quantize(&in,&new,256);
		num=write_dib( &new  );
		free(&in.r);
		free(&in.g);
		free(&in.b);
		free(&in.cmap);
		free(&new.r);
		free(&new.g);
		free(&new.b);
		free(&new.cmap);
		
	}
	if (num<3) num=3;
	c=LSB(width);
	fputc(c,fp);
	c=MSB(width);
	fputc(c,fp);
	c=LSB(height);
	fputc(c,fp);
	c=MSB(height);
	fputc(c,fp);
	fflush(fp);
	resolution=blockPtr->pixelSize;
	
	fflush(fp);

	c= (1 << 7) | (no_bits(num) << 4) | (no_bits(num));
	fputc(c,fp);
	resolution = no_bits(num)+1;

	numcolormap=1 << resolution;
	c=0;

       /*  background color */
	
	fputc(c,fp);

       /*  cero for future expansion  */

	fputc(c,fp);
	fflush(fp);

	for(x=0; x<numcolormap ;x++)
		fprintf(fp,"%c%c%c",mapa[0][x],mapa[1][x],mapa[2][x]);
		
 	c='!'; fputc(c,fp); 
 	c=254; fputc(c,fp); 
	c=46; fputc(c,fp); 

	cadena="CREACION DE LOLO. Ultima modificacion 10/20/95";
	for(x=0;x<=45;x++)
		fputc(cadena[x],fp);

	c=0; fputc(c,fp);	

	c=',';
	fputc(c,fp);
	c=LSB(top);
	fputc(c,fp);
	c=MSB(top);
	fputc(c,fp);
	c=LSB(left);
	fputc(c,fp);
	c=MSB(left);
	fputc(c,fp);

	c=LSB(width);
	fputc(c,fp);
	c=MSB(width);
	fputc(c,fp);

	c=LSB(height);
	fputc(c,fp);
	c=MSB(height);
	fputc(c,fp);

	c=0;
	fputc(c,fp);
	c=resolution;
	fputc(c,fp);

        ssize = blockPtr->width * blockPtr->height;
	compress(resolution+1,fp,ReadValue);  
	fputc(0,fp);
	fputc(';',fp);

	fclose(fp);
	return TCL_OK;	
}

int color(red, green, blue)
    int red;
    int green;
    int blue;
{
	int x;
	for(x=0;x<=256;x++)
	{
		if ((mapa[0][x]==red)&&(mapa[1][x]==green)&&(mapa[2][x]==blue))
		{	return (x) ; }
		
	}
	return -1;
}


int nuevo(red, green, blue)
  int red,green,blue;
{
	int x;
	for(x=0;x<num;x++)
	{
		if ((mapa[0][x]==red)&&(mapa[1][x]==green)&&(mapa[2][x]==blue))
			{  return 0; }
	}
	return 1;
}

 int savemap(blockPtr)
 
  Tk_PhotoImageBlock *blockPtr;

{
 unsigned char  *colores,c;
 int x,y,z;
 FILE *kk;
 unsigned int  red,green,blue;
 unsigned int  xx,yy,zz;
 



	colores=blockPtr->pixelPtr;

	num=0;
	for(x=0;x<blockPtr->width;x++)
	{
		for(y=0;y<blockPtr->height;y++)
		{
			red = *colores;
				colores++;
			green = *colores;
				colores++;
			blue = *colores;
			if(nuevo (red,green,blue))
			{
				if (num>255) 
					return -1;

				mapa[0][num]=red;
				mapa[1][num]=green;
				mapa[2][num]=blue;
				num++;
			}

			if (x<blockPtr->width) 
				colores++;
		}
	}
	return num-1;

}

int  ReadValue()
{
        unsigned int  y,z, x;

        if (ssize == 0 ) {
                return EOF;
        }
        x = *pixelo++;
        y = *pixelo++;
        z = *pixelo++;
	ssize--;

        return (color(x,y,z));
}


/*
 * Return the number of bits ( -1 ) to represent a given
 * number of colors ( ex: 256 colors => 7 ).
 */
no_bits( colors )
int colors;
{
  register int bits= 0;

  colors--;
  while ( colors >> bits )
    bits++;

  return (bits-1);
}


/*
 *
 * GIF Image compression - modified 'compress'
 *
 * Based on: compress.c - File compression ala IEEE Computer, June 1984.
 *
 * By Authors:  Spencer W. Thomas       (decvax!harpo!utah-cs!utah-gr!thomas)
 *              Jim McKie               (decvax!mcvax!jim)
 *              Steve Davies            (decvax!vax135!petsd!peora!srd)
 *              Ken Turkowski           (decvax!decwrl!turtlevax!ken)
 *              James A. Woods          (decvax!ihnp4!ames!jaw)
 *              Joe Orost               (decvax!vax135!petsd!joe)
 *
 */
#include <ctype.h>

static output();
static cl_block();
static cl_hash();
static writeerr();
static char_init();
static char_out();
static flush_char();

static int n_bits;                        /* number of bits/code */
static int maxbits = GIFBITS;                /* user settable max # bits/code */
static code_int maxcode;                  /* maximum code, given n_bits */
static code_int maxmaxcode = (code_int)1 << GIFBITS; /* should NEVER generate this code */
#ifdef COMPATIBLE               /* But wrong! */
# define MAXCODE(n_bits)        ((code_int) 1 << (n_bits) - 1)
#else
# define MAXCODE(n_bits)        (((code_int) 1 << (n_bits)) - 1)
#endif /* COMPATIBLE */

static count_int htab [HSIZE];
static unsigned short codetab [HSIZE];
#define HashTabOf(i)       htab[i]
#define CodeTabOf(i)    codetab[i]

static code_int hsize = HSIZE;                 /* for dynamic table sizing */
static count_int fsize;

/*
 * To save much memory, we overlay the table used by compress() with those
 * used by decompress().  The tab_prefix table is the same size and type
 * as the codetab.  The tab_suffix table needs 2**GIFBITS characters.  We
 * get this from the beginning of htab.  The output stack uses the rest
 * of htab, and contains characters.  There is plenty of room for any
 * possible stack (stack used to be 8000 characters).
 */

#define tab_prefixof(i) CodeTabOf(i)
#define tab_suffixof(i)        ((char_type *)(htab))[i]
#define de_stack               ((char_type *)&tab_suffixof((code_int)1<<GIFBITS))

static code_int free_ent = 0;                  /* first unused entry */
static int exit_stat = 0;

/*
 * block compression parameters -- after all codes are used up,
 * and compression rate changes, start over.
 */
static int clear_flg = 0;

static int offset;
static long int in_count = 1;            /* length of input */
static long int out_count = 0;           /* # of codes output (for debugging) */

/*
 * compress stdin to stdout
 *
 * Algorithm:  use open addressing double hashing (no chaining) on the
 * prefix code / next character combination.  We do a variant of Knuth's
 * algorithm D (vol. 3, sec. 6.4) along with G. Knott's relatively-prime
 * secondary probe.  Here, the modular division first probe is gives way
 * to a faster exclusive-or manipulation.  Also do block compression with
 * an adaptive reset, whereby the code table is cleared when the compression
 * ratio decreases, but after the table fills.  The variable-length output
 * codes are re-sized at this point, and a special CLEAR code is generated
 * for the decompressor.  Late addition:  construct the table according to
 * file size for noticeable speed improvement on small files.  Please direct
 * questions about this implementation to ames!jaw.
 */

static int g_init_bits;
static FILE *g_outfile;

static int ClearCode;
static int EOFCode;

compress( init_bits, outfile, ReadValue )
int init_bits;
FILE *outfile;
ifunptr ReadValue;
{
    register long fcode;
    register code_int i = 0;
    register int c;
    register code_int ent;
    register code_int disp;
    register code_int hsize_reg;
    register int hshift;

    /*
     * Set up the globals:  g_init_bits - initial number of bits
     *                      g_outfile   - pointer to output file
     */
    g_init_bits = init_bits;
    g_outfile = outfile;

    /*
     * Set up the necessary values
     */
    offset = 0;
    out_count = 0;
    clear_flg = 0;
    in_count = 1;
    maxcode = MAXCODE(n_bits = g_init_bits);

    ClearCode = (1 << (init_bits - 1));
    EOFCode = ClearCode + 1;
    free_ent = ClearCode + 2;

    char_init();

    ent = ReadValue();

    hshift = 0;
    for ( fcode = (long) hsize;  fcode < 65536L; fcode *= 2L )
        hshift++;
    hshift = 8 - hshift;                /* set hash code range bound */

    hsize_reg = hsize;
    cl_hash( (count_int) hsize_reg);            /* clear hash table */

    output( (code_int)ClearCode );

#ifdef SIGNED_COMPARE_SLOW
    while ( (c = ReadValue() ) != (unsigned) EOF ) {
#else
    while ( (c = ReadValue()) != EOF ) {
#endif

        in_count++;

        fcode = (long) (((long) c << maxbits) + ent);
        i = (((code_int)c << hshift) ^ ent);    /* xor hashing */

        if ( HashTabOf (i) == fcode ) {
            ent = CodeTabOf (i);
            continue;
        } else if ( (long)HashTabOf (i) < 0 )      /* empty slot */
            goto nomatch;
        disp = hsize_reg - i;           /* secondary hash (after G. Knott) */
        if ( i == 0 )
            disp = 1;
probe:
        if ( (i -= disp) < 0 )
            i += hsize_reg;

        if ( HashTabOf (i) == fcode ) {
            ent = CodeTabOf (i);
            continue;
        }
        if ( (long)HashTabOf (i) > 0 )
            goto probe;
nomatch:
        output ( (code_int) ent );
        out_count++;
        ent = c;
#ifdef SIGNED_COMPARE_SLOW
        if ( (unsigned) free_ent < (unsigned) maxmaxcode) {
#else
        if ( free_ent < maxmaxcode ) {
#endif
            CodeTabOf (i) = free_ent++; /* code -> hashtable */
            HashTabOf (i) = fcode;
        } else
                cl_block();
    }
    /*
     * Put out the final code.
     */
    output( (code_int)ent );
    out_count++;
    output( (code_int) EOFCode );

    return;
}

/*****************************************************************
 * TAG( output )
 *
 * Output the given code.
 * Inputs:
 *      code:   A n_bits-bit integer.  If == -1, then EOF.  This assumes
 *              that n_bits =< (long)wordsize - 1.
 * Outputs:
 *      Outputs code to the file.
 * Assumptions:
 *      Chars are 8 bits long.
 * Algorithm:
 *      Maintain a GIFBITS character long buffer (so that 8 codes will
 * fit in it exactly).  Use the VAX insv instruction to insert each
 * code in turn.  When the buffer fills up empty it and start over.
 */

static unsigned long cur_accum = 0;
static int  cur_bits = 0;

static
unsigned long masks[] = { 0x0000, 0x0001, 0x0003, 0x0007, 0x000F,
                                  0x001F, 0x003F, 0x007F, 0x00FF,
                                  0x01FF, 0x03FF, 0x07FF, 0x0FFF,
                                  0x1FFF, 0x3FFF, 0x7FFF, 0xFFFF };

static
output( code )
code_int  code;
{
    cur_accum &= masks[ cur_bits ];

    if( cur_bits > 0 )
        cur_accum |= ((long)code << cur_bits);
    else
        cur_accum = code;

    cur_bits += n_bits;

    while( cur_bits >= 8 ) {
        char_out( (unsigned int)(cur_accum & 0xff) );
        cur_accum >>= 8;
        cur_bits -= 8;
    }

    /*
     * If the next entry is going to be too big for the code size,
     * then increase it, if possible.
     */
   if ( free_ent > maxcode || clear_flg ) {

            if( clear_flg ) {
        
                maxcode = MAXCODE (n_bits = g_init_bits);
                clear_flg = 0;
                
            } else {
        
                n_bits++;
                if ( n_bits == maxbits )
                    maxcode = maxmaxcode;
                else
                    maxcode = MAXCODE(n_bits);
            }
        }
        
    if( code == EOFCode ) {
        /*
         * At EOF, write the rest of the buffer.
         */
        while( cur_bits > 0 ) {
                char_out( (unsigned int)(cur_accum & 0xff) );
                cur_accum >>= 8;
                cur_bits -= 8;
        }

        flush_char();
        fflush(g_outfile);

        if( ferror( g_outfile ) )
                writeerr();
    }
}

/*
 * Clear out the hash table
 */
static
cl_block ()             /* table clear for block compress */
{

        cl_hash ( (count_int) hsize );
        free_ent = ClearCode + 2;
        clear_flg = 1;

        output( (code_int)ClearCode );
}

static
cl_hash(hsize)          /* reset code table */
register count_int hsize;
{

        register count_int *htab_p = htab+hsize;

        register long i;
        register long m1 = -1;

        i = hsize - 16;
        do {                            /* might use Sys V memset(3) here */
                *(htab_p-16) = m1;
                *(htab_p-15) = m1;
                *(htab_p-14) = m1;
                *(htab_p-13) = m1;
                *(htab_p-12) = m1;
                *(htab_p-11) = m1;
                *(htab_p-10) = m1;
                *(htab_p-9) = m1;
                *(htab_p-8) = m1;
                *(htab_p-7) = m1;
                *(htab_p-6) = m1;
                *(htab_p-5) = m1;
                *(htab_p-4) = m1;
                *(htab_p-3) = m1;
                *(htab_p-2) = m1;
                *(htab_p-1) = m1;
                htab_p -= 16;
        } while ((i -= 16) >= 0);

        for ( i += 16; i > 0; i-- )
                *--htab_p = m1;
}

static
writeerr()
{
        printf( "error writing output file\n" );
        exit(1);
}

/******************************************************************************
 *
 * GIF Specific routines
 *
 ******************************************************************************/

/*
 * Number of characters so far in this 'packet'
 */
static int a_count;

/*
 * Set up the 'byte output' routine
 */
static
char_init()
{
        a_count = 0;
        cur_accum = 0;
        cur_bits = 0;
}

/*
 * Define the storage for the packet accumulator
 */
static char accum[ 256 ];

/*
 * Add a character to the end of the current packet, and if it is 254
 * characters, flush the packet to disk.
 */
static
char_out( c )
int c;
{
        accum[ a_count++ ] = c;
        if( a_count >= 254 )
                flush_char();
}

/*
 * Flush the packet to disk, and reset the accumulator
 */
static
flush_char()
{
        if( a_count > 0 ) {
                fputc( a_count, g_outfile );
                fwrite( accum, 1, a_count, g_outfile );
                a_count = 0;
        }
}       

/* The End */

/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */


/*
 * quantize.c - quantize a bitmap.
 *
 * Author:          Xiaolin Wu
 *                  Dept. of Computer Science
 *                  Univ. of Western Ontario
 *                  London, Ontario N6A 5B7
 *                  wu@csd.uwo.ca
 *
 * FileWriteGIF's author:    	Lolo
 *                  		University of Oviedo
 * Date:            		Fri Oct 20 1995 
*/

/**********************************************************************
            C Implementation of Wu's Color Quantizer (v. 2)
            (see Graphics Gems vol. II, pp. 126-133)

Author: Xiaolin Wu
        Dept. of Computer Science
        Univ. of Western Ontario
        London, Ontario N6A 5B7
        wu@csd.uwo.ca

Algorithm: Greedy orthogonal bipartition of RGB space for variance
           minimization aided by inclusion-exclusion tricks.
           For speed no nearest neighbor search is done. Slightly
           better performance can be expected by more sophisticated
           but more expensive versions.

The author thanks Tom Lane at Tom_Lane@G.GP.CS.CMU.EDU for much of
additional documentation and a cure to a previous bug.

Free to distribute, comments and suggestions are appreciated.
**********************************************************************/
/*
 * History:
 *
 * Raul Rivero - Fri Jun 26 1992
 *      One problem was detected when multiple calls to the 
 *      quantizer. The resulting color map was incoherent. The 
 *      problem was fixed reseting all the static buffers.
 *
 */


#define MAXCOLOR        256
#define RED             2
#define GREEN           1
#define BLUE            0

struct box {
    int r0;                      /* min value, exclusive */
    int r1;                      /* max value, inclusive */
    int g0;
    int g1;
    int b0;
    int b1;
    int vol;
};

/*
 * Histogram is in elements 1..HISTSIZE along each axis,
 * element 0 is for base or marginal value
 * NB: these must start out 0!
 */

float           m2[33][33][33];
long int        wt[33][33][33], mr[33][33][33], mg[33][33][33], mb[33][33][33];
byte            *Ir, *Ig, *Ib;
int             size; /*image size*/
int             K;    /*color look-up table size*/
 ushort          *Qadd; 
extern int      LUGverbose;

float Var();
float Maximize();
long Vol();
long Bottom();
long Top();

quantize(inbitmap, outbitmap, no_colors)
bitmap_hdr *inbitmap;
bitmap_hdr *outbitmap;
int no_colors;
{
  struct box cube[MAXCOLOR];
  byte *tag;
  byte lut_r[MAXCOLOR], lut_g[MAXCOLOR], lut_b[MAXCOLOR];
  int next;
  register long int i, weight;
  register int k;
  float vv[MAXCOLOR], temp;
  byte *ptr;
  if ( inbitmap->magic != LUGUSED )
	return TCL_ERROR;
  /*  error( 19 ); */

  /* set the image */
  K  = no_colors;
  Ir = inbitmap->r;
  Ig = inbitmap->g;
  Ib = inbitmap->b;
  size = inbitmap->xsize * inbitmap->ysize;

  /* Reset the buffers */
  memset( m2, 0, 33*33*33*sizeof(float) );
  memset( wt, 0, 33*33*33*sizeof(long) );
  memset( mr, 0, 33*33*33*sizeof(long) );
  memset( mg, 0, 33*33*33*sizeof(long) );
  memset( mb, 0, 33*33*33*sizeof(long) );

  Hist3d(wt, mr, mg, mb, m2); /* Histogram done */

  M3d(wt, mr, mg, mb, m2);    /* Moments done */

  cube[0].r0 = cube[0].g0 = cube[0].b0 = 0;
  cube[0].r1 = cube[0].g1 = cube[0].b1 = 32;
  next = 0;
  for( i = 1; i < K; ++i ) {
    if ( Cut(&cube[next], &cube[i]) ) {
      /* volume test ensures we won't try to cut one-cell box */
      vv[next] = (cube[next].vol>1) ? Var(&cube[next]) : 0.0;
      vv[i] = (cube[i].vol>1) ? Var(&cube[i]) : 0.0;
    } else {
      vv[next] = 0.0;   /* don't try to split this box again */
      i--;              /* didn't create box i */
    }
    next = 0; temp = vv[0];
    for ( k = 1; k <= i; ++k )
      if ( vv[k] > temp ) {
        temp = vv[k]; next = k;
      }
      if (temp <= 0.0) {
        K = i+1;
        break;
      }
    }
    /*  Partition done  */

    /* the space for array m2 can be freed now */

    tag = (byte *) Malloc(33*33*33);
    for(k = 0; k < K; ++k ){
      Mark( &cube[k], k, tag );
      weight = Vol( &cube[k], wt );
      if ( weight ) {
        lut_r[k] = Vol(&cube[k], mr) / weight;
        lut_g[k] = Vol(&cube[k], mg) / weight;
        lut_b[k] = Vol(&cube[k], mb) / weight;
      } else {
        lut_r[k] = lut_g[k] = lut_b[k] = 0;
      }
    }

    outbitmap->magic = LUGUSED;
    outbitmap->xsize = inbitmap->xsize;
    outbitmap->ysize = inbitmap->ysize;
    outbitmap->depth = 1 + no_bits( no_colors );
    outbitmap->colors = no_colors;
    ptr = outbitmap->cmap = (byte *) Malloc( 3 * no_colors );
    for( k = 0; k < K; ++k ) {
      *ptr++ = lut_r[k];
      *ptr++ = lut_g[k];
      *ptr++ = lut_b[k];
    }

    ptr = outbitmap->r = (byte *) Malloc( size );
    for( i = 0; i < size; ++i )
      *ptr++ = tag[Qadd[i]];

    free(Qadd);
    free(tag);
}

Hist3d(vwt, vmr, vmg, vmb, m2)
/* build 3-D color histogram of counts, r/g/b, c^2 */
long int *vwt, *vmr, *vmg, *vmb;
float   *m2;
{
  register int ind, r, g, b;
  int inr, ing, inb, table[256];
  register long int i;

  for ( i = 0; i < 256; ++i )
    table[i] = i * i;
  Qadd = (ushort *) Malloc( sizeof(short) * size ); 
  for( i = 0; i < size; ++i ) {
    r = Ir[i]; g = Ig[i]; b = Ib[i];
    inr = (r>>3) + 1;
    ing = (g>>3) + 1;
    inb = (b>>3) + 1;
    Qadd[i] = ind = (inr<<10) + (inr<<6) + inr + (ing<<5) + ing + inb;
             /*[inr][ing][inb]*/
    ++vwt[ind];
    vmr[ind] += r;
    vmg[ind] += g;
    vmb[ind] += b;
    m2[ind] += (float)(table[r]+table[g]+table[b]);
  }
}

/* At conclusion of the histogram step, we can interpret
 *   wt[r][g][b] = sum over voxel of P(c)
 *   mr[r][g][b] = sum over voxel of r*P(c)  ,  similarly for mg, mb
 *   m2[r][g][b] = sum over voxel of c^2*P(c)
 * Actually each of these should be divided by 'size' to give the usual
 * interpretation of P() as ranging from 0 to 1, but we needn't do that here.
 */

/* We now convert histogram into moments so that we can rapidly calculate
 * the sums of the above quantities over any desired box.
 */

M3d(vwt, vmr, vmg, vmb, m2) /* compute cumulative moments. */
long int *vwt, *vmr, *vmg, *vmb;
float *m2;
{
  register unsigned short int ind1, ind2;
  register byte i, r, g, b;
  long int line, line_r, line_g, line_b,
           area[33], area_r[33], area_g[33], area_b[33];
  float line2, area2[33];

  for ( r = 1; r <= 32; ++r ) {
    for ( i = 0; i <= 32; ++i )
      area2[i] = area[i] = area_r[i] = area_g[i] = area_b[i] = 0;
    for ( g = 1; g <= 32; ++g ) {
      line2 = line = line_r = line_g = line_b = 0;
      for ( b = 1; b <= 32; ++b ) {
        ind1 = (r<<10) + (r<<6) + r + (g<<5) + g + b; /* [r][g][b] */
        line += vwt[ind1];
        line_r += vmr[ind1];
        line_g += vmg[ind1];
        line_b += vmb[ind1];
        line2 += m2[ind1];
        area[b] += line;
        area_r[b] += line_r;
        area_g[b] += line_g;
        area_b[b] += line_b;
        area2[b] += line2;
        ind2 = ind1 - 1089; /* [r-1][g][b] */
        vwt[ind1] = vwt[ind2] + area[b];
        vmr[ind1] = vmr[ind2] + area_r[b];
        vmg[ind1] = vmg[ind2] + area_g[b];
        vmb[ind1] = vmb[ind2] + area_b[b];
        m2[ind1] = m2[ind2] + area2[b];
      }
    }
  }
}

long
Vol(cube, mmt)
/* Compute sum over a box of any given statistic */
struct box *cube;
long int mmt[33][33][33];
{
  return( mmt[cube->r1][cube->g1][cube->b1]
          -mmt[cube->r1][cube->g1][cube->b0]
          -mmt[cube->r1][cube->g0][cube->b1]
          +mmt[cube->r1][cube->g0][cube->b0]
          -mmt[cube->r0][cube->g1][cube->b1]
          +mmt[cube->r0][cube->g1][cube->b0]
          +mmt[cube->r0][cube->g0][cube->b1]
          -mmt[cube->r0][cube->g0][cube->b0] );
}

/* The next two routines allow a slightly more efficient calculation
 * of Vol() for a proposed subbox of a given box.  The sum of Top()
 * and Bottom() is the Vol() of a subbox split in the given direction
 * and with the specified new upper bound.
 */

long
Bottom(cube, dir, mmt)
/* Compute part of Vol(cube, mmt) that doesn't depend on r1, g1, or b1 */
/* (depending on dir) */
struct box *cube;
byte dir;
long int mmt[33][33][33];
{
  switch(dir){
    case RED:
            return( -mmt[cube->r0][cube->g1][cube->b1]
                    +mmt[cube->r0][cube->g1][cube->b0]
                    +mmt[cube->r0][cube->g0][cube->b1]
                    -mmt[cube->r0][cube->g0][cube->b0] );
            break;
    case GREEN:
            return( -mmt[cube->r1][cube->g0][cube->b1]
                    +mmt[cube->r1][cube->g0][cube->b0]
                    +mmt[cube->r0][cube->g0][cube->b1]
                    -mmt[cube->r0][cube->g0][cube->b0] );
            break;
    case BLUE:
            return( -mmt[cube->r1][cube->g1][cube->b0]
                    +mmt[cube->r1][cube->g0][cube->b0]
                    +mmt[cube->r0][cube->g1][cube->b0]
                    -mmt[cube->r0][cube->g0][cube->b0] );
            break;
  }
}

long
Top(cube, dir, pos, mmt)
/* Compute remainder of Vol(cube, mmt), substituting pos for */
/* r1, g1, or b1 (depending on dir) */
struct box *cube;
byte dir;
int   pos;
long int mmt[33][33][33];
{
  switch(dir){
    case RED:
            return( mmt[pos][cube->g1][cube->b1]
                   -mmt[pos][cube->g1][cube->b0]
                   -mmt[pos][cube->g0][cube->b1]
                   +mmt[pos][cube->g0][cube->b0] );
            break;
    case GREEN:
            return( mmt[cube->r1][pos][cube->b1]
                   -mmt[cube->r1][pos][cube->b0]
                   -mmt[cube->r0][pos][cube->b1]
                   +mmt[cube->r0][pos][cube->b0] );
            break;
    case BLUE:
            return( mmt[cube->r1][cube->g1][pos]
                   -mmt[cube->r1][cube->g0][pos]
                   -mmt[cube->r0][cube->g1][pos]
                   +mmt[cube->r0][cube->g0][pos] );
            break;
  }
}

float
Var(cube)
/* Compute the weighted variance of a box */
/* NB: as with the raw statistics, this is really the variance * size */
struct box *cube;
{
  float dr, dg, db, xx;

  dr = Vol(cube, mr);
  dg = Vol(cube, mg);
  db = Vol(cube, mb);
  xx =  m2[cube->r1][cube->g1][cube->b1]
        -m2[cube->r1][cube->g1][cube->b0]
        -m2[cube->r1][cube->g0][cube->b1]
        +m2[cube->r1][cube->g0][cube->b0]
        -m2[cube->r0][cube->g1][cube->b1]
        +m2[cube->r0][cube->g1][cube->b0]
        +m2[cube->r0][cube->g0][cube->b1]
        -m2[cube->r0][cube->g0][cube->b0];

  return( xx - (dr*dr+dg*dg+db*db) / (float)Vol(cube,wt) );
}

/* We want to minimize the sum of the variances of two subboxes.
 * The sum(c^2) terms can be ignored since their sum over both subboxes
 * is the same (the sum for the whole box) no matter where we split.
 * The remaining terms have a minus sign in the variance formula,
 * so we drop the minus sign and MAXIMIZE the sum of the two terms.
 */
/* Se intenta minimizar la suma de las varianzas de las dos subcajas
 * Los terminos de suma de cuadrados puede ser ignorado si la suma de ambas
 * subcajas es la misma (la suma para todo la caja) sin importar donde se haya
 * dividido. Los terminos restantes son negativos en la formula de la varianza
 * por lo que no se toma en cuenta el signo y se maximiza la suma de los dos
 * terminos
 */

float
Maximize(cube, dir, first, last, cut,
         whole_r, whole_g, whole_b, whole_w)
struct box *cube;
byte dir;
int first, last, *cut;
long int whole_r, whole_g, whole_b, whole_w;
{
  register long int half_r, half_g, half_b, half_w;
  long int base_r, base_g, base_b, base_w;
  register int i;
  register float temp, max;

  base_r = Bottom(cube, dir, mr);
  base_g = Bottom(cube, dir, mg);
  base_b = Bottom(cube, dir, mb);
  base_w = Bottom(cube, dir, wt);
  max = 0.0;
  *cut = -1;
  for( i = first; i < last; ++i ) {
    half_r = base_r + Top(cube, dir, i, mr);
    half_g = base_g + Top(cube, dir, i, mg);
    half_b = base_b + Top(cube, dir, i, mb);
    half_w = base_w + Top(cube, dir, i, wt);
    /* now half_x is sum over lower half of box, if split at i */
    if ( half_w == 0 ) {      /* subbox could be empty of pixels! */
      continue;             /* never split into an empty box */
    } else
        temp = ((float)half_r*half_r + (float)half_g*half_g +
                (float)half_b*half_b)/half_w;

    half_r = whole_r - half_r;
    half_g = whole_g - half_g;
    half_b = whole_b - half_b;
    half_w = whole_w - half_w;
    if ( half_w == 0 ) {      /* subbox could be empty of pixels! */
      continue;             /* never split into an empty box */
    } else
       temp += ((float)half_r*half_r + (float)half_g*half_g +
                (float)half_b*half_b)/half_w;

    if ( temp > max ) {
      max = temp;
      *cut=i;
    }
  }
  return(max);
}

int
Cut(set1, set2)
struct box *set1, *set2;
{
  byte dir;
  int cutr, cutg, cutb;
  float maxr, maxg, maxb;
  long int whole_r, whole_g, whole_b, whole_w;

  whole_r = Vol(set1, mr);
  whole_g = Vol(set1, mg);
  whole_b = Vol(set1, mb);
  whole_w = Vol(set1, wt);

  maxr = Maximize( set1, RED, set1->r0+1, set1->r1, &cutr,
                   whole_r, whole_g, whole_b, whole_w );
  maxg = Maximize( set1, GREEN, set1->g0+1, set1->g1, &cutg,
                   whole_r, whole_g, whole_b, whole_w );
  maxb = Maximize( set1, BLUE, set1->b0+1, set1->b1, &cutb,
                   whole_r, whole_g, whole_b, whole_w );

  if( (maxr >= maxg) && (maxr >= maxb) ) {
    dir = RED;
    if ( cutr < 0 )
      return 0; /* can't split the box */
  } else
    if( (maxg >= maxr) && (maxg >= maxb) )
      dir = GREEN;
    else
      dir = BLUE;

  set2->r1 = set1->r1;
  set2->g1 = set1->g1;
  set2->b1 = set1->b1;

  switch (dir){
    case RED:
            set2->r0 = set1->r1 = cutr;
            set2->g0 = set1->g0;
            set2->b0 = set1->b0;
            break;
    case GREEN:
            set2->g0 = set1->g1 = cutg;
            set2->r0 = set1->r0;
            set2->b0 = set1->b0;
            break;
    case BLUE:
            set2->b0 = set1->b1 = cutb;
            set2->r0 = set1->r0;
            set2->g0 = set1->g0;
            break;
  }

  set1->vol = (set1->r1-set1->r0) * (set1->g1-set1->g0) * (set1->b1-set1->b0);
  set2->vol = (set2->r1-set2->r0) * (set2->g1-set2->g0) * (set2->b1-set2->b0);

  return 1;
}

Mark(cube, label, tag)
struct box *cube;
int label;
byte *tag;
{
  register int r, g, b;

  for ( r = cube->r0+1; r <= cube->r1; ++r )
    for( g = cube->g0+1; g <= cube->g1; ++g )
      for( b = cube->b0+1; b <= cube->b1; ++b )
        tag[ (r<<10) + (r<<6) + r + (g<<5) + g + b ] = label;
}

int read_dib( image,xsize,ysize )

bitmap_hdr *image;
int xsize,ysize;
{
	int totalsize,i,j;
	byte *rb,*gb,*bb;
	byte *line;

	image->depth=24;
	totalsize=xsize*ysize;
	
	image->xsize=xsize;
	image->ysize=ysize;
	image->r=(byte *)Malloc(totalsize);
	image->g=(byte *)Malloc(totalsize);
	image->b=(byte *)Malloc(totalsize);

	image->magic = LUGUSED;

	rb=image->r;
	gb=image->g;
	bb=image->b;
	

	line=pixelo;

	for(i=0;i<ysize;i++)
	{
		j=xsize;
		while( j--)
		{
			*rb++ = *line++;
			*gb++ = *line++;
			*bb++ = *line++;
		}
	}
	return 0;
}

int write_dib ( image )

bitmap_hdr *image;
{

	int x,totalsize;
	byte *rb,*gb,*bb;
	byte *line;
	byte *map;

	line=pixelo;

	totalsize=image->xsize*image->ysize;
	map=image->cmap;
	rb=image->r;
	gb=image->g;
	bb=image->b;
	while (totalsize--)
	{
		*line++=map[(*rb)*3];
		*line++=map[(*rb)*3+1];
		*line++=map[(*rb++)*3+2];
	}
	for (x=0;x<image->colors;x++)
	{
		mapa[0][x]=map[x*3];
		mapa[1][x]=map[x*3+1];
		mapa[2][x]=map[x*3+2];
	}
	return x-1;
}
	

char *Malloc(size)
int size;
{
  char *ptr;

  ptr = (char *) malloc(size);

  /*
   * Usually compilers fill buffers with zeros,
   * but ...
   */
  memset( ptr, 0, size );
  return ptr;
}
