/* 
 * tkConfig.c --
 *
 *	This file contains the Tk_ConfigureWidget procedure.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkConfig.c 1.52 96/02/15 18:52:39
 */

#include "tkPort.h"
#include "tk.h"

/*
 * Values for "flags" field of Tk_ConfigSpec structures.  Be sure
 * to coordinate these values with those defined in tk.h
 * (TK_CONFIG_COLOR_ONLY, etc.).  There must not be overlap!
 *
 * INIT -		Non-zero means (char *) things have been
 *			converted to Tk_Uid's.
 */

#define INIT		0x20

/*
 * Forward declarations for procedures defined later in this file:
 */

static int		DoConfig _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Window tkwin, Tk_ConfigSpec *specPtr,
			    Tk_Uid value, int valueIsUid, char *widgRec));
static Tk_ConfigSpec *	FindConfigSpec _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_ConfigSpec *specs, char *argvName,
			    int needFlags, int hateFlags));
static char *		FormatConfigInfo _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Window tkwin, Tk_ConfigSpec *specPtr,
			    char *widgRec));
static char *		FormatConfigValue _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Window tkwin, Tk_ConfigSpec *specPtr,
			    char *widgRec, char *buffer,
			    Tcl_FreeProc **freeProcPtr));

static Tk_ConfigProc *configProc[] = {
	TK_CONFIG_NULL,
	Tk_ConfigBoolean,
	Tk_ConfigInt,
	Tk_ConfigDouble,
	Tk_ConfigString,
	Tk_ConfigUid,
	Tk_ConfigColor,
	Tk_ConfigFont,
	Tk_ConfigBitmap,
	Tk_ConfigBorder,
	Tk_ConfigRelief,
	Tk_ConfigCursor,
	Tk_ConfigActiveCursor,
	Tk_ConfigJustify,
	Tk_ConfigAnchor,
	TK_CONFIG_SYNONYM,
	Tk_ConfigCapStyle,
	Tk_ConfigJoinStyle,
	Tk_ConfigPixels,
	Tk_ConfigMM,
	Tk_ConfigWindow,
	Tk_ConfigCustom
};


/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigBoolean --
 *
 *	Process the TK_CONFIG_BOOLEAN option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigBoolean(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (boolean) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register int *intPtr = (int *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *intPtr = 0;
	    break;
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID:
	    return Tcl_GetBoolean(interp, value, intPtr);
	case TK_OPTION_PRINT:
	    interp->result[0] = (*intPtr)?'1':'0';
	    interp->result[1] = '\0';
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigBoolean";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigInt --
 *
 *	Process the TK_CONFIG_INT option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigInt(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register int *intPtr = (int *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *intPtr = 0;
	    break;
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID:
	    return Tcl_GetInt(interp, value, intPtr);
	case TK_OPTION_PRINT:
	    sprintf(interp->result, "%d", *intPtr);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigInt";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigDouble --
 *
 *	Process the TK_CONFIG_DOUBLE option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigDouble(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (double) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register double *doublePtr = (double *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *doublePtr = 0.0;
	    break;
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID:
	    return Tcl_GetDouble(interp, value, doublePtr);
	case TK_OPTION_PRINT:
	    Tcl_PrintDouble(interp, *doublePtr, interp->result);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigDouble";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigString --
 *
 *	Process the TK_CONFIG_STRING option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigString(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (string) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register char **stringPtr = (char **)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	    if (*stringPtr!=NULL) {
		ckfree(*stringPtr);
	    }
	case TK_OPTION_INIT:
	    *stringPtr = NULL;
	    break;
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID: {
	    char *new;
	    if (value!=NULL) {
		new = (char *) ckalloc((unsigned) (strlen(value) + 1));
		strcpy(new, value);
	    } else {
		new = NULL;
	    }
	    if (*stringPtr) {
		ckfree(*stringPtr);
	    }
	    *stringPtr = new;
	    break;
	}
	case TK_OPTION_PRINT:
	    if (*stringPtr!=NULL) {
		interp->result = *stringPtr;
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigString";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigUid --
 *
 *	Process the TK_CONFIG_UID option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigUid(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (uid) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register Tk_Uid *uidPtr = (Tk_Uid *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *uidPtr = NULL;
	    break;
	case TK_OPTION_PARSE:
	    if (value != NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID:
	    *uidPtr = (Tk_Uid) value;
	    break;
	case TK_OPTION_PRINT:
	    if (*uidPtr!=NULL) {
		interp->result = *uidPtr;
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigUid";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigColor --
 *
 *	Process the TK_CONFIG_COLOR option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigColor(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (color) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register XColor **colorPtr = (XColor **)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	    if (*colorPtr != NULL) {
		Tk_FreeColor(*colorPtr);
	    }
	case TK_OPTION_INIT:
	    *colorPtr = NULL;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    XColor *new;
	    if (value!=NULL) {
		new = Tk_GetColor(interp, tkwin, value);
		if (new == NULL) {
		    return TCL_ERROR;
		}
	    } else {
		new = NULL;
	    }
	    if (*colorPtr != NULL) {
		Tk_FreeColor(*colorPtr);
	    }
	    *colorPtr = new;
	    break;
	}
	case TK_OPTION_PRINT:
	    if (*colorPtr != NULL) {
		interp->result = Tk_NameOfColor(*colorPtr);
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigColor";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigFont --
 *
 *	Process the TK_CONFIG_FONT option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigFont(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (font) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register XFontStruct **fontPtr = (XFontStruct **)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	    if (*fontPtr != NULL) {
		Tk_FreeFontStruct(*fontPtr);
	    }
	case TK_OPTION_INIT:
	    *fontPtr = NULL;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    XFontStruct *new;
	    if (value!=NULL) {
		new = Tk_GetFontStruct(interp, tkwin, value);
		if (new == NULL) {
		    return TCL_ERROR;
		}
	    } else {
		new = NULL;
	    }
	    if (*fontPtr != NULL) {
		Tk_FreeFontStruct(*fontPtr);
	    }
	    *fontPtr = new;
	    break;
	}
	case TK_OPTION_PRINT:
	    if (*fontPtr != NULL) {
		interp->result = Tk_NameOfFontStruct(*fontPtr);
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigFont";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigBitmap --
 *
 *	Process the TK_CONFIG_BITMAP option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigBitmap(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* not used*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register Pixmap *pixmapPtr = (Pixmap *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	    if (*pixmapPtr != None) {
		Tk_FreeBitmap((Display *)tkwin,*pixmapPtr);
	    }
	case TK_OPTION_INIT:
	    *pixmapPtr = None;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    Pixmap new;
	    if (value!=NULL) {
		new = Tk_GetBitmap(interp, tkwin, value);
		if (new == None) {
		    return TCL_ERROR;
		}
	    } else {
		new = None;
	    }
	    if (*pixmapPtr != None) {
		Tk_FreeBitmap(Tk_Display(tkwin),*pixmapPtr);
	    }
	    *pixmapPtr = new;
	    break;
	}
	case TK_OPTION_PRINT:
	    if (*pixmapPtr != None) {
		interp->result = Tk_NameOfBitmap(Tk_Display(tkwin),*pixmapPtr);
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigBitmap";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigBorder --
 *
 *	Process the TK_CONFIG_BORDER option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigBorder(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register Tk_3DBorder *borderPtr = (Tk_3DBorder *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	    if (*borderPtr != None) {
		Tk_Free3DBorder(*borderPtr);
	    }
	case TK_OPTION_INIT:
	    *borderPtr = None;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    Tk_3DBorder new;
	    if (value!=NULL) {
		new = Tk_Get3DBorder(interp, tkwin, value);
		if (new == None) {
		    return TCL_ERROR;
		}
	    } else {
		new = None;
	    }
	    if (*borderPtr != None) {
		Tk_Free3DBorder(*borderPtr);
	    }
	    *borderPtr = new;
	    break;
	}
	case TK_OPTION_PRINT:
	    if (*borderPtr != None) {
		interp->result = Tk_NameOf3DBorder(*borderPtr);
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigBorder";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigRelief --
 *
 *	Process the TK_CONFIG_RELIEF option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigRelief(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register int *intPtr = (int *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *intPtr = 0;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    if (Tk_GetRelief(interp, value, intPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;
	}
	case TK_OPTION_PRINT:
	    interp->result = Tk_NameOfRelief(*intPtr);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigRelief";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigCursor --
 *
 *	Process the TK_CONFIG_CURSOR option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigCursor(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* not used */
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register Tk_Cursor *cursorPtr = (Tk_Cursor *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	    if (*cursorPtr != None) {
		Tk_FreeCursor((Display *)tkwin,*cursorPtr);
	    }
	case TK_OPTION_INIT:
	    *cursorPtr = None;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    Tk_Cursor new;
	    if (value!=NULL) {
		new = Tk_GetCursor(interp, tkwin, value);
		if (new == None) {
		    return TCL_ERROR;
		}
	    } else {
		new = None;
	    }
	    if (*cursorPtr != None) {
		Tk_FreeCursor(Tk_Display(tkwin),*cursorPtr);
	    }
	    *cursorPtr = new;
	    if (clientData!=NULL) {
		Tk_DefineCursor(tkwin, new);
	    }
	    break;
	}
	case TK_OPTION_PRINT:
	    if (*cursorPtr != None) {
		interp->result = Tk_NameOfCursor(Tk_Display(tkwin),*cursorPtr);
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigBorder";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigActiveCursor --
 *
 *	Process the TK_CONFIG_ACTIVE_CURSOR option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigActiveCursor(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    return Tk_ConfigCursor(function, (ClientData)1, interp, tkwin, value, widgRec, offset);
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigJustify --
 *
 *	Process the TK_CONFIG_JUSTIFY option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigJustify(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register Tk_Justify *justPtr = (Tk_Justify *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *justPtr = TK_JUSTIFY_LEFT;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    if (Tk_GetJustify(interp, value, justPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;
	}
	case TK_OPTION_PRINT:
	    interp->result = Tk_NameOfJustify(*justPtr);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigJustify";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigAnchor --
 *
 *	Process the TK_CONFIG_ANCHOR option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigAnchor(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register Tk_Anchor *anchorPtr = (Tk_Anchor *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *anchorPtr = TK_ANCHOR_N;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    if (Tk_GetAnchor(interp, value, anchorPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;
	}
	case TK_OPTION_PRINT:
	    interp->result = Tk_NameOfAnchor(*anchorPtr);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigAnchor";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigCapStype --
 *
 *	Process the TK_CONFIG_CAP_STYLE option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigCapStyle(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register int *intPtr = (int *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *intPtr = 0;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    if (Tk_GetCapStyle(interp, value, intPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;
	}
	case TK_OPTION_PRINT:
	    interp->result = Tk_NameOfCapStyle(*intPtr);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigCapStyle";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigJoinStyle --
 *
 *	Process the TK_CONFIG_JOINSTYLE option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigJoinStyle(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register int *intPtr = (int *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *intPtr = 0;
	    break;
	case TK_OPTION_PARSE:
	    if (value!=NULL) {
		value = Tk_GetUid(value);
	    }
	case TK_OPTION_PARSE_UID: {
	    if (Tk_GetJoinStyle(interp, value, intPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;
	}
	case TK_OPTION_PRINT:
	    interp->result = Tk_NameOfJoinStyle(*intPtr);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigJoinStyle";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigPixels --
 *
 *	Process the TK_CONFIG_PIXELS option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigPixels(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register int *intPtr = (int *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *intPtr = 0;
	    break;
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID:
	    return Tk_GetPixels(interp, tkwin, value, intPtr);
	case TK_OPTION_PRINT:
	    sprintf(interp->result, "%d", *intPtr);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigPixels";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigMM --
 *
 *	Process the TK_CONFIG_MM option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigMM(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register double *doublePtr = (double *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *doublePtr = 0.0;
	    break;
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID:
	    return Tk_GetScreenMM(interp, tkwin, value, doublePtr);
	case TK_OPTION_PRINT:
	    Tcl_PrintDouble(interp, *doublePtr, interp->result);
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigMM";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigWindow --
 *
 *	Process the TK_CONFIG_WINDOW option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigWindow(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    register Tk_Window *winPtr = (Tk_Window *)(widgRec + offset);
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    *winPtr = NULL;
	    break;
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID:
	    if (value==NULL) {
		*winPtr = NULL;
	    } else {
		if ((*winPtr = Tk_NameToWindow(interp, value, tkwin))==NULL) {
		    return TCL_ERROR;
		}
	    }
	    break;
	case TK_OPTION_PRINT:
	    if (*winPtr != NULL) {
		interp->result = Tk_PathName(*winPtr);
	    }
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigWindow";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigCustom --
 *
 *	Process the TK_CONFIG_CUSTOM option
 *
 * Results:
 *	Depends on function.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigCustom(function, clientData, interp, tkwin, value, widgRec, offset)
    int function;			/* which function to perform */
    ClientData clientData;		/* contains (Tk_CustomOption *).*/
    Tcl_Interp *interp;			/* Used for reporting. */
    Tk_Window tkwin;			/* Window containing canvas widget. */
    char *value;			/* Value of option (int) */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item. */
{
    switch (function) {
	case TK_OPTION_FREE:
	case TK_OPTION_INIT:
	    return TCL_ERROR; /* not implemented for old-style custom options */
	case TK_OPTION_PARSE:
	case TK_OPTION_PARSE_UID:
	    if ((*((Tk_CustomOption *)clientData)->parseProc)(
		    ((Tk_CustomOption *)clientData)->clientData, interp, tkwin,
		    value, widgRec, offset) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;
	case TK_OPTION_PRINT:
	    interp->result = (*((Tk_CustomOption *)clientData)->printProc)(
		    ((Tk_CustomOption *)clientData)->clientData, tkwin, widgRec,
		    offset, &(interp->freeProc));
	    break;
	default:
	    interp->result = "error: wrong function in Tk_ConfigCustom";
	    return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tk_ConfigureWidget --
 *
 *	Process command-line options and database options to
 *	fill in fields of a widget record with resources and
 *	other parameters.
 *
 * Results:
 *	A standard Tcl return value.  In case of an error,
 *	interp->result will hold an error message.
 *
 * Side effects:
 *	The fields of widgRec get filled in with information
 *	from argc/argv and the option database.  Old information
 *	in widgRec's fields gets recycled.
 *
 *--------------------------------------------------------------
 */

int
Tk_ConfigureWidget(interp, tkwin, specs, argc, argv, widgRec, flags)
    Tcl_Interp *interp;		/* Interpreter for error reporting. */
    Tk_Window tkwin;		/* Window containing widget (needed to
				 * set up X resources). */
    Tk_ConfigSpec *specs;	/* Describes legal options. */
    int argc;			/* Number of elements in argv. */
    char **argv;		/* Command-line options. */
    char *widgRec;		/* Record whose fields are to be
				 * modified.  Values must be properly
				 * initialized. */
    int flags;			/* Used to specify additional flags
				 * that must be present in config specs
				 * for them to be considered.  Also,
				 * may have TK_CONFIG_ARGV_ONLY set. */
{
    register Tk_ConfigSpec *specPtr;
    Tk_Uid value;		/* Value of option from database. */
    int needFlags;		/* Specs must contain this set of flags
				 * or else they are not considered. */
    int hateFlags;		/* If a spec contains any bits here, it's
				 * not considered. */

    needFlags = flags & ~(TK_CONFIG_USER_BIT - 1);
    if (Tk_Depth(tkwin) <= 1) {
	hateFlags = TK_CONFIG_COLOR_ONLY;
    } else {
	hateFlags = TK_CONFIG_MONO_ONLY;
    }

    /*
     * Pass one:  scan through all the option specs, replacing strings
     * with Tk_Uids (if this hasn't been done already) and clearing
     * the TK_CONFIG_OPTION_SPECIFIED flags.
     */

    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	if (!(specPtr->specFlags & INIT) && (specPtr->argvName != NULL)) {
	    if (((int *)specPtr->type>(int *)TK_CONFIG_NULL)&&((int *)specPtr->type<(int *)TK_CONFIG_END)) {
		specPtr->type = configProc[(int)specPtr->type];
	    }
	    if (specPtr->dbName != NULL) {
		specPtr->dbName = Tk_GetUid(specPtr->dbName);
	    }
	    if (specPtr->dbClass != NULL) {
		specPtr->dbClass = Tk_GetUid(specPtr->dbClass);
	    }
	    if (specPtr->defValue != NULL) {
		specPtr->defValue = Tk_GetUid(specPtr->defValue);
	    }
	}
	specPtr->specFlags = (specPtr->specFlags & ~TK_CONFIG_OPTION_SPECIFIED)
		| INIT;
    }

    /*
     * Pass two:  scan through all of the arguments, processing those
     * that match entries in the specs.
     */

    for ( ; argc > 0; argc -= 2, argv += 2) {
	specPtr = FindConfigSpec(interp, specs, *argv, needFlags, hateFlags);
	if (specPtr == NULL) {
	    return TCL_ERROR;
	}

	/*
	 * Process the entry.
	 */

	if (argc < 2) {
	    Tcl_AppendResult(interp, "value for \"", *argv,
		    "\" missing", (char *) NULL);
	    return TCL_ERROR;
	}
	if (DoConfig(interp, tkwin, specPtr, argv[1], 0, widgRec) != TCL_OK) {
	    char msg[100];

	    sprintf(msg, "\n    (processing \"%.40s\" option)",
		    specPtr->argvName);
	    Tcl_AddErrorInfo(interp, msg);
	    return TCL_ERROR;
	}
	specPtr->specFlags |= TK_CONFIG_OPTION_SPECIFIED;
    }

    /*
     * Pass three:  scan through all of the specs again;  if no
     * command-line argument matched a spec, then check for info
     * in the option database.  If there was nothing in the
     * database, then use the default.
     */

    if (!(flags & TK_CONFIG_ARGV_ONLY)) {
	for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	    if ((specPtr->specFlags & TK_CONFIG_OPTION_SPECIFIED)
		    || (specPtr->argvName == NULL)
		    || (specPtr->type == TK_CONFIG_SYNONYM)) {
		continue;
	    }
	    if (((specPtr->specFlags & needFlags) != needFlags)
		    || (specPtr->specFlags & hateFlags)) {
		continue;
	    }
	    value = NULL;
	    if (specPtr->dbName != NULL) {
		value = Tk_GetOption(tkwin, specPtr->dbName, specPtr->dbClass);
	    }
	    if (value != NULL) {
		if (DoConfig(interp, tkwin, specPtr, value, 1, widgRec) !=
			TCL_OK) {
		    char msg[200];
    
		    sprintf(msg, "\n    (%s \"%.50s\" in widget \"%.50s\")",
			    "database entry for",
			    specPtr->dbName, Tk_PathName(tkwin));
		    Tcl_AddErrorInfo(interp, msg);
		    return TCL_ERROR;
		}
	    } else {
		value = specPtr->defValue;
		if ((value != NULL) && !(specPtr->specFlags
			& TK_CONFIG_DONT_SET_DEFAULT)) {
		    if (DoConfig(interp, tkwin, specPtr, value, 1, widgRec) !=
			    TCL_OK) {
			char msg[200];
	
			sprintf(msg,
				"\n    (%s \"%.50s\" in widget \"%.50s\")",
				"default value for",
				specPtr->dbName, Tk_PathName(tkwin));
			Tcl_AddErrorInfo(interp, msg);
			return TCL_ERROR;
		    }
		}
	    }
	}
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * FindConfigSpec --
 *
 *	Search through a table of configuration specs, looking for
 *	one that matches a given argvName.
 *
 * Results:
 *	The return value is a pointer to the matching entry, or NULL
 *	if nothing matched.  In that case an error message is left
 *	in interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static Tk_ConfigSpec *
FindConfigSpec(interp, specs, argvName, needFlags, hateFlags)
    Tcl_Interp *interp;		/* Used for reporting errors. */
    Tk_ConfigSpec *specs;	/* Pointer to table of configuration
				 * specifications for a widget. */
    char *argvName;		/* Name (suitable for use in a "config"
				 * command) identifying particular option. */
    int needFlags;		/* Flags that must be present in matching
				 * entry. */
    int hateFlags;		/* Flags that must NOT be present in
				 * matching entry. */
{
    register Tk_ConfigSpec *specPtr;
    register char c;		/* First character of current argument. */
    Tk_ConfigSpec *matchPtr;	/* Matching spec, or NULL. */
    size_t length;

    c = argvName[1];
    length = strlen(argvName);
    matchPtr = NULL;
    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	if (specPtr->argvName == NULL) {
	    continue;
	}
	if ((specPtr->argvName[1] != c)
		|| (strncmp(specPtr->argvName, argvName, length) != 0)) {
	    continue;
	}
	if (((specPtr->specFlags & needFlags) != needFlags)
		|| (specPtr->specFlags & hateFlags)) {
	    continue;
	}
	if (specPtr->argvName[length] == 0) {
	    matchPtr = specPtr;
	    goto gotMatch;
	}
	if (matchPtr != NULL) {
	    Tcl_AppendResult(interp, "ambiguous option \"", argvName,
		    "\"", (char *) NULL);
	    return (Tk_ConfigSpec *) NULL;
	}
	matchPtr = specPtr;
    }

    if (matchPtr == NULL) {
	Tcl_AppendResult(interp, "unknown option \"", argvName,
		"\"", (char *) NULL);
	return (Tk_ConfigSpec *) NULL;
    }

    /*
     * Found a matching entry.  If it's a synonym, then find the
     * entry that it's a synonym for.
     */

    gotMatch:
    specPtr = matchPtr;
    if (specPtr->type == TK_CONFIG_SYNONYM) {
	for (specPtr = specs; ; specPtr++) {
	    if (specPtr->type == TK_CONFIG_END) {
		Tcl_AppendResult(interp,
			"couldn't find synonym for option \"",
			argvName, "\"", (char *) NULL);
		return (Tk_ConfigSpec *) NULL;
	    }
	    if ((specPtr->dbName == matchPtr->dbName) 
		    && (specPtr->type != TK_CONFIG_SYNONYM)
		    && ((specPtr->specFlags & needFlags) == needFlags)
		    && !(specPtr->specFlags & hateFlags)) {
		break;
	    }
	}
    }
    return specPtr;
}

/*
 *--------------------------------------------------------------
 *
 * DoConfig --
 *
 *	This procedure applies a single configuration option
 *	to a widget record.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	WidgRec is modified as indicated by specPtr and value.
 *	The old value is recycled, if that is appropriate for
 *	the value type.
 *
 *--------------------------------------------------------------
 */

static int
DoConfig(interp, tkwin, specPtr, value, valueIsUid, widgRec)
    Tcl_Interp *interp;		/* Interpreter for error reporting. */
    Tk_Window tkwin;		/* Window containing widget (needed to
				 * set up X resources). */
    Tk_ConfigSpec *specPtr;	/* Specifier to apply. */
    char *value;		/* Value to use to fill in widgRec. */
    int valueIsUid;		/* Non-zero means value is a Tk_Uid;
				 * zero means it's an ordinary string. */
    char *widgRec;		/* Record whose fields are to be
				 * modified.  Values must be properly
				 * initialized. */
{
    char *newValue;
    int error;
    Tk_Uid uid;
    int nullValue;

    nullValue = 0; newValue = value;
    if (*value == 0) {
	if(specPtr->specFlags & TK_CONFIG_NULL_OK) {
	    nullValue = 1; newValue = NULL;
	} else {
	    newValue = "";
	}
    }

    do {
	if (((int *)specPtr->type<(int *)TK_CONFIG_NULL)||((int *)specPtr->type>(int *)TK_CONFIG_END)) {
	    if (specPtr->type(TK_OPTION_PARSE+(valueIsUid!=0),(ClientData)specPtr->customPtr,
		    interp, tkwin, newValue, widgRec, specPtr->offset)!=TCL_OK) {
		return TCL_ERROR;
	    }
	} else {
	    sprintf(interp->result, "bad config table: unknown type %d",
		    specPtr->type);
	    return TCL_ERROR;
	}
	specPtr++;
    } while ((specPtr->argvName == NULL) && (specPtr->type != TK_CONFIG_END));
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tk_ConfigureInfo --
 *
 *	Return information about the configuration options
 *	for a window, and their current values.
 *
 * Results:
 *	Always returns TCL_OK.  Interp->result will be modified
 *	hold a description of either a single configuration option
 *	available for "widgRec" via "specs", or all the configuration
 *	options available.  In the "all" case, the result will
 *	available for "widgRec" via "specs".  The result will
 *	be a list, each of whose entries describes one option.
 *	Each entry will itself be a list containing the option's
 *	name for use on command lines, database name, database
 *	class, default value, and current value (empty string
 *	if none).  For options that are synonyms, the list will
 *	contain only two values:  name and synonym name.  If the
 *	"name" argument is non-NULL, then the only information
 *	returned is that for the named argument (i.e. the corresponding
 *	entry in the overall list is returned).
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
Tk_ConfigureInfo(interp, tkwin, specs, widgRec, argvName, flags)
    Tcl_Interp *interp;		/* Interpreter for error reporting. */
    Tk_Window tkwin;		/* Window corresponding to widgRec. */
    Tk_ConfigSpec *specs;	/* Describes legal options. */
    char *widgRec;		/* Record whose fields contain current
				 * values for options. */
    char *argvName;		/* If non-NULL, indicates a single option
				 * whose info is to be returned.  Otherwise
				 * info is returned for all options. */
    int flags;			/* Used to specify additional flags
				 * that must be present in config specs
				 * for them to be considered. */
{
    register Tk_ConfigSpec *specPtr;
    int needFlags, hateFlags;
    char *list;
    char *leader = "{";

    needFlags = flags & ~(TK_CONFIG_USER_BIT - 1);
    if (Tk_Depth(tkwin) <= 1) {
	hateFlags = TK_CONFIG_COLOR_ONLY;
    } else {
	hateFlags = TK_CONFIG_MONO_ONLY;
    }

    /*
     * If information is only wanted for a single configuration
     * spec, then handle that one spec specially.
     */

    Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
    if (argvName != NULL) {
	specPtr = FindConfigSpec(interp, specs, argvName, needFlags,
		hateFlags);
	if (specPtr == NULL) {
	    return TCL_ERROR;
	}
	interp->result = FormatConfigInfo(interp, tkwin, specPtr, widgRec);
	interp->freeProc = TCL_DYNAMIC;
	return TCL_OK;
    }

    /*
     * Loop through all the specs, creating a big list with all
     * their information.
     */

    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	if ((argvName != NULL) && (specPtr->argvName != argvName)) {
	    continue;
	}
	if (((specPtr->specFlags & needFlags) != needFlags)
		|| (specPtr->specFlags & hateFlags)) {
	    continue;
	}
	if (specPtr->argvName == NULL) {
	    continue;
	}
	list = FormatConfigInfo(interp, tkwin, specPtr, widgRec);
	Tcl_AppendResult(interp, leader, list, "}", (char *) NULL);
	ckfree(list);
	leader = " {";
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * FormatConfigInfo --
 *
 *	Create a valid Tcl list holding the configuration information
 *	for a single configuration option.
 *
 * Results:
 *	A Tcl list, dynamically allocated.  The caller is expected to
 *	arrange for this list to be freed eventually.
 *
 * Side effects:
 *	Memory is allocated.
 *
 *--------------------------------------------------------------
 */

static char *
FormatConfigInfo(interp, tkwin, specPtr, widgRec)
    Tcl_Interp *interp;			/* Interpreter to use for things
					 * like floating-point precision. */
    Tk_Window tkwin;			/* Window corresponding to widget. */
    register Tk_ConfigSpec *specPtr;	/* Pointer to information describing
					 * option. */
    char *widgRec;			/* Pointer to record holding current
					 * values of info for widget. */
{
    char *argv[6], *result;
    char buffer[200];
    Tcl_FreeProc *freeProc = (Tcl_FreeProc *) NULL;

    argv[0] = specPtr->argvName;
    argv[1] = specPtr->dbName;
    argv[2] = specPtr->dbClass;
    argv[3] = specPtr->defValue;
    if (specPtr->type == TK_CONFIG_SYNONYM) {
	return Tcl_Merge(2, argv);
    }
    argv[4] = FormatConfigValue(interp, tkwin, specPtr, widgRec, buffer,
	    &freeProc);
    if (argv[1] == NULL) {
	argv[1] = "";
    }
    if (argv[2] == NULL) {
	argv[2] = "";
    }
    if (argv[3] == NULL) {
	argv[3] = "";
    }
    if (argv[4] == NULL) {
	argv[4] = "";
    }
    result = Tcl_Merge(5, argv);
    if (freeProc != NULL) {
	if ((freeProc == TCL_DYNAMIC) || (freeProc == (Tcl_FreeProc *) free)) {
	    ckfree(argv[4]);
	} else {
	    (*freeProc)(argv[4]);
	}
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * FormatConfigValue --
 *
 *	This procedure formats the current value of a configuration
 *	option.
 *
 * Results:
 *	The return value is the formatted value of the option given
 *	by specPtr and widgRec.  If the value is static, so that it
 *	need not be freed, *freeProcPtr will be set to NULL;  otherwise
 *	*freeProcPtr will be set to the address of a procedure to
 *	free the result, and the caller must invoke this procedure
 *	when it is finished with the result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
FormatConfigValue(interp, tkwin, specPtr, widgRec, buffer, freeProcPtr)
    Tcl_Interp *interp;		/* Interpreter for use in real conversions. */
    Tk_Window tkwin;		/* Window corresponding to widget. */
    Tk_ConfigSpec *specPtr;	/* Pointer to information describing option.
				 * Must not point to a synonym option. */
    char *widgRec;		/* Pointer to record holding current
				 * values of info for widget. */
    char *buffer;		/* Static buffer to use for small values.
				 * Must have at least 200 bytes of storage. */
    Tcl_FreeProc **freeProcPtr;	/* Pointer to word to fill in with address
				 * of procedure to free the result, or NULL
				 * if result is static. */
{
    char *result;

    *freeProcPtr = NULL;
    if (((int *)specPtr->type>(int *)TK_CONFIG_NULL)&&((int *)specPtr->type<(int *)TK_CONFIG_END)) {
		specPtr->type = configProc[(int)specPtr->type];
    }
    if (((int *)specPtr->type<(int *)TK_CONFIG_NULL)||((int *)specPtr->type>(int *)TK_CONFIG_END)) {
	char *savedResult = interp->result;
	Tcl_FreeProc *savedFreeProc = interp->freeProc;
	interp->result = buffer; buffer[0]='\0';
	interp->freeProc = NULL;
	specPtr->type(TK_OPTION_PRINT,(ClientData)specPtr->customPtr,
		interp, tkwin, (char *)NULL,
		widgRec, specPtr->offset);
	result = interp->result;
	*freeProcPtr = interp->freeProc;
	interp->result = savedResult;
	interp->freeProc = savedFreeProc;
    } else {
	result = "?? unknown type ??";
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_ConfigureValue --
 *
 *	This procedure returns the current value of a configuration
 *	option for a widget.
 *
 * Results:
 *	The return value is a standard Tcl completion code (TCL_OK or
 *	TCL_ERROR).  Interp->result will be set to hold either the value
 *	of the option given by argvName (if TCL_OK is returned) or
 *	an error message (if TCL_ERROR is returned).
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_ConfigureValue(interp, tkwin, specs, widgRec, argvName, flags)
    Tcl_Interp *interp;		/* Interpreter for error reporting. */
    Tk_Window tkwin;		/* Window corresponding to widgRec. */
    Tk_ConfigSpec *specs;	/* Describes legal options. */
    char *widgRec;		/* Record whose fields contain current
				 * values for options. */
    char *argvName;		/* Gives the command-line name for the
				 * option whose value is to be returned. */
    int flags;			/* Used to specify additional flags
				 * that must be present in config specs
				 * for them to be considered. */
{
    Tk_ConfigSpec *specPtr;
    int needFlags, hateFlags;

    needFlags = flags & ~(TK_CONFIG_USER_BIT - 1);
    if (Tk_Depth(tkwin) <= 1) {
	hateFlags = TK_CONFIG_COLOR_ONLY;
    } else {
	hateFlags = TK_CONFIG_MONO_ONLY;
    }
    specPtr = FindConfigSpec(interp, specs, argvName, needFlags, hateFlags);
    if (specPtr == NULL) {
	return TCL_ERROR;
    }
    interp->result = FormatConfigValue(interp, tkwin, specPtr, widgRec,
	    interp->result, &interp->freeProc);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_FreeOptions --
 *
 *	Free up all resources associated with configuration options.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Any resource in widgRec that is controlled by a configuration
 *	option (e.g. a Tk_3DBorder or XColor) is freed in the appropriate
 *	fashion.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
void
Tk_FreeOptions(specs, widgRec, display, needFlags)
    Tk_ConfigSpec *specs;	/* Describes legal options. */
    char *widgRec;		/* Record whose fields contain current
				 * values for options. */
    Display *display;		/* X display; needed for freeing some
				 * resources. */
    int needFlags;		/* Used to specify additional flags
				 * that must be present in config specs
				 * for them to be considered. */
{
    register Tk_ConfigSpec *specPtr;

    for (specPtr = specs; specPtr->type != TK_CONFIG_END; specPtr++) {
	if ((specPtr->specFlags & needFlags) != needFlags) {
	    continue;
	}
	if (((int *)specPtr->type>(int *)TK_CONFIG_NULL)&&((int *)specPtr->type<(int *)TK_CONFIG_END)) {
	    specPtr->type = configProc[(int)specPtr->type];
	}
	if (((int *)specPtr->type<(int *)TK_CONFIG_NULL)||((int *)specPtr->type>(int *)TK_CONFIG_END)) {
	    specPtr->type(TK_OPTION_FREE,(ClientData)specPtr->customPtr,
		    (Tcl_Interp *)NULL, (Tk_Window) display, (char *)NULL,
		    widgRec, specPtr->offset);
	} else if (specPtr->type!=TK_CONFIG_SYNONYM) {
	    printf("bad config table: unknown type %d\n",
			specPtr->type);
	    exit(1);
	}
    }
}
