
/*
 * opt.c - implementation of Tcl-Option facilities
 *
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: option.c,v 1.3 1996/05/04 21:00:42 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include "optionInt.h"


/*
 *------------------------------------------------------*
 *
 *	Option_Init --
 *
 *	------------------------------------------------*
 *	Initializes this extension.
 *	Entrypoint used by dynamic loader.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The extension is registered as package.
 *
 *	Result:
 *		A standard Tcl error code.
 *
 *------------------------------------------------------*
 */

EXTERN int
Option_Init (interp)
Tcl_Interp*     interp;     /* interpreter for messages */
{
  if (Opt_IsInitialized (interp))
    {
      /* catch multiple initialization of an interpreter
       */
      return TCL_OK;
    }

  Tcl_SetAssocData (interp, ASSOC_KEY,
		    (Tcl_InterpDeleteProc*) NULL,
		    (ClientData) 1);

  return Tcl_PkgProvide (interp, "Option", OPT_VERSION);
}

/*
 *------------------------------------------------------*
 *
 *	Opt_IsInitialized --
 *
 *	------------------------------------------------*
 *	Check, wether this extension was initialized in
 *	the specified interpreter or not.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a boolean value.
 *
 *------------------------------------------------------*
 */

int
Opt_IsInitialized (interp)
Tcl_Interp *interp;	/* interpreter to check for initialization */
{
  long id = (long) Tcl_GetAssocData (interp, ASSOC_KEY, NULL);

  return (id == 1);
}

/*
 *------------------------------------------------------*
 *
 *	Opt_CompileArray --
 *
 *	------------------------------------------------*
 *	The definitions in the specified array are
 *	converted into an internal format.  This format
 *	is easier to use by the argument parser.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated.
 *
 *	Result:
 *		Opt_Table handle != NULL: compilation was sucessful.
 *		NULL:                     compilation failed.
 *					  an error message is left
 *					  in the specified interpreter.
 *
 *------------------------------------------------------*
 */

EXTERN Opt_Table
Opt_CompileArray (interp, numOptions, optDef)
Tcl_Interp*     interp;     /* interpreter for messages */
int             numOptions; /* #options to compile */
CONST Opt_Spec* optDef;     /* array of definitions */
{
  Opt_ITable*    table = ckalloc (sizeof (Opt_ITable));
  int            i, new;
  Tcl_HashEntry* hPtr;

  if (! table)
    {
      Tcl_AppendResult (interp, "not enough memory", 0);
      return NULL;
    }

  table->slave = NULL;
  Tcl_InitHashTable (&table->def, TCL_STRING_KEYS);

  for (i=0; i < numOptions; i++)
    {
      hPtr = Tcl_CreateHashEntry (&table->def, optDef [i].key, &new);
      Tcl_SetHashValue (hPtr, optDef [i].f);
    }

  return (Opt_Table) table;
}

/*
 *------------------------------------------------------*
 *
 *	Opt_DeleteTable --
 *
 *	------------------------------------------------*
 *	All memory allocated by the specified option
 *	table is released. The handle is not valid
 *	anymore after the call.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is freed.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

EXTERN void
Opt_DeleteTable (options)
Opt_Table options;
{
  Opt_ITable* optDef = (Opt_ITable*) options;

  assert (optDef);

  Tcl_DeleteHashTable (&optDef->def);
  ckfree (optDef);
}

/*
 *------------------------------------------------------*
 *
 *	Opt_LinkTables --
 *
 *	------------------------------------------------*
 *	The second table is defined as 'slave' of the
 *	first.  The argument parser will use the 'slave'
 *	if an option was not found in the master.  A slave
 *	is allowed to be the master of another slave.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

EXTERN void
Opt_LinkTables (master, slave)
Opt_Table master; /* table to link the second to */
Opt_Table slave;  /* table to link into the first */
{
  Opt_ITable* optDef = (Opt_ITable*) master;

  assert (optDef);
  assert (slave);

  optDef->slave = slave;
}

/*
 *------------------------------------------------------*
 *
 *	Opt_LinkIntoChain --
 *
 *	------------------------------------------------*
 *	The second table is defined as 'slave' of the
 *	first.  The argument parser will use the 'slave'
 *	if an option was not found in the master.  A slave
 *	is allowed to be the master of another slave.
 *	In contrast to 'Opt_LinkTables' the slave is
 *	linked at the end of the chain of already linked
 *	slaves.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

EXTERN void
Opt_LinkIntoChain (master, slave)
Opt_Table master; /* table to link the second to */
Opt_Table slave;  /* table to link into the first */
{
  Opt_ITable* optDef = (Opt_ITable*) master;

  assert (optDef);
  assert (slave);

  while (optDef->slave != NULL)
    {
      optDef = (Opt_ITable*) optDef->slave;
    }

  optDef->slave = slave;
}

/*
 *------------------------------------------------------*
 *
 *	Opt_Unlink --
 *
 *	------------------------------------------------*
 *	The connection of the specified table to a slave
 *	is cut.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

EXTERN void
Opt_Unlink (master)
Opt_Table master;
{
  Opt_ITable* optDef = (Opt_ITable*) master;

  assert (optDef);

  optDef->slave = NULL;
}

/*
 *------------------------------------------------------*
 *
 *	Opt_Lookup --
 *
 *	------------------------------------------------*
 *	The procedure locates the definition of option
 *	'name' and returns the associated handler procedure.
 *	If no definition is found, the chain of slaves is
 *	searched until a definition is found or the chain
 *	is exhausted.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		NULL:    option is not defined
 *		!= NULL: option is defined and the
 *			 return value references
 *			 the handler procedure
 *
 *------------------------------------------------------*
 */

EXTERN Opt_HandlerProc
Opt_Lookup (table, name)
Opt_Table   table; /* table to search for option */
CONST char* name;  /* option to look for (without preceding '-') */
{
  Opt_ITable*    optDef = (Opt_ITable*) table;
  Tcl_HashEntry* hPtr;

  assert (optDef);
  assert (name);

  while (optDef)
    {
      hPtr = Tcl_FindHashEntry (&optDef->def, (char*) name);

      if (! hPtr)
	{
	  /* option not defined in table */

	  if (! optDef->slave)
	    /* chain exhausted, signal failure */
	    return NULL;
	  else
	    /* try next node in chain of linked tables */
	    optDef = (Opt_ITable*) optDef->slave;
	}
      else
	{
	  /* option found */
	  return (Opt_HandlerProc) Tcl_GetHashValue (hPtr);
	}
    }

  /* control should not come here */
  assert (0);

  /* satisfy gcc */
  return NULL;
}

/*
 *------------------------------------------------------*
 *
 *	Opt_ParseArgv --
 *
 *	------------------------------------------------*
 *	Scans the specified arguments and handles all
 *	detected options.  The set of legal options is
 *	defined by 'options'.  Any detected option not in
 *	there causes an error.  In addition '--' is known
 *	and forces all arguments behind it to be treated
 *	as non-options.  Caution: This property is lost,
 *	if '--' is accepted by an handler procedure as
 *	argument to the handled option!
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Almost anything, depending on the option
 *		handlers.  'argv' is modified to hold the
 *		non-option arguments after the call.
 *		'argc' is modified accordingly. *		
 *
 *	Result:
 *		TCL_OK:    parsing successfully completed.
 *		TCL_ERROR: parsing failed, an error message
 *			   is left in the interpreter result
 *			   area.
 *		
 *
 *------------------------------------------------------*
 */

EXTERN int
Opt_ParseArgv (interp, options, argc, argv, clientData)
Tcl_Interp* interp;     /* interpreter for messages */
Opt_Table   options;    /* known options */
int*        argc;       /* number of arguments */
char**      argv;       /* text of arguments */
ClientData  clientData; /* arbitrary context, passed to handler procedures */
{
  Opt_HandlerProc handler;
  int             forceNonOption = FALSE;
  int         num;
  int         i, j, len, skip, res;

  assert (options);
  assert (interp);
  assert (argc);

  num = *argc;

#define IsNoOption(str) (forceNonOption || ('-' != str [0]))

  for (i=j=0; i < num; i++)
    {
      if (IsNoOption (argv [i]))
	{
	  /* save non-option in area of already processed arguments */

	  argv [j] = argv [i];
	  j++;
	  continue;
	}

      /* handle (possible) option */

      len = strlen (argv [i]);
      if (len < 2)
	{
	  Tcl_AppendResult (interp, "'-' is no legal option", 0);
	  return TCL_ERROR;
	}

      if ((2 == len) && ('-' == argv [i][1]))
	{
	  /* '--', force everything behind to be treated as non-option */
	  forceNonOption = TRUE;
	  continue;
	}

      /* lookup associated handler */

      handler = Opt_Lookup (options, argv [i] + 1 /* skip over '-' */);

      if (handler == NULL)
	{
	  /* option not known, throw error to caller */
	  Tcl_AppendResult (interp, "unknown option '", argv [i], "'", 0);
	  return TCL_ERROR;
	}


      /* option known, invoke handler to process any option-arguments
       */

      skip = 0;
      res = handler (clientData, interp,
		     argv [i] +1,  /* skip over '-' */
		     num  - (i+1), /* #arguments behind! argv [i] */
		     argv + (i+1), /* see above */
		     &skip);
      if (res != TCL_OK)
	return res;

      /* skip over processed option arguments */
      i += skip;
    }

  /* store number of non options back into caller-variable */
  *argc = j;
  return TCL_OK;

#undef IsNoOption
}

