
/*
 * symmetric.c --
 *
 *	Implementation of gneric kernel for symmetric cryptosystems.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: symmetric.c,v 1.2 1996/05/22 19:50:06 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include <blobXInt.h>


/*
 * declaration of internal types
 */

/*
 * Every command realizing a symmetric crptosystem is given
 * an instance of the following structure as its clientdata.
 * It enables the generic procedure to call the realization
 * of the special cryptosystem.
 */

typedef struct ScsData {
  BlobX_SCS          crypt;       /* realization of en-/decryption */
  BlobX_SCSInit      init;        /* init of additional option structure */
  ClientData         clientData;  /* context to use during calls to 'crypt' */
  Tcl_CmdDeleteProc* cleanup;     /* proc to cleanup context */
  Opt_Table          optExt;      /* additional options */
  int                optDataSize; /* additional required storage
				   * by 'optExt' handlers */
  Tcl_CmdDeleteProc* cleanOpt;    /* proc to cleanup data collected
				     via 'optExt' */
} ScsData;


/*
 * Internal procedures
 */

static void
DeleteSCS _ANSI_ARGS_ ((ClientData  clientData));

static int
HandleSCS _ANSI_ARGS_ ((ClientData  clientData,
			Tcl_Interp* interp,
			int         argc,
			char**      argv));

/*
 * Option handlers
 */

static int
HandleEncrypt _ANSI_ARGS_ ((ClientData  clientData,
			    Tcl_Interp* interp,
			    CONST char* name,
			    int         argc,
			    char**      argv,
			    int*        processed));

static int
HandleDecrypt _ANSI_ARGS_ ((ClientData  clientData,
			    Tcl_Interp* interp,
			    CONST char* name,
			    int         argc,
			    char**      argv,
			    int*        processed));

static int
HandleKey _ANSI_ARGS_ ((ClientData  clientData,
			Tcl_Interp* interp,
			CONST char* name,
			int         argc,
			char**      argv,
			int*        processed));

static int
HandleData _ANSI_ARGS_ ((ClientData  clientData,
			 Tcl_Interp* interp,
			 CONST char* name,
			 int         argc,
			 char**      argv,
			 int*        processed));

/*
 *------------------------------------------------------*
 *
 *	BlobX_SCSOptions --
 *
 *	------------------------------------------------*
 *	Retrieves the table of options known by symmetric
 *	cryptosystems.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated.
 *
 *	Result:
 *		The created Opt_Table
 *
 *------------------------------------------------------*
 */

Opt_Table
BlobX_SCSOptions (interp)
Tcl_Interp* interp;
{
  static Opt_Table scsOptions = 0;

  if (! scsOptions)
    {
      static Opt_Spec scs [] =
	{
	  {"encrypt", HandleEncrypt, "request encryption of data"},
	  {"decrypt", HandleDecrypt, "request decryption of data"},
	  {"key",     HandleKey,     "key to use in en-/decryption"},
	  {"data",    HandleData,    "data to en-/decrypt"}
	};

      scsOptions = Opt_CompileArray (interp,
				     sizeof (scs) / sizeof (Opt_Spec),
				     scs);
      assert (scsOptions);
    }

  return scsOptions;
}

/*
 *------------------------------------------------------*
 *
 *	BlobX_RegisterSCS --
 *
 *	------------------------------------------------*
 *	See header 'blobX.h'
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

int
BlobX_RegisterSCS (interp, name, crypt, init, clientData,
		   cleanup, optExt, optDataSize, cleanOpt)
Tcl_Interp*        interp;      /* interpreter to extend */
CONST char*        name;        /* name of new command */
BlobX_SCS          crypt;       /* procedure realizing the cryptosystem */
BlobX_SCSInit      init;        /* option init */
ClientData         clientData;  /* context to use in calls to 'f' */
Tcl_CmdDeleteProc* cleanup;     /* proc to cleanup context */
Opt_Table          optExt;      /* additional options */
int                optDataSize; /* additional required storage
				   by 'optExt' handlers */
Tcl_CmdDeleteProc* cleanOpt;    /* procedure to delete option data
				   collected via 'optExt' */
{
  ScsData*         scsDataPtr = (ScsData*) ckalloc (sizeof (ScsData));
  Opt_Table        options;

  if (! scsDataPtr)
    {
      Tcl_AppendResult (interp,
			"not enough memory to register '",
			name, "'", 0);
      return TCL_ERROR;
    }

  options = (optExt ? optExt : BlobX_SCSOptions (interp));

  scsDataPtr->crypt       = crypt;
  scsDataPtr->init        = init;
  scsDataPtr->clientData  = clientData;
  scsDataPtr->cleanup     = cleanup;
  scsDataPtr->optExt      = options;
  scsDataPtr->optDataSize = optDataSize;
  scsDataPtr->cleanOpt    = cleanOpt;

  Tcl_CreateCommand (interp, (char*) name,
		     HandleSCS, (ClientData) scsDataPtr, DeleteSCS);
  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	DeleteSCS --
 *
 *	------------------------------------------------*
 *	Frees all memory associated to a command
 *	realizing a SCS.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		None
 *
 *------------------------------------------------------*
 */

static void
DeleteSCS (clientData)
ClientData  clientData;
{
  ScsData* scsDataPtr = (ScsData*) clientData;

  if (scsDataPtr->cleanup)
    (*scsDataPtr->cleanup) (scsDataPtr->clientData);

  ckfree ((ScsData*) clientData);
}

/*
 *------------------------------------------------------*
 *
 *	HandleSCS --
 *
 *	------------------------------------------------*
 *	This procedure implements the general argument
 *	parsing of all SCS's.  After successful retrieval
 *	of result, key and data the function realizing
 *	the actual SCS is called to do the real
 *	en-/decryption.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		As of Blob_GetInput, Blob_Create, memory
 *		is allocated and freed
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleSCS (clientData, interp, argc, argv)
ClientData  clientData; /* reference to implementation of SCS */
Tcl_Interp* interp;     /* interpreter we are working in */
int         argc;       /* number of arguments to command */
char**      argv;       /* arguments to process */
{
  ScsData*             scsDataPtr  = (ScsData*) clientData;
  char*                cmdName     = argv [0];
  Opt_Table            options     = scsDataPtr->optExt;
  BlobX_ScsOptionData* optInfo;
  int                  optInfoSize = (sizeof (BlobX_ScsOptionData) +
				      scsDataPtr->optDataSize);
  char*                resultName;
  Blob                 resultBlob;
  int                  blobCreated = FALSE;
  int res;

  /* make command name prefix to possible error messages
   */
  Tcl_AppendResult (interp, cmdName, ": ", 0);


  /* setup structure to collect option information into.
   */
  optInfo = ckalloc (optInfoSize);
  
  if (! optInfo)
    {
      Tcl_AppendResult (interp, "not enough memory", 0);
      return TCL_ERROR;
    }

  memset ((VOID*) optInfo, '\0', optInfoSize);

  if (scsDataPtr->init)
    (*scsDataPtr->init) (optInfo, scsDataPtr->clientData);

  /* scan command line, skip over command itself */
  argc --;
  argv ++;
  res = Opt_ParseArgv (interp, options,
		       &argc, argv,
		       (ClientData) optInfo);

  if (res != TCL_OK)
    goto clean;

  /* additional argument checks */
  if (argc < 1)
    {
      Tcl_AppendResult (interp, "specification of result missing", 0);
      res = TCL_ERROR;
      goto clean;
    }
  else if (optInfo->direction == NOCRYPT)
    {
      Tcl_AppendResult (interp, "direction not defined", 0);
      res = TCL_ERROR;
      goto clean;
    }
  else if (optInfo->key.data == 0)
    {
      Tcl_AppendResult (interp, "key missing", 0);
      res = TCL_ERROR;
      goto clean;
    }

  /* Although it is tempting to do so we are not allowed to
   * check for the existence of data here yet !!  The special
   * cryptosystem may employ other forms operation beyond
   * encryption/decryption not requiring such.
   *
   * Example: blockciphers.
   * - They define the operation '-schedule-key' wich requires a key, but no data.
   */

#if 0
  else if (optInfo->data.data == 0)
    {
      Tcl_AppendResult (interp, "data missing", 0);
      res = TCL_ERROR;
      goto clean;
    }
#endif

  /* now lets do the work (skip over name of result) */

  resultName = argv [0];

  argc --;
  argv ++;

  res =  (*scsDataPtr->crypt) (scsDataPtr->clientData,
			       optInfo, interp, argc, argv);
  if (res != TCL_OK)
    goto clean;


  /* access blob to store result into, create if necessary */

  res = Blob_GetBlobHandle (interp, resultName, &resultBlob);
  if (res != TCL_OK)
    {
      res = Tcl_VarEval (interp, "blob create -secure", resultName, 0);
      if (res != TCL_OK)
	goto clean;

      res = Blob_GetBlobHandle (interp, resultName, &resultBlob);
      assert (res == TCL_OK);

      blobCreated = TRUE;
    }


  /* closing work: store transformed data */

  res = Blob_SetData (resultBlob, optInfo->data.length, optInfo->data.data);
  if (res != BLOB_OK)
    {
      Tcl_AppendResult (interp, Blob_LastError (resultBlob), 0);
      res = TCL_ERROR;

      if (blobCreated)
	{
	  Tcl_DString save;

	  Tcl_DStringInit (&save);
	  Tcl_DStringGetResult (interp, &save);

	  Tcl_VarEval (interp, "rename ", resultName, " {}", 0);
	  
	  Tcl_DStringResult (interp, &save);
	  Tcl_DStringFree (&save);
	}

      goto clean;
    }

  /* everything went fine */

  Tcl_ResetResult (interp);
  Tcl_AppendResult (interp, resultName, 0);
  res = TCL_OK;


 clean:
  if (optInfo->key.data && optInfo->key.isAlloc)
    ckfree (optInfo->key.data);

  if (optInfo->data.data)
    {
      assert (optInfo->data.isAlloc);
      ckfree (optInfo->data.data);
    }

  if (scsDataPtr->cleanOpt)
    (*scsDataPtr->cleanOpt) (optInfo);

  ckfree (optInfo);

  return res;
}

/*
 *------------------------------------------------------*
 *
 *	HandleEncrypt --
 *
 *	------------------------------------------------*
 *	process option '-encrypt'. no additional
 *	arguments are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleEncrypt (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_ScsOptionData* optInfo = (BlobX_ScsOptionData*) clientData;

  optInfo->direction = ENCRYPT;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleDecrypt --
 *
 *	------------------------------------------------*
 *	process option '-encrypt'. no additional
 *	arguments are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleDecrypt (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_ScsOptionData* optInfo = (BlobX_ScsOptionData*) clientData;

  optInfo->direction = DECRYPT;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleKey --
 *
 *	------------------------------------------------*
 *	process option '-key'. the following 2-5 arguments
 *	have to follow the syntax accepted by 'Blob_GetInput'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleKey (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_ScsOptionData* optInfo = (BlobX_ScsOptionData*) clientData;
  int                  res;
  int   length;
  int   isAlloc;
  char* data;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "-key without enough arguments", 0);
      return TCL_ERROR;
    }

  res = Blob_GetInput (interp, argc, argv,
		       0, "-key",
		       &length, &data, &isAlloc, processed);

  if (res != TCL_OK)
    {
      Tcl_AppendResult (interp, " (-key)", 0);
      return res;
    }

  optInfo->key.length  = length;
  optInfo->key.data    = data;
  optInfo->key.isAlloc = isAlloc;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleData --
 *
 *	------------------------------------------------*
 *	process option '-data'. the following 2-5 arguments
 *	have to follow the syntax accepted by 'Blob_GetInput'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleData (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_ScsOptionData* optInfo = (BlobX_ScsOptionData*) clientData;
  int                  res;

  int   length;
  int   isAlloc;
  char* data;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "-data without enough arguments", 0);
      return TCL_ERROR;
    }

  res = Blob_GetInput (interp, argc, argv,
		       0, "-data",
		       &length, &data, &isAlloc, processed);

  if (res != TCL_OK)
    {
      Tcl_AppendResult (interp, " (-data)", 0);
      return res;
    }

  /* We are ensure now that the data stored in 'optInfo' is
   * independent of its source.  This is required due to the
   * fact that the cryptosystem will overwrite this storage
   * during its operation and our intent not to interfere
   * with the data sources.
   */

  if (isAlloc)
    {
      /* free old memory, then use new one provided by 'GetInput' */

      if (optInfo->data.data)
	ckfree (optInfo->data.data);

      optInfo->data.data    = data;
      optInfo->data.length  = length;
      optInfo->data.isAlloc = TRUE;

      return TCL_OK;
    }


  /* data by 'GetInput' not allocated, try to reuse existing storage */

  if (! optInfo->data.data)
    /* no reuse possible, copy data */
    optInfo->data.data  = ckalloc (length);
  else
    {
      assert (optInfo->data.isAlloc);

      if (length > optInfo->data.length)
	{
	  optInfo->data.data = ckrealloc (optInfo->data.data, length);
	}
    }

  optInfo->data.isAlloc = TRUE;
  optInfo->data.length = length;

  memcpy ((VOID*) optInfo->data.data, (VOID*) data, length);

  return TCL_OK;
}

