
/*
 * safertcl.c --
 *
 *	Implementation of interface to SAFER encryption algorithm.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: safertcl.c,v 1.3 1996/05/19 14:58:47 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>

#include <safer/safer.h>
#include <blobXInt.h>


/*
 * Internal procedures.
 */

#define SAFER_BLOCKSIZE  (sizeof (safer_block_t)) /* 8 byte */
#define SAFER_KEYSIZE    (sizeof (safer_block_t)) /* at least, 16 is possib. */


static char*
safer_schedule _ANSI_ARGS_ ((BlobX_BcOptInfo* optInfo, int* length));

static void
safer_cipher _ANSI_ARGS_ ((VOID* schedule,
			   char* in,
			   char* out));

/*
 * Definition of keyschedule.
 */

typedef struct keyschedule {
  short       doEncryption;  /* direction information */
  safer_key_t saferSchedule; /* the real schedule of SAFER */
} keyschedule;


/*
 * SAFER introduces additional options.
 */

typedef struct saferOptions
{
  short useSK;  /* boolean flag indicating wether the
		 * strengthened schedule is to be
		 * used or not */
  short rounds; /* number of rounds to perform */
} saferOptions;


typedef struct SaferOptInfo
{
  BlobX_BcOptInfo bc;    /* blockcipher information */
  saferOptions    safer; /* safer specific information */
} SaferOptInfo;


static int
HandleStrong _ANSI_ARGS_ ((ClientData  clientData,
			   Tcl_Interp* interp,
			   CONST char* name,
			   int         argc,
			   char**      argv,
			   int*        processed));
static int
HandleRounds _ANSI_ARGS_ ((ClientData  clientData,
			   Tcl_Interp* interp,
			   CONST char* name,
			   int         argc,
			   char**      argv,
			   int*        processed));

/*
 *------------------------------------------------------*
 *
 *	BlobXRegisterSafer --
 *
 *	------------------------------------------------*
 *	Register interface to native SAFER implementation
 *	as block cipher.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

int
BlobXRegisterSafer (interp)
Tcl_Interp* interp;
{
  /* additional options provided by SAFER block cipher
   */

  static Opt_Table saferOpt = 0;

  if (saferOpt == 0)
    {
      static Opt_Spec soTable [] =
	{
	  {"strong", HandleStrong, "request strengthend keyschedule"},
	  {"rounds", HandleRounds, "request different number of rounds"}
	};

      saferOpt = Opt_CompileArray (interp,
				   sizeof (soTable) / sizeof (Opt_Spec),
				   soTable);
      assert (saferOpt);

      /* include block cipher options */
      Opt_LinkTables (saferOpt, BlobX_BCOptions (interp));

      Safer_Init_Module ();
    }


  return BlobX_RegisterBC (interp,
			   "safer",
			   SAFER_KEYSIZE,
			   SAFER_BLOCKSIZE,
			   safer_schedule,
			   safer_cipher,
			   0 /* no special cleanup, memset sufficient */,
			   0 /* no special context required */,
			   0 /* therefore no cleanup of context too */,
			   saferOpt,
			   sizeof (saferOptions),
			   0 /* no cleanup of additional option data */);
}

/*
 *------------------------------------------------------*
 *
 *	safer_schedule --
 *
 *	------------------------------------------------*
 *	Interfaces native SAFER implementation with
 *	Tcl-BlobX according to block cipher interface.
 *
 *	This is the keyscheduler.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Loads key area.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static char*
safer_schedule (bcOptInfo, length)
BlobX_BcOptInfo* bcOptInfo;
int*             length;
{
  SaferOptInfo*  optInfo  = (SaferOptInfo*) bcOptInfo;
  short          smallKey = optInfo->bc.scs.key.length < (2*SAFER_KEYSIZE);
  keyschedule*   schedule;
  safer_block_t  b1;
  safer_block_t  b2;


  schedule = ckalloc (sizeof (keyschedule));
  assert (schedule);

  /* at least 8 bytes of user key are in existence
   */
  memcpy (b1, optInfo->bc.scs.key.data, SAFER_KEYSIZE);

  if (smallKey)
    {
      /* specified key shorter than 128 bit, assume a 64 bit
       * key and put it into both halves as specified in 'safer.h'.
       */

      memcpy (b2, optInfo->bc.scs.key.data, SAFER_KEYSIZE);
    }
  else
    {
      /* use available second half of user key
       */

      memcpy (b2, optInfo->bc.scs.key.data + SAFER_KEYSIZE, SAFER_KEYSIZE);
    }


  if (optInfo->safer.rounds == 0)
    {
      /* specify default number of rounds,
       * differentiate by actual keysize and chosen schedule
       */

      if (optInfo->safer.useSK)
	optInfo->safer.rounds = (smallKey                       ?
				 SAFER_SK64_DEFAULT_NOF_ROUNDS  :
				 SAFER_SK128_DEFAULT_NOF_ROUNDS);
      else
	optInfo->safer.rounds = (smallKey                      ?
				 SAFER_K64_DEFAULT_NOF_ROUNDS  :
				 SAFER_K128_DEFAULT_NOF_ROUNDS);
    }


  schedule->doEncryption = (optInfo->bc.scs.direction == ENCRYPT);

  Safer_Expand_Userkey (b1, b2,
			optInfo->safer.rounds,
			optInfo->safer.useSK,
			schedule->saferSchedule);

  *length = sizeof (keyschedule);
  return (char*) schedule;

}

/*
 *------------------------------------------------------*
 *
 *	safer_cipher --
 *
 *	------------------------------------------------*
 *	Interfaces native SAFER implementation with
 *	Tcl-BlobX according to block cipher interface.
 *
 *	This is the encryption / decryption algorithm.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		'out' is set up to be the encryption or
 *		decryption of 'in' (dependent on the
 *		arguments to the key scheduler (see above)).
 *
 *	Result:
 *		out.
 *
 *------------------------------------------------------*
 */

static void
safer_cipher (schedule, in, out)
VOID* schedule;
char* in;
char* out;
{
  safer_block_t i, o;

  keyschedule* ks = (keyschedule*) schedule;


  memcpy (&i, in, SAFER_BLOCKSIZE);

  if (ks->doEncryption)
    {
      Safer_Encrypt_Block (i, ks->saferSchedule, o);
    }
  else
    {
      Safer_Decrypt_Block (i, ks->saferSchedule, o);
    }

  memcpy (out, &o, SAFER_BLOCKSIZE);
}

/*
 * Handle additional options.
 */

/*
 *------------------------------------------------------*
 *
 *	HandleStrong --
 *
 *	------------------------------------------------*
 *	Processes option "-strong", recording the request
 *	to use the strenghtened key schedule in the safer
 *	specific part of the option structure.
 *	no additional arguments are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleStrong (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  SaferOptInfo* optInfo = (SaferOptInfo*) clientData;

  optInfo->safer.useSK = TRUE;
  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleRounds --
 *
 *	------------------------------------------------*
 *	Process option "-rounds" and record the number
 *	requested iterations to perform by SAFER.
 *	A numeric argument is required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleRounds (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  SaferOptInfo* optInfo = (SaferOptInfo*) clientData;
  int           rounds;
  int           res;

  if (argc < 1)
    {
      Tcl_AppendResult (interp, "-rounds without argument", 0);
      return TCL_ERROR;
    }

  res = Tcl_GetInt (interp, argv [0], &rounds);
  if (res != TCL_OK)
    {
      Tcl_AppendResult (interp, " (-rounds)", 0);
      return res;
    }

  if (rounds < 6)
    {
      Tcl_AppendResult (interp, "-rounds argument to small (< 6)", 0);
      return TCL_ERROR;
    }
  else if (rounds > SAFER_MAX_NOF_ROUNDS)
    {
      Tcl_AppendResult (interp, "-rounds argument to big (> 13)", 0);
      return TCL_ERROR;
    }
  /* else: ok, data in set boundaries */

  optInfo->safer.rounds = rounds;
  *processed = 1;
  return TCL_OK;
}

