
/*
 * ideatcl.c --
 *
 *	Implementation of interface to IDEA encryption algorithm.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: ideatcl.c,v 1.2 1996/02/11 00:03:15 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>

#include <idea/idea.h>
#include <blobXInt.h>


/*
 * Declaration of internal procedures.
 */

#define IDEA_KEYSIZE   (16)  /* byte */
#define IDEA_BLOCKSIZE  (8)  /* byte */

static char*
idea_schedule _ANSI_ARGS_ ((BlobX_BcOptInfo* optInfo, int* length));

static void
idea_cipher _ANSI_ARGS_ ((VOID* keyschedule,
			  char* in,
			  char* out));


#ifdef WORDS_BIGENDIAN
#define FLIP(buf,n)
#else
  /* input is assumed to be big endian,
   * transform to little endian of actual
   * machine
   */

#define FLIP(buf,n) {int _i; for (_i=0; _i < n; _i++) buf [_i] = htons (buf [_i]);}
#endif

/*
 *------------------------------------------------------*
 *
 *	BlobXRegisterIdea --
 *
 *	------------------------------------------------*
 *	Register interface to native IDEA implementation
 *	as block cipher.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

int
BlobXRegisterIdea (interp)
Tcl_Interp* interp;
{
  return BlobX_RegisterBC (interp,
			   "idea",
			   IDEA_KEYSIZE,
			   IDEA_BLOCKSIZE,
			   idea_schedule,
			   idea_cipher,
			   0 /* no special cleanup, memset sufficient */,
			   0 /* no special context required */,
			   0 /* therefore no cleanup of context too */,
			   0 /* no additional options */,
			   0 /* therefore no additional option storage */,
			   0 /* no cleanup of additional option data */);
}

/*
 *------------------------------------------------------*
 *
 *	idea_schedule --
 *
 *	------------------------------------------------*
 *	Interfaces native IDEA implementation with
 *	Tcl-BlobX according to block cipher interface.
 *
 *	This is the keyscheduler.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Creates area containing internal key.
 *
 *	Result:
 *		Reference to allocated memory + length of that.
 *
 *------------------------------------------------------*
 */

static char*
idea_schedule (optInfo, length)
BlobX_BcOptInfo* optInfo;
int* length;
{
  Idea_UserKey K;
  Idea_Key     Z;
  Idea_Key*    schedule;

  schedule = ckalloc (sizeof (Idea_Key));
  assert (schedule);

  memcpy (K, optInfo->scs.key.data, sizeof (Idea_UserKey));
  FLIP (K, Idea_userKeyLen);

  Idea_ExpandUserKey (K, Z);

  if (optInfo->scs.direction == DECRYPT)
    {
      Idea_InvertKey (Z, Z);
    }

  memcpy (schedule, Z, sizeof (Idea_Key));

  *length = sizeof (Idea_Key);
  return (char*) schedule;
}

/*
 *------------------------------------------------------*
 *
 *	idea_cipher --
 *
 *	------------------------------------------------*
 *	Interfaces native IDEA implementation with
 *	Tcl-BlobX according to block cipher interface.
 *
 *	This is the encryption / decryption algorithm.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		'out' is set up to be the encryption or
 *		decryption of 'in' (dependent on the
 *		arguments to the key scheduler (see above)).
 *
 *	Result:
 *		out.
 *
 *------------------------------------------------------*
 */

static void
idea_cipher (schedule, in, out)
VOID* schedule;
char* in;
char* out;
{
  Idea_Key* ks = schedule;
  Idea_Data i, o;

  memcpy (i, in, sizeof (Idea_Data));
  FLIP (i, Idea_dataLen);

  Idea_Crypt (i, o, *ks);

  FLIP (o, Idea_dataLen);
  memcpy (out, o, sizeof (Idea_Data));
}

