
/*
 * genericRI.c --
 *
 *	Implementation of procedure realizing 'cmd result INPUT'
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: genericRI.c,v 1.4 1996/05/22 19:50:04 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>

#include <blobXInt.h>


/*
 * declaration of internal procedures.
 */

static int
GenericRICmd _ANSI_ARGS_ ((ClientData  dummy,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));

/*
 *------------------------------------------------------*
 *
 *	BlobX_RegisterUnaryOp --
 *
 *	------------------------------------------------*
 *	Registers a new command following the syntax
 *	'cmd result INPUT'.  From now on the system will
 *	execute function 'f' whenever the command 'name'
 *	is called and successfully parses its arguments.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		A command is created.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

void
BlobX_RegisterUnaryOp (interp, name, f)
Tcl_Interp*         interp; /* interpreter we are working in */
CONST char*         name;   /* name of command to create */
BlobX_UnaryOperator f;      /* function to call */
{
  Tcl_CreateCommand (interp, (char*) name,
		     GenericRICmd, (ClientData) f, 0);
}

/*
 *------------------------------------------------------*
 *
 *	GenericRICmd --
 *
 *	------------------------------------------------*
 *	Realization of generic command 'cmd result INPUT'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The provided function is evaluated given
 *		the specified input, its result is stored
 *		in a blob.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
GenericRICmd (dummy, interp, argc, argv)
ClientData  dummy;	/* function to evaluate */
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;	/* number of arguments to command */
char**      argv;	/* array referencing the arguments */
{
  /*
   * expected syntax: command result INPUT
   */

  char*   resultName;
  char*   cmdName;

  int     length;
  char*   data;
  int     dataIsAllocated = 0;

  int     res, num;
  int     resBlobCreated = 0;
  Blob    resBlob        = 0;

  BlobX_UnaryOperator f = (BlobX_UnaryOperator) dummy;


  cmdName = argv [0];

  if (argc < 4)
    {
      Tcl_AppendResult (interp,
			"wrong # args for ", cmdName,
			": must be '", cmdName," result INPUT'",
			0);
      return TCL_ERROR;
    }

  resultName = argv [1];

  argv += 2;
  argc -= 2;

  res = Blob_GetBlobHandle (interp, resultName, &resBlob);
  if (res != TCL_OK)
    {
      Tcl_ResetResult (interp);

      res = Tcl_VarEval (interp, "blob create -secure ", resultName, 0);
      if (res != TCL_OK)
	goto error;

      res = Blob_GetBlobHandle (interp, resultName, &resBlob);

      if (res != TCL_OK)
	return TCL_ERROR;

      assert (res == TCL_OK);
      resBlobCreated = 1;

      Tcl_ResetResult (interp);
    }

  res = Blob_GetInput (interp, argc, argv, resBlob, resultName,
		       &length, &data, &dataIsAllocated, &num);

  if (res != TCL_OK)
    goto error;
#if 0
  if (length == 0)
    {
      Tcl_AppendResult (interp, cmdName, ": can't operate on empty input", 0);
      goto error;
    }
#endif

  /* at last: evaluate function realizing the operator */
  res = (*f) (interp, length, data, resBlob);
  if (res != TCL_OK)
    goto error;

  Tcl_AppendResult (interp, resultName, 0);
  goto clean;

 error:
  res = TCL_ERROR;
  if (resBlob && resBlobCreated)
    {
      /* delete generated but unused blob
       * save interpreter result area and restore after deletion.
       */

      Tcl_DString save;

      Tcl_DStringInit (&save);
      Tcl_DStringGetResult (interp, &save);

      Tcl_VarEval (interp, "rename ", resultName, " {}", 0);

      Tcl_DStringResult (interp, &save);
      Tcl_DStringFree (&save);
    }

 clean:
  if (dataIsAllocated)
    ckfree (data);

  return res;
}
