
/*
 * crc.c --
 *
 *	Implementation of interface to CRC checksum generator
 *
 * Part of this code is taken from 'pgparmor.c' of PGPTools.  The code
 * there was originally taken from 'armor.c' of PGP.  Copyright holder
 * is therefore Phillip Zimmerman (marked: with -*- PGP -*-).
 * -------------------------------------------------------------------
 *
 * Everything else:
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: crc.c,v 1.2 1996/02/12 20:48:05 aku Exp $
 */

#include <blobXInt.h>

/*
 * Declaration of internal procedures and helper macros.
 */

/* -*- PGP -*- */
#ifdef __alpha
#define crcword unsigned int		/* if CRCBITS is 24 or 32 */
#else
#define crcword unsigned long		/* if CRCBITS is 24 or 32 */
#endif

#define CRCBITS 24      /* may be 16, 24, or 32 */

/* #define maskcrc(crc) ((crcword)(crc)) */     /* if CRCBITS is 16 or 32 */
#define maskcrc(crc) ((crc) & 0xffffffL)        /* if CRCBITS is 24 */

#define CRCHIBIT ((crcword) (1L<<(CRCBITS-1))) /* 0x8000 if CRCBITS is 16 */
#define CRCSHIFTS (CRCBITS-8)

#define CCITTCRC 0x1021   /* CCITT's 16-bit CRC generator polynomial */
#define PRZCRC 0x864cfbL  /* PRZ's   24-bit CRC generator polynomial */
#define CRCINIT 0xB704CEL /* Init value for CRC accumulator          */

/*      Notes on making a good 24-bit CRC--
	The primitive irreducible polynomial of degree 23 over GF(2),
	040435651 (octal), comes from Appendix C of "Error Correcting Codes,
	2nd edition" by Peterson and Weldon, page 490.  This polynomial was
	chosen for its uniform density of ones and zeros, which has better
	error detection properties than polynomials with a minimal number of
	nonzero terms.  Multiplying this primitive degree-23 polynomial by
	the polynomial x+1 yields the additional property of detecting any
	odd number of bits in error, which means it adds parity.  This 
	approach was recommended by Neal Glover.

	To multiply the polynomial 040435651 by x+1, shift it left 1 bit and
	bitwise add (xor) the unshifted version back in.  Dropping the unused 
	upper bit (bit 24) produces a CRC-24 generator bitmask of 041446373 
	octal, or 0x864cfb hex.  

	You can detect spurious leading zeros or framing errors in the 
	message by initializing the CRC accumulator to some agreed-upon 
	nonzero "random-like" value, but this is a bit nonstandard.  
*/

/* Table for speeding up CRC's */
static crcword CrcTable [256];

/* -*- PGP -*- */

#define CRCBYTES 3 /* may be 2, 3, or 4 */


static void
GenCrcLookupTable _ANSI_ARGS_ ((crcword polynomial));



/*
 *------------------------------------------------------*
 *
 *	PgpCrcOperator --
 *
 *	------------------------------------------------*
 *	Accumulate a buffer's worth of bytes into a CRC
 *	accumulator, returning the new CRC value.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code + see above.
 *
 *------------------------------------------------------*
 */

int
PgpCrcOperator (interp, len, buf, resBlob)
Tcl_Interp* interp;	/* interpreter we are working in */
int         len;	/* length of data to hash */
CONST char* buf;	/* information to check */
Blob        resBlob;    /* store computed crc checksum here */
{
  int     res;
  crcword accum = CRCINIT;
  crcword crc;
  char    outbuf [CRCBYTES];

  static int beenHere = 0;
  if (! beenHere)
    {
      beenHere = 1;
      GenCrcLookupTable (PRZCRC);
    }

  /* -*- PGP -*-, was 'crcbytes' */
  do {
    accum = (accum << 8) ^ CrcTable [ (unsigned char) (accum >> CRCSHIFTS) ^ *buf++ ];
  } while (--len);

  crc = maskcrc (accum);
  /* -*- PGP -*- */

  /* -*- PGP -*-, was outcrc, BIGENDIAN output */
  outbuf [0] = (crc >> 16) & 0xff;
  outbuf [1] = (crc >>  8) & 0xff;
  outbuf [2] = (crc >>  0) & 0xff;
  /* -*- PGP -*- */

  res = Blob_SetData (resBlob, CRCBYTES, outbuf);
  if (res != BLOB_OK)
    {
      Tcl_AppendResult (interp, "crc: ", Blob_LastError (resBlob), 0);
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	GenCrcLookupTable --
 *
 *	------------------------------------------------*
 *	The procedure derives a CRC lookup table from
 *	the CRC polynomial.  The table is used later by
 *	the 'PgpCrcOperator' function given above.  It
 *	only needs to be called once at the dawn of time.
 *
 *	The theory behind 'GenCrcLookupTable' is that
 *	CrcTable [i] is initialized with the CRC of i,
 *	and this is related to the CRC of i>>1, so the
 *	CRC of i>>1 (pointed to by p) can be used to
 *	derive the CRC of i (pointed to by q).
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The global variable 'CrcTable' is filled.
 *
 *	Result:
 *		See 'Sideeffects'.
 *
 *------------------------------------------------------*
 */

static void
GenCrcLookupTable (poly)
crcword poly;
{
  /* -*- PGP -*-, was 'mk_crctbl' */
  int i;
  crcword t, *p, *q;
  p = q = CrcTable;

  *q++ = 0;
  *q++ = poly;

  for (i = 1; i < 128; i++)
    {
      t = *++p;
      if (t & CRCHIBIT)
	{
	  t <<= 1;
	  *q++ = t ^ poly;
	  *q++ = t;
	}
      else
	{
	  t <<= 1;
	  *q++ = t;
	  *q++ = t ^ poly;
	}
    }
  /* -*- PGP -*- */
}

