
/*
 * blowfishtcl.c --
 *
 *	Implementation of interface to BLOWFISH encryption algorithm.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: blowfishtcl.c,v 1.2 1996/05/22 19:50:02 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>

#include <blowfish.h>
#include <blobXInt.h>


/*
 * Declaration of internal procedures.
 */

#define BLOWFISH_KEYSIZE    (1)  /* byte (minimal!, maximum is 56) */
#define BLOWFISH_BLOCKSIZE  (8)  /* byte */

static char*
blowfish_schedule _ANSI_ARGS_ ((BlobX_BcOptInfo* optInfo, int* length));

static void
blowfish_cipher _ANSI_ARGS_ ((VOID* keyschedule,
			 char* in,
			 char* out));

/*
 * Definition of keyschedule.
 */

typedef struct keyschedule {
  short                doEncryption; /* direction information */
  Blowfish_keyschedule blowfishSchedule;  /* the real schedule of BLOWFISH */
} keyschedule;


/*
 *------------------------------------------------------*
 *
 *	BlobXRegisterBlowfish --
 *
 *	------------------------------------------------*
 *	Register interface to native BLOWFISH implementation
 *	as block cipher.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

int
BlobXRegisterBlowfish (interp)
Tcl_Interp* interp;
{
  return BlobX_RegisterBC (interp,
			   "blowfish",
			   BLOWFISH_KEYSIZE,
			   BLOWFISH_BLOCKSIZE,
			   blowfish_schedule,
			   blowfish_cipher,
			   0 /* no special cleanup, memset sufficient */,
			   0 /* no special context required */,
			   0 /* therefore no cleanup of context too */,
			   0 /* no additional options */,
			   0 /* therefore no additional option storage */,
			   0 /* and no cleanup of additional option data */);
}

/*
 *------------------------------------------------------*
 *
 *	blowfish_schedule --
 *
 *	------------------------------------------------*
 *	Interfaces native BLOWFISH implementation with
 *	Tcl-BlobX according to block cipher interface.
 *
 *	This is the keyscheduler.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Creates area containing internal key.
 *
 *	Result:
 *		Reference to allocated memory + length of that.
 *
 *------------------------------------------------------*
 */

static char*
blowfish_schedule (optInfo, length)
BlobX_BcOptInfo* optInfo;
int* length;
{
  keyschedule* schedule;

  schedule = ckalloc (sizeof (keyschedule));
  assert (schedule);

  schedule->doEncryption = (optInfo->scs.direction == ENCRYPT);

  InitializeBlowfish (optInfo->scs.key.data,
		      (short) optInfo->scs.key.length,
		      &schedule->blowfishSchedule);

  *length = sizeof (keyschedule);
  return (char*) schedule;
}

/*
 *------------------------------------------------------*
 *
 *	blowfish_cipher --
 *
 *	------------------------------------------------*
 *	Interfaces native BLOWFISH implementation with
 *	Tcl-BlobX according to block cipher interface.
 *
 *	This is the encryption / decryption algorithm.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		'out' is set up to be the encryption or
 *		decryption of 'in' (dependent on the
 *		arguments to the key scheduler (see above)).
 *
 *	Result:
 *		out.
 *
 *------------------------------------------------------*
 */

static void
blowfish_cipher (schedule, in, out)
VOID* schedule;
char* in;
char* out;
{
  keyschedule* ks = (keyschedule*) schedule;

  union aword left;
  union aword right;

  /* assume implicit bigendian order of incoming byte stream */

  left.word     = 0;
  right.word    = 0;

  left.w.byte0  = in [0];
  left.w.byte1  = in [1];
  left.w.byte2  = in [2];
  left.w.byte3  = in [3];
  right.w.byte0 = in [4];
  right.w.byte1 = in [5];
  right.w.byte2 = in [6];
  right.w.byte3 = in [7];

  if (ks->doEncryption)
    {
      Blowfish_encipher (&ks->blowfishSchedule,
			 (UWORD_32bits*) &left,
			 (UWORD_32bits*) &right);
    }
  else
    {
      Blowfish_decipher (&ks->blowfishSchedule,
			 (UWORD_32bits*) &left,
			 (UWORD_32bits*) &right);
    }

  out [0] = left.w.byte0;
  out [1] = left.w.byte1;
  out [2] = left.w.byte2;
  out [3] = left.w.byte3;
  out [4] = right.w.byte0;
  out [5] = right.w.byte1;
  out [6] = right.w.byte2;
  out [7] = right.w.byte3;
}

#if 0

#ifndef WORDS_BIGENDIAN
#ifdef __alpha
  {
    char tmp;
#define FLIP(a, b)  tmp = out [a]; out [a] = out [b]; out [b] = tmp;
    FLIP (0,3);
    FLIP (1,2);
    FLIP (4,7);
    FLIP (5,6);
#undef FLIP
  }
#else
  * ((long*) out)   = ntohl (* ((long*) out));
  * ((long*) out+4) = ntohl (* ((long*) out+4));
#endif
#endif

  if (in != out)
    {
      /* transfer block */
      memcpy (out, in, BLOWFISH_BLOCKSIZE);
    }
#endif
