
/*
 * blockcipher.c --
 *
 *	Implementation of procedures realizing the generic parts
 *	of block ciphers.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: blockcipher.c,v 1.3 1996/05/22 19:50:01 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include <blobXInt.h>

/*
 * declaration of internal types
 */

/*
 * Every command realizing a block cipher is given
 * an instance of the following structure as clientdata.
 * It enables the generic procedures to call the realization
 * of the special cipher.
 */

typedef struct BcData {
  char*               name;        /* Name of the blockcipher.  Allocated
				    * of the heap.  Required during
				    * destruction to access the hashtable
				    * in 'BcInterpData'. */
  int                 keySize;     /* (minimal) size of key (in bytes) */
  int                 blockSize;   /* Size of single block (in bytes)  */
  BlobX_BCScheduleKey schedule;    /* Proc. to compile external key     */
  BlobX_BCCipher      cipher;      /* Proc. to en-/decrypt              */
  BlobX_BCClean       clean;       /* Proc. to remove dangerous info    */

  ClientData          clientData;  /* Arbitrary context */
  Tcl_CmdDeleteProc*  cleanup;     /* Proc. to cleanup context */
  Opt_Table           optExt;      /* Additional options */
  int                 optDataSize; /* Additional required storage
				    * by 'optExt' handlers */
  Tcl_CmdDeleteProc*  cleanOpt;    /* Proc. to cleanup data collected
				      via 'optExt' */
} BcData;


/*
 * An instance of the following struct is associated to
 * every interpreter with registered blockciphers.
 */

typedef struct BcInterpData {

  Tcl_HashTable ciphers; /* Mapping from the names of all
			  * registered blockciphers to their
			  * definition structures (of type 'BcData').
			  */
} BcInterpData;


/*
 * The next number to use as handle for a crypt-channel.
 * Initialized with 0.  Nothing is done currently to prevent
 * overflow and resulting collisions.
 *
 * A global variable is used to prevent collisions for handles in
 * different interpreters (as it is possible to share a channel-
 * handle between interpreters).
 */

static unsigned long serial = 0;

/*
 * State information maintained by the in/out procedures
 * called from the generic layer.
 */

typedef struct BcChanControl {
  char* buf;        /* Storage area to buffer incomplete and
		     * therefore unprocessed blocks. */
  int   maxLength;  /* Maximum number of bytes the buffer can
		     * hold. */
  int   pending;    /* Number of bytes currently waiting in
		     * the buffer */
  int   state;      /* buffer state, see #define's below. */
} BcChanControl;


/*
 * Possible buffer states. Different for read- and write-buffers.
 *
 * Readbuffer:
 *
 * BCB_WAIT_FOR_READ_REQEUEST = The buffer contains already decrypted
 *                              data waiting to be read by the general
 *                              layer and the user of the channel.
 *
 * BCB_WAIT_FOR_DATA = The buffer contains an incomplete block.  The
 *                     system is waiting for the arrival of more data
 *                     from the source to be able to proceed with the
 *                     decryption process.
 *
 * Writebuffer:
 *
 * The 'state' field is not used!
 */

#define BCB_WAIT_FOR_DATA         (0) /* initial state */
#define BCB_WAIT_FOR_READ_REQUEST (1)


/*
 * An instance of the following struct is associated with every
 * crypt channel in existence.  It contains all necessary state
 * information to encrypt/decrypt outgoing/incoming data.
 */

typedef struct BcChannel {
  Tcl_Channel      source;         /* The channel to read from/write to. */
  BlobX_BcOptInfo* optInfo;        /* reference to optiondata collected
				    * during creation. */
  BcData*          bcDef;          /* Replication of information in 'optInfo->def',
				    * but access via with this element
				    * eliminates a redirection (and a cast) */
  BlobX_Data       scheduleE;      /* Keyschedule to use for encryption
				    * upon writing to the channel. */
  BlobX_Data       scheduleD;      /* Keyschedule to use for decryption
				    * upon reading from the channel. */
  BcChanControl    out;            /* State information used by 'CryptOutput' */
  BcChanControl    in;             /* State information used by 'CryptInput' */
} BcChannel;


/* Defines to use for argument 'data_required'
 * of procedure 'CheckOptionData'.
 */

#define DONT_CHECK_DATA (0)
#define CHECK_DATA      (1)


/*
 * declaration of internal procedures
 */

static int
CipherCmd _ANSI_ARGS_ ((ClientData           clientData,
			BlobX_ScsOptionData* optInfo,
			Tcl_Interp*          interp,
			int                  argc,
			char**               argv));
static void
InitBC _ANSI_ARGS_ ((BlobX_ScsOptionData* opt,
		     ClientData           clientData));

static void
DeleteBC _ANSI_ARGS_ ((ClientData clientData));

static void
CleanOpt _ANSI_ARGS_ ((ClientData clientData));

static int
CheckOptionData _ANSI_ARGS_ ((Tcl_Interp*      interp,
			      BlobX_BcOptInfo* optInfo,
			      BcData*          bcDef,
			      int              dataRequired));

/*
 * Option handler
 */

static int
HandleECB _ANSI_ARGS_ ((ClientData  clientData,
			Tcl_Interp* interp,
			CONST char* name,
			int         argc,
			char**      argv,
			int*        processed));
static int
HandleCBC _ANSI_ARGS_ ((ClientData  clientData,
			Tcl_Interp* interp,
			CONST char* name,
			int         argc,
			char**      argv,
			int*        processed));
static int
HandleCFB _ANSI_ARGS_ ((ClientData  clientData,
			Tcl_Interp* interp,
			CONST char* name,
			int         argc,
			char**      argv,
			int*        processed));
static int
HandleOFB _ANSI_ARGS_ ((ClientData  clientData,
			Tcl_Interp* interp,
			CONST char* name,
			int         argc,
			char**      argv,
			int*        processed));
static int
HandleSchedule _ANSI_ARGS_ ((ClientData  clientData,
			     Tcl_Interp* interp,
			     CONST char* name,
			     int         argc,
			     char**      argv,
			     int*        processed));
static int
HandleIV _ANSI_ARGS_ ((ClientData  clientData,
		       Tcl_Interp* interp,
		       CONST char* name,
		       int         argc,
		       char**      argv,
		       int*        processed));
static int
HandleShift _ANSI_ARGS_ ((ClientData  clientData,
			  Tcl_Interp* interp,
			  CONST char* name,
			  int         argc,
			  char**      argv,
			  int*        processed));
static int
HandleKey _ANSI_ARGS_ ((ClientData  clientData,
			Tcl_Interp* interp,
			CONST char* name,
			int         argc,
			char**      argv,
			int*        processed));
static int
HandleKeySchedule _ANSI_ARGS_ ((ClientData  clientData,
				Tcl_Interp* interp,
				CONST char* name,
				int         argc,
				char**      argv,
				int*        processed));

/*
 * Generic mode handling
 */

static void
ExecECB _ANSI_ARGS_ ((BlobX_BcOptInfo* opt));

static void
ExecCBC _ANSI_ARGS_ ((BlobX_BcOptInfo* opt));

static void
ExecCFB _ANSI_ARGS_ ((BlobX_BcOptInfo* opt));

static void
ExecOFB _ANSI_ARGS_ ((BlobX_BcOptInfo* opt));


/* The next 2 procedures were taken from PGPTools:idea.c
 * Original names are 'cfbshift' and 'xorbuf':
 *
 * >>	Can be applied for any block encryption algorithm,
 * >>	with any block size, such as the DES or the IDEA cipher.
 */

static void
ShiftBuf _ANSI_ARGS_ ((char* buf, char* in,
		       int shiftWidth, int blockSize));

static void
XorBuf _ANSI_ARGS_ ((char* buf, char* in, int blockSize));


static void
TooShort _ANSI_ARGS_ ((Tcl_Interp* interp,
		       CONST char* name,
		       int         minSize));

/*
 * Channel procedures.
 */

static Opt_Table
ChannelOptions _ANSI_ARGS_ ((Tcl_Interp* interp));

static BcInterpData*
GetInterpData _ANSI_ARGS_ ((Tcl_Interp* interp));

static void
DeleteInterpData _ANSI_ARGS_ ((ClientData  clientData,
			       Tcl_Interp* interp));

static BcChannel*
ProcessChannelConfiguration _ANSI_ARGS_ ((Tcl_Interp* interp,
					  const char* systemName,
					  int         argc,
					  char**      argv));

static void
DeleteCryptInfo _ANSI_ARGS_ ((BcChannel* crypt));

static void
CryptSrcClosed _ANSI_ARGS_ ((ClientData clientData));

static int
CryptClose _ANSI_ARGS_ ((ClientData  clientData,
			 Tcl_Interp* interp,
			 Tcl_File    inFile,
			 Tcl_File    outFile));

static int
CryptInput _ANSI_ARGS_ ((ClientData  clientData,
			 Tcl_File    inFile,
			 char*       buf,
			 int         bufSize,
			 int*        errorCodePtr));

static int
CryptOutput _ANSI_ARGS_ ((ClientData  clientData,
			  Tcl_File    outFile,
			  char*       buf,
			  int         toWrite,
			  int*        errorCodePtr));

static void
ExpandBuffer _ANSI_ARGS_ ((BcChanControl* ctrl, int newSize));

static void
DecryptBuffer _ANSI_ARGS_ ((BlobX_BcOptInfo* opt,
			    BcChannel*       chan,
			    BcChanControl*   ctrl,
			    int              num));
static void
EncryptBuffer _ANSI_ARGS_ ((BlobX_BcOptInfo* opt,
			    BcChannel*       chan,
			    BcChanControl*   ctrl,
			    int              num));

/*
 *------------------------------------------------------*
 *
 *	BlobX_BCOptions --
 *
 *	------------------------------------------------*
 *	Retrieves the table of options known by block ciphers.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated.
 *
 *	Result:
 *		The created Opt_Table
 *
 *------------------------------------------------------*
 */

Opt_Table
BlobX_BCOptions (interp)
Tcl_Interp* interp;
{
  static Opt_Table bcOptions = 0;

  if (! bcOptions)
    {
      static Opt_Spec bc [] =
	{
	  {"keyschedule",  HandleKeySchedule, "define internal key to use"},
	  {"key",          HandleKey,         "define key to use"},
	  /* 'key' overloads definition in SCS */

	  {"schedule-key", HandleSchedule, "request precompilation of key"},

	  {"ecb",          HandleECB,      "request electronic code book"},
	  {"cbc",          HandleCBC,      "request cipher block chaining"},
	  {"cfb",          HandleCFB,      "request cipher feedback"},
	  {"ofb",          HandleOFB,      "request output feedback"},

	  {"iv",           HandleIV,     "define initialization vector"},
	  {"shift",        HandleShift,  "define shiftwidth of feedback modes"}
	};

      bcOptions = Opt_CompileArray (interp,
				     sizeof (bc) / sizeof (Opt_Spec),
				     bc);
      assert (bcOptions);

      /* include standard options of SCS */
      Opt_LinkTables (bcOptions, BlobX_SCSOptions (interp));
    }

  return bcOptions;
}

/*
 *------------------------------------------------------*
 *
 *	BlobX_RegisterBC --
 *
 *	------------------------------------------------*
 *	The procedure creates a new command with name
 *	'name'.  At every successful invocation the
 *	internal general block-cipher procedure is
 *	executed to en-/decrypt the provided data with
 *	the specified key using the actual block-cipher.
 *
 *	See manual for calling syntax.
 *
 *	In addition the cipher is registrated in the
 *	structures used by the channel code, making it
 *	available for crypt-channels too.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above. 
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

EXTERN int
BlobX_RegisterBC (interp, name, keySize, blockSize,
		  schedule, cipher, clean, clientData,
		  cleanup, optExt, optDataSize, cleanOpt)

Tcl_Interp*         interp;      /* interpreter to extend */
CONST char*         name;        /* name of new command */
int                 keySize;     /* expected size of key (in bytes) */
int                 blockSize;   /* size of single block (in bytes) */
BlobX_BCScheduleKey schedule;    /* proc to do key scheduling */
BlobX_BCCipher      cipher;      /* proc to doen-/decryption */
BlobX_BCClean       clean;       /* proc to remove dangerous info */
ClientData          clientData;  /* arbitrary context */
Tcl_CmdDeleteProc*  cleanup;     /* proc to cleanup context */
Opt_Table           optExt;      /* additional options */
int                 optDataSize; /* additional required storage by
				    'optExt' handlers */
Tcl_CmdDeleteProc* cleanOpt;     /* procedure to delete option data
				    collected via 'optExt'. allowed
				    to be NULL */
{
  BcData*        bcDataPtr = (BcData*) ckalloc (sizeof (BcData));
  Opt_Table      options;
  int            res;
  BcInterpData*  idata;
  Tcl_HashEntry* hPtr;
  int            new;


  if (! bcDataPtr)
    {
      Tcl_AppendResult (interp,
			"not enough memory to register '",
			name, "'", 0);
      return TCL_ERROR;
    }

  options = (optExt ? optExt : BlobX_BCOptions (interp));

  bcDataPtr->keySize     = keySize;
  bcDataPtr->blockSize   = blockSize;
  bcDataPtr->schedule    = schedule;
  bcDataPtr->cipher      = cipher;
  bcDataPtr->clean       = clean;
  bcDataPtr->clientData  = clientData;
  bcDataPtr->cleanup     = cleanup;
  bcDataPtr->optExt      = options;
  bcDataPtr->optDataSize = optDataSize;
  bcDataPtr->cleanOpt    = cleanOpt;


  /* Register blockcipher in data structures used by channel code.
   */

  idata = GetInterpData (interp);
  hPtr  = Tcl_CreateHashEntry(&idata->ciphers, (char*) name, &new);

  if (! new)
    {
      Tcl_AppendResult (interp,
			"Multiple registration of blockcipher '",
			name, "'", 0);

      return TCL_ERROR;
    }

  Tcl_SetHashValue (hPtr, bcDataPtr);


  /* Do registration of blockcipher as specialized SCS.
   */

  res = BlobX_RegisterSCS (interp, name, CipherCmd, InitBC,
			   (ClientData) bcDataPtr, DeleteBC, options,
			   sizeof (BlobX_BcOptionData) + optDataSize,
			   CleanOpt);

  if (res != TCL_OK)
    {
      ckfree (bcDataPtr);
      return res;
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	CipherCmd --
 *
 *	------------------------------------------------*
 *	This procedure realizes the generic part of BC
 *	operation (argument parsing and mode operation).
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
CipherCmd (clientData, extOptInfo, interp, argc, argv)
ClientData           clientData; /* arbitrary information required
				    by actual function (defined at
				    registration time) */
BlobX_ScsOptionData* extOptInfo; /* information defined via options */
Tcl_Interp*          interp;     /* interpreter we are working in */
int                  argc;       /* number of trailing arguments   (UNUSED) */
char**               argv;       /* arguments with additional info (UNUSED) */
{
  /*
   * The generic SCS code handled the option processing before.
   * Now we are able to do additional checks. Main task however
   * is the execution of the requested cipher.
   */

  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) extOptInfo;
  BcData*    bcDef         = (BcData*) clientData;
  int        res;


  if (optInfo->scs.direction == SCHEDULE)
    {
      /* required is '-key'.
       */

      if (optInfo->bc.keytype != KEY_BLOCK)
	{
	  Tcl_AppendResult (interp, "can't schedule a schedule", 0);
	  return TCL_ERROR;
	}
      else if (! optInfo->scs.key.data)
	{
	  Tcl_AppendResult (interp, "no key specified for schedule", 0);
	  return TCL_ERROR;
	}

      /* remove possibly defined '-data' */
      if (optInfo->scs.data.data)
	{
	  assert (optInfo->scs.data.isAlloc);
	  ckfree (optInfo->scs.data.data);
	}

      /* do schedule and store result in the
       * place expected by the SCS code
       */

      optInfo->scs.data.data = (*bcDef->schedule) (optInfo,
						   &optInfo->scs.data.length);
      optInfo->scs.data.isAlloc = TRUE;

      return TCL_OK;
    }

  /* encrypt / decrypt from now on:
   */

  res = CheckOptionData (interp, optInfo, bcDef, CHECK_DATA);

  if (res != TCL_OK)
    return res;

  /* checks (BC) done now.
   * Of course the block cipher is allowed to do additional checks.
   * These have to be done in 'schedule'.
   *
   * -- start cipher execution with compilation of key into schedule.
   */


  if (optInfo->bc.keytype == KEY_BLOCK)
    {
      /* compute schedule
       */

      int   length;
      char* data;
      int   direction;

      direction = optInfo->scs.direction;

      /* feedback modes operate cipher in encryption mode on both sides.
       * the real choice is saved and restored later.
       */
      if ((optInfo->bc.mode == CFB) || (optInfo->bc.mode == OFB))
	optInfo->scs.direction = ENCRYPT;

      data = (*bcDef->schedule) (optInfo, &length);

      optInfo->scs.direction = direction;


      if (! data)
	return TCL_ERROR;

      if (optInfo->scs.key.isAlloc)
	ckfree (optInfo->scs.key.data);

      optInfo->scs.key.data    = data;
      optInfo->scs.key.length  = length;
      optInfo->scs.key.isAlloc = TRUE;
    }

  /* -- dispatch according to mode */

  switch (optInfo->bc.mode)
    {
    case ECB:
      ExecECB (optInfo);
      break;

    case CBC:
      ExecCBC (optInfo);
      break;

    case CFB:
      ExecCFB (optInfo);
      break;

    case OFB:
      ExecOFB (optInfo);
      break;

    default:
      /* should not happen */
      assert (0);
    }


  /* -- closing work */

  if (optInfo->bc.keytype == KEY_BLOCK)
    {
      /* a temporary key schedule was made, zap it now
       */

      if (bcDef->clean)
	(*bcDef->clean) (optInfo->scs.key.data);
      else
	memset (optInfo->scs.key.data, '\0', optInfo->scs.key.length);

      /* release is done by SCS code */
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	InitBC --
 *
 *	------------------------------------------------*
 *	Initialize block cipher specific option structure
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
InitBC (opt, clientData)
BlobX_ScsOptionData* opt;
ClientData           clientData;
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) opt;  
  optInfo->bc.def          = clientData;
}

/*
 *------------------------------------------------------*
 *
 *	DeleteBC --
 *
 *	------------------------------------------------*
 *	does nothing // Frees all memory associated to a command
 *	anymore      // realizing a BC
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		None
 *
 *------------------------------------------------------*
 */

static void
DeleteBC (clientData)
ClientData  clientData;
{
  /*  ckfree ((BcData*) clientData); */

  /* The 'BcData' structure is not only used by a tcl-command
   * anymore, but by the channel code too.  Deleting it after
   * removal of the command could leave leave dangling pointers
   * there.  Ownership is therefore transfered too
   * --> 'BcInterpData.cipher'.  The deletion takes place at
   * interpreter destruction from now on.
   */
}

/*
 *------------------------------------------------------*
 *
 *	CleanOpt --
 *
 *	------------------------------------------------*
 *	Free memory associated to BC option information
 *	and everything defined by additional cipher options.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
CleanOpt (clientData)
ClientData  clientData;
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;
  BcData*    bcDef         = (BcData*)    optInfo->bc.def;

  /* scs.key, scs.data were handled before by SCS code.
   * Now handle initialization vector.
   */

  if (optInfo->bc.iv.isAlloc)
    ckfree (optInfo->bc.iv.data);

  if (bcDef->cleanOpt)
    (*bcDef->cleanOpt) (optInfo);
}

/*
 * Option handler
 */

/*
 *------------------------------------------------------*
 *
 *	HandleECB --
 *
 *	------------------------------------------------*
 *	process option '-ecb'. no additional arguments
 *	are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleECB (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;

  optInfo->bc.mode = ECB;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleCBC --
 *
 *	------------------------------------------------*
 *	process option '-cbc'. no additional arguments
 *	are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleCBC (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;

  optInfo->bc.mode = CBC;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleCFB --
 *
 *	------------------------------------------------*
 *	process option '-cfb'. no additional arguments
 *	are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleCFB (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;

  optInfo->bc.mode = CFB;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleOFB --
 *
 *	------------------------------------------------*
 *	process option '-ofb'. no additional arguments
 *	are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleOFB (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;

  optInfo->bc.mode = OFB;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleSchedule --
 *
 *	------------------------------------------------*
 *	process option '-schedule-key'. no additional
 *	arguments are required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleSchedule (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;

  optInfo->scs.direction = SCHEDULE;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleIV --
 *
 *	------------------------------------------------*
 *	process option '-iv'. the following 2-5 arguments
 *	have to follow the syntax accepted by 'Blob_GetInput'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleIV (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;
  BcData*    bcDef         = (BcData*)    optInfo->bc.def;
  int        res;

  int   length;
  int   isAlloc;
  char* data;


  if (argc < 2)
    {
      Tcl_AppendResult (interp, "-iv without enough arguments", 0);
      return TCL_ERROR;
    }


  res = Blob_GetInput (interp, argc, argv, 0, "-iv",
		       &length, &data, &isAlloc, processed);
  if (res != TCL_OK)
    {
      Tcl_AppendResult (interp, " (-iv)", 0);
      return res;
    }


  /* preliminary checks */

  if (length < bcDef->blockSize)
    {
      TooShort (interp, "iv", bcDef->blockSize);
      return TCL_ERROR;
    }


  /* store data for use later on */

  if (optInfo->bc.iv.data)
    {
      /* there was an earlier definition, reuse occupied space */
      assert (optInfo->bc.iv.isAlloc);

      memcpy ((VOID*) optInfo->bc.iv.data,
	      (VOID*) data,
	      bcDef->blockSize);

      if (isAlloc)
	ckfree (data);
    }
  else if (! isAlloc)
    {
      /* Make a copy of the IV. This is required to
       * avoid overwriting the source (which might be a blob).
       * Superfluous bytes are cut here too.
       */

      char* tmp = ckalloc (bcDef->blockSize);
      
      if (! tmp)
	{
	  Tcl_AppendResult (interp, "(-iv) not enough memory", 0);
	  return TCL_ERROR;
	}

      memcpy ((VOID*) tmp, (VOID*) data, bcDef->blockSize);

      data    = tmp;
      length  = bcDef->blockSize;
      isAlloc = TRUE;
    }
  /* else isAlloc: we can use info directly */

  optInfo->bc.iv.isAlloc = isAlloc;
  optInfo->bc.iv.data    = data;
  optInfo->bc.iv.length  = length;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleShift --
 *
 *	------------------------------------------------*
 *	process option '-shift'. a single integer number
 *	is required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleShift (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;
  BcData*    bcDef         = (BcData*)    optInfo->bc.def;
  int sw, res;

  if (argc < 1)
    {
      Tcl_AppendResult (interp, "-shift without argument", 0);
    }

  /* a temporary ('sw') is used to preserve an already defined
   * value as long as possible
   */

  res = Tcl_GetInt (interp, argv [0], &sw);
  if (res != TCL_OK)
    {
      Tcl_AppendResult (interp, " (-shift)", 0);
      return res;
    }


  /* preliminary checks */

  if (sw > bcDef->blockSize)
    {
      Tcl_AppendResult (interp,
			"shiftvalue greater than size of blocks", 0);
      return TCL_ERROR;
    }
  else if (sw <= 0)
    {
      Tcl_AppendResult (interp,
			"illegal value <= 0 presented to -shift", 0);
      return TCL_ERROR;
    }
  else if (0 != (bcDef->blockSize % sw))
    {
      Tcl_AppendResult (interp,
			"shiftvalue is no divisor of blocksize", 0);
      return TCL_ERROR;
    }

  /* accept new value */

  optInfo->bc.shiftwidth = sw;
  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleKey --
 *
 *	------------------------------------------------*
 *	process option '-key'. the following 2-5 arguments
 *	have to follow the syntax accepted by 'Blob_GetInput'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleKey (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;
  BcData*    bcDef         = (BcData*)    optInfo->bc.def;
  int        res;

  int   length;
  int   isAlloc;
  char* data;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "-key without enough arguments", 0);
      return TCL_ERROR;
    }

  res = Blob_GetInput (interp, argc, argv,
		       0, "-key",
		       &length, &data, &isAlloc, processed);

  if (res != TCL_OK)
    {
      Tcl_AppendResult (interp, " (-key)", 0);
      return res;
    }

  /* preliminary checks */

  if (length < bcDef->keySize)
    {
      TooShort (interp, "key", bcDef->keySize);
      return TCL_ERROR;
    }


  if (optInfo->scs.key.data && optInfo->scs.key.isAlloc)
    ckfree (optInfo->scs.key.data);

  optInfo->scs.key.length  = length;
  optInfo->scs.key.data    = data;
  optInfo->scs.key.isAlloc = isAlloc;

  optInfo->bc.keytype      = KEY_BLOCK;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	HandleKeyschedule --
 *
 *	------------------------------------------------*
 *	process option '-key-schedule'. the following 2-5
 *	arguments have to follow the syntax accepted by
 *	'Blob_GetInput'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
HandleKeySchedule (clientData, interp, name, argc, argv, processed)
ClientData  clientData;  /* arbitrary context */
Tcl_Interp* interp;      /* interpreter for messages */
CONST char* name;        /* name of option */
int         argc;        /* #arguments behind option */
char**      argv;        /* trailing arguments, possibly
			    arguments to option itself */
int*        processed;   /* number of processed option-
			    arguments, preset by caller to 0 */
{
  BlobX_BcOptInfo* optInfo = (BlobX_BcOptInfo*) clientData;
  int        res;

  int   length;
  int   isAlloc;
  char* data;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "-key-schedule without enough arguments", 0);
      return TCL_ERROR;
    }

  res = Blob_GetInput (interp, argc, argv,
		       0, "-key",
		       &length, &data, &isAlloc, processed);

  if (res != TCL_OK)
    {
      Tcl_AppendResult (interp, " (-key-schedule)", 0);
      return res;
    }

  if (optInfo->scs.key.data && optInfo->scs.key.isAlloc)
    ckfree (optInfo->scs.key.data);

  optInfo->scs.key.length  = length;
  optInfo->scs.key.data    = data;
  optInfo->scs.key.isAlloc = isAlloc;

  optInfo->bc.keytype      = KEY_SCHED;

  return TCL_OK;
}

/*
 * Generic code of cipher modes.
 */

/*
 *------------------------------------------------------*
 *
 *	ExecECB --
 *
 *	------------------------------------------------*
 *	Implementation of generic mode ECB ==  Electronic CodeBook.
 *	All blocks are handled independently.  This is
 *	an insecure mode.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
ExecECB (opt)
BlobX_BcOptInfo* opt;
{
  BcData*    bcDef  = (BcData*) opt->bc.def;
  int        blocks = opt->scs.data.length / bcDef->blockSize;
  char*      data   = opt->scs.data.data;
  int        i;

  for (i=0; i < blocks; i++, data += bcDef->blockSize)
    {
      (*bcDef->cipher) (opt->scs.key.data, data, data);
    }
}

/*
 *------------------------------------------------------*
 *
 *	ExecCBC --
 *
 *	------------------------------------------------*
 *	Implementation of generic mode CBC == Cipher Block Chaining.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
ExecCBC (opt)
BlobX_BcOptInfo* opt;
{
  BcData*    bcDef  = (BcData*) opt->bc.def;
  int        blocks = opt->scs.data.length / bcDef->blockSize;
  char*      data   = opt->scs.data.data;
  int        i;

  if (opt->scs.direction == ENCRYPT)
    {
      for (i=0; i < blocks; i++, data += bcDef->blockSize)
	{
	  XorBuf (data, opt->bc.iv.data, bcDef->blockSize);

	  (*bcDef->cipher) (opt->scs.key.data, data, data);
	  memcpy ((VOID*) opt->bc.iv.data, (VOID*) data, bcDef->blockSize);
	}
    }
  else
    {
      /* create temporary register */
      char* tmp = ckalloc (bcDef->blockSize);
      assert (tmp);

      for (i=0; i < blocks; i++, data += bcDef->blockSize)
	{
	  /* save encrypted block, as it is the next IV */
	  memcpy ((VOID*) tmp, (VOID*) data, bcDef->blockSize);

	  (*bcDef->cipher) (opt->scs.key.data, data, data);

	  XorBuf (data, opt->bc.iv.data, bcDef->blockSize);
	  memcpy ((VOID*) opt->bc.iv.data, (VOID*) tmp, bcDef->blockSize);
	}

      ckfree (tmp);
    }
}

/*
 *------------------------------------------------------*
 *
 *	ExecCFB --
 *
 *	------------------------------------------------*
 *	Implementation of generic mode CFB == Cipher FeedBack.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
ExecCFB (opt)
BlobX_BcOptInfo* opt;
{
  BcData*    bcDef  = (BcData*) opt->bc.def;
  int        blocks = opt->scs.data.length / opt->bc.shiftwidth;
  char*      data   = opt->scs.data.data;
  int        i;

  /* create temporary register for feedback */
  char* tmp = ckalloc (bcDef->blockSize);
  assert (tmp);

  if (opt->scs.direction == ENCRYPT)
    {
      for (i=0; i < blocks; i++, data += opt->bc.shiftwidth)
	{
	  (*bcDef->cipher) (opt->scs.key.data, opt->bc.iv.data, tmp);

	  XorBuf   (data,            tmp,  opt->bc.shiftwidth);
	  ShiftBuf (opt->bc.iv.data, data, opt->bc.shiftwidth, bcDef->blockSize);
	}
    }
  else
    {
      for (i=0; i < blocks; i++, data += opt->bc.shiftwidth)
	{
	  (*bcDef->cipher) (opt->scs.key.data, opt->bc.iv.data, tmp);

	  ShiftBuf (opt->bc.iv.data, data, opt->bc.shiftwidth, bcDef->blockSize);
	  XorBuf   (data,            tmp,  opt->bc.shiftwidth);
	}
    }

  ckfree (tmp);
}

/*
 *------------------------------------------------------*
 *
 *	ExecOFB --
 *
 *	------------------------------------------------*
 *	Implementation of generic mode OFB == Output FeedBack.
 *	It is said to be insecure for a shiftwidth
 *	smaller than the blocksize.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
ExecOFB (opt)
BlobX_BcOptInfo* opt;
{
  BcData*    bcDef  = (BcData*) opt->bc.def;
  int        blocks = opt->scs.data.length / bcDef->blockSize;
  char*      data   = opt->scs.data.data;
  int        i;

  /* create temporary register for feedback */
  char* tmp = ckalloc (bcDef->blockSize);
  assert (tmp);

  for (i=0; i < blocks; i++, data += opt->bc.shiftwidth)
    {
      (*bcDef->cipher) (opt->scs.key.data, opt->bc.iv.data, tmp);

      XorBuf   (data,            tmp, opt->bc.shiftwidth);
      ShiftBuf (opt->bc.iv.data, tmp, opt->bc.shiftwidth, bcDef->blockSize);
    }

  ckfree (tmp);
}

/*
 *------------------------------------------------------*
 *
 *	ShiftBuf --
 *
 *	------------------------------------------------*
 *	Take the 'shiftWidth' leftmost bytes of 'in' and
 *	shift them into the rightmost bytes of 'buf'.
 *	The leftmost bytes of 'buf' are lost.  Both
 *	buffers are assumed to be of size 'blockSize'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
ShiftBuf (buf, in, shiftWidth, blockSize)
char* buf;        /* register to shift */
char* in;         /* data to shift into the register */
int   shiftWidth; /* number of bytes to shift in */
int   blockSize;  /* size of both registers */
{
  assert (shiftWidth > 0);

  if (shiftWidth == blockSize)
    {
      /* special case of dropping the whole old register contents
       * added by me (aku).
       */
      memcpy ((VOID*) buf, (VOID*) in, blockSize);
    }
  else
    {
      int retained;

      /* number bytes in 'buf' to retain */
      retained = blockSize - shiftWidth;

      /* left-shift retained bytes of 'buf' over by
       * 'shiftWidth' bytes to make room
       */
      while (retained --)
	{
	  *buf = *(buf + shiftWidth);
	  buf ++;
	}

      /* now copy 'shiftwidth' bytes from 'input' to shifted tail of 'buf' */
      do
	{
	  *buf++ = *in++;
	}
      while (--shiftWidth);
    }
}

/*
 *------------------------------------------------------*
 *
 *	XorBuf --
 *
 *	------------------------------------------------*
 *	Do an exclusive or of all 'blockSize' bytes of
 *	'buf' with the corresponding bytes in 'mask'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		None
 *
 *------------------------------------------------------*
 */

static void
XorBuf (buf, mask, blockSize)
char* buf;       /* register to manipulate */
char* mask;      /* data to mix with the register */
int   blockSize; /* size of both registers */
{
  assert (blockSize > 0);

  do
    {
      *buf++ ^= *mask++;
    }
  while (--blockSize);
}

/*
 *------------------------------------------------------*
 *
 *	TooShort --
 *
 *	------------------------------------------------*
 *	Generate errormessage for data being shorter than
 *	required.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The interpreter result is changed.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
TooShort (interp, name, minSize)
Tcl_Interp* interp;
CONST char* name;
int         minSize;
{
  char buf [50];
  sprintf (buf, "%d", minSize);

  Tcl_AppendResult (interp,
		    "length of specified ", name, " shorter than ",
		    buf, " bytes", 0);
}

/*
 *------------------------------------------------------*
 *
 * Code for integration of blockciphers into the channel
 * system of Tcl.
 */

/*
 * The structure describing crypt channels to the
 * generic channel management built into Tcl.
 */

static Tcl_ChannelType cryptChannelType = {
  "crypt",

  0, /* blocking mode dependent on used channel */

  CryptClose,
  CryptInput,
  CryptOutput,

  0, /* crypt channels are not seekable */
  0, /* and cannot be configured, therefore */
  0, /* no configuration is accessible */
};

/*
 *------------------------------------------------------*
 *
 *	ChannelOptions --
 *
 *	------------------------------------------------*
 *	Retrieves the table of options known by crypt channels.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated.
 *
 *	Result:
 *		The created Opt_Table
 *
 *------------------------------------------------------*
 */

static Opt_Table
ChannelOptions (interp)
Tcl_Interp* interp;	/* interpreter we are working in/with */
{
  static Opt_Table channelOptions = 0;

  if (! channelOptions)
    {
      /* Cryptchannels are configured with a reduced set
       * of options taken from the set of blockcipher options.
       *
       * Excluded are: -data, -keyschedule, -schedule-key
       */

      static Opt_Spec bc [] =
	{
	  {"key",     HandleKey,     "define key to use"},
	  {"ecb",     HandleECB,     "request electronic code book"},
	  {"cbc",     HandleCBC,     "request cipher block chaining"},
	  {"cfb",     HandleCFB,     "request cipher feedback"},
	  {"ofb",     HandleOFB,     "request output feedback"},
	  {"iv",      HandleIV,      "define initialization vector"},
	  {"shift",   HandleShift,   "define shiftwidth of feedback modes"},
	};

      channelOptions = Opt_CompileArray (interp,
					 sizeof (bc) / sizeof (Opt_Spec),
					 bc);
      assert (channelOptions);

      /* do NOT include standard options of SCS */
      /* Opt_LinkTables (channelOptions, BlobX_SCSOptions (interp)); */
    }

  return channelOptions;
}

/*
 *------------------------------------------------------*
 *
 *	GetInterpData --
 *
 *	------------------------------------------------*
 *	Retrieves the the BcInterpData structure
 *	associated to the specified interpreter.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The structure is allocated and registered,
 *		if its not already a part of the interpreter.
 *
 *	Result:
 *		The pointer to the structure.
 *
 *------------------------------------------------------*
 */

static BcInterpData*
GetInterpData (interp)
Tcl_Interp* interp;	/* interpreter we are working in/with */
{
  BcInterpData*         idata;
  Tcl_InterpDeleteProc* f = DeleteInterpData;

  idata = (BcInterpData*) Tcl_GetAssocData (interp, "blobX.bc", &f);

  if (idata == NULL)
    {
      idata = (BcInterpData*) ckalloc (sizeof (BcInterpData));

      assert (idata);

      Tcl_InitHashTable (&idata->ciphers, TCL_STRING_KEYS);

      Tcl_SetAssocData (interp, "blobX.bc", DeleteInterpData, (ClientData) idata);
    }

  return idata;
}

/*
 *------------------------------------------------------*
 *
 *	DeleteInterpData --
 *
 *	------------------------------------------------*
 *	Frees all memory associated to a BcInterpData structure.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is given back to the system.
 *
 *	Result:
 *		Nothing.
 *
 *------------------------------------------------------*
 */

static void
DeleteInterpData (clientData, interp)
ClientData  clientData;	/* registry of blockciphers */
Tcl_Interp* interp;	/* interpreter we are working in/with */
{
  BcInterpData*  idata = (BcInterpData*) clientData;
  Tcl_HashSearch hSearch;
  Tcl_HashEntry* hPtr;
  BcData*        bcDataPtr;

  /* The referenced structures (of type 'BcData') are NOT
   * owned by the tcl-commands realizing the described
   * blockciphers anymore.  Their deletion in --> 'DeleteBC'
   * is therefore deactivated and is done here from now on!
   */

  for (hPtr  = Tcl_FirstHashEntry (&idata->ciphers, &hSearch);
       hPtr != NULL;
       hPtr  = Tcl_NextHashEntry (&hSearch))
    {
      bcDataPtr = (BcData*) Tcl_GetHashValue (hPtr);
      ckfree (bcDataPtr);
    }

  Tcl_DeleteHashTable (&idata->ciphers);
  ckfree (idata);
}

/*
 *------------------------------------------------------*
 *
 *	BlobXCryptCmd --
 *
 *	------------------------------------------------*
 *	Realizes the 'crypt' command.
 *	See manual for calling syntax.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Generates a crypt channel, associates it
 *		to another channel, configures it.
 *
 *	Result:
 *		A standard Tcl error code.
 *
 *------------------------------------------------------*
 */

int
BlobXCryptCmd (clientData, interp, argc, argv)
ClientData  clientData;	/* not used */
Tcl_Interp* interp;	/* interpreter we are working with/in */
int         argc;	/* #arguments */
char**      argv;	/* texts of the arguments */
{
  /* called as:
   *
   * crypt channel ?options?
   */

  static char cryptName [50];

  Tcl_Channel source;
  int         sourceMode;
  char*       sourceName;
  char*       systemName;
  Tcl_File    in;
  Tcl_File    out;
  BcChannel*  cryptChannelInfo;
  Tcl_Channel cryptChannel;


  if (argc < 2)
    {
      Tcl_AppendResult (interp,
			"wrong # args for crypt, should be called ",
			"as 'crypt channel -system ?options?",
			0);
      return TCL_ERROR;
    }

  /*
   * Things to do.
   * - locate channel, determine read/write modes
   * - create instance data of new channel (part of option structures).
   * - process options.
   * - generate keyschedules.
   * - create the channel and let it go.
   */

  sourceName = argv [1];
  systemName = argv [2];

  argc -= 3;
  argv += 3;

  source = Tcl_GetChannel (interp, sourceName, &sourceMode);

  if (source == NULL)
    return TCL_ERROR;

  /* Fake out generic system.  Crypt channels do not have use for the
   * Tcl_File's as they operate on another channel.  Nevertheless the
   * generic layer requires them to determine wether reading and/or
   * writing is allowed on this channel.  I just take the files of the
   * underlying channel.
   */

  in  = Tcl_GetChannelFile (source, TCL_READABLE);
  out = Tcl_GetChannelFile (source, TCL_WRITABLE);

  if ((in == NULL) && (out == NULL))
    {
      Tcl_AppendResult (interp, "crypt cannot operate with channel ",
			sourceName, " (is unreadable & unwritable)", 0);
      return TCL_ERROR;
    }


  cryptChannelInfo = ProcessChannelConfiguration (interp, systemName, argc, argv);

  if (cryptChannelInfo == NULL)
    return TCL_ERROR;


  /* generate keyschedules to use,
   * eliminate basic key information afterward
   */

  cryptChannelInfo->bcDef = (BcData*) cryptChannelInfo->optInfo->bc.def;

  if (in != NULL)
    {
      int   length;
      char* data;

      /* feedback modes operate cipher in encryption mode on both sides.
       */

      if ((cryptChannelInfo->optInfo->bc.mode == CFB) ||
	  (cryptChannelInfo->optInfo->bc.mode == OFB))
	{
	  cryptChannelInfo->optInfo->scs.direction = ENCRYPT;
	}
      else
	{
	  cryptChannelInfo->optInfo->scs.direction = DECRYPT;
	}

      data = (*cryptChannelInfo->bcDef->schedule) (cryptChannelInfo->optInfo, &length);

      if (! data)
	{
	  DeleteCryptInfo (cryptChannelInfo);
	  return TCL_ERROR;
	}

      cryptChannelInfo->scheduleD.data    = data;
      cryptChannelInfo->scheduleD.length  = length;
      cryptChannelInfo->scheduleD.isAlloc = TRUE;
    }

  if (out != NULL)
    {
      int   length;
      char* data;

      cryptChannelInfo->optInfo->scs.direction = ENCRYPT;

      data = (*cryptChannelInfo->bcDef->schedule) (cryptChannelInfo->optInfo, &length);

      if (! data)
	{
	  DeleteCryptInfo (cryptChannelInfo);
	  return TCL_ERROR;
	}

      cryptChannelInfo->scheduleE.data    = data;
      cryptChannelInfo->scheduleE.length  = length;
      cryptChannelInfo->scheduleE.isAlloc = TRUE;
    }


  if (cryptChannelInfo->optInfo->scs.key.isAlloc)
    {
      ckfree (cryptChannelInfo->optInfo->scs.key.data);
    }

  cryptChannelInfo->optInfo->scs.key.isAlloc = 0;
  cryptChannelInfo->optInfo->scs.key.data    = NULL;


  sprintf (cryptName, "crypt%ld", serial);

  cryptChannel = Tcl_CreateChannel (&cryptChannelType, cryptName,
				    in, out, (ClientData) cryptChannelInfo);

  if (cryptChannel == NULL)
    {
      DeleteCryptInfo (cryptChannelInfo);
      return TCL_ERROR;
    }

  cryptChannelInfo->source = source;

  Tcl_CreateCloseHandler (source, CryptSrcClosed,
			  (ClientData) cryptChannelInfo);

  /* prepare for creation of another channel.
   */

  serial ++;

  /* Transfer ownership to interpreter
   * and return the name of the new channel
   */

  Tcl_RegisterChannel (interp, cryptChannel);
  Tcl_AppendResult    (interp, cryptName, 0);

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	ProcessChannelCaonfiguration --
 *
 *	------------------------------------------------*
 *	Determines the used cryptosystem, processes its
 *	configuration and generates the structures of a
 *	new channel instance.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated.
 *
 *	Result:
 *		The created channel structure or NULL in
 *		case of an error.
 *
 *------------------------------------------------------*
 */

static BcChannel*
ProcessChannelConfiguration (interp, systemName, argc, argv)
Tcl_Interp* interp;	/* interpreter we are working with */
const char* systemName;	/* name of chosen cryptosystem */
int         argc;	/* #configuration options */
char**      argv;	/* texts of configuration options */
{
  BcInterpData*    idata;
  Tcl_HashEntry*   hPtr;
  BcData*          bcDef;
  BcChannel*       chan;
  BlobX_BcOptInfo* opt;
  int              res;
  Opt_Table        standard;

  /*
   * -1  retrieve description of used cryptosystem.
   * -2  generate data structures for channel instance
   * -3  process options
   */

  /* -1- */

  if ('-' != systemName [0])
    {
      Tcl_AppendResult (interp,
			"crypt: name of cryptosystem missing",
			0);
      return NULL;
    }

  idata = GetInterpData (interp);
  hPtr  = Tcl_FindHashEntry (&idata->ciphers, (char*) systemName + 1);

  if (hPtr == NULL)
    {
      Tcl_AppendResult (interp,
			"crypt: unknown cryptosystem '",
			systemName + 1, "'", 0);
      return NULL;
    }

  bcDef = (BcData*) Tcl_GetHashValue (hPtr);


  /* -2- */

  chan = (BcChannel*) ckalloc (sizeof (BcChannel));
  assert (chan);
  memset ((VOID*) chan, '\0',  sizeof (BcChannel));


  opt =  (BlobX_BcOptInfo*) ckalloc (sizeof (BlobX_BcOptInfo) + bcDef->optDataSize);
  assert (opt);
  memset ((VOID*) opt, '\0',         sizeof (BlobX_BcOptInfo) + bcDef->optDataSize);


  opt->bc.def   = bcDef;
  chan->optInfo = opt;

  /* -3- */

  standard = BlobX_BCOptions (interp);

  if (bcDef->optExt == standard)
    {
      res = Opt_ParseArgv (interp,
			   ChannelOptions (interp),
			   &argc, argv,
			   (ClientData) opt);
    }
  else
    {
      /* link non-standard options of used blockcipher
       * temporarily with channel options.
       */

      Opt_LinkTables (bcDef->optExt, ChannelOptions (interp));

      res = Opt_ParseArgv (interp,
			   bcDef->optExt,
			   &argc, argv,
			   (ClientData) opt);

      /* restore old link */
      Opt_LinkTables (bcDef->optExt, BlobX_BCOptions (interp));
    }

  if (res != TCL_OK)
    {
      DeleteCryptInfo (chan);
      return NULL;
    }

  res = CheckOptionData (interp, opt, bcDef, DONT_CHECK_DATA);

  if (res != TCL_OK)
    {
      DeleteCryptInfo (chan);
      return NULL;
    }

  /* Make IV independent from static and/or temporary memory
   * This protect against dangling pointers and/or
   * inadvertently altered information.
   */

  if (! opt->bc.iv.isAlloc)
    {
      char* tmp = opt->bc.iv.data;
      
      opt->bc.iv.data = ckalloc (opt->bc.iv.length);
      assert (opt->bc.iv.data);

      memcpy ((VOID*) opt->bc.iv.data, (VOID*) tmp, opt->bc.iv.length);

      opt->bc.iv.isAlloc = TRUE;
    }


  /* The keyschedules will be generated by the caller.
   * We don't know about the allowed directions here.
   */

  return chan;
}

/*
 *------------------------------------------------------*
 *
 *	DeleteCryptInfo --
 *
 *	------------------------------------------------*
 *	Releases the memory associated to the instance
 *	structures of a cryptchannel. The buffer information
 *	is not touched! and has to be released by the caller.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		Nothing.
 *
 *------------------------------------------------------*
 */

static void
DeleteCryptInfo (crypt)
BcChannel* crypt;	/* channel structure to delete */
{
  /* Take advantage of existing methods to clean the option
   * structure.  We have to clean the SCS ourselves, as this
   * is NOT handled by the called function.
   */

  if (crypt->optInfo->scs.key.isAlloc)
    ckfree (crypt->optInfo->scs.key.data);

  CleanOpt ((ClientData) crypt->optInfo);

  /* we don't need to watch the source anymore */
  Tcl_DeleteCloseHandler (crypt->source, CryptSrcClosed,
			  (ClientData) crypt);

  /* now remove the main instance information */
  ckfree (crypt);
}

/*
 *------------------------------------------------------*
 *
 *	CheckOptionData --
 *
 *	------------------------------------------------*
 *	Do a consistency check on the information retrieved
 *	by the option processor.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The interpreter result area will contain
 *		an error message upon failure.
 *
 *	Result:
 *		A standard Tcl error code
 *
 *------------------------------------------------------*
 */

static int
CheckOptionData (interp, optInfo, bcDef, dataRequired)
Tcl_Interp*      interp;	/* interpreter we are working in */
BlobX_BcOptInfo* optInfo;	/* structure to check */
BcData*          bcDef;		/* description of used cryptosytem */
int              dataRequired;	/* flag: check existence of data ? */
{
  /*
   * general requirements:
   *
   *	-data
   *	-ecb, -cbc, -cfb or -ofb
   *	-key or -key-schedule 
   *
   * stream modes:   -iv
   * feedback modes: -iv and -shift
   *
   * all modes place additional constraints on the size of 'data'.
   */

  int modulus;

  if (optInfo->bc.mode == NO_MODE)
    {
      Tcl_AppendResult (interp, "mode not defined", 0);
      return TCL_ERROR;
    }
  else if (dataRequired && (optInfo->scs.data.data == 0))
    {
      Tcl_AppendResult (interp, "data missing", 0);
      return TCL_ERROR;
    }


  modulus = bcDef->blockSize;

  if (optInfo->bc.mode != ECB)
    {
      /* stream modes (CBC, CFB, OFB) */

      if (! optInfo->bc.iv.data)
	{
	  Tcl_AppendResult (interp, "init-vector not defined", 0);
	  return TCL_ERROR;
	}

      if (optInfo->bc.mode != CBC)
	{
	  /* feedback modes (CFB, OFB) */

	  if (optInfo->bc.shiftwidth == 0)
	    {
	      Tcl_AppendResult (interp, "shift not defined", 0);
	      return TCL_ERROR;
	    }

	  modulus = optInfo->bc.shiftwidth;
	}
    }

  /* modulus:
   * ECB, CBC: blocksize
   * CFB, OFB: shiftwidth
   */

  if (0 != (optInfo->scs.data.length % modulus))
    {
      char buf [50];
      sprintf (buf, "%d", modulus);

      Tcl_AppendResult (interp,
			"size of specified data is not a multiple of ",
			buf, " bytes", 0);
      return TCL_ERROR;
    }

#if 0 /* catched in SCS code already */
  if (! optInfo->scs.key.data)
    {
      Tcl_AppendResult (interp, "neither key nor schedule specified", 0);
      return TCL_ERROR;
    }
#endif

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	CryptClose --
 *
 *	------------------------------------------------*
 *	The channel was closed by the generic layer.
 *	Release all associated memory.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		A POSIX error code!
 *
 *------------------------------------------------------*
 */

static int
CryptClose (clientData, interp, inFile, outFile)
ClientData  clientData;	/* channel data */
Tcl_Interp* interp;	/* interpreter we are working in,
			 * possibly NULL! */
Tcl_File    inFile;	/* dummy, not used here */
Tcl_File    outFile;	/* dummy, not used here */
{
  BcChannel* chan = (BcChannel*) clientData;
  int        res  = 0;

  /* Release buffer information.
   */

  if (chan->out.maxLength > 0)
    {
      ckfree (chan->out.buf);

      if (chan->out.pending > 0)
	{
	  res = EPERM;
	  if (interp)
	    {
	      Tcl_AppendResult (interp,
				"incomplete block waiting for write", 0);
	    }
	}
    }

  if (chan->in.maxLength > 0)
    {
      ckfree (chan->in.buf);

      if (chan->in.pending > 0)
	{
	  if (interp)
	    {
	      if (res == EPERM)
		{
		  Tcl_AppendResult (interp, "\n", 0);
		}

	      Tcl_AppendResult (interp,
				"incomplete block waiting for read", 0);
	    }

	  res = EPERM;
	}
    }

  /* ** future ** */
  /* mark serial number (name) for reusage during create!  */

  /* Release everything else.
   */

  DeleteCryptInfo (chan);
  return 0;
}

/*
 *------------------------------------------------------*
 *
 *	CryptInput --
 *
 *	------------------------------------------------*
 *	Used by the generic layer to read data from the
 *	channel.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Forwards the message to the underlying
 *		channel and decrypts all data read from
 *		there into the specified buffer.
 *
 *	Result:
 *		The number of bytes read or -1 in case
 *		of a failure.
 *
 *------------------------------------------------------*
 */

static int
CryptInput (clientData, inFile, buf, bufSize, errorCodePtr)
ClientData  clientData;	/* channel structure */
Tcl_File    inFile;	/* dummy, not used here */
char*       buf;	/* buffer to write the read data into */
int         bufSize;	/* max. number of bytes the buffer can hold */
int*        errorCodePtr;	/* place to store the errorcode
				 * in case of a failure */
{
  BcChannel*       chan  = (BcChannel*) clientData;
  BcChanControl*   ctrl  = &chan->in;
  BcData*          bcDef = chan->bcDef;
  BlobX_BcOptInfo* opt   = chan->optInfo;

  /* A description of the operation of this procedure can be found in
   * the file 'doc/ReadBuffering'.  Marks of the form <X> refer to the
   * decision table in this file.
   */

  int bSize = (opt->bc.mode > CBC ? opt->bc.shiftwidth : bcDef->blockSize);

  int read  = 0; /* Number of bytes read during the invocation of this
		  * procedure. */


  /*fprintf (stderr, "-- CryptInput %p %p %p = %d (%d in, %d waiting)\n",
    chan, ctrl, bcDef, ctrl->state,  bufSize, ctrl->pending);*/

 recurse:
  switch (ctrl->state)
    {
    case BCB_WAIT_FOR_READ_REQUEST:
      {
	/* handle <A>, <B>
	 *
	 * The internal buffer contains information already decrypted
	 * and just waiting for retrieval by the generic layer of the
	 * channel system.
	 */

	if (bufSize <= ctrl->pending)
	  {
	    /* <A>
	     * The buffer provided by the generic layer is not big
	     * enough to hold all pending information.  Transfer the
	     * largest amount possible, then return and wait for the
	     * next request.
	     */
	    
	    /*fprintf (stderr, "<A>\n");*/

	    memcpy ((VOID*) buf, (VOID*) ctrl->buf, bufSize);
	    ctrl->pending -= bufSize;
	    read          += bufSize;
	  }
	else
	  {
	    /* <B>
	     * The buffer provided by the generic layer can hold more
	     * data than waiting here.  Transfer the information into
	     * the buffer, then jump into this decision table again
	     * to fill the rest of the buffer with data retrieved from
	     * the source channel.
	     */

	    /*fprintf (stderr, "<B>\n");*/

	    memcpy ((VOID*) buf, (VOID*) ctrl->buf, ctrl->pending);
	    read    += ctrl->pending;
	    buf     += ctrl->pending;
	    bufSize -= ctrl->pending;
	    ctrl->pending = 0;

	    ctrl->state = BCB_WAIT_FOR_DATA;
	    goto recurse;
	  }
      }
      break;

    case BCB_WAIT_FOR_DATA:
      {
	/* handle <D>, <E>, <H>
	 *
	 * The internal buffer is empty or contains an incomplete block
	 * of encrypted information.  The source channel is used to
	 * complete the buffered data and proceed with decryption.
	 */

	if ((ctrl->pending == 0) && (bufSize < bSize))
	  {
	    /* <D>
	     * The internal buffer is empty and the generic layer requests
	     * a chunk smaller than a single block.  So try to read single
	     * block from the source and decrypt it on success.  Then
	     * transfer it to the external buffer (by jumping back into
	     * the decision table).
	     */

	    /*fprintf (stderr, "<D>\n");*/

	    ExpandBuffer (ctrl, bSize);
	    ctrl->pending = Tcl_Read (chan->source, ctrl->buf, bSize);

	    /*fprintf (stderr, "  got %d of %d\n", ctrl->pending, bSize);*/

	    if (ctrl->pending == bSize)
	      {
		DecryptBuffer (opt, chan, ctrl, bSize);
		ctrl->state = BCB_WAIT_FOR_READ_REQUEST;
		goto recurse;
	      }
	  }
	else if ((ctrl->pending == 0) && (bufSize >= bSize))
	  {
	    /* <E>
	     * The internal buffer is empty and the generic layer requests
	     * a chunk greater than a single block.  Compute the number of
	     * blocks the provided buffer can hold, then read that many + 1
	     * blocks from the source.  Decrypt as many blocks as we really
	     * got and transfer them to the generic layer.  An incomplete
	     * block is left in the buffer for later processsing.  The same
	     * is true for a complete block which cannot be stored in the
	     * provided buffer anymore.
	     */

	    int blocks   = bufSize / bSize;
	    int required = bSize * (blocks+1);

	    /*fprintf (stderr, "<E>, wanting %d / %d\n", blocks, required);*/

	    ExpandBuffer (ctrl, required);

	    ctrl->pending = Tcl_Read (chan->source, ctrl->buf, required);

	    /*fprintf (stderr, "  got %d of %d\n", ctrl->pending, required);*/

	    blocks = ctrl->pending / bSize;

	    if (blocks > 0)
	      {
		int bytes    = bSize * blocks;
		int transfer = Min (bytes, bufSize);

		DecryptBuffer (opt, chan, ctrl, bytes);
		memcpy ((VOID*) buf, (VOID*) ctrl->buf, transfer);

		ctrl->pending -= transfer;
		read          += transfer;
		ctrl->state    = (ctrl->pending < bSize     ?
				  BCB_WAIT_FOR_DATA         :
				  BCB_WAIT_FOR_READ_REQUEST);

		if (ctrl->pending > 0)
		  {
		    /* Shift the unprocessed information
		     * up to the start of the buffer.
		     */

		    memmove ((VOID*) ctrl->buf,
			     (VOID*) (ctrl->buf + transfer),
			     ctrl->pending);
		  }
	      }
	  }
	else if ((0 < ctrl->pending) && (ctrl->pending < bSize))
	  {
	    /* <H>
	     * The internal buffer contains an incomplete block.  Try to
	     * complete it by reading data from the source.  If successful,
	     * decrypt the block, then transfer it into the provided buffer.
	     * The latter is done by recursion into the decision table.
	     *  This will take care of a buffer bigger than a single block too.
	     */

	    /*fprintf (stderr, "<H>\n");*/

	    ExpandBuffer (ctrl, bSize);

	    ctrl->pending += Tcl_Read (chan->source, ctrl->buf,
				       bSize - ctrl->pending);

	    /*fprintf (stderr, "  have %d,  wanted %d\n", ctrl->pending, bSize - ctrl->pending);*/
	    
	    if (ctrl->pending == bSize)
	      {
		DecryptBuffer (opt, chan, ctrl, bSize);
		ctrl->state = BCB_WAIT_FOR_READ_REQUEST;
		goto recurse;
	      }
	  }
	else
	  /* we should not come to this place */
	  assert (0);
      }
      break;

    default:
      /* we should not come to this point */
      assert (0);
      break;
    }

  /*fprintf (stderr, "-- %d read /%d waiting\n", read, ctrl->pending);*/

  return read;
}

/*
 *------------------------------------------------------*
 *
 *	CryptOutput --
 *
 *	------------------------------------------------*
 *	Used by the generic layer to write data into the
 *	channel.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Encrypts the data in the specified buffer
 *		and then forwards the message to the
 *		underlying channel.
 *
 *	Result:
 *		The number of written bytes or -1 on failure.
 *
 *------------------------------------------------------*
 */

static int
CryptOutput (clientData, outFile, buf, toWrite, errorCodePtr)
ClientData  clientData;	/* channel structure */
Tcl_File    outFile;	/* dummy, not used here */
char*       buf;	/* buffer containing the data to write */
int         toWrite;	/* number of bytes in this buffer */
int*        errorCodePtr;	/* place to store the errorcode
				 * in case of a failure */
{
  BcChannel*       chan  = (BcChannel*) clientData;
  BcChanControl*   ctrl  = &chan->out;
  BcData*          bcDef = chan->bcDef;
  BlobX_BcOptInfo* opt   = chan->optInfo;

  int toWriteInternal;
  int written;
  int bSize = (opt->bc.mode > CBC ? opt->bc.shiftwidth : bcDef->blockSize);

  /*fprintf (stderr, "-- CryptOutput %p %p %p = %d, %d\n", chan, ctrl, bcDef, ctrl->pending, toWrite);*/

  /* Transfer given data into internal buffer, encrypt as
   * many blocks as possible, then write these.  If there
   ** is a leftover, shift it up to the start of the buffer,
   * then wait for the next call.
   */

  ExpandBuffer (ctrl, ctrl->pending + toWrite);
  memcpy ((VOID*) (ctrl->buf + ctrl->pending), (VOID*) buf, toWrite);

  ctrl->pending += toWrite;

  toWriteInternal = (ctrl->pending / bSize) * bSize;

  /*fprintf (stderr, "really encrypt, write = %d\n", toWriteInternal);*/

  EncryptBuffer (opt, chan, ctrl, toWriteInternal);

  written = Tcl_Write (chan->source, ctrl->buf, toWriteInternal);

  /*fprintf (stderr, "written =               %d\n", written);*/

  if (written < 0)
    {
      *errorCodePtr = Tcl_GetErrno ();
      return -1;
    }

  ctrl->pending -= written;

  if (ctrl->pending > 0)
    {
      /*fprintf (stderr, "shift %d\n", ctrl->pending);*/

      memmove ((VOID*) ctrl->buf,
	       (VOID*) (ctrl->buf + written),
	       ctrl->pending);
    }

  return toWrite;
}

/*
 *------------------------------------------------------*
 *
 *	ExpandBuffer --
 *
 *	------------------------------------------------*
 *	Expand the buffer to a new maximal size.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory can be allocated.
 *
 *	Result:
 *		Nothing.
 *
 *------------------------------------------------------*
 */

static void
ExpandBuffer (ctrl, newSize)
BcChanControl* ctrl;	/* buffer to expand */
int            newSize;	/* required size */
{
  if (ctrl->maxLength < newSize)
    {
      if (ctrl->buf == 0)
	{
	  ctrl->buf = ckalloc (newSize);
	}
      else
	{
	  ctrl->buf = ckrealloc (ctrl->buf, newSize);
	}

      assert (ctrl->buf);

      ctrl->maxLength = newSize;
    }
}

/*
 *------------------------------------------------------*
 *
 *	DecryptBuffer --
 *
 *	------------------------------------------------*
 *	Decrypt the information contained in the channel
 *	buffer.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		Nothing.
 *
 *------------------------------------------------------*
 */

static void
DecryptBuffer (opt, chan, ctrl, num)
BlobX_BcOptInfo* opt;	/* option structure to setup */
BcChannel*       chan;	/* channel instance containing the keyschedule */
BcChanControl*   ctrl;  /* buffer containing the data to decrypt */
int              num;   /* number of bytes to process. */
{
  /* Prepare data structures for operation
   * (Generate the environment expected by the Exec... procedures).
   */

  opt->scs.direction   = DECRYPT;
  opt->scs.key.data    = chan->scheduleD.data;
  opt->scs.data.data   = ctrl->buf;
  opt->scs.data.length = num;

  /* now use standard functionality.
   */

  switch (opt->bc.mode)
    {
    case ECB:
      ExecECB (opt);
      break;

    case CBC:
      ExecCBC (opt);
      break;

    case CFB:
      ExecCFB (opt);
      break;

    case OFB:
      ExecOFB (opt);
      break;

    default:
      /* should not happen */
      assert (0);
    }

  /* clean opt structure to prevent
   * accidentally usage of invalid information.
   */

  opt->scs.direction   = -1;
  opt->scs.key.data    =  0;
  opt->scs.data.data   =  0;
  opt->scs.data.length =  0;
}

/*
 *------------------------------------------------------*
 *
 *	EncryptBuffer --
 *
 *	------------------------------------------------*
 *	Encrypt the information contained in the channel
 *	buffer.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		Nothing.
 *
 *------------------------------------------------------*
 */

static void
EncryptBuffer (opt, chan, ctrl, num)
BlobX_BcOptInfo* opt;	/* option structure to setup */
BcChannel*       chan;	/* channel instance containing the keyschedule */
BcChanControl*   ctrl;  /* buffer containing the data to encrypt */
int              num;   /* number of bytes to process. */
{
  /* Prepare data structures for operation
   * (Generate the environment expected by the Exec... procedures).
   */

  opt->scs.direction   = ENCRYPT;
  opt->scs.key.data    = chan->scheduleE.data;
  opt->scs.data.data   = ctrl->buf;
  opt->scs.data.length = num;

  /* now use standard functionality.
   */

  switch (opt->bc.mode)
    {
    case ECB:
      ExecECB (opt);
      break;

    case CBC:
      ExecCBC (opt);
      break;

    case CFB:
      ExecCFB (opt);
      break;

    case OFB:
      ExecOFB (opt);
      break;

    default:
      /* should not happen */
      assert (0);
    }

  /* clean opt structure to prevent
   * accidentally usage of invalid information.
   */

  opt->scs.direction   = -1;
  opt->scs.key.data    =  0;
  opt->scs.data.data   =  0;
  opt->scs.data.length =  0;
}

/*
 *------------------------------------------------------*
 *
 *	CryptSrcClosed --
 *
 *	------------------------------------------------*
 *	Called when the source of of a crypt channel was
 *	closed.  panics.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		See above.
 *
 *	Result:
 *		Nothing.
 *
 *------------------------------------------------------*
 */

static void
CryptSrcClosed (clientData)
ClientData clientData;
{
#if 0
  BcChannel*  chan  = (BcChannel*) clientData;  
#endif

  /* We are get called only, if the source channel does
   * no longer exist in no interpreter.
   *
   * Our problem here is, that the cryptchannel is possibly
   * shared too!  I found no direct mechanism (neither public
   * not private) to determine the interpreters the channel is
   * part of. An indirect approach would be iteration over all
   * existing interpreters, but to do that one has to access
   * internal information of the core (the hashtable associated
   * to "tclMasterRecord" for each interpreter and the static
   * variable 'interp' in tclMain.c.
   *
   * Because of this I decided to do nothing, but to panic.
   * Sorry.
   */

  EXTERN void panic();
  
  panic ("Source channel closed before crypt-channel");
}

