
/*
 * blobObjCmd.c --
 *
 *	Implementation of Tcl commands manipulating BLOBs.
 *	The object commands are realized here.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: blobObjCmd.c,v 1.7 1996/05/01 22:27:31 aku Exp $
 */

#include <assert.h>
#include <string.h>
#include <stdio.h>
#ifdef __STDC__
#include <stdlib.h>
#endif
#include <ctype.h>

#include "blobInt.h"


/*
 * Internal procedures realizing the different
 * subcommands of blob objects
 */

static int
AppendCmd    _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
ClearCmd     _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
EscapeCmd    _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
InsertCmd    _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
PrependCmd   _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
PadCmd       _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
RemoveCmd    _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
ReplaceCmd   _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
SetCmd       _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
SizeCmd      _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
StringCmd    _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
TruncateCmd  _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
UnpackCmd    _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));
static int
WriteCmd     _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));

static int
SetSecureCmd _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));

static int
IsSecureCmd  _ANSI_ARGS_ ((Blob        b,
			   CONST char* blobName,
			   Tcl_Interp* interp,
			   int         argc,
			   char**      argv));

static int
GetPosition _ANSI_ARGS_ ((Blob        b,
			  CONST char* blobName,
			  Tcl_Interp* interp,
			  CONST char* string,
			  int*        position,
			  int*        length));


/*
 *------------------------------------------------------*
 *
 *	BlobObjectCmd --
 *
 *	------------------------------------------------*
 *	Realizes the minor commands associated to an
 *	blob object and its command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		see Manual.
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

int
BlobObjectCmd (clientData, interp, argc, argv)
ClientData  clientData;	/* handle of blob to operate with */
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;	/* number of arguments to blob command */
char**      argv;	/* array referencing the arguments */
{
  BlobClientData* bcd = (BlobClientData*) clientData;
  char*      blobName = argv [0];
  char*      minorCommand;
  int        len;
  char       c;

  Tcl_ResetResult (interp);

  assert (bcd);
  assert (bcd->b);

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "wrong # args for \"", blobName, "\"", 0);

      return TCL_ERROR;
    }

  minorCommand = argv [1];

  if (minorCommand [0] == '\0')
    {
    usage:

      Tcl_AppendResult (interp,
			"illegal minor command \"",
			minorCommand,
			"\", must be one of append, clear, escape, insert, pack, pad, prepend, remove, replace, set, secure, is-secure, size, string, truncate, unpack or write",
			0);
      return TCL_ERROR;
    }

  /*
   * Common argument checking now done, start dispatching.
   * Skip over name of blob we are operating on first.
   */

  len = strlen (minorCommand);
  c   = minorCommand [0];

  argc --;
  argv ++;

  switch (c)
    {
    case 'a':
      if (strncmp (minorCommand, "append", len) == 0)
	{
	  return AppendCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 'c':
      if (strncmp (minorCommand, "clear", len) == 0)
	{
	  return ClearCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 'e':
      if (strncmp (minorCommand, "escape", len) == 0)
	{
	  return EscapeCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 'i':
      if (len == 1)
	{
	  goto not_unique;
	}
      else if (strncmp (minorCommand, "insert", len) == 0)
	{
	  return InsertCmd (bcd->b, blobName, interp, argc, argv);
	}
      else if (strncmp (minorCommand, "is-secure", len) == 0)
	{
	  return IsSecureCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 'p':
      if (len == 1)
	{
	  goto not_unique;
	}
      else if (strncmp (minorCommand, "prepend", len) == 0)
	{
	  return PrependCmd (bcd->b, blobName, interp, argc, argv);
	}
      else if (strncmp (minorCommand, "pad", len) == 0)
	{
	  return PadCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 'r':
      if (len == 1)
	{
	  goto not_unique;
	}
      else if ((len == 2) && (0 == strcmp (minorCommand, "re")))
	{
	  goto not_unique;
	}

      if (strncmp (minorCommand, "remove", len) == 0)
	{
	  return RemoveCmd (bcd->b, blobName, interp, argc, argv);
	}

      if (strncmp (minorCommand, "replace", len) == 0)
	{
	  return ReplaceCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 's':
      if (len == 1)
	{
	  goto not_unique;
	}

      if (strncmp (minorCommand, "set", len) == 0)
	{
	  return SetCmd (bcd->b, blobName, interp, argc, argv);
	}

      if (strncmp (minorCommand, "size", len) == 0)
	{
	  return SizeCmd (bcd->b, blobName, interp, argc, argv);
	}

      if (strncmp (minorCommand, "string", len) == 0)
	{
	  return StringCmd (bcd->b, blobName, interp, argc, argv);
	}

      if (strncmp (minorCommand, "secure", len) == 0)
	{
	  return SetSecureCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 't':
      if (strncmp (minorCommand, "truncate", len) == 0)
	{
	  return TruncateCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 'u':
      if (strncmp (minorCommand, "unpack", len) == 0)
	{
	  return UnpackCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;

    case 'w':
      if (strncmp (minorCommand, "write", len) == 0)
	{
	  return WriteCmd (bcd->b, blobName, interp, argc, argv);
	}
      break;
    }

  /* failed to detect a legal minor command */
  goto usage;

 not_unique:
  Tcl_AppendResult (interp,
		    "non-unique abbreviation \"",
		    minorCommand,
		    "\" used as minor command",
		    0);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	SizeCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'size' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_Size
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
SizeCmd (b, blobName, interp, argc, argv)
Blob        b;		/* blob to operate on and/or with */
CONST char* blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;
char**      argv;
{
  int   blobLength;
  char *minorCommand = argv [0];

  assert (b);

  if (argc > 1)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			minorCommand,
			"\", must be \"", blobName, " size\"",
			0);
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_Size (b, &blobLength))
    {
      Tcl_AppendResult(interp, blobName, " size : ",
		       Blob_LastError (b), 0);

      return TCL_ERROR;
    }
  else
    {
      char buf [30];
      sprintf (buf, "%d", blobLength);

      Tcl_AppendResult(interp, buf, 0);

      return TCL_OK;
    }
}

/*
 *------------------------------------------------------*
 *
 *	EscapeCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'escape' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_(Get)Escape
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
EscapeCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  char *minorCommand  = argv [0];
  int   result;

  assert (b);

  if ((argc < 1) || (argc > 2))
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			minorCommand,
			"\", must be \"", blobName, " escape ?<character>?\"",
			0);
      return TCL_ERROR;
    }

  if (argc == 1)
    {
      result = Blob_GetEscape (b);

      if (result != BLOB_ERROR)
	{
	  char out [2];   
	  out [0] = (char) result;
	  out [1] = '\0';
	  
	  Tcl_AppendResult (interp, out, 0);
	  return TCL_OK;
	}
    }
  else
    {
      if ('\0' == argv [1][0])
	{
	  Tcl_AppendResult (interp,
			    "illegal use of '\\0' as escape character",
			    0);
	  return TCL_ERROR;
	}

      result = Blob_SetEscape (b, (int) argv [1][0]);
    }

  if (result == BLOB_ERROR)
    {
      Tcl_AppendResult (interp, Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	ClearCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'clear' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_SetData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
ClearCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  assert (b);

  if (argc > 1)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0],
			"\", must be \"", blobName, " clear\"",
			0);
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_SetData (b, 0, 0))
    {
      Tcl_AppendResult(interp, blobName, " clear : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }
  else
    {
      return TCL_OK;
    }
}

/*
 *------------------------------------------------------*
 *
 *	AppendCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'append' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_InsertData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */


static int
AppendCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int   length;
  char *newData;
  int   dataIsAllocated;
  int   blobLength;
  int   result;

  assert (b);

  if (argc < 3)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0],
			"\", must be \"", blobName, " append INPUT\"",
			0);
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_Size (b, &blobLength))
    {
      Tcl_AppendResult(interp, blobName, " append : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  argc --;
  argv ++;

  if (TCL_OK != Blob_GetInput (interp, argc, argv, b, blobName,
			       &length, &newData, &dataIsAllocated, NULL))
    {
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_InsertData (b, blobLength,
				  length, newData))
    {
      Tcl_AppendResult(interp, blobName, " append : ",
		       Blob_LastError (b), 0);
      result = TCL_ERROR;
    }
  else
    {
      result = TCL_OK;
    }

  if (dataIsAllocated)
    {
      ckfree (newData);
    }

  return result;
}

/*
 *------------------------------------------------------*
 *
 *	InsertCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'insert' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_InsertData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
InsertCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int   position;
  int   blobLength;
  int   length;
  char *newData;
  int   dataIsAllocated;
  int   result;

  assert (b);

  if (argc < 4)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0],
			"\", must be \"", blobName, " insert <at> INPUT\"",
			0);
      return TCL_ERROR;
    }

  if (TCL_OK != GetPosition (b, blobName, interp, argv [1],
			     &position, &blobLength))
     {
       return TCL_ERROR;
     }

  argc -= 2;
  argv += 2;

  if (TCL_OK != Blob_GetInput (interp, argc, argv, b, blobName,
			      &length, &newData, &dataIsAllocated, NULL))
    {
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_InsertData (b, position,
				  length, newData))
    {
      Tcl_AppendResult(interp, blobName, " append : ",
		       Blob_LastError (b), 0);
      result = TCL_ERROR;
    }
  else
    {
      result = TCL_OK;
    }

  if (dataIsAllocated)
    {
      ckfree (newData);
    }

  return result;
}

/*
 *------------------------------------------------------*
 *
 *	PrependCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'prepend' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_InsertData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
PrependCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int   length;
  char *newData;
  int   dataIsAllocated;
  int   result;

  assert (b);

  if (argc < 3)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" prepend INPUT\"",
			0);
      return TCL_ERROR;
    }

  argc --;
  argv ++;

  if (TCL_OK != Blob_GetInput (interp, argc, argv, b, blobName,
			      &length, &newData, &dataIsAllocated, NULL))
    {
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_InsertData (b, 0,
				  length, newData))
    {
      Tcl_AppendResult(interp, blobName, " prepend : ",
		       Blob_LastError (b), 0);
      result = TCL_ERROR;
    }
  else
    {
      result = TCL_OK;
    }

  if (dataIsAllocated)
    {
      ckfree (newData);
    }

  return result;
}

/*
 *------------------------------------------------------*
 *
 *	PadCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'pad' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_InsertData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
PadCmd (b, blobName, interp, argc, argv)
Blob         b;	        /* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int   length;
  char *newData;
  int   dataIsAllocated;
  int   result   = TCL_OK;
  int   isModulo = 0;
  int   reqSize, padSize, blobLength;

  assert (b);

  if (argc < 4)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" pad ?-modulo? <size> INPUT\"",
			0);
      return TCL_ERROR;
    }

  argc --;
  argv ++;

  /* option processing */

  if (argv [0][0] == '-')
    {
      int len = strlen (argv [0] +1);

      if (0 != strncmp ("modulo", argv [0] +1, len))
	{
	  Tcl_AppendResult (interp,
			    "unknown pad-option \"", argv [0],
			    "\"", 0);
	  return TCL_ERROR;
	}

      isModulo ++;
      argv ++;
      argc --;
    }

  if (TCL_OK != Tcl_GetInt (interp, argv [0], &reqSize))
    {
      return TCL_ERROR;
    }

  argv ++;
  argc --;

  if (BLOB_OK != Blob_Size (b, &blobLength))
    {
      Tcl_AppendResult(interp, blobName, " pad : ",
		       Blob_LastError (b), 0);

      return TCL_ERROR;
    }

  /* normalize '-modulo size' to simple size-request */
  if (isModulo)
    {
      int off = blobLength % reqSize;
      if (off)
	{
	  reqSize = blobLength + (reqSize - off);
	}

      isModulo = 0;
    }


  if (reqSize <= blobLength)
    {
      /* request < reality, nothing to do */
      return TCL_OK;
    }

  if (TCL_OK != Blob_GetInput (interp, argc, argv, b, blobName,
			      &length, &newData, &dataIsAllocated, NULL))
    {
      return TCL_ERROR;
    }

  padSize = reqSize - blobLength;
  assert (padSize > 0);

  /* Do padding now.  If the input is shorter than required
   * append full copies of the specified pattern.  For an
   * input longer than required append a partial copy of
   * the specified pattern.
   */

  while (padSize >= length)
    {
      if (BLOB_OK != Blob_InsertData (b, blobLength,
				      length, newData))
	{
	  Tcl_AppendResult(interp, blobName, " prepend : ",
			   Blob_LastError (b), 0);
	  result = TCL_ERROR;
	  goto clean;
	}

      padSize    -= length;
      blobLength += length;
    }

  if (padSize < length)
    {
      if (BLOB_OK != Blob_InsertData (b, blobLength,
				      padSize, newData))
	{
	  Tcl_AppendResult(interp, blobName, " prepend : ",
			   Blob_LastError (b), 0);
	  result = TCL_ERROR;
	}
    }

 clean:
  if (dataIsAllocated)
    {
      ckfree (newData);
    }

  return result;
}

/*
 *------------------------------------------------------*
 *
 *	RemoveCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'remove' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_RemoveData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
RemoveCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int from;
  int to;
  int length;

  assert (b);

  if ((argc < 1) || (argc > 3))
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0],
			"\", must be \"", blobName,
			" remove ?<from> ?<to>??\"",
			0);
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_Size (b, &length))
    {
      Tcl_AppendResult(interp, blobName, " remove : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  argc --;
  argv ++;

  if (TCL_OK != BlobGetIntervalSpec (interp, argc, argv,
				     length, &from, &to))
    {
      return TCL_ERROR;
    }


  if (BLOB_OK != Blob_RemoveData (b, from, to))
    {
      Tcl_AppendResult(interp, blobName, " remove : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }
  else
    {
      return TCL_OK;
    }
}

/*
 *------------------------------------------------------*
 *
 *	ReplaceCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'replace' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_ReplaceData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
ReplaceCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int   from;
  int   to;
  int   blobLength;
  int   edges;
  int   length;
  char *newData;
  int   dataIsAllocated;
  int   result;

  assert (b);

  if (argc < 3)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0],
			"\", must be \"", blobName,
			" replace ?<from> ?<to>?? INPUT\"",
			0);
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_Size (b, &blobLength))
    {
      Tcl_AppendResult(interp, blobName, " replace : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  /* skip over minor command */
  argc --;
  argv ++;

  if (TCL_OK != BlobGetIntervalSpecInside (interp, argc, argv,
					   blobLength, &from, &to, &edges))
    {
      return TCL_ERROR;
    }

  /* skip over parsed range-spec. */
  argc -= edges;
  argv += edges;

  if (TCL_OK != Blob_GetInput (interp, argc, argv, b, blobName,
			      &length, &newData, &dataIsAllocated, NULL))
    {
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_ReplaceData (b, from, to,
				   length, newData))
    {
      Tcl_AppendResult(interp, blobName, " replace : ",
		       Blob_LastError (b), 0);
      result = TCL_ERROR;
    }
  else
    {
      result = TCL_OK;
    }

  if (dataIsAllocated)
    {
      ckfree (newData);
    }

  return result;
}

/*
 *------------------------------------------------------*
 *
 *	SetCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'set' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_SetData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
SetCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int   length;
  char *newData;
  int   dataIsAllocated;
  int   result;

  assert (b);

  if (argc < 3)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" set INPUT\"",
			0);
      return TCL_ERROR;
    }

  argc --;
  argv ++;

  if (TCL_OK != Blob_GetInput (interp, argc, argv, b, blobName,
			      &length, &newData, &dataIsAllocated, NULL))
    {
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_SetData (b, length, newData))
    {
      Tcl_AppendResult(interp, blobName, " set : ",
		       Blob_LastError (b), 0);
      result = TCL_ERROR;
    }
  else
    {
      result = TCL_OK;
    }

  if (dataIsAllocated)
    {
      ckfree (newData);
    }

  return result;
}

/*
 *------------------------------------------------------*
 *
 *	TruncateCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'truncate' minor command of an
 *	blob command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_RemoveData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
TruncateCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  int position;
  int length;

  assert (b);

  if (argc != 2)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" truncate <from>\"",
			0);
      return TCL_ERROR;
    }

  if (TCL_OK != GetPosition (b, blobName, interp, argv [1],
			     &position, &length))
     {
       return TCL_ERROR;
     }

  if (position < 0)
    {
      position = 0;
    }
  else if (position >= length)
    {
      return TCL_OK;
    }

  if (BLOB_OK != Blob_RemoveData (b, position, length))
    {
      Tcl_AppendResult(interp, blobName, " size : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }
  else
    {
      return TCL_OK;
    }
}

/*
 *------------------------------------------------------*
 *
 *	WriteCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'write' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_GetData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
WriteCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  char*       data;
  int         length;
  int         from;
  int         to;
  int         actual;
  int         mode;
  Tcl_Channel chan;

  assert (b);

  if ((argc < 2) || (argc > 4))
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" write <filehandle> ?<from> ?<to>??\"",
			0);
      return TCL_ERROR;
    }

  chan = Tcl_GetChannel (interp, argv [1], &mode);

  if (chan == (Tcl_Channel) NULL)
    {
      return TCL_ERROR;
    }

  if ((mode & TCL_WRITABLE) == 0)
    {
      Tcl_AppendResult(interp, "channel \"", argv [1],
		       "\" wasn't opened for writing", (char *) NULL);
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_Size (b, &length))
    {
      Tcl_AppendResult(interp, blobName, " write : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  argc -= 2;
  argv += 2;

  if (TCL_OK != BlobGetIntervalSpec (interp, argc, argv,
				     length, &from, &to))
    {
      return TCL_ERROR;
    }

  if (BLOB_OK != Blob_GetData (b, from, to,
			       &length, &data))
    {
      Tcl_AppendResult(interp, blobName, " string : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }
  else
    {
      actual = Tcl_Write(chan,data,length);
      sprintf(interp->result,"%d",actual);
      return TCL_OK;
    }
}

/*
 *------------------------------------------------------*
 *
 *	SetSecureCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'secure' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_SetSecure
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
SetSecureCmd (b, blobName, interp, argc, argv)
Blob         b;	        /* blob to operate on and/or with */
CONST char*  blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp*  interp;	/* interpreter we are working in */
int          argc;
char**       argv;
{
  assert (b);

  if (argc > 1)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" secure",
			0);
      return TCL_ERROR;
    }

  Blob_SetSecure (b);

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	IsSecureCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'is-secure' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_GetSecure
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
IsSecureCmd (b, blobName, interp, argc, argv)
Blob         b;	        /* blob to operate on and/or with */
CONST char*  blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp*  interp;	/* interpreter we are working in */
int          argc;
char**       argv;
{
  assert (b);

  if (argc > 1)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" is-secure",
			0);
      return TCL_ERROR;
    }

  Tcl_AppendResult (interp,
		    (Blob_GetSecure (b) ?
		     "1" :
		     "0"), 0);

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	StringCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'string' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_GetData
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
StringCmd (b, blobName, interp, argc, argv)
Blob         b;	/* blob to operate on and/or with */
CONST char *blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp *interp;	/* interpreter we are working in */
int         argc;
char      **argv;
{
  char *data;
  int   length;
  int   from;
  int   to;
  int   result;

  Blob_CvtBin2String cvtFun;

  assert (b);

  /* argv [0] == "string" */

  if ((argc < 1) || (argc > 4))
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0], "\", must be \"", blobName,
			" string ?-no|-hex|-esc? ?<from> ?<to>??\"",
			0);
      return TCL_ERROR;
    }

  /* conversion option specified ?
   * check number of arguments too
   * ("x string" -> (argc == 1) -> (argv [1] == 0)!
   */
  if ((argc > 1) && ('-' == argv [1][0]))
    {
      char* option = argv [1];

      /* skip over '-' */
      option ++;

      if (TCL_OK != BlobFindConversion (interp, option, NULL, &cvtFun))
	return TCL_ERROR;

      /* skip over legal option */
      argv ++;
      argc --;
    }
  else if (TCL_OK != BlobFindConversion (interp, "esc", NULL, &cvtFun))
    return TCL_ERROR;


  if (BLOB_OK != Blob_Size (b, &length))
    {
      Tcl_AppendResult(interp, blobName, " string : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  argc --;
  argv ++;

  if (TCL_OK != BlobGetIntervalSpec (interp, argc, argv,
				     length, &from, &to))
    {
      return TCL_ERROR;
    }

  if (length == 0)
    /* empty blob -> empty string */
    return TCL_OK;

  if (BLOB_OK != Blob_GetData (b, from, to,
			       &length, &data))
    {
      Tcl_AppendResult(interp, blobName, " string : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  /*
   * Now do the specified conversion
   */

  result = (*cvtFun) (interp, b, length, data, &data);
  if (result == TCL_OK)
    {
      /*
       * The conversion procedure allocated 'data',
       * so its a dynamic result. 
       */

      Tcl_SetResult (interp, data, TCL_DYNAMIC);      
      return TCL_OK;
    }
  else
    {
      return TCL_ERROR;
    }
}

/*
 *------------------------------------------------------*
 *
 *	UnpackCmd --
 *
 *	------------------------------------------------*
 *	realizes the 'unpack' minor command of an blob
 *	command.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
UnpackCmd (b, blobName, interp, argc, argv)
Blob        b;		/* blob to operate on and/or with */
CONST char* blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;
char**      argv;
{
  char* result;
  int   res;

  assert (b);

  if (argc != 2)
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"", blobName, " ",
			argv [0],
			"\", must be \"", blobName, " unpack format\"",
			0);
      return TCL_ERROR;
    }

  res = BlobUnpack (interp, b, argv [1], &result);
  if (res != TCL_OK)
    return res;

  Tcl_SetResult (interp, result, TCL_DYNAMIC);
  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *      GetPosition --
 *
 *	------------------------------------------------*
 *	Retrieves the size of the specified blob and
 *	determines a position inside the blob from the
 *	given argument.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The Interpreter result may change.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
GetPosition (b, blobName, interp, string, position, length)
Blob        b;		/* blob to operate on and/or with */
CONST char* blobName;	/* external name of blob we are
			 * operating on and/or with */
Tcl_Interp* interp;	/* interpreter we are working in */
CONST char* string;	/* argument defining the position */
int*        position;	/* here goes the calculated position */
int*        length;	/* here goes the calculated blob length */
{
  if (BLOB_OK != Blob_Size (b, length))
    {
      Tcl_AppendResult(interp, blobName, " size : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  if (0 == strcmp (string, "end"))
    {
      *position = *length;
      return TCL_OK;
    }
  else if (TCL_OK != Tcl_GetInt (interp, (char *) string, position))
    {
      return TCL_ERROR;
    }

  return TCL_OK;
}

