
/*
 * blobCmd.c --
 *
 *	Implementation of Tcl commands manipulating BLOBs.
 *	The class commands are realized here.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: blobCmd.c,v 1.7 1996/05/04 16:32:08 aku Exp $
 */

#include <assert.h>
#include <string.h>
#include <stdio.h>
#ifdef __STDC__
#include <stdlib.h>
#endif

#include "blobInt.h"

/*
 * Internally used procedures.
 */

static int
BlobCmd _ANSI_ARGS_ ((ClientData  dummy,
		      Tcl_Interp* interp,
		      int         argc,
		      char**      argv));
static int
CreateCmd _ANSI_ARGS_ ((ClientData  dummy,
			Tcl_Interp* interp,
			int         argc,
			char**      argv));
static int
InfoCmd _ANSI_ARGS_ ((ClientData  dummy,
		      Tcl_Interp* interp,
		      int         argc,
		      char**      argv));

static void
HandleInterpreterDestruction _ANSI_ARGS_ ((ClientData  clientData,
					   Tcl_Interp* interp));

static void
BlobDeleteCmd _ANSI_ARGS_ ((ClientData clientData));


/*
 *------------------------------------------------------*
 *
 *	Blob_Init --
 *
 *	------------------------------------------------*
 *	creates all commands of the BLOB extension
 *	visible on TCL level.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Tcl_CreateCommand
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

int
Blob_Init (interp)
Tcl_Interp *interp;	/* interpreter to extend with
			 * blob functionality */
{
  int             res;
  BlobInterpData* iData;

  if (Blob_IsInitialized (interp))
    {
      /* catch multiple initialization of an interpreter
       */
      return TCL_OK;
    }


  iData = (BlobInterpData*) ckalloc (sizeof (BlobInterpData));

  if (! iData)
    {
      Tcl_AppendResult (interp,
			"Not enough memory to initialize extension \"Tcl-Blob\"",
			0);

      return TCL_ERROR;
    }

  Tcl_CreateCommand (interp, "blob", BlobCmd, 0, 0);

  Tcl_InitHashTable (&iData->blobSet, TCL_ONE_WORD_KEYS);
  Tcl_InitHashTable (&iData->convSet, TCL_STRING_KEYS);

  Tcl_SetAssocData (interp, ASSOC_KEY,
		    (Tcl_InterpDeleteProc*) HandleInterpreterDestruction,
		    (ClientData) iData);

  res = BlobAddStdConversionMethods (interp);
  if (TCL_OK != res)
    {
      return res;
    }

  /* register extension as now available package */
  Tcl_PkgProvide (interp, "Blob", BLOB_VERSION);

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	Blob_SafeInit --
 *
 *	------------------------------------------------*
 *	creates all commands of the BLOB extension
 *	visible on TCL level.
 *	VARIANT for safe interpreter
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Tcl_CreateCommand
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

int
Blob_SafeInit (interp)
Tcl_Interp *interp;	/* interpreter to extend with
			 * blob functionality */
{
  /* There is nothing security relevant in the Tcl-Blob
   * extension.  Yes, we are using file/channel handles,
   * but their security is handled by Tcl itself already.
   */

  return Blob_Init (interp);
}

/*
 *------------------------------------------------------*
 *
 *	Blob_IsInitialized --
 *
 *	------------------------------------------------*
 *	Check, wether this extension was initialized in
 *	the specified interpreter or not.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a boolean value.
 *
 *------------------------------------------------------*
 */

int
Blob_IsInitialized (interp)
Tcl_Interp *interp;	/* interpreter to check for initialization */
{
  BlobInterpData* iData = Tcl_GetAssocData (interp, ASSOC_KEY, NULL);

  return (iData != NULL);
}

/*
 *------------------------------------------------------*
 *
 *	Blob_AddConversion --
 *
 *	------------------------------------------------*
 *	Register new conversion method in interpreter.
 *	From now on the method is accessible via option
 *	'-name'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		see above.
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

int
Blob_AddConversion (interp, name, s2bFun, b2sFun)
Tcl_Interp*        interp; /* interpreter to extend */
CONST char*        name;   /* name of conversion method */
Blob_CvtString2Bin s2bFun; /* function to convert string -> binary data */
Blob_CvtBin2String b2sFun; /* function to convert binary data -> string */
{
  BlobInterpData* iData;
  BlobConv*       conv;
  Tcl_HashEntry*  hPtr;
  int             new;

  iData = Tcl_GetAssocData (interp, ASSOC_KEY, NULL);
  hPtr  = Tcl_FindHashEntry (&iData->convSet, (char*) name);

  if (hPtr)
    {
      Tcl_AppendResult (interp,
			"conversion method \"", name,
			"\" already defined", 0);
      return TCL_ERROR;
    }

  conv  = (BlobConv*) ckalloc (sizeof (BlobConv));

  if (! conv)
    {
      Tcl_AppendResult (interp,
			"Not enough memory to add conversion method \"",
			name, "\"", 0);
      return TCL_ERROR;
    }

  hPtr = Tcl_CreateHashEntry (&iData->convSet, (char*) name, &new);
  assert (new);

  Tcl_SetHashValue (hPtr, conv);

  conv->s2bFun = s2bFun;
  conv->b2sFun = b2sFun;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	Blob_GetBlobHandle --
 *
 *	------------------------------------------------*
 *	Retrieve handle of blob associated to command
 *	'name'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

EXTERN int
Blob_GetBlobHandle (interp, name, result)
Tcl_Interp* interp; /* interpreter to use during inquiry */
CONST char* name;   /* name of blob to retrieve */
Blob*       result; /* storage to write the handle to */
{
  Tcl_CmdInfo cmdInfo;
  int         ok;

  assert (result);
  assert (name);

  ok = Tcl_GetCommandInfo(interp, (char*) name, &cmdInfo);
  if (! ok)
    {
      Tcl_ResetResult  (interp);
      Tcl_AppendResult (interp, "\"", name, "\" does not exist", 0);

      return TCL_ERROR;
    }

  if (cmdInfo.proc != BlobObjectCmd)
    {
      Tcl_ResetResult  (interp);
      Tcl_AppendResult (interp, "\"", name, "\" is not a blob", 0);

      return TCL_ERROR;
    }

  *result = ((BlobClientData*) cmdInfo.clientData)->b;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	BlobCmd --
 *
 *	------------------------------------------------*
 *	realizes the command 'blob' and all its minor
 *	commands.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of the called subroutines.
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
BlobCmd (dummy, interp, argc, argv)
ClientData  dummy;	/* irrelevant */
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;	/* number of arguments to command */
char**      argv;	/* array referencing the arguments */
{
  int   len;
  char  c;
  char* minorCommand;

  Tcl_ResetResult (interp);

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "wrong # args for \"blob\"", 0);

      return TCL_ERROR;
    }

  minorCommand = argv [1];

  if (! minorCommand [0])
    {
    usage:
      Tcl_AppendResult (interp,
			"illegal minor command \"",
			minorCommand,
			"\", must be one of create, info, endian or version",
			0);
      return TCL_ERROR;
    }

  /*
   * common argument checking now done,
   * start dispatching the minor commands.
   * skip over the major command name ("blob") beforehand.
   */

  len = strlen (minorCommand);
  c   = minorCommand [0];

  argc --;
  argv ++;

  switch (c)
    {
    case 'c':
      if (strncmp (minorCommand, "create", len) == 0)
	{
	  return CreateCmd (dummy, interp, argc, argv);
	}
      break;

    case 'i':
      if (strncmp (minorCommand, "info", len) == 0)
	{
	  return InfoCmd (dummy, interp, argc, argv);
	}
      break;

    case 'v':
      if (strncmp (minorCommand, "version", len) == 0)
	{
	  if (argc > 1)
	    {
	      Tcl_AppendResult (interp,
				"wrong # args for \"blob version\"",
				0);
	      return TCL_ERROR;
	    }

	  Tcl_AppendResult (interp, BLOB_VERSION, 0);
	  return TCL_OK;
	}
      break;

    case 'e':
      if (strncmp (minorCommand, "endian", len) == 0)
	{
	  if (argc > 1)
	    {
	      Tcl_AppendResult (interp,
				"wrong # args for \"blob endian\"",
				0);
	      return TCL_ERROR;
	    }

#ifdef WORDS_BIGENDIAN
	  Tcl_AppendResult (interp, "big", 0);
#else
	  Tcl_AppendResult (interp, "little", 0);
#endif
	  return TCL_OK;
	}
      break;
    }

  goto usage;

#if CURRENTLY_NOT_USED
  /*
   * This code is maybe necessary after the creation of
   * additional minor commands.
   */

 not_unique:
  Tcl_AppendResult (interp,
		    "non-unique abbreviation \"",
		    minorCommand,
		    "\" used as minor command",
		    0);
  return TCL_ERROR;
#endif
}

/*
 *------------------------------------------------------*
 *
 *	CreateCmd --
 *
 *	------------------------------------------------*
 *	realizes 'blob create'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		see manual.
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
CreateCmd (dummy, interp, argc, argv)
ClientData  dummy;	/* irrelevant */
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;	/* number of arguments to minor
			 * command */
char**      argv;	/* array referencing the arguments */
{
  char* minorCommand = argv [0];

  if ((argc != 2) && (argc != 3))
    {
    arg_error:
      Tcl_AppendResult (interp,
			"wrong # args for \"blob ",
			minorCommand,
			"\", must be \"blob create ?-secure? <name>\"",
			0);
      return TCL_ERROR;
    }
  else
    {
      char*           blobName = argv [1];
      Blob            b;
      BlobInterpData* iData;
      Tcl_Command     cmd;
      BlobClientData* bcd;
      Tcl_HashEntry*  hPtr;
      int             new;
      int             is_secure = FALSE;

      if (0 == strcmp (blobName, "-secure")) 
	{
	  if (argc == 2)
	    goto arg_error;

	  is_secure = TRUE;
	  blobName = argv [2];
	}
      else if (argc > 2)
	goto arg_error;

      b = Blob_Create ();

      if (b == 0)
	{
	  Tcl_AppendResult(interp,
			   "\"blob ", minorCommand,
			   ": could not create blob \"", blobName, "\"",
			   0);
	  return TCL_ERROR;
	}

      if (is_secure)
	{
	  Blob_SetSecure (b);
	}

      iData = Tcl_GetAssocData (interp, ASSOC_KEY, NULL);

      bcd = (BlobClientData*) ckalloc (sizeof (BlobClientData));

      bcd->blobSet = &iData->blobSet;
      bcd->b       = b;

      cmd = Tcl_CreateCommand (interp, blobName, BlobObjectCmd,
			       (ClientData) bcd, BlobDeleteCmd);
      
      hPtr = Tcl_CreateHashEntry (&iData->blobSet, (char*) bcd, &new);

      /*
       * Tcl_CreateCommand should have thrown away any existent
       * blob command and associated hashtable entry!
       */
      assert (new);

      Tcl_SetHashValue (hPtr, cmd);
      Tcl_AppendResult (interp, blobName, 0);
      return TCL_OK;
    }
}

/*
 *------------------------------------------------------*
 *
 *	InfoCmd --
 *
 *	------------------------------------------------*
 *	realizes 'blob info'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
InfoCmd (dummy, interp, argc, argv)
ClientData  dummy;	/* irrelevant */
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;	/* number of arguments to minor
			 * command */
char**      argv;	/* array referencing the arguments */
{
  char*           minorCommand = argv [0];
  char*           blobName;
  char*           pattern;
  Tcl_Command     token;
  BlobInterpData* iData;
  Tcl_HashSearch  s;
  Tcl_HashEntry*  hPtr;


  if ((argc < 1) || (argc > 2))
    {
      Tcl_AppendResult (interp,
			"wrong # args for \"blob ",
			minorCommand,
			"\", must be \"blob info ?pattern?\"",
			0);
      return TCL_ERROR;
    }


  pattern = (argc == 2 ? argv [1] : "*");


  /*
   * Scan table of defined blobs and
   * remember the names of the associated
   * commands matching the pattern.
   */

  iData = Tcl_GetAssocData (interp, ASSOC_KEY, NULL);

  for (hPtr  = Tcl_FirstHashEntry (&iData->blobSet, &s);
       hPtr != 0;
       hPtr  = Tcl_NextHashEntry (&s))
    {
      token    = (Tcl_Command) Tcl_GetHashValue (hPtr);
      blobName = Tcl_GetCommandName (interp, token);

      if (Tcl_StringMatch (blobName, pattern))
	{
	  Tcl_AppendElement(interp, blobName);
	}
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	BlobDeleteCmd --
 *
 *	------------------------------------------------*
 *	Called during the disposal of the object
 *	command of some blob.  Destroys associated blob
 *	and information stored in interpreter.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of Blob_Delete ()
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
BlobDeleteCmd (clientData)
ClientData  clientData;	/* BlobClientData reference */
{
  BlobClientData* bcd  = (BlobClientData*) clientData;
  Tcl_HashEntry*  hPtr = Tcl_FindHashEntry (bcd->blobSet, (char*) bcd);

  Tcl_DeleteHashEntry (hPtr);
  Blob_Delete         (bcd->b);

  ckfree (bcd);
}

/*
 *------------------------------------------------------*
 *
 *	HandleInterpreterDestruction --
 *
 *	------------------------------------------------*
 *	Deletes all extension structures associated to
 *	the interpreter.  ATTENTION: The commands are
 *	deleted already, so there is no need to destroy
 *	the blob handles explicitly.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		see above.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
HandleInterpreterDestruction (dummy, interp)
ClientData  dummy;	/* the extension structure to delete */
Tcl_Interp *interp;	/* deleted interpreter */
{
  BlobInterpData* iData = (BlobInterpData*) dummy;

  Tcl_DeleteHashTable (&iData->blobSet);
  Tcl_DeleteHashTable (&iData->convSet);

  ckfree (iData);
}
