/*
 * blob.c --
 *
 *	Implements the C level procedures handling BLOBS.
 *
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: blob.c,v 1.4 1996/05/01 22:27:27 aku Exp $
 */

#include <assert.h>
#include <string.h>
#include <stdio.h>
#ifdef __STDC__
#include <stdlib.h>
#endif

#include "blobInt.h"

/*
 * Import of flag-variable used by system calls to indicate
 * error conditions.
 */

EXTERN int errno;


/*
 * ---------- ---------- ---------- ----------
 *
 * Definition of internal blob-type.
 *
 * For every blob a structure of the following type is
 * allocated off the heap.  Externally visible handle
 * is the pointer to this structure.
 */

typedef struct _iBlob_
{
  int         length;	/* Size of the stored data
			 * (in bytes). */
  char*       data;	/* Reference to memory holding
			 * the stored information. */
  int         escape;	/* The character to use as
			 * escape of '\0'. A value of
			 * '\0' indicates usage of the
			 * global escape character. */
  Tcl_DString* lastErr;	/* Reference to string holding
			 * the last generated error message.
			 */
  short        secure;  /* boolean flag. TRUE commands
			 * this blob not to leave its contents
			 * lying around during removal
			 * or changes in size.
			 */
} _iBlob, *iBlob;



#define BSIZE (sizeof (_iBlob))

/*
 * Forward declarations of internal procedures.
 */

static int
DefineBlobSize _ANSI_ARGS_ ((iBlob blob, int newLength));


/*
 *------------------------------------------------------
 *
 *	Blob_Create --
 *
 *	------------------------------------------------
 *	a new blob is created and initialized as empty
 *	(and insecure).
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		memory is allocated.
 *
 *	Result:
 *		a blob-handle or NULL in case of a
 *		failure.
 *
 *------------------------------------------------------
 */

Blob
Blob_Create ()
{
  /*
   * Allocate new blob and initialize its memory.
   */

  iBlob newBlob = ckalloc (BSIZE);

  if (newBlob == NULL)
    {
      return (Blob) 0;
    }

  memset (newBlob, 0, BSIZE);
  newBlob->escape = '@';

  return (Blob) newBlob;
}

/*
 *------------------------------------------------------
 *
 *	Blob_Delete --
 *
 *	------------------------------------------------
 *	destroys the given blob and its contents
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		allocated memory is freed again.
 *
 *	Result:
 *		a standard BLOB error code
 *
 *------------------------------------------------------
 */

void
Blob_Delete (blobE)
Blob blobE;
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  if (blob->data)
    {
      if (blob->secure)
	{
	  /* wipe data of secure blob before release */
	  memset (blob->data, 0, blob->length);
	}

      ckfree (blob->data);
    }

  if (blob->lastErr)
    {
      Tcl_DStringFree (blob->lastErr);
      ckfree          (blob->lastErr);
    }

  if (blob->secure)
    {
      /* wipe management information too */
      memset (blob, 0, BSIZE);
    }

  ckfree (blob);
}

/*
 *------------------------------------------------------
 *
 *	Blob_SetSecure --
 *
 *	------------------------------------------------
 *	mark the given blob as secure.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		All functions operating on the blob will
 *		clean all released memory from now on.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------
 */

void
Blob_SetSecure (blobE)
Blob blobE;
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  blob->secure = TRUE;
}

/*
 *------------------------------------------------------
 *
 *	Blob_GetSecure --
 *
 *	------------------------------------------------
 *	Query the current state of the objects security-flag.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		See above.
 *
 *------------------------------------------------------
 */

int
Blob_GetSecure (blobE)
Blob blobE;
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  return blob->secure;
}

/*
 *------------------------------------------------------
 *
 *	Blob_RemoveData --
 *
 *	------------------------------------------------
 *	Deletes the specified part of blob contents.
 *
 *	to   <= from		=> do nothing
 *	to   == length		=> truncate <from>
 *	from == 0, to == length => clear all
 *
 *	"from" specifies the position of the first byte
 *	to be deleted. "to" specifies the position of
 *	the byte AFTER the last byte to be deleted.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		see above
 *
 *	Result:
 *		a standard BLOB error code
 *
 *------------------------------------------------------
 */

EXTERN int
Blob_RemoveData (blobE, from, to)
Blob blobE;	/* blob to manipulate */
int  from;	/* see above */
int  to;	/* see above */
{
  iBlob blob = (iBlob) blobE;
  assert (blob);


  /*
   * Check for indices out of bounds
   */

  if ((from < 0) || (to > blob->length))
    {
      char intBuf [40];
      if (! blob->lastErr)
	{
	  blob->lastErr = ckalloc (sizeof (Tcl_DString));
	  Tcl_DStringInit (blob->lastErr);
	}
      else 
	{
	  Tcl_DStringFree (blob->lastErr);
	}

      sprintf (intBuf, "%d..%d)", from, to);
      Tcl_DStringAppend (blob->lastErr,
			 "Blob_RemoveData: illegal interval [", -1);
      Tcl_DStringAppend (blob->lastErr, intBuf, -1);

      return BLOB_ERROR;
    }


  /*
   * Check for specification of empty interval.
   * Do nothing, if such is given,
   */

  if (to <= from)
    {
      return BLOB_OK;
    }


  /*
   * Now remove the interval.
   *
   * This degenerates to "Truncate" or "Clear"
   * if the interval goes up to the end of the blob
   */

  if (to == blob->length)
    {
      /*
       * truncate / clear data
       */

      if (BLOB_OK != DefineBlobSize (blob, from))
	{
	  return BLOB_ERROR;
	}      
    }
  else
    {
      /*
       * Remove inner slice from array.
       *
       * "len" == length of block residing
       * behind the slice to be removed
       */

      int len  = blob->length - to;

      memmove ((VOID *) &(blob->data [from]),
	       (VOID *) &(blob->data [to]),
	       (int)    len);

      if (BLOB_OK != DefineBlobSize (blob, from+len))
	{
	  return BLOB_ERROR;
	}      
    }

  return BLOB_OK;
}

/*
 *------------------------------------------------------
 *
 *	Blob_InsertData --
 *
 *	------------------------------------------------
 *	Insert the given data into the blob contents.
 *	This insertion is made just BEFORE the given
 *	position.
 *
 *	at == length ==> append data.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		see above
 *
 *	Result:
 *		a standard BLOB error code
 *
 *------------------------------------------------------
 */

EXTERN int
Blob_InsertData (blobE, at, length, newData)
Blob   blobE;	/* blob to manipulate */
int    at;
int    length;
char*  newData;
{
  int   oldLength;
  iBlob blob = (iBlob) blobE;

  assert (blob);

  if ((length <= 0) || (newData == 0))
    {
      /*
       * Nothing to do, no data to insert was given.
       */

      return BLOB_OK;
    }


  /*
   * Check for position out of bounds
   */

  if ((at < 0) || (at > blob->length))
    {
      char intBuf [20];
      if (! blob->lastErr)
	{
	  blob->lastErr = ckalloc (sizeof (Tcl_DString));
	  Tcl_DStringInit (blob->lastErr);
	}
      else 
	{
	  Tcl_DStringFree (blob->lastErr);
	}

      sprintf (intBuf, "%d", at);
      Tcl_DStringAppend (blob->lastErr,
			 "Blob_InsertData: illegal insert position ", -1);
      Tcl_DStringAppend (blob->lastErr, intBuf, -1);

      return BLOB_ERROR;
    }


  /*
   * Now do the insertion.
   * Extend the data storage first, then transfer given information.
   */

  oldLength = blob->length;

  if (BLOB_OK != DefineBlobSize (blob, blob->length + length))
    {
      return BLOB_ERROR;
    }


  if (at < oldLength)
    {
      /*
       * Generate place for new information by shifting backward
       * the information residing behind the point of insertion.
       *
       * ATTENTION: use OLD length of blob to determine the amount
       * of data to be shifted.
       */

      memmove ((VOID *) &(blob->data [at+length]),
	       (VOID *) &(blob->data [at]),
	       (int)    (oldLength - at));
    }
  /* else if (at == oldLength)
   *   -> Append, so don't touch current contents.
   */

  memcpy ((VOID *) &(blob->data [at]),
	  (VOID *) newData,
	  (int)    length);

  return BLOB_OK;
}

/*------------------------------------------------------
 *
 *	Blob_ReplaceData --
 *
 *	------------------------------------------------
 *	Exchanges the specified part of blob contents
 *	against the new information.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		as of Blob_RemoveData, Blob_InsertData
 *
 *	Result:
 *		a standard BLOB error code
 *
 *------------------------------------------------------
 */

EXTERN int
Blob_ReplaceData (blobE, from, to, length, newData)
Blob   blobE;	/* blob to manipulate */
int    from;
int    to;
int    length;
char*  newData;
{
  if (BLOB_OK != Blob_RemoveData (blobE, from, to))
    {
      return BLOB_ERROR;
    }

  return Blob_InsertData (blobE, from, length, newData);
}

/*
 *------------------------------------------------------
 *
 *	Blob_SetData --
 *
 *	------------------------------------------------
 *	the actual contents of the given blob are
 *	deleted and replaced with the new information.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		memory is freed and allocated.
 *
 *	Result:
 *		a standard BLOB error code
 *
 *------------------------------------------------------
 */

int
Blob_SetData (blobE, len, newData)
Blob   blobE;	/* blob to manipulate */
int    len;
char*  newData;
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  /*
   * Adjust data storage to the needs of the specified
   * information, then transfer it.
   */

  if (BLOB_OK != DefineBlobSize (blob, len))
    {
      return BLOB_ERROR;
    }

  if (len > 0)
    {
      memcpy ((VOID *) blob->data, (VOID *) newData, len);
    }

  return BLOB_OK;
}

/*
 *------------------------------------------------------
 *
 *	Blob_GetData --
 *
 *	------------------------------------------------
 *	retrieves all or part of the data stored in the
 *	BLOB.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard BLOB error code
 *
 *	ATTENTION:
 *		The retrieval buffer refers INTO the
 *		blob structure AND MUST NOT be written
 *
 *------------------------------------------------------
 */

int
Blob_GetData (blobE, from, to, len, data)
Blob    blobE;	/* blob to look at */
int     from;
int     to;
int*    len;	/* length of retrieved data */
char**  data;	/* reference to buffer containing
		 * the retrieved information. */
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  /*
   * Check for indices out of bounds
   */

  if ((from < 0) || (to > blob->length))
    {
      char intBuf [40];
      if (! blob->lastErr)
	{
	  blob->lastErr = ckalloc (sizeof (Tcl_DString));
	  Tcl_DStringInit (blob->lastErr);
	}
      else 
	{
	  Tcl_DStringFree (blob->lastErr);
	}

      sprintf (intBuf, "%d..%d)", from, to);
      Tcl_DStringAppend (blob->lastErr,
			 "Blob_GetData: illegal interval [", -1);
      Tcl_DStringAppend (blob->lastErr, intBuf, -1);

      return BLOB_ERROR;
    }


  if (to <= from)
    {
      /*
       * An empty interval was specified.
       */

      *len  = 0;
      *data = 0;
    }
  else
    {
      *len  = to - from;

      /* ATTENTION */
      /* set pointer into data storage of blob */
      *data = &(blob->data [from]);
    }

  return BLOB_OK;
}

/*
 *------------------------------------------------------
 *
 *	Blob_SetEscape --
 *
 *	------------------------------------------------
 *	the escape character of the given blob is set.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		see above.
 *
 *	Result:
 *		a standard BLOB error code
 *
 *------------------------------------------------------
 */

int
Blob_SetEscape (blobE, escape)
Blob blobE;	/* blob to manipulate */
int  escape;
{
  iBlob blob = (iBlob) blobE;
  assert (blob);
  assert (escape);

  blob->escape = escape;

  return BLOB_OK;
}

/*
 *------------------------------------------------------
 *
 *	Blob_GetEscape --
 *
 *	------------------------------------------------
 *	retrieve escape character for given blob
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		the escape character or BLOB_ERROR in
 *		case of failure.
 *
 *------------------------------------------------------
 */

int
Blob_GetEscape (blobE)
Blob blobE;	/* blob to look at */
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  return blob->escape;
}

/*
 *------------------------------------------------------
 *
 *	Blob_LastError --
 *
 *	------------------------------------------------
 *	returns string containing the last generated
 *	error message for specified blob.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		pointer to a string
 *
 *------------------------------------------------------
 */

CONST char *
Blob_LastError (blobE)
Blob blobE;
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  if (! blob->lastErr)
    {
      return "";
    }
  else
    {
      return Tcl_DStringValue (blob->lastErr);
    }
}

/*
 *------------------------------------------------------
 *
 *	Blob_Size --
 *
 *	------------------------------------------------
 *	retrieves the size of the data area of the
 *	specified blob.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard BLOB error code.
 *
 *------------------------------------------------------
 */

int
Blob_Size (blobE, size)
Blob blobE;	/* blob to look at */
int* size;
{
  iBlob blob = (iBlob) blobE;
  assert (blob);

  *size = blob->length;

  return BLOB_OK;
}

/*
 *------------------------------------------------------
 *
 *	DefineBlobSize --
 *
 *	------------------------------------------------
 *	the data area of the blob is expanded or
 *	shrunked to be of size 'newLength'.
 *	------------------------------------------------
 *
 *	Sideeffects:
 *		memory is allocated and/or freed.
 *
 *	Result:
 *		a standard BLOB error code
 *
 *------------------------------------------------------
 */

static int
DefineBlobSize (blob, newLength)
iBlob blob;
int   newLength;
{
  /* 2 possibilities:
   *
   * <> Memory holding the data was allocated before,
   *    simply expand or shrink it to the needs of the
   *    new contents.
   *
   * <> No memory is available, so allocate it now.
   */

  if (blob->data)
    {
      if (blob->length > newLength)
	{
	  /*
	   * Shrink memory, maybe free it completely.
	   */

	  if (blob->secure)
	    {
	      /* wipe released part of memory */
	      memset (blob->data + newLength, 0, blob->length - newLength);
	    }

	  if (newLength == 0)
	    {
	      ckfree (blob->data);
	      blob->data = 0;
	    }
	  else
	    {
	      blob->data = ckrealloc (blob->data, newLength);
	    }
	}
      else if (blob->length < newLength)
	{
	  if (blob->secure)
	    {
	      /* for a secure blob a simple 'ckrealloc' is
	       * NOT sufficient.  if the system choses to
	       * relocate the allocated area, the old area is
	       * most certainly not wiped!  so we have to
	       * do the reallocation ourselves!
	       */

	      char* new_data = ckalloc (newLength);

	      memcpy (new_data, blob->data, blob->length);
	      memset (blob->data, 0, blob->length);

	      ckfree (blob->data);

	      blob->data = new_data;
	    }
	  else
	    {
	      blob->data = ckrealloc (blob->data, newLength);
	    }
	}
    }
  else
    {
      blob->data = ckalloc (newLength);
    }

  blob->length = newLength;

  return BLOB_OK;
}
