set rcsId {$Id: filesel.tcl,v 1.3 1996/05/19 20:31:03 jfontain Exp $}

source scrolist.tcl
source combolab.tcl

proc fileSelector::fileSelector {this parentPath args} composite {[new toplevel $parentPath] $args} {
    # create current directory display
    composite::manage $this [new frame $widget($this,path)] directory
    pack $composite($this,directory,path) -anchor w
    composite::manage $this\
        [new label $composite($this,directory,path) -text {Current directory:}] directoryLabel\
        [new label $composite($this,directory,path) -text [pwd]] directoryCurrent
    pack $composite($this,directoryLabel,path) $composite($this,directoryCurrent,path) -side left

    # create selected file and entry display
    composite::manage $this\
        [new label $widget($this,path) -text {Selected file:}] fileLabel [new entry $widget($this,path)] fileEntry
    pack $composite($this,fileLabel,path) -anchor w
    pack $composite($this,fileEntry,path) -fill x

    # create directories and files lists
    composite::manage $this [new frame $widget($this,path)] lists
    pack $composite($this,lists,path) -fill both -expand 1
    composite::manage $this\
        [new label $composite($this,lists,path) -text {Directories:}] directoriesLabel\
        [new scrollList $composite($this,lists,path)] directoriesList\
        [new label $composite($this,lists,path) -text {Files:}] filesLabel\
        [new scrollList $composite($this,lists,path)] filesList
    grid $composite($this,directoriesLabel,path) -column 0 -row 0 -sticky nw
    grid $composite($this,directoriesList,path) -column 0 -row 1 -sticky nsew
    grid $composite($this,filesLabel,path) -column 1 -row 0 -sticky nw
    grid $composite($this,filesList,path) -column 1 -row 1 -sticky nsew
    grid columnconfigure $composite($this,lists,path) 0 -weight 1
    grid columnconfigure $composite($this,lists,path) 1 -weight 1
    grid rowconfigure $composite($this,lists,path) 1 -weight 1

    # create file type and combo label display
    composite::manage $this\
        [new label $widget($this,path) -text {File type:}] typeLabel\
        [new comboLabel $widget($this,path) -command "fileSelector::filterChange $this"] typeCombo
    pack $composite($this,typeLabel,path) -anchor w
    pack $composite($this,typeCombo,path) -fill x

    # create separator
    composite::manage $this [new frame $widget($this,path) -relief sunken -borderwidth 1 -height 2] separator
    pack $composite($this,separator,path) -fill x -pady 5

    # create buttons
    composite::manage $this\
        [new button $widget($this,path) -state disabled -text OK -command "fileSelector::selected $this"] ok\
        [new button $widget($this,path) -text Cancel -command "fileSelector::canceled $this"] cancel
    pack $composite($this,ok,path) -side left -expand 1 -pady 3
    pack $composite($this,cancel,path) -side right -expand 1 -pady 3

    composite::complete $this

    # setup file entry bindings
    bind $composite($this,fileEntry,path) <KeyRelease> "fileSelector::updateButtons $this"

    # setup listbox selection bindings
    set path $composite($composite($this,directoriesList),listbox,path)
    bind $path <Double-1> "fileSelector::selectDirectory $this"
    bind $path <space> "fileSelector::selectDirectory $this"
    set path $composite($composite($this,filesList),listbox,path)
    bind $path <ButtonRelease-1> "fileSelector::selectFile $this"
    bind $path <space> "fileSelector::selectFile $this"
    bind $path <Double-1> "fileSelector::selected $this"

    # set return and escape bindings
    bind $widget($this,path) <Return> "$composite($this,ok,path) invoke"
    bind $widget($this,path) <Escape> "$composite($this,cancel,path) invoke"
    # handle destruction from window manager
    wm protocol $widget($this,path) WM_DELETE_WINDOW "$composite($this,cancel,path) invoke"

    set fileSelector($this,patterns) {}
    fileSelector::updateDirectories $this
    fileSelector::updateFiles $this
}

proc fileSelector::~fileSelector {this} {}

proc fileSelector::options {this} {
    return {{-command command Command {}} {-filters filters Filters {}}}
}

# do nothing, command will be stored at the composite level
proc fileSelector::set-command {this value} {}

proc fileSelector::set-filters {this value} {
    widget::configure $composite($this,typeCombo) -list $value
}

proc fileSelector::updateDirectories {this} {
    set list $composite($this,directoriesList)
    scrollList::delete $list 0 end
    foreach file [lsort [glob .* *]] {
        if {![file isdirectory $file]} continue
        if {[string compare $file .]==0} continue
        if {([string compare [pwd] /]==0)&&([string compare $file ..]==0)} continue
        scrollList::insert $list end $file
    }
}

proc fileSelector::updateFiles {this} {
    set list $composite($this,filesList)
    scrollList::delete $list 0 end
    set files {}
    if {[llength $fileSelector($this,patterns)]==0} {
        catch {set files [lsort [glob .* *]]}
    } else {
        catch {set files [lsort [eval glob $fileSelector($this,patterns)]]}
    }
    foreach file $files {
        # filter out directories
        if {![file isdirectory $file]} {
            scrollList::insert $list end $file
        }
    }
}

proc fileSelector::filterChange {this entry} {
    # look for parenthesized pattern (such as (*.txt)) in selected entry
    if {![regexp {\((.+)\)} $entry dummy pattern]} {
        set pattern {}
    }
    fileSelector::setFilter $this $pattern
}

proc fileSelector::setFilter {this pattern} {
    widget::configure $composite($this,base) -cursor watch
    update idletasks
    if {([string length $pattern]==0)||([string compare $pattern *]==0)} {
        # a * pattern is like no filter
        set fileSelector($this,patterns) {}
    } else {
        if {[string compare [string index $pattern 0] .]==0} {
            set fileSelector($this,patterns) $pattern
        } else {
            # handle file names beginning with a dot
            set fileSelector($this,patterns) ".$pattern $pattern"
        }
    }
    $composite($this,fileEntry,path) delete 0 end
    fileSelector::updateFiles $this
    fileSelector::updateButtons $this
    widget::configure $composite($this,base) -cursor {}
}

proc fileSelector::selectDirectory {this} {
    set selections [scrollList::curselection $composite($this,directoriesList)]
    if {[llength $selections]!=1} {
        return
    }
    set directory [scrollList::get $composite($this,directoriesList) [lindex $selections 0]]
    widget::configure $composite($this,base) -cursor watch
    update idletasks
    $composite($this,fileEntry,path) delete 0 end
    if {[catch {cd $directory} message]} {
        tk_dialog .error Error "Cannot change directory to \"$directory\": $message" error 0 OK
    } else {
        widget::configure $composite($this,directoryCurrent) -text [pwd]
        fileSelector::updateDirectories $this
        fileSelector::updateFiles $this
    }
    fileSelector::updateButtons $this
    widget::configure $composite($this,base) -cursor {}
}

proc fileSelector::selectFile {this} {
    set selections [scrollList::curselection $composite($this,filesList)]
    if {[llength $selections]!=1} {
        return
    }
    set file [scrollList::get $composite($this,filesList) [lindex $selections 0]]
    set path $composite($this,fileEntry,path)
    $path delete 0 end
    if {[string compare [pwd] /]!=0} {
        $path insert 0 [pwd]
    }
    $path insert end /$file
    fileSelector::updateButtons $this
}

proc fileSelector::updateButtons {this} {
    if {[string length [string trim [$composite($this,fileEntry,path) get]]]==0} {
        widget::configure $composite($this,ok) -state disabled
    } else {
        widget::configure $composite($this,ok) -state normal
    }
}

proc fileSelector::selected {this} {
    eval $composite($this,-command) \{[string trim [$composite($this,fileEntry,path) get]]\}
    delete $this
}

proc fileSelector::canceled {this} {
    $composite($this,-command) {}
    delete $this
}
