static char rcsid[]="$Id: tkCanvDline.c,v 1.1 94/02/09 10:13:52 mangin Exp $";

/* Modified by Frank Mangin, 94/01/06, to
 *  support dashed lines

 * 
 * tkCanvDline.c --
 *
 *	This file implements dline items for canvas widgets.
 *
 * Copyright (c) 1991-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <tkInt.h>
#include "tkCanvas.h"
#include <tkConfig.h>

/*
 * The structure below defines the record for each dline item.
 */

typedef struct DlineItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    Tk_Canvas *canvasPtr;	/* Canvas containing item.  Needed for
				 * parsing arrow shapes. */
    int numPoints;		/* Number of points in dline (always >= 2). */
    double *coordPtr;		/* Pointer to malloc-ed array containing
				 * x- and y-coords of all points in dline.
				 * X-coords are even-valued indices, y-coords
				 * are corresponding odd-valued indices. If
				 * the dline has arrowheads then the first
				 * and last points have been adjusted to refer
				 * to the necks of the arrowheads rather than
				 * their tips.  The actual endpoints are
				 * stored in the *firstArrowPtr and
				 * *lastArrowPtr, if they exist. */
    int width;			/* Width of dline. */
    XColor *fg;			/* Foreground color for dline. */
    Pixmap fillStipple;		/* Stipple bitmap for filling dline. */
    int capStyle;		/* Cap style for dline. */
    int joinStyle;		/* Join style for dline. */
    GC gc;			/* Graphics context for filling dline. */
    Tk_Uid arrow;		/* Indicates whether or not to draw arrowheads:
				 * "none", "first", "last", or "both". */
    float arrowShapeA;		/* Distance from tip of arrowhead to center. */
    float arrowShapeB;		/* Distance from tip of arrowhead to trailing
				 * point, measured along shaft. */
    float arrowShapeC;		/* Distance of trailing points from outside
				 * edge of shaft. */
    double *firstArrowPtr;	/* Points to array of PTS_IN_ARROW points
				 * describing polygon for arrowhead at first
				 * point in dline.  First point of arrowhead
				 * is tip.  Malloc'ed.  NULL means no arrowhead
				 * at first point. */
    double *lastArrowPtr;	/* Points to polygon for arrowhead at last
				 * point in dline (PTS_IN_ARROW points, first
				 * of which is tip).  Malloc'ed.  NULL means
				 * no arrowhead at last point. */
    int smooth;			/* Non-zero means draw dline smoothed (i.e.
				 * with Bezier splines). */
    int splineSteps;		/* Number of steps in each spline segment. */
    int dashNumber;		/* Number of elements in dashList */
    char *dashList;		/* Specifies dash style. Malloc'd, a value of */
    				/* NULL means no dashes */
} DlineItem;

/*
 * Number of points in an arrowHead:
 */

#define PTS_IN_ARROW 6

/*
 * Prototypes for procedures defined in this file:
 */

static int		ArrowheadPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    DlineItem *dlinePtr, double *arrowPtr,
			    Tk_PostscriptInfo *psInfoPtr));
static void		ComputeDlineBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    DlineItem *dlinePtr));
static int		ConfigureDline _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		ConfigureArrows _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    DlineItem *dlinePtr));
static int		CreateDline _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		DeleteDline _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr));
static void		DisplayDline _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static int		DlineCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		DlineToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *rectPtr));
static double		DlineToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *coordPtr));
static int		DlineToPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Tk_PostscriptInfo *psInfoPtr));
static int		ParseArrowShape _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, Tk_Window tkwin, char *value,
			    char *recordPtr, int offset));
static char *		PrintArrowShape _ANSI_ARGS_((ClientData clientData,
			    Tk_Window tkwin, char *recordPtr, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		ParseDashes _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, Tk_Window tkwin, char *value,
			    char *recordPtr, int offset));
static char *		PrintDashes _ANSI_ARGS_((ClientData clientData,
			    Tk_Window tkwin, char *recordPtr, int offset,
			    Tcl_FreeProc **freeProcPtr));
static void		ScaleDline _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslateDline _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateDline.
 */

static Tk_CustomOption arrowShapeOption = {ParseArrowShape,
	PrintArrowShape, (ClientData) NULL};

static Tk_CustomOption dashesOption = {ParseDashes, PrintDashes, (ClientData) NULL};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_UID, "-arrow", (char *) NULL, (char *) NULL,
	"none", Tk_Offset(DlineItem, arrow), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-arrowshape", (char *) NULL, (char *) NULL,
	"8 10 3", Tk_Offset(DlineItem, arrowShapeA),
	TK_CONFIG_DONT_SET_DEFAULT, &arrowShapeOption},
    {TK_CONFIG_CAP_STYLE, "-capstyle", (char *) NULL, (char *) NULL,
	"butt", Tk_Offset(DlineItem, capStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(DlineItem, fg), TK_CONFIG_NULL_OK},
    {TK_CONFIG_JOIN_STYLE, "-joinstyle", (char *) NULL, (char *) NULL,
	"round", Tk_Offset(DlineItem, joinStyle), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BOOLEAN, "-smooth", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(DlineItem, smooth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-splinesteps", (char *) NULL, (char *) NULL,
	"12", Tk_Offset(DlineItem, splineSteps), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(DlineItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_PIXELS, "-width", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(DlineItem, width), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-dashes", (char *) NULL, (char *) NULL,
	"", Tk_Offset(DlineItem, dashList),
	TK_CONFIG_DONT_SET_DEFAULT, &dashesOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the dline item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkDlineType = {
    "dline",				/* name */
    sizeof(DlineItem),			/* itemSize */
    CreateDline,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureDline,			/* configureProc */
    DlineCoords,				/* coordProc */
    DeleteDline,				/* deleteProc */
    DisplayDline,			/* displayProc */
    0,					/* alwaysRedraw */
    DlineToPoint,			/* pointProc */
    DlineToArea,				/* areaProc */
    DlineToPostscript,			/* postscriptProc */
    ScaleDline,				/* scaleProc */
    TranslateDline,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

/*
 * The Tk_Uid's below refer to uids for the various arrow types:
 */

static Tk_Uid noneUid = NULL;
static Tk_Uid firstUid = NULL;
static Tk_Uid lastUid = NULL;
static Tk_Uid bothUid = NULL;

/*
 * The definition below determines how large are static arrays
 * used to hold spline points (splines larger than this have to
 * have their arrays malloc-ed).
 */

#define MAX_STATIC_POINTS 200
/***************************************************************
 ****          Canvas ddline item creation                   ****
 ***************************************************************/

void DlineItemTypeCreate()
{
  Tk_CreateItemType(&TkDlineType);
}


/*
 *--------------------------------------------------------------
 *
 * CreateDline --
 *
 *	This procedure is invoked to create a new dline item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new dline item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateDline(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing dline. */
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;
    int i;

    if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin), "\" create ",
		itemPtr->typePtr->name, " x1 y1 x2 y2 ?x3 y3 ...? ?options?",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    dlinePtr->canvasPtr = canvasPtr;
    dlinePtr->numPoints = 0;
    dlinePtr->coordPtr = NULL;
    dlinePtr->width = 1;
    dlinePtr->fg = None;
    dlinePtr->fillStipple = None;
    dlinePtr->capStyle = CapButt;
    dlinePtr->joinStyle = JoinRound;
    dlinePtr->gc = None;
    if (noneUid == NULL) {
	noneUid = Tk_GetUid("none");
	firstUid = Tk_GetUid("first");
	lastUid = Tk_GetUid("last");
	bothUid = Tk_GetUid("both");
    }
    dlinePtr->arrow = noneUid;
    dlinePtr->arrowShapeA = 8.0;
    dlinePtr->arrowShapeB = 10.0;
    dlinePtr->arrowShapeC = 3.0;
    dlinePtr->firstArrowPtr = NULL;
    dlinePtr->lastArrowPtr = NULL;
    dlinePtr->smooth = 0;
    dlinePtr->splineSteps = 12;
    dlinePtr->dashNumber = 0;
    dlinePtr->dashList = (char *)0;
    
    /*
     * Count the number of points and then parse them into a point
     * array.  Leading arguments are assumed to be points if they
     * start with a digit or a minus sign followed by a digit.
     */

    for (i = 4; i < (argc-1); i+=2) {
	if ((!isdigit(UCHAR(argv[i][0]))) &&
		((argv[i][0] != '-') || (!isdigit(UCHAR(argv[i][1]))))) {
	    break;
	}
    }
    if (DlineCoords(canvasPtr, itemPtr, i, argv) != TCL_OK) {
	goto error;
    }
    if (ConfigureDline(canvasPtr, itemPtr, argc-i, argv+i, 0) == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteDline(canvasPtr, itemPtr);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * DlineCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on dlines.  See the user documentation for details
 *	on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
DlineCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;
    char cx[TCL_DOUBLE_SPACE], cy[TCL_DOUBLE_SPACE];
    int i, numPoints;

    if (argc == 0) {
	double *coordPtr;
	int numCoords;

	numCoords = 2*dlinePtr->numPoints;
	if (dlinePtr->firstArrowPtr != NULL) {
	    coordPtr = dlinePtr->firstArrowPtr;
	} else {
	    coordPtr = dlinePtr->coordPtr;
	}
	for (i = 0; i < numCoords; i++, coordPtr++) {
	    if (i == 2) {
		coordPtr = dlinePtr->coordPtr+2;
	    }
	    if ((dlinePtr->lastArrowPtr != NULL) && (i == (numCoords-2))) {
		coordPtr = dlinePtr->lastArrowPtr;
	    }
	    Tcl_PrintDouble(canvasPtr->interp, *coordPtr, cx);
	    Tcl_AppendElement(canvasPtr->interp, cx);
	}
    } else if ((argc == 1) &&
	       !strcmp(argv[0], "-bbox")) {
      double *coordPtr;
      int numCoords;
      double x1, y1, x2, y2;

      numCoords = 2*dlinePtr->numPoints;
      if (dlinePtr->firstArrowPtr != NULL) {
	coordPtr = dlinePtr->firstArrowPtr;
      } else {
	coordPtr = dlinePtr->coordPtr;
      }
      x1 = x2 = *coordPtr;
      y1 = y2 = coordPtr[1];
      coordPtr = dlinePtr->coordPtr+2;
      
      for (i = 2; i < numCoords; i+=2) {
	if ((dlinePtr->lastArrowPtr != NULL) && (i == (numCoords-2))) {
	  coordPtr = dlinePtr->lastArrowPtr;
	}
	if (*coordPtr < x1) x1 = *coordPtr;
	if (*coordPtr > x2) x2 = *coordPtr;
	coordPtr++;
	if (*coordPtr < y1) y1 = *coordPtr;
	if (*coordPtr > y2) y2 = *coordPtr;
	coordPtr++;
      }
      Tcl_PrintDouble(canvasPtr->interp, x1, cx);
      Tcl_PrintDouble(canvasPtr->interp, y1, cy);
      Tcl_AppendResult(canvasPtr->interp, cx, " ", cy, (char *)0);
      Tcl_PrintDouble(canvasPtr->interp, x2, cx);
      Tcl_PrintDouble(canvasPtr->interp, y2, cy);
      Tcl_AppendResult(canvasPtr->interp, " ", cx, " ", cy, (char *)0);
      return(TCL_OK);
    } else if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp,
		"too few coordinates for dline:  must have at least 4",
		(char *) NULL);
	return TCL_ERROR;
    } else if (argc & 1) {
	Tcl_AppendResult(canvasPtr->interp,
		"odd number of coordinates specified for dline",
		(char *) NULL);
	return TCL_ERROR;
    } else {
	numPoints = argc/2;
	if (dlinePtr->numPoints != numPoints) {
	    if (dlinePtr->coordPtr != NULL) {
		ckfree((char *) dlinePtr->coordPtr);
	    }
	    dlinePtr->coordPtr = (double *) ckalloc((unsigned)
		    (sizeof(double) * argc));
	    dlinePtr->numPoints = numPoints;
	}
	for (i = argc-1; i >= 0; i--) {
	    if (TkGetCanvasCoord(canvasPtr, argv[i], &dlinePtr->coordPtr[i])
		    != TCL_OK) {
		return TCL_ERROR;
	    }
	}

	/*
	 * Update arrowheads by throwing away any existing arrow-head
	 * information and calling ConfigureArrows to recompute it.
	 */

	if (dlinePtr->firstArrowPtr != NULL) {
	    ckfree((char *) dlinePtr->firstArrowPtr);
	    dlinePtr->firstArrowPtr = NULL;
	}
	if (dlinePtr->lastArrowPtr != NULL) {
	    ckfree((char *) dlinePtr->lastArrowPtr);
	    dlinePtr->lastArrowPtr = NULL;
	}
	if (dlinePtr->arrow != noneUid) {
	    ConfigureArrows(canvasPtr, dlinePtr);
	}
	ComputeDlineBbox(canvasPtr, dlinePtr);
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureDline --
 *
 *	This procedure is invoked to configure various aspects
 *	of a dline item such as its background color.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureDline(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Dline item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;

    if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
	    configSpecs, argc, argv, (char *) dlinePtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (dlinePtr->fg == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = dlinePtr->fg->pixel;
	gcValues.join_style = dlinePtr->joinStyle;
	if (dlinePtr->width < 0) {
	    dlinePtr->width = 1;
	}
	gcValues.line_width = dlinePtr->width;
	mask = GCForeground|GCJoinStyle|GCLineWidth;
	if (dlinePtr->fillStipple != None) {
	    gcValues.stipple = dlinePtr->fillStipple;
	    gcValues.fill_style = FillStippled;
	    mask |= GCStipple|GCFillStyle;
	}
	if (dlinePtr->arrow == noneUid) {
	    gcValues.cap_style = dlinePtr->capStyle;
	    mask |= GCCapStyle;
	}
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
      }
    if (dlinePtr->gc != None) {
	Tk_FreeGC(canvasPtr->display, dlinePtr->gc);
    }
    dlinePtr->gc = newGC;

    /*
     * Keep spline parameters within reasonable limits.
     */

    if (dlinePtr->splineSteps < 1) {
	dlinePtr->splineSteps = 1;
    } else if (dlinePtr->splineSteps > 100) {
	dlinePtr->splineSteps = 100;
    }

    /*
     * Setup arrowheads, if needed.  If arrowheads are turned off,
     * restore the dline's endpoints (they were shortened when the
     * arrowheads were added).
     */

    if ((dlinePtr->firstArrowPtr != NULL) && (dlinePtr->arrow != firstUid)
	    && (dlinePtr->arrow != bothUid)) {
	dlinePtr->coordPtr[0] = dlinePtr->firstArrowPtr[0];
	dlinePtr->coordPtr[1] = dlinePtr->firstArrowPtr[1];
	ckfree((char *) dlinePtr->firstArrowPtr);
	dlinePtr->firstArrowPtr = NULL;
    }
    if ((dlinePtr->lastArrowPtr != NULL) && (dlinePtr->arrow != lastUid)
	    && (dlinePtr->arrow != bothUid)) {
	int index;

	index = 2*(dlinePtr->numPoints-1);
	dlinePtr->coordPtr[index] = dlinePtr->lastArrowPtr[0];
	dlinePtr->coordPtr[index+1] = dlinePtr->lastArrowPtr[1];
	ckfree((char *) dlinePtr->lastArrowPtr);
	dlinePtr->lastArrowPtr = NULL;
    }
    if (dlinePtr->arrow != noneUid) {
	if ((dlinePtr->arrow != firstUid) && (dlinePtr->arrow != lastUid)
		&& (dlinePtr->arrow != bothUid)) {
	    Tcl_AppendResult(canvasPtr->interp, "bad arrow spec \"",
		    dlinePtr->arrow, "\": must be none, first, last, or both",
		    (char *) NULL);
	    dlinePtr->arrow = noneUid;
	    return TCL_ERROR;
	}
	ConfigureArrows(canvasPtr, dlinePtr);
    }

    /*
     * Recompute bounding box for dline.
     */

    ComputeDlineBbox(canvasPtr, dlinePtr);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteDline --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a dline item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteDline(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;

    if (dlinePtr->coordPtr != NULL) {
	ckfree((char *) dlinePtr->coordPtr);
    }
    if (dlinePtr->fg != NULL) {
	Tk_FreeColor(dlinePtr->fg);
    }
    if (dlinePtr->fillStipple != None) {
	Tk_FreeBitmap(canvasPtr->display, dlinePtr->fillStipple);
    }
    if (dlinePtr->gc != None) {
	Tk_FreeGC(canvasPtr->display, dlinePtr->gc);
    }
    if (dlinePtr->firstArrowPtr != NULL) {
	ckfree((char *) dlinePtr->firstArrowPtr);
    }
    if (dlinePtr->lastArrowPtr != NULL) {
	ckfree((char *) dlinePtr->lastArrowPtr);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeDlineBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a dline.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeDlineBbox(canvasPtr, dlinePtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    DlineItem *dlinePtr;			/* Item whose bbos is to be
					 * recomputed. */
{
    register double *coordPtr;
    int i;

    coordPtr = dlinePtr->coordPtr;
    dlinePtr->header.x1 = dlinePtr->header.x2 = *coordPtr;
    dlinePtr->header.y1 = dlinePtr->header.y2 = coordPtr[1];

    /*
     * Compute the bounding box of all the points in the dline,
     * then expand in all directions by the dline's width to take
     * care of butting or rounded corners and projecting or
     * rounded caps.  This expansion is an overestimate (worst-case
     * is square root of two over two) but it's simple.  Don't do
     * anything special for curves.  This causes an additional
     * overestimate in the bounding box, but is faster.
     */

    for (i = 1, coordPtr = dlinePtr->coordPtr+2; i < dlinePtr->numPoints;
	    i++, coordPtr += 2) {
	TkIncludePoint(canvasPtr, (Tk_Item *) dlinePtr, coordPtr);
    }
    dlinePtr->header.x1 -= dlinePtr->width;
    dlinePtr->header.x2 += dlinePtr->width;
    dlinePtr->header.y1 -= dlinePtr->width;
    dlinePtr->header.y2 += dlinePtr->width;

    /*
     * For mitered dlines, make a second pass through all the points.
     * Compute the locations of the two miter vertex points and add
     * those into the bounding box.
     */

    if (dlinePtr->joinStyle == JoinMiter) {
	for (i = dlinePtr->numPoints, coordPtr = dlinePtr->coordPtr; i >= 3;
		i--, coordPtr += 2) {
	    double miter[4];
	    int j;
    
	    if (TkGetMiterPoints(coordPtr, coordPtr+2, coordPtr+4,
		    (double) dlinePtr->width, miter, miter+2)) {
		for (j = 0; j < 4; j += 2) {
		    TkIncludePoint(canvasPtr, (Tk_Item *) dlinePtr, miter+j);
		}
	    }
	}
    }

    /*
     * Add in the sizes of arrowheads, if any.
     */

    if (dlinePtr->arrow != noneUid) {
	if (dlinePtr->arrow != lastUid) {
	    for (i = 0, coordPtr = dlinePtr->firstArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(canvasPtr, (Tk_Item *) dlinePtr, coordPtr);
	    }
	}
	if (dlinePtr->arrow != firstUid) {
	    for (i = 0, coordPtr = dlinePtr->lastArrowPtr; i < PTS_IN_ARROW;
		    i++, coordPtr += 2) {
		TkIncludePoint(canvasPtr, (Tk_Item *) dlinePtr, coordPtr);
	    }
	}
    }

    /*
     * Add one more pixel of fudge factor just to be safe (e.g.
     * X may round differently than we do).
     */

    dlinePtr->header.x1 -= 1;
    dlinePtr->header.x2 += 1;
    dlinePtr->header.y1 -= 1;
    dlinePtr->header.y2 += 1;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayDline --
 *
 *	This procedure is invoked to draw a dline item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayDline(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;
    XPoint staticPoints[MAX_STATIC_POINTS];
    XPoint *pointPtr;
    register XPoint *pPtr;
    register double *coordPtr;
    int i, numPoints;
    unsigned long valuemask;
    XGCValues savedLineStyle;

    if (dlinePtr->gc == None) {
	return;
    }

    /*
     * Build up an array of points in screen coordinates.  Use a
     * static array unless the dline has an enormous number of points;
     * in this case, dynamically allocate an array.  For smoothed dlines,
     * generate the curve points on each redisplay.
     */

    if ((dlinePtr->smooth) && (dlinePtr->numPoints > 2)) {
	numPoints = 1 + dlinePtr->numPoints*dlinePtr->splineSteps;
    } else {
	numPoints = dlinePtr->numPoints;
    }

    if (numPoints <= MAX_STATIC_POINTS) {
	pointPtr = staticPoints;
    } else {
	pointPtr = (XPoint *) ckalloc((unsigned) (numPoints * sizeof(XPoint)));
    }

    if (dlinePtr->smooth) {
	numPoints = TkMakeBezierCurve(canvasPtr, dlinePtr->coordPtr,
		dlinePtr->numPoints, dlinePtr->splineSteps, pointPtr,
		(double *) NULL);
    } else {
	for (i = 0, coordPtr = dlinePtr->coordPtr, pPtr = pointPtr;
		i < dlinePtr->numPoints;  i += 1, coordPtr += 2, pPtr++) {
	    pPtr->x = SCREEN_X(canvasPtr, *coordPtr);
	    pPtr->y = SCREEN_Y(canvasPtr, coordPtr[1]);
	}
    }

    /*
     * Display dline, the free up dline storage if it was dynamically
     * allocated.  If we're stippling, then modify the stipple offset
     * in the GC.  Be sure to reset the offset when done, since the
     * GC is supposed to be read-only.
     * If dash style != solid, then modify the GC dash style.
     * Be sure to reset the dash style when done, since the
     * GC is supposed to be read-only. Note that if an XSetDashes had been
     * performed on the GC before this point, it won't be restored.
     */

    if (dlinePtr->dashNumber > 0) {
      XGCValues newLineStyle;

      valuemask = GCLineStyle;
      XGetGCValues(canvasPtr->display, dlinePtr->gc,
                   valuemask, &savedLineStyle);
      newLineStyle.line_style = LineOnOffDash;
      XChangeGC(canvasPtr->display, dlinePtr->gc,
                valuemask, &newLineStyle);
      XSetDashes(canvasPtr->display, dlinePtr->gc,
                 0, dlinePtr->dashList, dlinePtr->dashNumber);
    }

    if (dlinePtr->fillStipple != None) {
	XSetTSOrigin(Tk_Display(canvasPtr->tkwin), dlinePtr->gc,
		-canvasPtr->drawableXOrigin, -canvasPtr->drawableYOrigin);
    }
    XDrawLines(Tk_Display(canvasPtr->tkwin), drawable, dlinePtr->gc,
	    pointPtr, numPoints, CoordModeOrigin);
    if (pointPtr != staticPoints) {
	ckfree((char *) pointPtr);
    }

    /*
     * Display arrowheads, if they are wanted.
     */

    if (dlinePtr->arrow != noneUid) {
	if (dlinePtr->arrow != lastUid) {
	    TkFillPolygon(canvasPtr, dlinePtr->firstArrowPtr, PTS_IN_ARROW,
		    drawable, dlinePtr->gc);
	}
	if (dlinePtr->arrow != firstUid) {
	    TkFillPolygon(canvasPtr, dlinePtr->lastArrowPtr, PTS_IN_ARROW,
		    drawable, dlinePtr->gc);
	}
    }

    if (dlinePtr->dashNumber > 0) {
      XChangeGC(canvasPtr->display, dlinePtr->gc,
                valuemask, &savedLineStyle);
      /* Here's the place for a XSetDashes ( savedDashes ). */
    }            

    if (dlinePtr->fillStipple != None) {
	XSetTSOrigin(Tk_Display(canvasPtr->tkwin), dlinePtr->gc, 0, 0);
    }
}

/*
 *--------------------------------------------------------------
 *
 * DlineToPoint --
 *
 *	Computes the distance from a given point to a given
 *	dline, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the dline.  If the
 *	point isn't inside the dline then the return value is the
 *	distance from the point to the dline.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
DlineToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;
    register double *coordPtr, *dlinePoints;
    double staticSpace[2*MAX_STATIC_POINTS];
    double poly[10];
    double bestDist, dist;
    int numPoints, count;
    int changedMiterToBevel;	/* Non-zero means that a mitered corner
				 * had to be treated as beveled after all
				 * because the angle was < 11 degrees. */

    bestDist = 1.0e40;

    /*
     * Handle smoothed dlines by generating an expanded set of points
     * against which to do the check.
     */

    if ((dlinePtr->smooth) && (dlinePtr->numPoints > 2)) {
	numPoints = 1 + dlinePtr->numPoints*dlinePtr->splineSteps;
	if (numPoints <= MAX_STATIC_POINTS) {
	    dlinePoints = staticSpace;
	} else {
	    dlinePoints = (double *) ckalloc((unsigned)
		    (2*numPoints*sizeof(double)));
	}
	numPoints = TkMakeBezierCurve(canvasPtr, dlinePtr->coordPtr,
		dlinePtr->numPoints, dlinePtr->splineSteps, (XPoint *) NULL,
		dlinePoints);
    } else {
	numPoints = dlinePtr->numPoints;
	dlinePoints = dlinePtr->coordPtr;
    }

    /*
     * The overall idea is to iterate through all of the edges of
     * the dline, computing a polygon for each edge and testing the
     * point against that polygon.  In addition, there are additional
     * tests to deal with rounded joints and caps.
     */

    changedMiterToBevel = 0;
    for (count = numPoints, coordPtr = dlinePoints; count >= 2;
	    count--, coordPtr += 2) {

	/*
	 * If rounding is done around the first point then compute
	 * the distance between the point and the point.
	 */

	if (((dlinePtr->capStyle == CapRound) && (count == numPoints))
		|| ((dlinePtr->joinStyle == JoinRound)
			&& (count != numPoints))) {
	    dist = hypot(coordPtr[0] - pointPtr[0], coordPtr[1] - pointPtr[1])
		    - dlinePtr->width/2.0;
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}

	/*
	 * Compute the polygonal shape corresponding to this edge,
	 * consisting of two points for the first point of the edge
	 * and two points for the last point of the edge.
	 */

	if (count == numPoints) {
	    TkGetButtPoints(coordPtr+2, coordPtr, (double) dlinePtr->width,
		    dlinePtr->capStyle == CapProjecting, poly, poly+2);
	} else if ((dlinePtr->joinStyle == JoinMiter) && !changedMiterToBevel) {
	    poly[0] = poly[6];
	    poly[1] = poly[7];
	    poly[2] = poly[4];
	    poly[3] = poly[5];
	} else {
	    TkGetButtPoints(coordPtr+2, coordPtr, (double) dlinePtr->width, 0,
		    poly, poly+2);

	    /*
	     * If this dline uses beveled joints, then check the distance
	     * to a polygon comprising the last two points of the previous
	     * polygon and the first two from this polygon;  this checks
	     * the wedges that fill the mitered joint.
	     */

	    if ((dlinePtr->joinStyle == JoinBevel) || changedMiterToBevel) {
		poly[8] = poly[0];
		poly[9] = poly[1];
		dist = TkPolygonToPoint(poly, 5, pointPtr);
		if (dist <= 0.0) {
		    bestDist = 0.0;
		    goto done;
		} else if (dist < bestDist) {
		    bestDist = dist;
		}
		changedMiterToBevel = 0;
	    }
	}
	if (count == 2) {
	    TkGetButtPoints(coordPtr, coordPtr+2, (double) dlinePtr->width,
		    dlinePtr->capStyle == CapProjecting, poly+4, poly+6);
	} else if (dlinePtr->joinStyle == JoinMiter) {
	    if (TkGetMiterPoints(coordPtr, coordPtr+2, coordPtr+4,
		    (double) dlinePtr->width, poly+4, poly+6) == 0) {
		changedMiterToBevel = 1;
		TkGetButtPoints(coordPtr, coordPtr+2, (double) dlinePtr->width,
			0, poly+4, poly+6);
	    }
	} else {
	    TkGetButtPoints(coordPtr, coordPtr+2, (double) dlinePtr->width, 0,
		    poly+4, poly+6);
	}
	poly[8] = poly[0];
	poly[9] = poly[1];
	dist = TkPolygonToPoint(poly, 5, pointPtr);
	if (dist <= 0.0) {
	    bestDist = 0.0;
	    goto done;
	} else if (dist < bestDist) {
	    bestDist = dist;
	}
    }

    /*
     * If caps are rounded, check the distance to the cap around the
     * final end point of the dline.
     */

    if (dlinePtr->capStyle == CapRound) {
	dist = hypot(coordPtr[0] - pointPtr[0], coordPtr[1] - pointPtr[1])
		- dlinePtr->width/2.0;
	if (dist <= 0.0) {
	    bestDist = 0.0;
	    goto done;
	} else if (dist < bestDist) {
	    bestDist = dist;
	}
    }

    /*
     * If there are arrowheads, check the distance to the arrowheads.
     */

    if (dlinePtr->arrow != noneUid) {
	if (dlinePtr->arrow != lastUid) {
	    dist = TkPolygonToPoint(dlinePtr->firstArrowPtr, PTS_IN_ARROW,
		    pointPtr);
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}
	if (dlinePtr->arrow != firstUid) {
	    dist = TkPolygonToPoint(dlinePtr->lastArrowPtr, PTS_IN_ARROW,
		    pointPtr);
	    if (dist <= 0.0) {
		bestDist = 0.0;
		goto done;
	    } else if (dist < bestDist) {
		bestDist = dist;
	    }
	}
    }

    done:
    if ((dlinePoints != staticSpace) && (dlinePoints != dlinePtr->coordPtr)) {
	ckfree((char *) dlinePoints);
    }
    return bestDist;
}

/*
 *--------------------------------------------------------------
 *
 * DlineToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the
 *	area, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
DlineToArea(canvasPtr, itemPtr, rectPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against dline. */
    double *rectPtr;
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;
    register double *coordPtr;
    double staticSpace[2*MAX_STATIC_POINTS];
    double *dlinePoints, poly[10];
    double radius;
    int numPoints, count;
    int changedMiterToBevel;	/* Non-zero means that a mitered corner
				 * had to be treated as beveled after all
				 * because the angle was < 11 degrees. */
    int inside;			/* Tentative guess about what to return,
				 * based on all points seen so far:  one
				 * means everything seen so far was
				 * inside the area;  -1 means everything
				 * was outside the area.  0 means overlap
				 * has been found. */ 

    radius = dlinePtr->width/2.0;
    inside = -1;

    /*
     * Handle smoothed dlines by generating an expanded set of points
     * against which to do the check.
     */

    if ((dlinePtr->smooth) && (dlinePtr->numPoints > 2)) {
	numPoints = 1 + dlinePtr->numPoints*dlinePtr->splineSteps;
	if (numPoints <= MAX_STATIC_POINTS) {
	    dlinePoints = staticSpace;
	} else {
	    dlinePoints = (double *) ckalloc((unsigned)
		    (2*numPoints*sizeof(double)));
	}
	numPoints = TkMakeBezierCurve(canvasPtr, dlinePtr->coordPtr,
		dlinePtr->numPoints, dlinePtr->splineSteps, (XPoint *) NULL,
		dlinePoints);
    } else {
	numPoints = dlinePtr->numPoints;
	dlinePoints = dlinePtr->coordPtr;
    }

    coordPtr = dlinePoints;
    if ((coordPtr[0] >= rectPtr[0]) && (coordPtr[0] <= rectPtr[2])
	    && (coordPtr[1] >= rectPtr[1]) && (coordPtr[1] <= rectPtr[3])) {
	inside = 1;
    }

    /*
     * Iterate through all of the edges of the dline, computing a polygon
     * for each edge and testing the area against that polygon.  In
     * addition, there are additional tests to deal with rounded joints
     * and caps.
     */

    changedMiterToBevel = 0;
    for (count = numPoints; count >= 2; count--, coordPtr += 2) {

	/*
	 * If rounding is done around the first point of the edge
	 * then test a circular region around the point with the
	 * area.
	 */

	if (((dlinePtr->capStyle == CapRound) && (count == numPoints))
		|| ((dlinePtr->joinStyle == JoinRound)
		&& (count != numPoints))) {
	    poly[0] = coordPtr[0] - radius;
	    poly[1] = coordPtr[1] - radius;
	    poly[2] = coordPtr[0] + radius;
	    poly[3] = coordPtr[1] + radius;
	    if (TkOvalToArea(poly, rectPtr) != inside) {
		inside = 0;
		goto done;
	    }
	}

	/*
	 * Compute the polygonal shape corresponding to this edge,
	 * consisting of two points for the first point of the edge
	 * and two points for the last point of the edge.
	 */

	if (count == numPoints) {
	    TkGetButtPoints(coordPtr+2, coordPtr, (double) dlinePtr->width,
		    dlinePtr->capStyle == CapProjecting, poly, poly+2);
	} else if ((dlinePtr->joinStyle == JoinMiter) && !changedMiterToBevel) {
	    poly[0] = poly[6];
	    poly[1] = poly[7];
	    poly[2] = poly[4];
	    poly[3] = poly[5];
	} else {
	    TkGetButtPoints(coordPtr+2, coordPtr, (double) dlinePtr->width, 0,
		    poly, poly+2);

	    /*
	     * If the last joint was beveled, then also check a
	     * polygon comprising the last two points of the previous
	     * polygon and the first two from this polygon;  this checks
	     * the wedges that fill the beveled joint.
	     */

	    if ((dlinePtr->joinStyle == JoinBevel) || changedMiterToBevel) {
		poly[8] = poly[0];
		poly[9] = poly[1];
		if (TkPolygonToArea(poly, 5, rectPtr) != inside) {
		    inside = 0;
		    goto done;
		}
		changedMiterToBevel = 0;
	    }
	}
	if (count == 2) {
	    TkGetButtPoints(coordPtr, coordPtr+2, (double) dlinePtr->width,
		    dlinePtr->capStyle == CapProjecting, poly+4, poly+6);
	} else if (dlinePtr->joinStyle == JoinMiter) {
	    if (TkGetMiterPoints(coordPtr, coordPtr+2, coordPtr+4,
		    (double) dlinePtr->width, poly+4, poly+6) == 0) {
		changedMiterToBevel = 1;
		TkGetButtPoints(coordPtr, coordPtr+2, (double) dlinePtr->width,
			0, poly+4, poly+6);
	    }
	} else {
	    TkGetButtPoints(coordPtr, coordPtr+2, (double) dlinePtr->width, 0,
		    poly+4, poly+6);
	}
	poly[8] = poly[0];
	poly[9] = poly[1];
	if (TkPolygonToArea(poly, 5, rectPtr) != inside) {
	    inside = 0;
	    goto done;
	}
    }

    /*
     * If caps are rounded, check the cap around the final point
     * of the dline.
     */

    if (dlinePtr->capStyle == CapRound) {
	poly[0] = coordPtr[0] - radius;
	poly[1] = coordPtr[1] - radius;
	poly[2] = coordPtr[0] + radius;
	poly[3] = coordPtr[1] + radius;
	if (TkOvalToArea(poly, rectPtr) != inside) {
	    inside = 0;
	    goto done;
	}
    }

    /*
     * Check arrowheads, if any.
     */

    if (dlinePtr->arrow != noneUid) {
	if (dlinePtr->arrow != lastUid) {
	    if (TkPolygonToArea(dlinePtr->firstArrowPtr, PTS_IN_ARROW,
		    rectPtr) != inside) {
		inside = 0;
		goto done;
	    }
	}
	if (dlinePtr->arrow != firstUid) {
	    if (TkPolygonToArea(dlinePtr->lastArrowPtr, PTS_IN_ARROW,
		    rectPtr) != inside) {
		inside = 0;
		goto done;
	    }
	}
    }

    done:
    if ((dlinePoints != staticSpace) && (dlinePoints != dlinePtr->coordPtr)) {
	ckfree((char *) dlinePoints);
    }
    return inside;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleDline --
 *
 *	This procedure is invoked to rescale a dline item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The dline referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point
 *	coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScaleDline(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing dline. */
    Tk_Item *itemPtr;			/* Dline to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    DlineItem *dlinePtr = (DlineItem *) itemPtr;
    register double *coordPtr;
    int i;

    for (i = 0, coordPtr = dlinePtr->coordPtr; i < dlinePtr->numPoints;
	    i++, coordPtr += 2) {
	coordPtr[0] = originX + scaleX*(*coordPtr - originX);
	coordPtr[1] = originY + scaleY*(coordPtr[1] - originY);
    }
    if (dlinePtr->firstArrowPtr != NULL) {
	for (i = 0, coordPtr = dlinePtr->firstArrowPtr; i < PTS_IN_ARROW;
		i++, coordPtr += 2) {
	    coordPtr[0] = originX + scaleX*(coordPtr[0] - originX);
	    coordPtr[1] = originY + scaleY*(coordPtr[1] - originY);
	}
    }
    if (dlinePtr->lastArrowPtr != NULL) {
	for (i = 0, coordPtr = dlinePtr->lastArrowPtr; i < PTS_IN_ARROW;
		i++, coordPtr += 2) {
	    coordPtr[0] = originX + scaleX*(coordPtr[0] - originX);
	    coordPtr[1] = originY + scaleY*(coordPtr[1] - originY);
	}
    }
    ComputeDlineBbox(canvasPtr, dlinePtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateDline --
 *
 *	This procedure is called to move a dline by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the dline is offset by (xDelta, yDelta), and
 *	the bounding box is updated in the generic part of the item
 *	structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateDline(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    DlineItem *dlinePtr = (DlineItem *) itemPtr;
    register double *coordPtr;
    int i;

    for (i = 0, coordPtr = dlinePtr->coordPtr; i < dlinePtr->numPoints;
	    i++, coordPtr += 2) {
	coordPtr[0] += deltaX;
	coordPtr[1] += deltaY;
    }
    if (dlinePtr->firstArrowPtr != NULL) {
	for (i = 0, coordPtr = dlinePtr->firstArrowPtr; i < PTS_IN_ARROW;
		i++, coordPtr += 2) {
	    coordPtr[0] += deltaX;
	    coordPtr[1] += deltaY;
	}
    }
    if (dlinePtr->lastArrowPtr != NULL) {
	for (i = 0, coordPtr = dlinePtr->lastArrowPtr; i < PTS_IN_ARROW;
		i++, coordPtr += 2) {
	    coordPtr[0] += deltaX;
	    coordPtr[1] += deltaY;
	}
    }
    ComputeDlineBbox(canvasPtr, dlinePtr);
}

/*
 *--------------------------------------------------------------
 *
 * ParseArrowShape --
 *
 *	This procedure is called back during option parsing to
 *	parse arrow shape information.
 *
 * Results:
 *	The return value is a standard Tcl result:  TCL_OK means
 *	that the arrow shape information was parsed ok, and
 *	TCL_ERROR means it couldn't be parsed.
 *
 * Side effects:
 *	Arrow information in recordPtr is updated.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ParseArrowShape(clientData, interp, tkwin, value, recordPtr, offset)
    ClientData clientData;	/* Not used. */
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Window tkwin;		/* Not used. */
    char *value;		/* Textual specification of arrow shape. */
    char *recordPtr;		/* Pointer to item record in which to
				 * store arrow information. */
    int offset;			/* Offset of shape information in widget
				 * record. */
{
    DlineItem *dlinePtr = (DlineItem *) recordPtr;
    double a, b, c;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(DlineItem, arrowShapeA)) {
	panic("ParseArrowShape received bogus offset");
    }

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad arrow shape \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    if ((TkGetCanvasCoord(dlinePtr->canvasPtr, argv[0], &a) != TCL_OK)
	    || (TkGetCanvasCoord(dlinePtr->canvasPtr, argv[1], &b) != TCL_OK)
	    || (TkGetCanvasCoord(dlinePtr->canvasPtr, argv[2], &c) != TCL_OK)) {
	goto syntaxError;
    }
    dlinePtr->arrowShapeA = a;
    dlinePtr->arrowShapeB = b;
    dlinePtr->arrowShapeC = c;
    ckfree((char *) argv);
    return TCL_OK;
}

/****  Same as previous for dashes option  ****/

static int
ParseDashes(clientData, interp, tkwin, value, recordPtr, offset)
    ClientData clientData;	/* Not used. */
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Window tkwin;		/* Not used. */
    char *value;		/* Textual specification of arrow shape. */
    char *recordPtr;		/* Pointer to item record in which to
				 * store arrow information. */
    int offset;			/* Offset of shape information in widget
				 * record. */
{
    DlineItem *dlinePtr = (DlineItem *) recordPtr;
    int argc;
    char **argv = NULL;
    
    if (offset != Tk_Offset(DlineItem, dashList)) {
	panic("ParseDashes received bogus offset");
    }

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad dash list \"", value,
		"\": must be even length list of numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc%2 != 0) {
	goto syntaxError;
    }

    if (dlinePtr->dashList != (char *)0)
      free(dlinePtr->dashList);
    dlinePtr->dashList = (char *)0;
    dlinePtr->dashNumber = argc;

    if (argc > 0) {
      /**  fill in the dashList array  **/
      int ibuf, i;
      dlinePtr->dashList = (char *)malloc(argc * sizeof(char));
      for (i = 0; i < argc; i++) {
	if (Tcl_GetInt(interp, argv[i], &ibuf) != TCL_OK) {
	  free(dlinePtr->dashList);
	  dlinePtr->dashList = (char *)0;
	  goto syntaxError;
	}
	dlinePtr->dashList[i] = (char)ibuf;
      }
    }

    ckfree((char *) argv);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PrintArrowShape --
 *
 *	This procedure is a callback invoked by the configuration
 *	code to return a printable value describing an arrow shape.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

    /* ARGSUSED */
static char *
PrintArrowShape(clientData, tkwin, recordPtr, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Window associated with dlinePtr's widget. */
    char *recordPtr;		/* Pointer to item record containing current
				 * shape information. */
    int offset;			/* Offset of arrow information in record. */
    Tcl_FreeProc **freeProcPtr;	/* Store address of procedure to call to
				 * free string here. */
{
    DlineItem *dlinePtr = (DlineItem *) recordPtr;
    char *buffer;

    buffer = ckalloc(120);
    sprintf(buffer, "%.5g %.5g %.5g", dlinePtr->arrowShapeA,
	    dlinePtr->arrowShapeB, dlinePtr->arrowShapeC);
    *freeProcPtr = (Tcl_FreeProc *) free;
    return buffer;
}


/****  Same as above for dashList  ****/

static char *
PrintDashes(clientData, tkwin, recordPtr, offset, freeProcPtr)
    ClientData clientData;	/* Not used. */
    Tk_Window tkwin;		/* Window associated with dlinePtr's widget. */
    char *recordPtr;		/* Pointer to item record containing current
				 * shape information. */
    int offset;			/* Offset of arrow information in record. */
    Tcl_FreeProc **freeProcPtr;	/* Store address of procedure to call to
				 * free string here. */
{
    DlineItem *dlinePtr = (DlineItem *) recordPtr;
    char *buffer;

    if (dlinePtr->dashNumber == 0) {
      buffer = ckalloc(1);
      *buffer = '\0';
    } else {
      int i;
      
      buffer = ckalloc((4*dlinePtr->dashNumber+1) * sizeof(char));
      sprintf(buffer, "%d", dlinePtr->dashList[0]);
      for (i = 1; i < dlinePtr->dashNumber; i++)
	sprintf(buffer + strlen(buffer), " %d", dlinePtr->dashList[i]);
    }
    *freeProcPtr = (Tcl_FreeProc *) free;
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureArrows --
 *
 *	If arrowheads have been requested for a dline, this
 *	procedure makes arrangements for the arrowheads.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Information in dlinePtr is set up for one or two arrowheads.
 *	the firstArrowPtr and lastArrowPtr polygons are allocated
 *	and initialized, if need be, and the end points of the dline
 *	are adjusted so that a thick dline doesn't stick out past
 *	the arrowheads.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ConfigureArrows(canvasPtr, dlinePtr)
    Tk_Canvas *canvasPtr;		/* Canvas in which arrows will be
					 * displayed (interp and tkwin
					 * fields are needed). */
    register DlineItem *dlinePtr;		/* Item to configure for arrows. */
{
    double *poly, *coordPtr;
    double dx, dy, length, sinTheta, cosTheta, temp, shapeC;
    double fracHeight;			/* Dline width as fraction of
					 * arrowhead width. */
    double backup;			/* Distance to backup end points
					 * so the dline ends in the middle
					 * of the arrowhead. */
    double vertX, vertY;		/* Position of arrowhead vertex. */

    /*
     * If there's an arrowhead on the first point of the dline, compute
     * its polygon and adjust the first point of the dline so that the
     * dline doesn't stick out past the leading edge of the arrowhead.
     */

    shapeC = dlinePtr->arrowShapeC + dlinePtr->width/2.0;
    fracHeight = (dlinePtr->width/2.0)/shapeC;
    backup = fracHeight*dlinePtr->arrowShapeB
	    + dlinePtr->arrowShapeA*(1.0 - fracHeight)/2.0;
    if (dlinePtr->arrow != lastUid) {
	poly = dlinePtr->firstArrowPtr;
	if (poly == NULL) {
	    poly = (double *) ckalloc((unsigned)
		    (2*PTS_IN_ARROW*sizeof(double)));
	    poly[0] = poly[10] = dlinePtr->coordPtr[0];
	    poly[1] = poly[11] = dlinePtr->coordPtr[1];
	    dlinePtr->firstArrowPtr = poly;
	}
	dx = poly[0] - dlinePtr->coordPtr[2];
	dy = poly[1] - dlinePtr->coordPtr[3];
	length = hypot(dx, dy);
	if (length == 0) {
	    sinTheta = cosTheta = 0.0;
	} else {
	    sinTheta = dy/length;
	    cosTheta = dx/length;
	}
	vertX = poly[0] - dlinePtr->arrowShapeA*cosTheta;
	vertY = poly[1] - dlinePtr->arrowShapeA*sinTheta;
	temp = shapeC*sinTheta;
	poly[2] = poly[0] - dlinePtr->arrowShapeB*cosTheta + temp;
	poly[8] = poly[2] - 2*temp;
	temp = shapeC*cosTheta;
	poly[3] = poly[1] - dlinePtr->arrowShapeB*sinTheta - temp;
	poly[9] = poly[3] + 2*temp;
	poly[4] = poly[2]*fracHeight + vertX*(1.0-fracHeight);
	poly[5] = poly[3]*fracHeight + vertY*(1.0-fracHeight);
	poly[6] = poly[8]*fracHeight + vertX*(1.0-fracHeight);
	poly[7] = poly[9]*fracHeight + vertY*(1.0-fracHeight);

	/*
	 * Polygon done.  Now move the first point towards the second so
	 * that the corners at the end of the dline are inside the
	 * arrowhead.
	 */

	dlinePtr->coordPtr[0] = poly[0] - backup*cosTheta;
	dlinePtr->coordPtr[1] = poly[1] - backup*sinTheta;
    }

    /*
     * Similar arrowhead calculation for the last point of the dline.
     */

    if (dlinePtr->arrow != firstUid) {
	coordPtr = dlinePtr->coordPtr + 2*(dlinePtr->numPoints-2);
	poly = dlinePtr->lastArrowPtr;
	if (poly == NULL) {
	    poly = (double *) ckalloc((unsigned)
		    (2*PTS_IN_ARROW*sizeof(double)));
	    poly[0] = poly[10] = coordPtr[2];
	    poly[1] = poly[11] = coordPtr[3];
	    dlinePtr->lastArrowPtr = poly;
	}
	dx = poly[0] - coordPtr[0];
	dy = poly[1] - coordPtr[1];
	length = hypot(dx, dy);
	if (length == 0) {
	    sinTheta = cosTheta = 0.0;
	} else {
	    sinTheta = dy/length;
	    cosTheta = dx/length;
	}
	vertX = poly[0] - dlinePtr->arrowShapeA*cosTheta;
	vertY = poly[1] - dlinePtr->arrowShapeA*sinTheta;
	temp = shapeC*sinTheta;
	poly[2] = poly[0] - dlinePtr->arrowShapeB*cosTheta + temp;
	poly[8] = poly[2] - 2*temp;
	temp = shapeC*cosTheta;
	poly[3] = poly[1] - dlinePtr->arrowShapeB*sinTheta - temp;
	poly[9] = poly[3] + 2*temp;
	poly[4] = poly[2]*fracHeight + vertX*(1.0-fracHeight);
	poly[5] = poly[3]*fracHeight + vertY*(1.0-fracHeight);
	poly[6] = poly[8]*fracHeight + vertX*(1.0-fracHeight);
	poly[7] = poly[9]*fracHeight + vertY*(1.0-fracHeight);
	coordPtr[2] = poly[0] - backup*cosTheta;
	coordPtr[3] = poly[1] - backup*sinTheta;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DlineToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	dline items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
DlineToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
    register DlineItem *dlinePtr = (DlineItem *) itemPtr;
    char buffer[200];
    char *style;

    if (dlinePtr->fg == NULL) {
	return TCL_OK;
    }

    /*
     * Generate a path for the dline's center-dline (do this differently
     * for straight dlines and smoothed dlines).
     */

    if (!dlinePtr->smooth) {
	TkCanvPsPath(canvasPtr->interp, dlinePtr->coordPtr, dlinePtr->numPoints,
		psInfoPtr);
    } else {
	if (dlinePtr->fillStipple == None) {
	    TkMakeBezierPostscript(canvasPtr->interp, dlinePtr->coordPtr,
		    dlinePtr->numPoints, psInfoPtr);
	} else {
	    /*
	     * Special hack: Postscript printers don't appear to be able
	     * to turn a path drawn with "curveto"s into a clipping path
	     * without exceeding resource limits, so TkMakeBezierPostscript
	     * won't work for stippled curves.  Instead, generate all of
	     * the intermediate points here and output them into the
	     * Postscript file with "dlineto"s instead.
	     */

	    double staticPoints[2*MAX_STATIC_POINTS];
	    double *pointPtr;
	    int numPoints;

	    numPoints = 1 + dlinePtr->numPoints*dlinePtr->splineSteps;
	    pointPtr = staticPoints;
	    if (numPoints > MAX_STATIC_POINTS) {
		pointPtr = (double *) ckalloc((unsigned)
			(numPoints * 2 * sizeof(double)));
	    }
	    numPoints = TkMakeBezierCurve(canvasPtr, dlinePtr->coordPtr,
		    dlinePtr->numPoints, dlinePtr->splineSteps, (XPoint *) NULL,
		    pointPtr);
	    TkCanvPsPath(canvasPtr->interp, pointPtr, numPoints, psInfoPtr);
	    if (pointPtr != staticPoints) {
		ckfree((char *) pointPtr);
	    }
	}
    }

    /*
     * Set other dline-drawing parameters and stroke out the dline.
     */

    sprintf(buffer, "%d setlinewidth\n", dlinePtr->width);
    Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
    style = "0 setlinecap\n";
    if (dlinePtr->capStyle == CapRound) {
	style = "1 setlinecap\n";
    } else if (dlinePtr->capStyle == CapProjecting) {
	style = "2 setlinecap\n";
    }
    Tcl_AppendResult(canvasPtr->interp, style, (char *) NULL);
    style = "0 setlinejoin\n";
    if (dlinePtr->joinStyle == JoinRound) {
	style = "1 setlinejoin\n";
    } else if (dlinePtr->joinStyle == JoinBevel) {
	style = "2 setlinejoin\n";
    }
    Tcl_AppendResult(canvasPtr->interp, style, (char *) NULL);
    if (TkCanvPsColor(canvasPtr, psInfoPtr, dlinePtr->fg) != TCL_OK) {
	return TCL_ERROR;
    };
    if (dlinePtr->dashNumber > 0) {
      int i;
      sprintf(buffer, "[%d", dlinePtr->dashList[0]);
      for (i = 1; i < dlinePtr->dashNumber; i++)
	sprintf(buffer + strlen(buffer), " %d", dlinePtr->dashList[i]);
      strcat(buffer, "] 0 setdash\n");
      Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
    }	     
    if (dlinePtr->fillStipple != None) {
	if (TkCanvPsStipple(canvasPtr, psInfoPtr, dlinePtr->fillStipple, 0)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
    }

    /*
     * Output polygons for the arrowheads, if there are any.
     */

    if (dlinePtr->firstArrowPtr != NULL) {
	if (ArrowheadPostscript(canvasPtr, dlinePtr, dlinePtr->firstArrowPtr,
		psInfoPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    if (dlinePtr->lastArrowPtr != NULL) {
	if (ArrowheadPostscript(canvasPtr, dlinePtr, dlinePtr->lastArrowPtr,
		psInfoPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ArrowheadPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	an arrowhead for a dline item.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	arrowhead is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ArrowheadPostscript(canvasPtr, dlinePtr, arrowPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    DlineItem *dlinePtr;			/* Dline item for which Postscript is
					 * being generated. */
    double *arrowPtr;			/* Pointer to first of five points
					 * describing arrowhead polygon. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
    TkCanvPsPath(canvasPtr->interp, arrowPtr, PTS_IN_ARROW, psInfoPtr);
    if (dlinePtr->fillStipple != None) {
	if (TkCanvPsStipple(canvasPtr, psInfoPtr, dlinePtr->fillStipple, 1)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
    }
    return TCL_OK;
}
