#
# $Id: form.tcl,v 1.8 1995/06/27 10:27:09 sls Exp $
#
# This software is copyright (C) 1995 by the Lawrence Berkeley Laboratory.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that: (1) source code distributions
# retain the above copyright notice and this paragraph in its entirety, (2)
# distributions including binary code include the above copyright notice and
# this paragraph in its entirety in the documentation or other materials
# provided with the distribution, and (3) all advertising materials mentioning
# features or use of this software display the following acknowledgement:
# ``This product includes software developed by the University of California,
# Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
# the University nor the names of its contributors may be used to endorse
# or promote products derived from this software without specific prior
# written permission.
# 
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
# form support
#
# form_priv(name) -- name of current form
#

document_title form "deal with HTML+ forms" {

    These procedures deal with HTML forms, and are layered over
    the HTML primitives in #html.tcl#.  Each of the widget commands
    understands a set of flags, these flags correspond to the HTML
    widget options.  For example, #text -value "Hi There!"# will
    output #<INPUT TYPE=text VALUE="Hi There!">#.

    For more information about HTML forms, see
    http://www.w3.org/hypertext/WWW/MarkUp/html-spec/html-spec_8.html.

}

document_proc form_begin {
    begins a new form.  Understand the #-action#, #-method#, and
    #-enctype# flags.  `name' names the form, see #form_end# for its
    meaning.
}
proc form_begin {name args} {
    global form_priv
    msg "form_begin $name"
    args $args {-action -method -enctype}
    set txt "<FORM"
    ifexists action { append txt " ACTION=$action" }
    ifexists enctype { append txt " ENCTYPE=$enctype" }
    ifexists method { append txt " METHOD=$method" }
    append txt ">"
    catch {unset form_priv}
    set form_priv(name) $name
    html $txt
}

document_proc form_listvar {
    declares that each of the form variables in `args' should be
    accumulated in a list when the form is processed.
}
proc form_listvar args {
    global form_priv
    foreach arg $args {
	lappend form_priv(listvars) $arg
    }
}

document_proc form_end {

    ends a form.  If `body' is present then write out a form handler
    to the file `name' (where `name' is the name of the form.  The
    form handler is composed of "cgi_post_hdr.tcl", followed by
    `body', followed by "cgi_post_trl.tcl".  #form_end# searches for
    the .tcl files in $#sntl_library# if it exists, otherwise in the
    current directory.

}
proc form_end {{body "*no-body*"}} {
    global form_priv
    msg [list form_end $form_priv(name)]
    html "</FORM>"
    if [string compare $body "*no-body*"] {
	form_write_handler $form_priv(name) $body
    }
}

# look for a library file in . or in $stl_library
proc form_find_file file {
    global stl_library
    if [file exists $file] {
	return $file
    }
    if [info exists stl_library] {
	if [file exists $stl_library/$file] {
	    return $stl_library/$file
	}
    }
    return $file
}

proc form_write_handler {name body} {
    global form_priv
    set fp [open $name w]
    puts $fp [read_file [form_find_file "cgi_post_hdr.tcl"]]
    if [info exists form_priv(listvars)] {
	foreach var $form_priv(listvars) {
	    puts $fp [list set artype($var) list]
	}
    }
    puts $fp "cgi_read \$cgi_debug"
    puts $fp [list set form_priv(handler_name) $name]
    puts $fp $body
    puts $fp [read_file [form_find_file "cgi_post_trl.tcl"]]
    close $fp
    exec chmod +rx $name
}

#
# INPUT tag types
#
document_proc form_text {
    prints the HTML that creates a text entry widget.  Understands
    the #-value#, #-name#, #-maxlength#, and #-size# flags.
}
proc form_text {args} {
    args $args {-value -name -size -maxlength}
    set txt "<INPUT TYPE=\"text\""
    ifexists value { append txt " VALUE=\"$value\"" }
    ifexists name { append txt " NAME=\"$name\"" }
    ifexists size { append txt " SIZE=\"$size\"" }
    ifexists maxlength { append txt " MAXLENGTH=\"$maxlength\"" }
    append txt ">"
    html $txt
}

document_proc form_hidden {
    prints the HTML that creates a hidden widget.  Understands the #-value#,
    and #-name# flags.  #hidden# properly quotes the value, using
    #cgi_hex_quote#.
}
proc form_hidden {args} {
    args $args {-name -value}
    set txt "<INPUT TYPE=\"hidden\" SIZE=1"
    ifexists value { append txt " VALUE=\"[cgi_hex_quote $value]\"" }
    ifexists name { append txt " NAME=\"$name\"" }
    append txt ">"
    html $txt
}

document_proc form_password {
    prints the HTML that creates a password widget.  Understands the
    #-value#, #-name#, #-size#, and #-maxlength# flags.
}
proc form_password {args} {
    args $args {-value -name -size -maxlength}
    set txt "<INPUT TYPE=\"password\""
    ifexists value { append txt " VALUE=\"$value\"" }
    ifexists name { append txt " NAME=\"$name\"" }
    ifexists size { append txt " SIZE=\"$size\"" }
    ifexists maxlength { append txt " MAXLENGTH=\"$maxlength\"" }
    append txt ">"
    html $txt
}

document_proc form_checkbox {
    prints the HTML that creates a checkbox widget.  Understands the
    #-value#, #-name#, and #-checked# flags.
}
proc form_checkbox {args} {
    args $args {-name -checked -value}
    set txt "<INPUT TYPE=\"checkbox\""
    ifexists name { append txt " NAME=\"$name\"" }
    ifexists value { append txt " VALUE=\"$value\"" }
    ifexists checked { append txt " CHECKED" }
    append txt ">"
    html $txt
}

document_proc form_radio {
    prints the HTML that creates a radio button widget.  Understands the
    #-value#, #-name#, and #-checked# flags.
}
proc form_radio {args} {
    args $args {-name -checked -value}
    set txt "<INPUT TYPE=\"radio\""
    ifexists name { append txt " NAME=\"$name\"" }
    ifexists value { append txt " VALUE=\"$value\"" }
    ifexists checked { append txt " CHECKED" }
    append txt ">"
    html $txt
}

document_proc form_submit {
    prints the HTML that creates a submit button.  Understands the
    #-value# flag.
}
proc form_submit {args} {
    args $args -value
    set txt "<INPUT TYPE=\"submit\""
    ifexists value { append txt " VALUE=\"$value\"" }
    append txt ">"
    html $txt
}

document_proc form_reset {
    prints the HTML that creates a reset button.  Understands the
    #-value# flag.
}
proc form_reset {args} {
    args $args -value
    set txt "<INPUT TYPE=\"reset\""
    ifexists value { append txt " VALUE=\"$value\"" }
    append txt ">"
    html $txt
}

#
# SELECT box
#
document_proc form_select_begin {
    prints the HTML that starts a selection widget.  Understands the #-name#,
    #-size#, and #-multiple# options.
}
proc form_select_begin {args} {
    args $args {-name -size -multiple}
    set txt "<SELECT"
    ifexists name { append txt " NAME=\"$name\"" }
    ifexists size { append txt " SIZE=\"$size\"" }
    ifexists multiple { append txt " MULTIPLE" }
    append txt ">"
    html $txt
}

document_proc form_option {
    prints the HTML that creates an option in a selection widget.
    Understands the #-selected# and #-value# flags.
}
proc form_option {args} {
    args $args {{-selected 0} -value}
    set txt "<OPTION"
    ifexists selected { append txt " SELECTED" }
    ifexists value { append txt " VALUE=$value" }
    append txt ">"
    html $txt
}

document_proc form_select_end {
    prints the HTML that ends a selection widget.
}
proc form_select_end {args} {
    args $args {}
    html "</SELECT>"
}

#
# TEXTAREA
#
document_proc form_textarea_begin {
    prints the HTML that starts a text area widget.  Understands the #-name#,
    #-rows#, and #-cols# options.
}
proc form_textarea_begin {args} {
    args $args {-name -rows -cols}
    set txt "<TEXTAREA"
    ifexists name { append txt " NAME=\"$name\"" }
    ifexists rows { append txt " ROWS=\"$rows\"" }
    ifexists cols { append txt " COLS=\"$cols\"" }
    append txt ">"
    html $txt
}

document_proc form_textarea_end {
    prints the HTML that ends a text area widget.
}
proc form_textarea_end {args} {
    args $args {}
    html "</TEXTAREA>"
}

#
# procs for form-handlers
#

document_proc cgi_read {
    reads variables via CGI.  #cgi_read# reads from stdin if #REQUEST_METHOD#
    is #POST#, otherwise it extracts them from #QUERY_STRING#.  It expects
    input to be a sequence of assigments separated by &'s.  Assignments are
    of the form `variable'=`value'.  Variables will be stored in the global
    array #ar#.  If the global element #artype#(`variable') exists and is
    equal to the string #list#, then multiple instances of `variable' will
    be #lappend#-ed onto #ar#(`variable').
}
proc cgi_read {{debug 0}} {
    global env ar artype
    set method [string tolower $env(REQUEST_METHOD)]
    if {$method == "post"} {
	set txt [read stdin $env(CONTENT_LENGTH)]
    } else {
	set txt $env(QUERY_STRING)
    }
    foreach assignment [split $txt &] {
	set assignment [split $assignment =]
	set var [lindex $assignment 0]
	set val [cgi_hex_unquote [lindex $assignment 1]]
	if $debug {
	    puts "<CODE> [list $var = $val] </CODE><P>"
	}
	if {[info exists artype($var)] && $artype($var) == "list"} {
	    lappend ar($var) $val
	} else {
	    set ar($var) $val
	}
    }
}

document_proc cgi_hex_quote {
    quotes `txt' for inclusion in a URL and returns the quoted text.
    The quoting mechanism is to replace special characters with
    %`xx', where `xx' is the ASCII code of the character.
    Currently &, %, double-quote, <, >, newline, and tab are quoted.
}
proc cgi_hex_quote {txt} {
    regsub -all "%" $txt "%25" txt
    regsub -all "\"" $txt "%22" txt
    regsub -all "<" $txt "%3C" txt
    regsub -all ">" $txt "%3E" txt
    regsub -all "\n" $txt "%0A" txt
    regsub -all "\t" $txt "%09" txt
    regsub -all "&" $txt "%26" txt
    return $txt
}

document_proc cgi_hex_unquote {
    unquotes `txt'.
}
proc cgi_hex_unquote {txt} {
    regsub -all "\\+" $txt " " txt
    while {[regexp -nocase "%\[0-9A-F]\[0-9A-F]" $txt match]} {
	scan $match "%%%x" n
	set ch [format "%c" $n]
	if {![string compare $ch "&"]} {
	    set ch "\\&"
	}
	regsub -all $match $txt $ch txt
    }
    return $txt
}
