# tixBalloon -
#
#	The help widget. It provides both "balloon" type of help message
# and "status bar" type of help message. You can use this widget to indicate
# the function of the widgets inside your application.
#
#
tixWidgetClass tixBalloon {
    -classname TixBalloon
    -superclass tixShell
    -method {
	bind post unbind
    }
    -flag {
	-installcolormap -initwait -state -statusbar
    }
    -configspec {
	{-installcolormap installColormap InstallColormap false}
	{-initwait initWait InitWait 200}
	{-state state State both}
	{-statusbar statusBar StatusBar {}}

 	{-cursor cursor Cursur left_ptr}
    }
    -default {
	{*background 			#ffff60}
	{*foreground 			black}
	{*borderWidth 			0}
	{.borderWidth 			1}
	{.background 			black}
    }
}


proc tixBalloon::InitWidgetRec {w} {
    upvar #0 $w data

    tixChainMethod $w InitWidgetRec

    set data(popped)    0
    set data(fakeLeave) 0
    set data(statusSet) 0
    set data(serial)    0
    set data(fakeEnter) 0
    set data(curWidget) {}
}

proc tixBalloon::ConstructWidget {w} {
    upvar #0 $w data

    tixChainMethod $w ConstructWidget

    wm overrideredirect $w 1
    wm withdraw $w

    # Frame 1 : arrow
    frame $w.f1 -bd 0
    set data(w:label) [label $w.f1.lab -bd 0 -relief flat \
		       -bitmap [tix getbitmap balArrow]]
    pack $data(w:label) -side left -padx 1 -pady 1
    
    # Frame 2 : Message
    frame $w.f2 -bd 0
    set data(w:message) [message $w.f2.message -padx 0 -pady 0 -bd 0]
    pack $data(w:message) -side left -expand yes -fill both -padx 10 -pady 1

    # Pack all
    pack $w.f1 -fill both
    pack $w.f2 -fill both    
}

bind TixBalloon <Leave> "tixBalloon::Leave %W %X %Y"
bind TixBalloon <Visibility> "raise %W"

#----------------------------------------------------------------------
# Config:
#----------------------------------------------------------------------

proc tixBalloon::config-state {w value} {
    upvar #0 $w data

    case $value {
	{none balloon status both} {}
	default {
	   error "invalid value $value, must be none, balloon, status, or both"
	}
    }
}

#----------------------------------------------------------------------
# PrivateMethods:
#----------------------------------------------------------------------

proc tixBalloon::ClientDestroy {w client} {
    upvar #0 $w data

    if {$data(curWidget) == $client} {
	tixBalloon::Popdown $w
    }

    # Maybe thses have already been unset by the Destroy method
    #
    catch {unset data(m:$client)}
    catch {unset data(s:$client)}
}

# Handle the mouse pointer entering the client widget
#
proc tixBalloon::Enter {w client} {
    upvar #0 $w data

    if {$data(fakeEnter) > 0} {
	# The mouse pointer just left either the balloon window or the
	# client window: do nothing; otherwise the balloon will flash
	#
	set data(fakeEnter) 0
	return
    }
    if {$data(-state) != "none"} {
    	set data(popped)    0
    	set data(statusSet) 0
	set data(curWidget) $client
	incr data(serial)
    	after $data(-initwait) tixBalloon::Activate $w $data(serial)
    }
}

proc tixBalloon::post {w client} {
    upvar #0 $w data

    if {![info exists data(m:$client)]} {
	return
    }
    tixBalloon::Enter $w $client
    incr data(fakeEnter)
}

proc tixBalloon::Within {wid rootX rootY} {
    set rx1 [winfo rootx $wid]
    set ry1 [winfo rooty $wid]
    set rw  [winfo width  $wid]
    set rh  [winfo height $wid]
    set rx2 [expr $rx1+$rw]
    set ry2 [expr $ry1+$rh]

    if {$rootX >= $rx1 && $rootX < $rx2 && $rootY >= $ry1 && $rootY < $ry2} {
	return 1
    } else {
	return 0
    }
}

proc tixBalloon::Leave {w rootX rootY} {
    upvar #0 $w data

    if {$data(curWidget) == ""} {
	return
    }
    if {$data(fakeLeave) == 1} {
	set data(fakeLeave) 0
	return
    }

    set cw [winfo containing $rootX $rootY]
    set mask [tixBalloon::GetMask $w $data(curWidget)]

    if [tixBalloon::Within $w $rootX $rootY] {
	# It is safe to do this because we know the balloon is always on top
	#
	set data(fakeEnter) 1
	return
    }
    if {$cw == $mask} {
	set data(fakeEnter) 1
	return
    }

    if {$data(popped) == 1 || $data(statusSet) == 1} {
	set data(fakeEnter) 0
	tixBalloon::Popdown $w
    } else {
	# have to make sure that previous popup's are cancelled
	# just make sure previous
	#
	incr data(serial)
    }
}

proc tixBalloon::Activate {w serial} {

    if {![winfo exists $w]} {
	return
    }
    upvar #0 $w data

    if {![winfo exists $data(curWidget)]} {
	return
    }
    
    if {$serial != $data(serial)} {
	# a new balloon will be activated by the latest call
	#
	return	
    }

    set mask [tixBalloon::GetMask $w $data(curWidget)]

    set tp [winfo toplevel $data(curWidget)]
    set x [expr [winfo rootx $data(curWidget)]-[winfo rootx $tp]]
    set y [expr [winfo rooty $data(curWidget)]-[winfo rooty $tp]]
    set W [winfo width  $data(curWidget)]
    set H [winfo height $data(curWidget)]

    tixMoveResizeWindow $mask $x $y $W $H
    tixMapWindow $mask
    raise $mask
    update

    if {$data(-state) == "both" || $data(-state) == "balloon"} {
	tixBalloon::Popup $w
    }
    if {$data(-state) == "both" || $data(-state) == "status"} {
	tixBalloon::SetStatus $w
    }
}

proc tixBalloon::Popup {w} {
    upvar #0 $w data

    if [tixGetBoolean -nocomplain $data(-installcolormap)] {
	wm colormapwindows [winfo toplevel $data(curWidget)] $w
    }

    # trick: the following lines allow the balloon window to
    # acquire a stable width and height when it is finally
    # put on the visible screen
    #
    set client $data(curWidget)
    $data(w:message) config -text $data(m:$client)
    wm geometry $w +10000+10000
    wm deiconify $w
    raise $w
    update

    # Put it on the visible screen
    #
    set x [expr [winfo rootx $client]+[winfo width  $client]/2]
    set y [expr int([winfo rooty $client]+[winfo height $client]/1.3)]

    wm geometry $w +$x+$y

    set data(popped) 1

    after 100 "tixBalloon::Verify $w $data(curWidget)"
}

# tixBalloon::Verify
#	Sometimes we "lose events" when the user moves the mouse pointer
#	rapidly. This routine continuously check whether the mouse
#	pointer is still in the balloon region. If not, it pops down the
#	balloon.
#
proc tixBalloon::Verify {w client} {
    upvar #0 $w data

    if {!$data(popped)} {
	return
    }
    if {$data(curWidget) != $client} {
	return
    }

    set rootX [winfo pointerx $client]
    set rootY [winfo pointery $client]

    if {$rootX == -1 || $rootY == -1} {
	# mouse pointercursor moved to another screen
	tixBalloon::Popdown $w
	return
    }

    set cw [winfo containing $rootX $rootY]
    set mask [tixBalloon::GetMask $w $data(curWidget)]

    if {[tixBalloon::Within $w $rootX $rootY] || $cw == $mask} {
	# mouse pointer position OK (still in either client or balloon)
	after 100 tixBalloon::Verify $w $client
    } else {
	tixBalloon::Popdown $w
    }
}

proc tixBalloon::Popdown {w} {
    upvar #0 $w data

    # Close the balloon
    #
    wm withdraw $w

    # Clear the status bar
    #
    if {$data(statusSet) == 1} {
	tixBalloon::ClearStatus $w
	set $data(statusSet) 0
    }

    # Withdraw the mask window
    #
    tixUnmapWindow [tixBalloon::GetMask $w $data(curWidget)]

    set data(popped) 0
}

proc tixBalloon::SetStatus {w} {
    upvar #0 $w data

    if {![winfo exists $data(-statusbar)]} {
	return
    }

    if {$data(-statusbar) != {}} {
	set vv [$data(-statusbar) cget -textvariable]
	if {$vv == ""} {
	    $data(-statusbar) config -text $data(s:$data(curWidget))
	} else {
	    uplevel #0 set $vv [list $data(s:$data(curWidget))]
	}
    }
    set data(statusSet) 1
}

proc tixBalloon::ClearStatus {w} {
    upvar #0 $w data

    if {![winfo exists $data(-statusbar)]} {
	return
    }

    # Clear the StatusBar widget
    #
    if {$data(-statusbar) != {}} {
	set vv [$data(-statusbar) cget -textvariable]
	if {$vv == ""} {
	    $data(-statusbar) config -text ""
	} else {
	    uplevel #0 set $vv [list ""]
	}
    }
}

proc tixBalloon::BindOneWidget {w client subwidget} {
    upvar #0 $w data

    if {![winfo exists $subwidget]} {
	return
    }

    set class [winfo class $subwidget]

    bind TixBalloon$client <Any-Enter>  "tixBalloon::Enter $w $client"
    bind TixBalloon$client <Destroy>    "tixBalloon::ClientDestroy $w $client"

    tixAppendBindTag $client TixBalloon$client
}
#----------------------------------------------------------------------
# Mask window handlng
#----------------------------------------------------------------------

# We need a "mask" window to put all over the client widget so that we can
# find out when the user presses the mouse buttons
#
# This is the most complicated code in all of Tix. If you don't understand 
# is going on, don't touch it.
#

set btn_fields  {
%% %# %a %b %c %d %f %h %k %m %o %p %s %t %w %x %y %A %B %E %K %N %R %S %T %W %X %Y
}

# Since the mask window overlays the client widget, it gets all the mouse 
# events of the client widget. We need to capture these events and resend
# them to the client widget.
# 
proc tixBalloon::InterceptMouseEvents {w mask client} {
    global btn_fields

    if {![winfo exists $client]} {
	return
    }

    set done 0
    foreach tag [bindtags $client] {
	foreach event [bind $tag] {
	    if [regexp {([1-3]>$)} $event] {
		# This is a button event

		bind $mask $event \
		    [concat tixBalloon::GenerateEvent $w $mask $event $btn_fields]
		set done 1
	    }
	}
    }

    # We want this for all widgets:
    #   pressing the left mouse button and the
    #   balloon goes away
    bind $mask <1> \
	[concat tixBalloon::GenerateEvent $w $mask <1> $btn_fields]
}

# When this function is called, we have intercepted a mouse event
# for the client widget. Let's send it to the client. But before
# that we have to substitute all the % stuff in the commands.
#
#
proc tixBalloon::GenerateEvent [concat w mask event $btn_fields] {
    upvar #0 $w data
    global btn_fields

    tixBalloon::Popdown $w

    set client [winfo containing [set %X] [set %Y]]

    set %W $client

    foreach tag [bindtags $client] {
	set command [bind $tag $event]

	if {$command  == {}} {
	    continue
	}

	foreach f $btn_fields {
	    regsub -all $f $command [set $f] command
	}

	eval $command
    }

    set data(fakeEnter) 1
    set data(fakeLeave) 1
}

proc tixBalloon::GetMask {w client} {
    if {![winfo exists $client]} {
	## Something insane has happened!
	set tp .
    } else {
	set tp [winfo toplevel $client]
    }

    if {$tp == "."} {
	set tp ""
    }

    set mask $tp.tixInt:bal

    if {![winfo exists $mask]} {
	tixInputOnly $mask
	bind $mask <Leave>  "tixBalloon::Leave $w %X %Y"
    }
    tixBalloon::InterceptMouseEvents $w $mask $client

    return $mask
}

#----------------------------------------------------------------------
# PublicMethods:
#----------------------------------------------------------------------

# %% if balloon is already popped-up for this client, change mesage
#
proc tixBalloon::bind {w client args} {
    upvar #0 $w data

    if [info exists data(m:$client)] {
	set alreadyBound 1
    } else {
	set alreadyBound 0
    }

    set opt(-balloonmsg) {}
    set opt(-statusmsg)  {}
    set opt(-msg)        {}

    tixHandleOptions opt {-balloonmsg -msg -statusmsg} $args

    if {$opt(-balloonmsg) != {}} {
	set data(m:$client) $opt(-balloonmsg)
    } else {
	set data(m:$client) $opt(-msg)
    }
    if {$opt(-statusmsg) != {}} {
	set data(s:$client) $opt(-statusmsg)
    } else {
	set data(s:$client) $opt(-msg)
    }

    # Set up the bindings of the widget, in which the balloon should appear
    #
    tixBalloon::BindOneWidget $w $client $client
}

proc tixBalloon::unbind {w client} {
    upvar #0 $w data

    if [info exists data(m:$client)] {
	catch {unset data(m:$client)}
	catch {unset data(s:$client)}

	if [winfo exists $client] {
	    tixDeleteBindTag $client TixBalloon$client
	}
    }
}

