#=========================================================================
# html.tcl
#	The HTML language and DTD should be regarded as a builtin part
#	of CW.  There are several HTML-specific aspects:
#		HTML can be used as an output and for rendering
#		The HTML DTD is likely to be used for some documents
#		Many DTDs (e.g. XML) will be derived from HTML
#=========================================================================

puts "Reading html.tcl..."

    global env

#=========================================================================
proc HTML::initialise {} {

    puts "Initialising HTML..."

    HTML::userInitialise

    SGML::addEmptyTags {BR HR IMG LINK}

# these are all defaulted in the DTD and cause havoc with non-standard
# SGML browsers as they include < and >

## uncomment this to avoid SDA everywhere
#    SGML::addSilentAttributes {SDAFORM SDARULE SDAPREF SDASUFF SDASUSP}
}

# just display all the contents as HTML
proc BODY::displayNode {parentWin address {control ""}} {
    set win $address
    set content [Node::getEventStream $address]
puts ==$content==
    Hypertext::display $parentWin $win $content $control
}

#------------------------------------------------------------------------

proc TITLE::processEvent {{address ""}} {
    
}

#------------------------------------------------------------------------
# gets current directory for relative filenames (either from original SGML
# document) or from last file used.  Rather hairy...
proc HTML::getCurrentDirectory {win} {

    global CW
    if {[info exists CW(HTEXTFILE,$win)]} {
        set dir [file dirname $CW(HTEXTFILE,$win)]
    } elseif {[info exists CW(DOCUMENT,FILENAME)]} {
        set dir [file dirname $CW(DOCUMENT,FILENAME)]
    } else {
        set dir ""
    }
    return $dir
}

# make each <A>...</A> fit on a single line
# assumes that all tags start on a new line
# remove blanks
proc HTML::normalise {lines} {
    set contents ""
    set temp ""
    foreach line $lines {
        if {[string trim $line] == ""} {
        } elseif {[string range $line 0 2] == "<A "} {
            set temp $line
        } elseif {$temp != ""} {
            append temp $line
            if {[string range $line 0 3] == "</A>"} {
                lappend contents $temp
                set temp ""
            }
        } else {
            lappend contents $line
        }
    }
    return $contents
}
                
# translates 'foreign' tags in a stream into HTML tags.  The syntax of
# $translation is a list where each element is:
#        <list of tags>     <prefix>  <html_tag>  <suffix>
# example (to turn ITALIC and EMPHASIS into <I>...</I>:
#        "ITALIC EMPHASIS"      ""  I ""

# tends to be very slow for large chunks of text
proc HTML::translateTags {htext translation} {

    foreach elem $translation {
        set tags [lindex $elem 0]
        foreach tag $tags {
            set tag [string toupper $tag]
            set HTML(PREF,$tag) [lindex $elem 1]
            set HTML(TAG,$tag)  [lindex $elem 2]
            set HTML(SUFF,$tag) [lindex $elem 3]
        }
    }

    set result ""
    while {1} {
        if {[regexp {([^<]*)<([/]*)([^ >]*) *([^>]*)>(.*)} \
            $htext junk first slash tag atts htext]} {
            set tag [string toupper $tag]
            if {[info exists HTML(TAG,$tag)]} {
                if {$slash == "/"} {
                    append result "$first</$HTML(TAG,$tag)>$HTML(SUFF,$tag)"
                } else {
                    append result "$first$HTML(PREF,$tag)<$HTML(TAG,$tag)$atts>"
                }
            } else {
                append result "$first<$slash$tag$atts>"
            }
        } else {
            append result $htext
            break
        }
    }
    return $result
}

# traverses tree for HTML subtree within document
# this treats HTML as a tree rather than an event stream
# at present the only specific thing is the rendering of
# the <A> tags (which I make come out on a single line
# <A>...</A>

proc getRecursiveHypertext {{address ""}} {

    global LEVEL
    global ACONT
    if {$address != ""} {
        set LEVEL 0
        selectNode node $address
    }
    incr LEVEL
    set result($LEVEL) ""
    set nodetype($LEVEL) [query nodetype]
    set gi($LEVEL) [query gi]
    if {$nodetype($LEVEL) == "CDATA"} {
        lappend result($LEVEL) [content]
    } else {
        if {$nodetype($LEVEL) == "EL"} {
            set startag "<$gi($LEVEL)"
            if {$gi($LEVEL) == "A"} {
# for <A NAME...> simply replicate
                set val [query attval NAME]
                if {$val != ""} {
                    append startag " NAME=\"$val\""
                }
# for <A HREF...> follow in various ways
                set val [query attval HREF]
# if REL = glossary (or anything else) prepend this to URL
                set rel [query attval REL]
                if {$rel != ""} {set rel $rel:}
                if {$val != ""} {
                    append startag " HREF=\"$rel$val\""
                }
            } elseif {$gi($LEVEL) == "IMG"} {
                set src [query attval SRC]
                if {$src != ""} {
                    append startag " SRC=\"$src\""
                }
                set ismap [query attval ISMAP]
                if {$ismap != ""} {
                    append startag " ISMAP"
                }
            }
            append startag ">"
            lappend result($LEVEL) $startag
        }
        foreachNode child {
            set child [getRecursiveHypertext]
            if {$child != ""} {
                set result($LEVEL) [concat $result($LEVEL) $child]
            }
        }
# add any closing tags
        if {$nodetype($LEVEL) == "EL"} {
            if {![SGML::isEmptyTag $gi($LEVEL)]} {
                lappend result($LEVEL) "</$gi($LEVEL)>"
            }
        }
    }
    set tmp $result($LEVEL)
    incr LEVEL -1
    return $tmp
}

proc expandHREFandREL {} {
    set href [query attval HREF]
    if {$href == ""} {return}
    if {[string first "#" $href] == 0} {
        set target [getNodeAddressForInternalTarget $href]
    }
    set rel [query attval REL]
# this is still not finalised
    return " HREF=\"$href\""
}

# finds target of HREF (==> NAME), strips '#' and returns address
# or finds target of HREF (==> ID) since some systems may use this
# (very messy)
proc getNodeAddressForInternalTarget {href} {
    regsub {^#} $href "" name
    set target ""
    withNode doctree withattval NAME $name {
        set target [query address]
    }
    if {$target == ""} {
        withNode doctree withattval ID $name {
            set target [query address]
        }
    }
    return $target
}

#-------------------------borrowed from HTMLlib--------------------------
proc HMset_image {win handle src} {
        global Url message
#        if {[string match /* $src]} {
#                set image $src
#        } else {
#                set image [file dirname $Url]/$src
#        }
    set image $src
    set message "fetching image $image"
    update
    if {[string first " $image " " [image names] "] >= 0} {
        HMgot_image $handle $image
    } else {
        set type photo
        if {[file extension $image] == ".bmp"} {set type bitmap}
        catch {image create $type $image -file $image} image
        HMgot_image $handle $image
    }
}


#---------------------------------INITIALISE------------------------------
# these routines can be overwritten (overloaded) by the local versions
# in xml_local.tcl
proc HTML::userInitialise {} {}
#---------------------------------INITIALISE------------------------------

#-------------------------------------------------------------------------
# additional user routines if supplied
#-------------------------------------------------------------------------

    if {[file exists $CW(TCLDIR,HTML)/local.tcl]} {
        source $CW(TCLDIR,HTML)/local.tcl
    }

#=========================================================================
    HTML::initialise
#=========================================================================

puts "... end of html.tcl"

