#graphical routines

puts "Reading graph.tcl..."

proc GI::getColor {gi} {
    global CW
    if {[info exists CW(COLOR,GI,$gi)]} {
        return $CW(COLOR,GI,$gi)
    } 
    return black
}

# Overrides html_library call - when hyperlink in HTML is clicked
proc HMlink_callback {win href} {
    
puts ..$win...
    global CW

    set fragment ""
    set file ""
# if href is a 'normal' file, try to call it
    if {[regexp {(.*)[\.](.*)} $href junk root suffix]} {
# does it have a fragment on the end?
        regexp {([^#]*)#(.+)} $suffix junk suffix fragment
# html file
        if {[regexp -nocase "htm" $suffix]} {
            set dir [HTML::getCurrentDirectory $win]
            set file $dir/$root.$suffix
puts stdout "displaying file: $file..."
            GUI::displayFile $file HTEXT $fragment
# asci or  sgml (raw)
        } elseif {[regexp -nocase "(txt|sgm)" $suffix]} {
            GUI::displayFile $href TEXT 
# gif
        } elseif {[regexp -nocase "(gif)" $suffix]} {
            exec $CW(GIF,DISPLAYCOMMAND) $href &
        } else {
            GUI::errorMessage "Don't know how to display: $href"
        }
    } else {
# if it's a fragment?
        if {[regexp {([^#]*)#(.+)} $href junk file fragment]} {
# this is local to this page, I think
            set target [getNodeAddressForInternalTarget $href]
            set winNode [Node::ungetOKID $win]
# kludge - make a new window...
Node::display $win $target
return
# is target in the same window? (I can't get this to work...)
            if {[query? node $target ancestor node $winNode]} {
                HMgoto $win $fragment
                return
            }
        }
        HTML::userLinkCallback $win $href
    }
}
# this is called (from html_library) when there is a hit on a HREF tag
# users may wish to overwrite it.
proc HTML::userLinkCallback {win href} {

# internal to the SGML file?
    if {[string first "#" $href] == 0} {
        set target [getNodeAddressForInternalTarget $href]
        Node::display $win $target
# ismap? (collects coords without an href)
    } elseif {[regexp {(.*)[\?]([0-9]*),([0-9]*)} $href junk address x y]} {
        set gi [query node $address gi]
        if {$gi == "Error"} return
        if {[info proc $gi::processISMAP] == ""} {
            errorMessage "No $gi::processISMAP: cannot process clickable map"
        } else {
            eval $gi::processISMAP $address $x $y
        }
    } elseif {[regexp -nocase "GLOSSARY:(.*)" $href junk glossref]} {
        Glossary::showEntry $glossref
    }
}


#----------------------------TopLevel-------------------------------------

proc TopLevel::createFromGI {parentNode gi address} {
# make a toplevel window of form .xhtml$address, etc (note lower case)

    set giWin [Node::createWindowName $gi $address]
    set parentTitle $parentNode
    TopLevel::createFromParent $parentNode $giWin $parentTitle $address

}

proc TopLevel::createFromParent {parent win {parentTitle ""} {address ""}} {
# make toplevel if it doesn't exist and bind it to die if parent dies
    global CW 

    if {![winfo exists $win]} {
        if {[catch "toplevel $win"]} {
            GUI::errorMessage "Can't make toplevel window: $win"
            return
        }
# top bar
        set frame [Widget::makeOrRenew frame $win.titlebar]
        $frame configure -border 3
        pack $frame -fill x
        set button [Widget::makeOrRenew button $frame.quit]
        $button configure -text QUIT -command "destroy $win"
        pack $button -side left -expand yes -fill x
# locate parent window
        set button [Widget::makeOrRenew button $frame.parent]
        $button configure -text "Parent ($parentTitle)" \
            -command "Window::highlightParent \"$parent\" \"$address\""
        pack $button -side left -expand yes -fill x
# locate object in text window
        if {$address != ""} {
            set button [Widget::makeOrRenew button $frame.text]
            $button configure -text "Locate Text" \
                -command "Node::locateInText $CW(MAINTOC) $address"
            pack $button -side left -expand yes -fill x
        }

        Window::updateListOfToplLevelChildren $parent $win
        bind $parent <Destroy> "Window::destroyChildren $parent"
        wm minsize $win 200 100
    }
    catch "wm deiconify $win"
    catch "raise $win"
    return $win
}

proc TopLevel::createEntry {msg} {
    global ee
    Widget::makeOrRenew toplevel .e
    label .e.l -text $msg
    pack .e.l -side left
    entry .e.e -relief sunken
    pack .e.e -side left
    bind .e.e <Return> {set ee [.e.e get]}
    tkwait variable ee
    if {[winfo exists .e]} {
        destroy .e
    }
    return $ee
}

#---------------------------------Node-------------------------------------
proc Node::createWindowName {gi address} {
    regsub -all {\.} $gi "" giWin
    return ".[string tolower $giWin][Node::getOKID $address]"
}

proc Node::displayContent {win address} {

    withNode node [Node::getAddress $address] {
        set content [Node::getMixedContent $address]
    }
    set label [Widget::makeOrRenew label $win.lab]
    $label configure -text Content -padx 0 -pady 0
    pack $label -fill x
    Hypertext::display $win "" $content
}

proc Node::displayAttributes {win address} {

    set tuples [Node::getTuples $address]
    set content ""
    foreach tuple $tuples {
        append content "<B>[lindex $tuple 0]</B>: [lindex $tuple 1]<BR>"
    }
    set label [Widget::makeOrRenew label $win.lab]
    $label configure -text Attributes -padx 0 -pady 0
    pack $label -fill x
    Hypertext::display $win "" $content
}

# launches a toplevel window for a node - kludgy inside
proc Node::makeToplevel {win address} {
    global CW
    set temp $CW(NODEACTION)
    set CW(NODEACTION) SUBTREE
    Node::display $win $address
    set CW(NODEACTION) $temp
}

# there are now several ways of displaying subtrees
proc Node::display {win address {frame ""} {control ""}} {

    global CW

    set gi [Node::getGI $address]

# move text to reveal current node
    Node::locateInText $CW(MAINTOC) $address

# title
    set title [Node::getMeaningfulTitle $address]/address/$gi

    if {$CW(NODEACTION) == "SUBTREE"} {
# make subframe
        if {$frame == ""} {
            set frame "[TopLevel::createFromGI $win $gi $address]"
            wm title $frame "${title}($frame)"
        }
    }
    
# display subnodes as panes
    if {$CW(NODEACTION) == "PANED"} {
        set nodeFrame $CW(SIDEWIN,$CW(SIDEWIN,CURRENT))
        incr CW(SIDEWIN,CURRENT) 
        if {$CW(SIDEWIN,CURRENT) >= $CW(SIDEWIN,NUMB)} {
            set CW(SIDEWIN,CURRENT) 0
        }
        catch "destroy $nodeFrame"
        frame $nodeFrame
        pack $nodeFrame -expand yes -fill both
        set panes [Tix::makePane $nodeFrame.panes {{node 75 350} {subnodes 25 75} {attribs 25 100}} horizontal]
        set displayFrame [Widget::makeOrRenew frame [lindex $panes 0].frame]
        pack $displayFrame -expand yes -fill both
    
# make a top bar for the displayFrame (magnify, hide panes, etc)
        set topbar [Widget::makeOrRenew frame $displayFrame.bar]
        pack $topbar -side top

        set button [Widget::makeOrRenew button $topbar.att]
        pack $button -side left
        $button configure -padx 0 -pady 0 -text A \
            -command "Window::togglePacking $nodeFrame.panes.attribs.frame"
        $button configure -state disabled
        
        set button [Widget::makeOrRenew button $topbar.cont]
        pack $button -side left
        $button configure -padx 0 -pady 0 -text C \
            -command "Window::togglePacking $nodeFrame.panes.subnodes.frame"
        $button configure -state disabled

        set button [Widget::makeOrRenew button $topbar.mag]
        pack $button -side left
        $button configure -padx 0 -pady 0 -text M\
            -command "Node::makeToplevel $win $address"

        set button [Widget::makeOrRenew button $topbar.loc]
        pack $button -side left
        $button configure -padx 0 -pady 0 -text L \
            -command "Node::locateInText $CW(MAINTOC) $address"

        set label [Widget::makeOrRenew label $topbar.lab]
        $label configure -padx 0 -pady 0 -text $title
        pack $label -fill x
    
# contents
        set contentFrame [Widget::makeOrRenew frame [lindex $panes 1].frame]
        pack $contentFrame -expand yes -fill both
        Node::displayContent $contentFrame $address

# attributes
        set attribFrame [Widget::makeOrRenew frame [lindex $panes 2].frame]
        pack $attribFrame -expand yes -fill both
        Node::displayAttributes $attribFrame $address


        set frame $displayFrame
    }

# common to both methods
    if {[info proc $gi::displayNode] != ""} {
        eval $gi::displayNode $frame $address $control
    } else {
        puts stderr "NO ROUTINE: $gi::displayNode"
        ANY::displayNode $frame $address $control
    }
}

proc Node::undisplay {address} {
    set win [Node::createWindowName [Node::getGI $address] $address]
    if {[winfo exists $win]} {destroy $win}
}

# moves text in text window to have node at top
proc Node::locateInText {win address} {
    global CW
    set line [Toc::getLineFromAddress $win $address]
    if {$line == ""} {
        Toc::showAncestry $win $address
        set line [Toc::getLineFromAddress $win $address]
        if {$line == ""} return
    }
# get line number
    set y [expr [lindex [split $line \.] 0]-1]
    $win yview $y
    set range [$win tag ranges $address]
    set start [lindex $range 0]
    set end [lindex $range 1]
    $win tag delete locate 
    $win tag add locate $start $end
    $win tag configure locate -background yellow
}

#----------------------------------Window---------------------------------
proc Window::highlightParent {parent address} {
    if {[winfo exists $parent.titlebar]} {
        catch "raise $parent"
        catch "wm deiconify $parent"
        for {set i 0} {$i < 10} {incr i} {
            $parent.titlebar.parent flash
        }
    } else {
        set parentAddress [Node::getParentAddress $address]
        if {$parentAddress != ""} {
            Node::display "dummy" $parentAddress
        }
    }
}

# makes a list of which toplevels are children
proc Window::updateListOfToplLevelChildren {parent frame} {
    global CW
    if {![info exists CW(TOPLEVELCHILDREN,$parent)]} {
        set CW(TOPLEVELCHILDREN,$parent) ""
    }
    if {[lsearch $CW(TOPLEVELCHILDREN,$parent) $frame] == -1} {
        lappend CW(TOPLEVELCHILDREN,$parent) $frame
    }
}
    
# destroys any toplevel which relates to ESIS children of window
proc Window::destroyChildren {win} {
    global CW
    if {[info exists CW(TOPLEVELCHILDREN,$win)]} {
        foreach win $CW(TOPLEVELCHILDREN,$win) {
            if {[winfo exists $win]} {
                destroy $win
            }
        }
    }
}

# window names must be lowercase and have no dots.  e.g. X.HTML==> xhtml
proc Window::makeNameFromGI {gi} {
    regsub -all {\.} $gi "" gi
    return [string tolower $gi]
}

proc Window::togglePacking {win} {
    if {[winfo ismapped $win]} {
        pack unpack $win
    } else {
        pack $win
    }
}

# window names must be lowercase and have no dots or slashes
proc Window::makeNameFromFile {fileName} {
    regsub -all {\.} $fileName "" fileName
    regsub -all {\/} $fileName "" fileName
    return [string tolower $fileName]
}

#------------------------------------ANY-----------------------------------
proc ANY::formatAttributes {address} {

    set tuples [Node::getTuples $address]

    Output::capture
    foreach tuple $tuples {
        set name [lindex $tuple 0]
        set val [lindex $tuple 1]
        if {$val != "" } {
            Output::append "[Output::format I $name:] $val [Output::format BR]"
        }
    }
    return [Output::capture]
}

# displays the 'content' and the subnodes 
proc ANY::displayNode {parentWin address {control ""}} {

    global CW

    set win $address
    
# all hypertext goes in one window
    set attributes [ANY::formatAttributes $address]
    if {$attributes != ""} {
        Hypertext::display $parentWin $win $attributes APPEND
    }

# display subnodes. 
    set children [Node::displaySubnodes $parentWin $address $control]
# if no children, display contents as hypertext
    if {$children == 0} {
        set content [Node::getContents $address]
        Hypertext::display $parentWin $win $content APPEND
    }
}

proc TextWindow::createWithScrollbar {win {width 80} {height 10}} {

    if {![winfo exists $win]} {
        scrollbar ${win}scroll  -command "$win yview"  -orient v
#    option add *Text.height $height startup
#    option add *Text.width $width startup
        text $win \
            -width $width -height $height \
            -yscrollcommand "${win}scroll set" \
            -padx 3 -pady 3 -takefocus 0
    } else {
        $win delete 1.0 end
    }
    ScrollBar::pack $win
}

proc ScrollBar::pack {win} {
    pack ${win}scroll -side left -expand 0 -fill y
    pack $win -side left -fill both -expand 1
}

# parentWin should be a frame:  winId is not used (? goof?)
proc Hypertext::display {parentWin winId contents {control ""}} {

# control is a *list* of keywords
# if control==LIST we need to join the elements with newlines
    set list [regexp LIST $control]
# if control==USER, apply userProcessHypertext
    set user [regexp USER $control]
# if control==APPEND we append to end of whatever is already there.
    set append [regexp APPEND $control]

# might wish to process the hypertext further (e.g. inlining)

    if {$user} {
        set contents [userProcessHypertext $contents]
    }
    if {$list} {
        set contents [join $contents \n]
    }

    if {$parentWin == "."} {set parentWin ""}
    set window [Widget::makeOrRenew frame $parentWin.htext]
    pack $window -fill both -expand yes
    if {$append} {
        set htwin $window.append
#        set contents "<HR>$contents"
        set contents "<BR>$contents"
    } else {
        set htwin $window.htext
    }
    if {[winfo exists $htwin]} {
        ScrollBar::pack $htwin
        if {!$append} {
            HMreset_win $htwin
        }
    } else {
        set nchar [string length $contents]
        set height 30; set width 80
        if {$nchar < 80} {
            set height 2; set width 40
        } elseif {$nchar < 400} {
            set height 6; set width 60
        } elseif {$nchar < 1000} {
            set height 15; set width 80
        } else {
            set height 25; set width 80
        }
        if {[catch "TextWindow::createWithScrollbar $htwin $width $height"]} {
            errorMessage "TextWindow::createWithScrollbar: $htwin"
        }
        HMinit_win $htwin
    }
    if {$contents != ""} {
# backslashes can cause Hippo to bomb, sometimes ??
        if {[string first \\ $contents] != -1} {
            regsub -all {\\} $contents {\&#92;} contents
        }
        HMparse_html $contents "HMrender $htwin"
        $htwin configure -background white
    } else {
        $htwin configure -background grey
    }
}

# omits some of the subnodes of a window
proc Node::includeOmitSubnodes {gi title controlList} {

    set omitList ""
    if {[lindex $controlList 0] == "OMIT"} {
        set omitList [lindex $controlList 1]
    }
    set includeList ""
    if {[lindex $controlList 0] == "INCLUDE"} {
        set includeList [lindex $controlList 1]
    }

    set omit 0
    if {$omitList != ""} {
        if {[lsearch $omitList $gi] != -1 ||
            [lsearch $omitList $title] != -1} {
            set omit 1
        }
    }
    if {$includeList != ""} {
        if {[lsearch $includeList $gi] == -1 && \
            [lsearch $includeList $title] == -1} {
            set omit 1
        }
    }
    return $omit
}

proc Node::displaySubnodes {parentWin address {controlList ""} {control ""}} {
# display the subnodes as sub-buttons of the parent.  If
# controlList is not "". it can OMIT or INCLUDE specific GIs

# if control contains APPEND, quit  - WHY???

    global CW
    if {[regexp APPEND $control]} {
        return
    }

    set children 0
# TOC-like display
    if {$CW(NODEDISPLAY) == "TOC"} {
        withNode node $address {
            set tocWin $parentWin.text
            Toc::display $tocWin $CW(NODE,DEPTH) 
        }

# display nodes as buttons...
    } elseif {$CW(NODEDISPLAY) == "TREE"} {

        set frame [Widget::makeOrRenew frame $parentWin.frame]
        $frame configure -border 3 -relief sunken -background white
        pack $frame -side top
        set count 0
        set bar 0
        set barframe [Widget::makeOrRenew frame $frame.bar$bar]
        pack $barframe
        set barwidth 0
        GUI::postStatus "Displaying Contained Nodes"
        set children 0
        foreachNode node $address child {
            set gi [query gi]
            set title [Node::getMeaningfulTitle]
            set childAddress [query address]
# the contents might not be nodes (e.g. #PCDATA)
            if {$gi != ""} {
                incr children
                set omit [Node::includeOmitSubnodes $gi $title $controlList]
                if {!$omit} {
                    set button $barframe.[Window::makeNameFromGI [query gi]]$count
                    Widget::makeOrRenew button $button
                    $button configure -padx 0 -pady 0
                    pack $button -side left
                    set buttonChild [Widget::makeOrRenew frame ${button}child]
                    pack $buttonChild -side top
                    $button configure -text $title -foreground [GI::getColor $gi] \
                        -background white \
                        -command "Node::subnodesButtonCommand $parentWin \
                            $frame $buttonChild $button \{$childAddress\}"
                    update
                    incr count
                    incr barwidth [winfo width $button]
                    if {$barwidth > 600} {
                        set barwidth 0
                        incr bar
                        set barframe [Widget::makeOrRenew frame $frame.bar$bar]
                        pack $barframe
                    }
                }
            }
        }
    } elseif {$CW(NODEDISPLAY) == "EVENT"} {
        set content [HTML::escapeDelimiters [Node::getEventStream $address]]
        if {$content != ""} {
            Hypertext::display $parentWin "" $content APPEND
        }
        set children 1
    }

    GUI::postStatus ""
    return $children
}

proc Node::subnodesButtonCommand {parent frame win button address} {

    Node::display $parent $address
#    $button configure -command "Window::togglePacking $win"
}

proc Widget::makeOrRenew {widget name} {
    if {![winfo exists $name]} {
# we have to catch it in case it has been destroyed just beforehand!
        catch "eval $widget $name"
    }
    if {$widget == "toplevel"} {
        wm minsize $name 300 150
        wm deiconify $name
        raise $name
    }
    return $name
}

#proc Widget::toggleVisib {win} {
#    if {[winfo visible $win]} {
#        catch "unpack $win"
#    } else {
#        catch "pack $win"
#    }
#}

# many windows spawn off toplevels to show child nodes.  Destroy these
proc Node::destroyToplevelChildren {parentWin} {
    foreach window [winfo children .] {
        if {$parentWin != $window && [string first $parentWin $window] == 0} {
            destroy $window
        }
    }
}

#--------------------------------GUI--------------------------------------
proc GUI::initialise {} {

# the GI colours are done in the user.tcl
}

# display either a text i(control=TEXT) or hypertext file (control=HTEXT) i
# in a new toplevel window
# if fragment != "", then go to that fragment
proc GUI::displayFile {file {control TEXT} {fragment ""}} {

    global CW

    set f [open $file r]
    set msg [read $f 100000]
    close $f
    set toplevel ".[Window::makeNameFromFile $file]"
    set toplevel [Widget::makeOrRenew toplevel $toplevel]
    wm minsize $toplevel 300 150
    set quitButton [Widget::makeOrRenew button $toplevel.quit]
    $quitButton configure -text "QUIT" -command "destroy $toplevel"
    pack $quitButton -side bottom
    set textFrame [Widget::makeOrRenew frame $toplevel.f]
    pack $textFrame -side bottom
    if {$control == "TEXT"} {
        set label [Widget::makeOrRenew label $toplevel.lab]
        $label configure -text "$file"
        pack $label -side top
        set textWin $textFrame.text
        TextWindow::createWithScrollbar $textWin 80 25
        $textWin insert 1.0 $msg
    } elseif {$control == "HTEXT"} {
# remember the filename
        if {[string range $file 0 0] == "/"} {
            set filename $file
        } else {
            set filename "[pwd]/$file"
        }
        set CW(HTEXTFILE,$textFrame.htext.htext) $filename"
        Hypertext::display $textFrame htext $msg 
        if {$fragment != ""} {
            HMgoto $textFrame.htext.htext $fragment
        }
    }
    return $toplevel
}

proc GUI::errorMessage {msg} {

    GUI::postStatus ""
    if {![winfo exists .err]} {
        toplevel .err
        message .err.l  -width 400
        pack .err.l -fill x -expand yes
        button .err.b -text OK -command "destroy .err"
        pack .err.b
    }
    .err.l configure -text $msg
    raise .err
    tkwait window .err
}

# I find menus difficult, so this helps.  Format is:
#	GUI::makeMenu <menuname> <menutitle> <config> <pack> <items>
#
#	menuname	Valid toplevel widget name (e.g. .fred)
#	menutitle	any string
#	config		Confugration of menu (e.g. -fg red)
#	pack		Packing options (e.g. -side left)
#	items 		list of items (see below)

# for each item: 
#	0 	command, checkbutton, radiobutton, label, separator
#	1	<label>
#	2	<variable name>
#	3	<variable value>
#	4	<options> (e.g. -foreground red)
proc GUI::makeMenu {menu title config pack items} {
    
    set menuitems $menu.m
    if {![winfo exists $menu]} {
        menubutton $menu
        menu $menuitems
    }
    if {$config != ""} {
        catch "$menu configure $config"
    }
    eval pack $menu $pack
    $menu configure -relief raised -bd 2 -text $title -menu $menuitems
    foreach item $items {
        set widget [lindex $item 0]
        if {$widget == "command"} {
            $menuitems add command -label [lindex $item 1] \
                -command [lindex $item 2]
        } elseif {$widget == "checkbutton"} {
# checkbutton title varname initial(0/1)
            eval $menuitems add checkbutton -label \"[lindex $item 1]\" \
                -variable [lindex $item 2] [lindex $item 4]
                global [lindex $item 2]
                eval set [lindex $item 2] \"[lindex $item 3]\"
        } elseif {$widget == "label"} {
            eval $menuitems add command -label \"[lindex $item 1]\" \
                -state disabled [lindex $item 4]
        } elseif {$widget == "radiobutton"} {
            eval $menuitems add radiobutton -label \"[lindex $item 1]\" \
                -variable [lindex $item 2] -value \"[lindex $item 3]\" \
                [lindex $item 4]
        } elseif {$widget == "separator"} {
            $menuitems add separator
        } else {
            GUI::errorMessage "Bad menu item: $widget"
        }
    }
}

proc GUI::postStatus {msg} {
    global CW
    if {$msg != ""} {
        Cursor::change . watch
        if {[info exists CW(MAINTOC)]} {Cursor::change $CW(MAINTOC) watch}
    } else {
        Cursor::change . ""
        if {[info exists CW(MAINTOC)]} {Cursor::change $CW(MAINTOC) ""}
    }
    set label [Widget::makeOrRenew label .label]
    .label configure -text $msg -relief sunken -border 1 
    pack .label -side bottom -fill x
    update
}

proc Cursor::change {win cursor} {
    if {[winfo exists $win]} {
        $win configure -cursor $cursor
    }
}

#-------------------------------Widgets-----------------------------------

# makes list of radiobuttons - horizontal or vertical
# list is name-value pairs with optional command, 
# varaible is name of (global) variable
# orient is 'top' or 'left'
proc Widget::makeRadio {frame list variable value orient} {
    set i 0
    foreach tuple $list {
        set radio [Widget::makeOrRenew radiobutton $frame.r$i]
        if {[lindex $tuple 2] != ""} {
            set command "-command \"[lindex $tuple 2]\""
        } else {
            set command ""
        }
        eval $radio configure -text [lindex $tuple 0] -value \
            [lindex $tuple 1] -variable $variable $command
        pack $radio -side $orient
        incr i
    }
    global [Global::root $variable]
    eval set $variable \"$value\"
}

proc Widget::makeCombo \
    {frame list label variable value {options {label.anchor e}}} {

    global [Global::root $variable]
    set options "listbox.background red"
    set labelWidth [string length $label]
    if {$labelWidth < 5} {set labelWidth 5}
    set listWidth [llength $list]
    if {$listWidth < 5} {set listWidth 5}
    if {[winfo exists $frame]} {
        destroy $frame
    }
    tixComboBox $frame \
        -label $label \
        -variable $variable \
        -dropdown true \
        -editable false \
        -options "listbox.height $listWidth listbox.width 15 \
            label.width $labelWidth"

    foreach item $list {
        $frame insert end $item
    }
    tixSetSilent $frame $value
}

proc Widget::makeDialog {message but1 {but2 ""} {grab YES}} {
    global dialogVar
    set dialog [Widget::makeOrRenew toplevel .dialog]
    wm minsize .dialog 100 100
    set msg [Widget::makeOrRenew message .dialog.msg]
    $msg configure -text $message -width 500
    pack $msg -side top
    set frame [Widget::makeOrRenew frame .dialog.frame]
    pack $frame -side top -expand yes
    set b1 [Widget::makeOrRenew button $frame.b1]
    $b1 configure -text $but1 -command "Widget::exitDialog \"$but1\""
    pack $b1 -side left -fill x 
    if {$but2 != ""} {
        set b2 [Widget::makeOrRenew button $frame.b2]
        $b2 configure -text $but2 -command "Widget::exitDialog \"$but2\""
        pack $b2 -side left -fill x 
    }
    if {$grab == "YES"} {
        grab set .dialog
    }
    tkwait window .dialog
    return $dialogVar
}

proc Widget::exitDialog {but} {
    global dialogVar
    set dialogVar $but
    destroy .dialog
}

proc Widget::makeInlineCheckbutton {widget text variable value} {
    global [Global::root $variable]

    set widget [Widget::makeOrRenew frame $widget]
    pack $widget -side left
    set label [Widget::makeOrRenew label $widget.l]
    $label configure -text $text
    pack $label -side left
    set check [Widget::makeOrRenew checkbutton $widget.c]
    pack $check -side left
    $check configure -variable $variable
    eval set variable $value
}

# list consists of {text command} [{text command}] ...
proc Widget::makeButtonBox {widget list} {
    if {[winfo exists $widget]} {
        destroy $widget
    }
    tixButtonBox $widget -orientation horizontal
    pack $widget -fill x
    set i 0
    foreach item $list {
        set text [lindex $item 0]
        set command [lindex $item 1]
        $widget add button$i -text $text -command $command
        incr i
    }
}

proc Widget::makeSeparator {parent frame {orient VERT} {value ""}} {
    set frame [Widget::makeOrRenew frame $parent$frame]
    if {$orient == "VERT"} {
        if {$value == ""} {set value 10}
        $frame configure -height $value -background grey
        pack $frame -fill x -side top
    } elseif {$orient == "HORIZ"} {
        if {$value == ""} {set value 10}
        $frame configure -width $value -background grey
        pack $frame -fill y -side left
    }
}
    
# make menu of radio buttons (tuples are name, initvalue)
proc Widget::makeRadioMenu {frame title variable tuples {defaultval ""}} {
    set buttons ""
    global [Global::root $variable]
    if {![info exists $variable]} {
        eval set $variable \"$defaultval\"
    }
    foreach tuple $tuples {
        lappend buttons \
            "radiobutton [lindex $tuple 0] $variable \"[lindex $tuple 1]\"" 
    }
    GUI::makeMenu $frame $title "" "-side left" $buttons
}

proc Widget::toggleButtonState {button} {
    set state [lindex [$button configure -state] 4]
    if {$state == "normal"} {
        $button configure -state disabled
    } elseif {$state == "disabled"} {
        $button configure -state normal
    }
}
#---------------------------------TIX-------------------------------------

proc Tix::getFilename {w {filetypes ""}} {
    global CW

    if {$filetypes == ""} {
        set filetypes {{{*} {Any files}}}
    }
    GUI::postStatus "Bringing up file browser"
    set filedlg [tix filedialog tixExFileSelectDialog]

    $filedlg subwidget fsbox config -command "Tix::setFilename " \
        -filetypes $filetypes
    $filedlg popup $w
    GUI::postStatus "loading file"
    tkwait variable CW(FILENAME)
    GUI::postStatus ""
    return $CW(FILENAME)
}

proc Tix::setFilename {file} {
    global CW
    set CW(FILENAME) $file
}

# makes a panes window.  returns a list of the panes (use them as frames)
# requires that $parent is a valid window name, has an exisiting parent
# but that $parent does not exist.
# names is a list of names with options:
#	name [<min>] [<size>], e.g. "fred 100 200"
proc Tix::makePane {parent names {orient vertical} {min 50}} {
    tixPanedWindow $parent -separatorbg gray50 -orientation $orient
    set size 100
    foreach name $names {
         set win [lindex $name 0]
         if {[lindex $name 1] != ""} {set min [lindex $name 1]}
         if {[lindex $name 2] != ""} {set size [lindex $name 2]}
         lappend panes [$parent add $win -min $min -size $size]
    }
    pack $parent -expand yes -fill both
    return $panes
}

# makes a tixNoteBook 
# Example: set frames [MkNoteBook $frame {{fred "FRED BOX"} {jim "JIM PAGE"}}]
#        text [lindex $frames 0].fred ;# and so on
proc TIX::noteBook {w pages} {

    tixNoteBook $w.nb -ipadx 6 -ipady 6
    pack $w.nb

    foreach page $pages {
        set frame [lindex $page 0]
        set label [lindex $page 1]
        $w.nb add $frame -label $label
# Create the page
        set newFrame [$w.nb subwidget $frame]
        lappend  newFrames $newFrame
    }
    return $newFrames 
}

#------------------------------Command Line------------------------------

# makes the command line box
proc CommandLine::construct {win} {
    global CW

    frame $win
    label $win.l -text "Command:"
    entry $win.entry -relief sunken -textvariable CW($win.entry)
    pack $win -fill x -side bottom
    pack $win.l -side left
    pack $win.entry -side left  -expand yes -fill x
    bind $win.entry <Return> "CommandLine::execute $win.entry"
    bind $win.entry <Up> "CommandHistory::recall $win.entry up"
    bind $win.entry <Down> "CommandHistory::recall $win.entry down"
}

# execute a command typed in
# NOTE: The user has access to the CW and env arrays.  Use them wisely...
# NOTE: The command is passed through the interpreter twice.  This means 
# that quoting can be difficult (for [] and others).  
# Example:
#	puts stdout [expr 2+2]
# needs to be:
#	puts stdout \[expr 2+2\]
# An alternative command is available to overcome these problems if the user
# simply wishes to execute a command:
#	DISPLAY <command>
# will transmit the string <command> to the interpreter, and output the result
# to a text widget.  STDOUT will do the same to stdout.


proc CommandLine::execute {win} {
    global CW env

    if {[regexp {\.entry$} $win]} {
        set text [$win get]
        $win delete 0 end
    } elseif {$win == ".script.t"} {
# not yet working
        set text [$win get 1.0 end]
        $win delete 1.0 end
    }
# trap STDOUT and DISPLAY.
    set firstwd ""
    if {[regexp {^[ ]*(STDOUT|DISPLAY)[ ]+(.*)} $text x firstwd rest]} {
        set text "puts stdout \[$rest\]"
puts TEXT:$text/$firstwd/
    }
    if {[catch "eval $text" msg]} {
        GUI::errorMessage "executeCommandLine: Bad command ($msg): $text"
    } else {
        lappend CW(COMMAND,HISTORY) $text
    }
    set CW(COMMAND,HISTORY,CURSORLINE) \
        [llength $CW(COMMAND,HISTORY)]
}

proc CommandHistory::recall {win arrow} {
    global CW

    if {![info exists CW(COMMAND,HISTORY)]} {
        set CW(COMMAND,HISTORY) ""
    }
    if {![info exists CW(COMMAND,HISTORY,CURSORLINE)]} {
        set CW(COMMAND,HISTORY,CURSORLINE) \
            [llength $CW(COMMAND,HISTORY)]
    }
    if {$arrow == "up"} {
        if {$CW(COMMAND,HISTORY,CURSORLINE) > 0} {
            incr CW(COMMAND,HISTORY,CURSORLINE) -1
        }
    } elseif {$arrow == "down"} {
        if {$CW(COMMAND,HISTORY,CURSORLINE) < \
            [expr [llength $CW(COMMAND,HISTORY)]-1]} {
            incr CW(COMMAND,HISTORY,CURSORLINE) 
        }
    }
    set cmd [lindex $CW(COMMAND,HISTORY) $CW(COMMAND,HISTORY,CURSORLINE)]
    set CW($win) $cmd
}

# text widget to receive a command line script and execute it
# NOT YET WORKING
proc GUI::displayCommandScript {win} {
    set script [Widget::makeOrRenew toplevel $win]
    set text $script.t
    text $text
    pack $text -side top
    set bottom [Widget::makeOrRenew frame $script.bottom]
    pack $bottom -side top -fill x -expand yes
    set ok [Widget::makeOrRenew button $bottom.ok]
    $ok configure -text OK -command "executeCommandLine $text"
    pack $ok -side left -fill x -expand yes
    set clear [Widget::makeOrRenew button $bottom.clear]
    $clear configure -text Clear -command "$text delete 1.0 end"
    pack $clear -side left -fill x -expand yes
    set cancel [Widget::makeOrRenew button $bottom.cancel]
    $cancel configure -text Cancel -command "destroy $script"
    pack $cancel -side left -fill x -expand yes
}

proc Log::output {msg} {
    global CW
    lappend CW(LOG,LINES) $msg
}

proc Log::display {} {
    global CW
    if {[info exists CW(LOG,LINES)]} {
        set toplevel [Widget::makeOrRenew toplevel .outputLog]
        TextWindow::createWithScrollbar $toplevel.text
        $toplevel.text insert 1.0 $CW(LOG,LINES)
    } else {
        GUI::errorMessage "No log output generated (yet?)"
    }
}

proc GIF::display {gif} {
    global CW
    CW::addPID [exec $CW(GIF,DISPLAYCOMMAND) $gif &]
}

