#-------------------------------------------------------------------------
# A set of core utilities for processing CML documents in costwish.
# some are generic
#-------------------------------------------------------------------------

puts "Reading cwlib.tcl..."

#----------------------------------SGML-----------------------------------

proc SGML::initialise {} {
    global env CW

    SGML::userInitialise 
}

proc SGML::readCatalogs {file} {
    set string ""
    set f [open $file r]
    while {[gets $f line] != -1} {
        append string " $line"
    }
    close $f
    set result ""
    while {1} {
        set comstart [string first "--" $string]
        if {$comstart == -1} {
            append result [string range $string 0 end]
            break
        }
        append result [string range $string 0 [expr $comstart-1]]
        set string [string range $string [expr $comstart+2] end]
        set comend [string first "--" $string]
        if {$comend == -1} {
            GUI::errorMessage "Unbalanced comments in catalogs file \"$file\""
            break
        }
        set string [string range $string [expr $comend+2] end]
    }
    set catalogs ""
    while {1} {
        set result [string trim $result]
        if {$result == ""} break
        if {[regexp -nocase \
            {^CATALOG  *SYSTEM  *"([^"]*)"(.*)} $result junk cat result]} {
            if {[file exists $cat]} {
                lappend catalogs $cat
            } else {
                GUI::errorMessage "File ($cat) in list of catalogs does not exist"
            }
        } else {
            GUI::errorMessage "Bad line ($result) in catalogs file"
            set catalogs ""
            break
        }
    }
    return $catalogs
}

proc SGML::setDTDtype {dtd} {
    global CW
    set CW(DTD,TYPE) $dtd
    CW::updateInfoBar dtd $dtd
}

proc SGML::getDTDtype {} {
    global CW
    if {[info exists CW(DTD,TYPE)]} {
        return $CW(DTD,TYPE)
    } 
}

# outputs *.SGM, *.ESIS or *.HTML
proc SGML::writeFile {{language SGML}} {
    global CW
    set fileH [Output::openFile]
    if {$fileH == ""} return
    set of [lindex $fileH 1]
    if {[lsearch {SGML HTML ESIS} $language] == -1} {
        GUI::errorMessage "SGML::writeFile: Cannot save as $language"
        return
    }
    Output::setLanguage $language
    GUI::postStatus "Writing $language tree to file"
    if {$language == "SGML"} {
        GUI::postStatus "WARNING. Extraneous NEWLINES inserted (in SGML). SORRY"
# alter this...
        Output::write [SGML::getDTDName DOCTYPE]
        recursiveTocDisplay 0 
    } elseif {$language == "ESIS"} {
# this is messy because we cannot easily pass these things through the args
        set CW(CONTENT) ""
        set CW(OUTFILEHANDLE) $of
        process SGML::esisDump
# this is the final "C" in the ESIS file
        Output::write C
    }
    Output::closeFile
    GUI::postStatus ""
}

proc SGML::getDTDName {{control ""}} {
    if {$control == ""} {
        return CML
    } elseif {control == "DOCTYPE"} {
        return "<!DOCTYPE cml PUBLIC \"-//CML//DTD CML V0.6G//EN\">
    }
}

# CoST event stream processing.  This is why we use global variables...
proc SGML::esisDump {event} {
    switch $event {
        START {
            global CW
            if {$CW(CONTENT) != ""} {
                Output::write "-$CW(CONTENT)"
            }
            set list ""
            foreachNode attlist {
                set attname [query attname]
                set attval [query parent attval $attname]
                if {$attval == ""} {
                    set string IMPLIED
                } else {
                    set string "TOKEN $attval"
                }
                lappend list "A$attname $string"
            }
            set l [expr [llength $list]-1]
            for {set i $l} {$i >= 0} {incr i -1} {
                 Output::write "[lindex $list $i]"
            }
            Output::write "([query gi]"
            set CW(CONTENT) ""
        }
        END {
            global CW
            if {$CW(CONTENT) != ""} {
                Output::write "-$CW(CONTENT)"
            }
            Output::write ")[query gi]"
            set CW(CONTENT) ""
        }
        RE {
            global CW
            append CW(CONTENT) "\\n"
        }
        CDATA {
            global CW
            append CW(CONTENT) [content]
        }
    }
}

# this is the command used for (say) sgmls
# type might be SGMLS or NORMALISED
proc SGML::setCommand {type command} {
    global CW
    set CW(SGML,$type) $command
}

proc SGML::parseFile {type} {
    global CW
    if {[info exists CW(SGML,$type)]} {
        return $CW(SGML,$type)
    } else {
        GUI::errorMessage "No Command set up for producing ESIS ($type)"
    }
}

# can load the following types at present: 
#    ESIS, NORM (normalised SGML w/o DTD), SGML (with DTD)
proc SGML::loadDocument {{type ESIS} {file ""}} {

    global CW
# try to determine file type from suffix
    if {$type == "UNK"} {
# is it ESIS?
        if {[regexp -nocase "\.esi\[s\]?" $file]} {
            set type ESIS
        } else {
# no, assume SGML with DTD
            set type SGML
        }
    }
    if {$type == "ESIS"} {
        if {$file == ""} {
            set file [Tix::getFilename . {{{*.esis} {ESIS}}}]
        }
        loadfile $file
        GUI::postStatus "Displaying ESIS tree"
    } elseif {$type == "NORM"} {
        if {$file == ""} {
            set file [Tix::getFilename . {{*.sgml SGML}}]
        }
        GUI::postStatus "parsing normalised SGML file: $file"
        set msg [SGML::loadFile $file NORM]
        if {$msg != ""} {
             GUI::errorMessage "Could not parse normalised file: $file"
             return
        }
        GUI::postStatus ""
    } elseif {$type == "SGML"} {
        if {$file == ""} {
            set file [Tix::getFilename . \
                {{*.sgml SGML} {*.cml CML} {*.html "HTML2.0"}}]
        }
        GUI::postStatus "loading and parsing SGML file: $file"
# I could use loaddoc here, but Joe says it has a memory leak (2.02a)
        set msg [SGML::loadFile $file SGML]
        if {$msg != ""} {
             GUI::errorMessage "SGMLS: could not parse file: $file"
             return
        }
        GUI::postStatus ""
    }
    set CW(DOCUMENT,FILENAME) $file
    CW::updateInfoBar file $file
    SGML::processESIS
    Toc::display $CW(MAINTOC) $CW(TOC,DEPTH) 1:1
    GUI::postStatus ""
}

# this carries out user procedures on the ESIS file - before display
proc SGML::processESIS {} {
    SGML::setDTDtype [query docroot child el gi]
    SGML::loadPostprocessor [SGML::getDTDtype]
    GI::makeHiddenMenu NODE
    GI::makeHiddenMenu CHILDREN
    GUI::postStatus "Processing ESIS tree after loading"
    CW::userProcessESIS
    GUI::postStatus "Making GI list"
    GI::makeList
    GUI::postStatus ""
}

proc SGML::loadFile {file {type SGML}} {

    set retval ""
    set command [SGML::parseFile $type]
    puts stdout "exec'ing $command $file"
# Joe says it is the 'close' command which needs to be 'caught'
    set f [open "|$command $file" r+]
    if {[catch "loadsgmls $f" msg]} {
        set retval $msg
    }
    if {[catch "close $f" msg]} {
        set msg0 [string range $msg 0 1000]...
        if {[Widget::makeDialog "Parsing errors ($type) in file :\n\n${msg0}\n\nDo you want to see the error messages in detail? (This may give some hints as to what may have gone wrong)" \
            YES NO NO] == "YES"} {
            SGML::interpretErrors $msg
        }
    }
    return $retval
}

proc SGML::interpretErrors {msg} {

    global CW env

# note - these msgs must not have explicit ( or ) in
    set errs {\
    "No DOCTYPE declaration; document type is unknown" \
    "DOCTYPE markup declaration not permitted here;" \
    "(.*) markup declaration not permitted here; declaration ended" \
    "Undefined (.*) start-tag GI ignored; not used in DTD" \
    "No element declaration for (.*) end-tag GI; end-tag ignored" \
    "(.*) start-tag implied by data; not minimizable" \
    "(.*) end-tag implied by data; not minimizable" \
    "(.*) end-tag implied by (.*) start-tag; not minimizable" \
    "(.*) end-tag ignored: doesn't end any open element .current is (.*)." \
    "(.*) start-tag implied by (.*) start-tag; not minimizable" \
    "(.*) end-tag implied by (.*) end-tag; not minimizable" \
    "Start-tag omitted from (.*) with empty content" \
    "Possible attributes treated as data because none were defined" \
    "(.*) element not allowed at this point in (.*) element" \
    "Data not allowed at this point in (.*) element" \
    "(.*) element ended prematurely; required subelement omitted " \
    "Out-of-context (.*) start-tag ended (.*) document element .and parse." \
    "End of file in comment" \
    "(.*) IDREF attribute ignored; referenced ID does not exist" \
    "Undelimited attribute value requires .SHORTTAG YES." \
    "Out-of-context data ended (.*) document element .and parse." \
    "Document element end tag can only occur in document element because entity end not allowed in other prolog" \
    "Incorrect character in markup; markup terminated" \
    "Unclosed start or end tag requires .SHORTTAG YES." \
    "Attribute name omission requires .SHORTTAG YES."\
    "(.*) IDREF attribute ignored: referenced ID does not exist" \
    ".PUBLIC. or .SYSTEM. required; declaration terminated" \
    "Could not find external document type .(.*)." \
    "No definition for (.*) document type; .(.*) O O ANY. assumed" \
    "Invalid character.s. ignored. attempting to resume DOCTYPE subset" \
    "No such file or directory" \
    "Entity (.*) terminated: could not read file" \
    "Document ended invalidly within prolog; parsing ended" \
    "Parameter entity name longer than .NAMELEN-1.; truncated" \
    "Length of name, number, or token exceeded NAMELEN or LITLEN limit" \
    "Could not find external general entity .(.*)." \
    "(.*) = .(.*). attribute ignored: not defined for this element" \
    "(.*) = .(.*). attribute value defaulted: invalid character" \
    "Could not open file for entity .(.*).; entity reference ignored" \
    "(.*) element ended prematurely; required (.*) omitted" \
    "Net-enabling start tag requires .SHORTTAG YES." \
    }
 
    set errnos {\
    nodoctype \
    doctypenothere \
    markupnothere \
    undefinedstarttag \
    endefendtag \
    impliedstarttagdata \
    impliedendtagdata \
    impliedendtag \
    ignoredendtag \
    impliedstarttagstarttag \
    impliedendtagendtag \
    starttagnocontent \
    attributesasdata \
    elementnothere \
    datanothere \
    elementprematureend \
    outcontextstart \
    eofincomment \
    noidref \
    shorttagyes \
    outcontextdata \
    docendentity \
    incorrectchar \
    unclosestartend \
    attnameomit \
    refidnonexist \
    nopubsys \
    noextdoc \
    nodefdoc \
    invalidchar \
    nofiledir \
    entityterm \
    endprolog \
    parenttoolong \
    nametoktoolong \
    noextgenent \
    undefattrib \
    badattribchar \
    noentfile \
    premelemend \
    netenable \
    }

# error window
    set errwin [Widget::makeOrRenew toplevel .sgmlerror]
# bottom bar
    set bar [Widget::makeOrRenew frame $errwin.bar]
    pack $bar -side bottom -expand yes
    set button [Widget::makeOrRenew button $bar.quit]
    $button configure -text QUIT -command "destroy $errwin"
    pack $button -fill x 
# text window
    set text $errwin.text
    TextWindow::createWithScrollbar $text 80 25
    wm withdraw $errwin

    set lines [split $msg \n]
    set nlines [llength $lines]
    if {$nlines > 50} {set nlines 50}
    set count 0
    while {$count < $nlines} {
        set line [lindex $lines $count]
        set file ""; set lineno -1; set char ""; set decpar "" ; set file1 ""
# DOCTYPE problems
        if {[regexp -nocase\
          {sgmls:.*(warning|error) *at (.*), line (.*) in declaration parameter (.*)} \
            $line junk severity file lineno decpar]} {
        } elseif {[regexp \
            {sgmls: Error accessing ["]*(.*)["]*:} \
            $line junk file]} {
            incr count;    # skip the next line (Entity...)
        } elseif {[regexp \
            {sgmls: SGML error at (.*), line (.*) at ["]*(.*)["]*:} \
            $line junk file lineno char]} {
            if {$char == "record start"} {set char "^"}
        } elseif {[regexp \
            {sgmls: Error at (.*), line (.*) accessing ["]*(.*)["]*:} \
            $line junk file lineno file1]} { \
            incr count;    # skip the next line (Entity...)
        } elseif {[regexp {sgmls: Catalog error at (.*), line (.*):} \
            $line junk file lineno]} {
        } elseif {[regexp "Entity ..." $line]} {
            continue
        } else {
            GUI::errorMessage "SGML::interpretErrors: cannot parse: $line"
            break
        }
        set errmes [lindex $lines [incr count]]
        set errno 0 
        set errfound -1
# search through errors
        foreach err $errs {
            if {[regexp $err $errmes]} {
                set errfound $errno
                break
            }
            incr errno
        }
# special routine for line 1 (i.e. most drastic error)
        if {$count == 1 && $errfound != -1} {
            set errtype [lindex $errnos $errfound]
            set badLine [SGML::getRawLine $file $lineno]
# couldn't find DTD
            if {$errtype == "noextdoc"} {
                set text "FAILURE TO FIND OR RESOLVE DTD\n\nDo you want guidance?"
                set ans [Widget::makeDialog $text YES NO]
                if {$ans == "YES"} {
                    if {[regexp -nocase {<!DOCTYPE *([^ ]*) *(SYSTEM|PUBLIC) *(.*)} $badLine junk dtd idType rest]} {
                        set idType [string toupper [string trim $idType]]
                        if {$idType == "SYSTEM"} {
                            regsub -all {['">]} \
                                [lindex [split $rest " "] 0] "" file
                            set sgmlPath ""
                            if {[info exists env(SGML_PATH)]} {
                                set sgmlPath $env(SGML_PATH)
                            }
                            GUI::errorMessage "You are trying to use the file \"$file\" as a DTD but it cannot be located.  Check if it exists.  If so, sgmls uses SGML_PATH to search for DTDs and this may not point to where \"$file\" is located.  (Your SGML_PATH is \"$sgmlPath\" - but this may not be obviously useful unless you know sgmls, I'm afraid).\nYou might wish to think about using the catalog and FPIs rather than SYSTEM for your DTDs"
puts //$dtd//$idType//$rest//
                        } elseif {$idType == "PUBLIC"} {
                            if {![regexp {"([^"]*)"} $rest junk fpi]} {
                                GUI::errorMessage "You must have a valid FPI (in quotes) after the word PUBLIC"
                                break 
                            }
                            GUI::errorMessage "You are trying to use the FPI \"$fpi\" to locate your DTD but it cannot be resolved.  This could be because:\n - you haven't used the right catalog\n - you have misspelt your FPI\n - the FPI in the catalog points to an unlocatable file."
                        }
                    } else {
                        GUI::errorMessage "Your DOCTYPE line is:\n\n$badLine\n\nIt seems seriously wrong.  Is it of the form:\n\n<!DOCTYPE foo SYSTEM ...\n\nor\n\n<!DOCTYPE foo PUBLIC ...?"
                    }
                    catch "destroy $errwin"
                    break
                }
            }
        }
        catch "wm deiconify $errwin"
        $text insert end "$errmes in $file (l.$lineno)"
        if {$file1 != ""} {
            $text insert end " (missing file: $file1)"
        }
        if {$errfound != -1} {
            set button [Widget::makeOrRenew button $text.bm$count]
            $text window create end -window $button
            $button configure -text "Help?" \
                -command "HTML::display $CW(DOCDIR)/sgmlerr.html#[lindex $errnos $errfound]"
        }
        incr count
        set button [Widget::makeOrRenew button $text.b$count]
        $text window create end -window $button
        set args [list $file $lineno $char $errmes]
        $button configure -text "View" \
            -command "SGML::viewRawFile $args"
        $text insert end \n
    }
}

proc SGML::getRawLine {file lineno} {
    set rawLine ""
    set count 0
    set f [open $file r]
    while {[gets $f line] != -1} {
        if {[incr count] == $lineno} {
            set rawLine $line
            break
        }
    }
    close $f
    return $rawLine
}

proc SGML::viewRawFile {args} {

    set file [lindex $args 0]
    set lineno [lindex $args 1]
    set char [lindex $args 2]
    set errmes [lindex $args 3]

    set win [Widget::makeOrRenew toplevel .sgmlfile]
    set text $win.text
    TextWindow::createWithScrollbar $text 80 25
    set count 0
    set f [open $file r]
    while {[gets $f line] != -1} {
        if {[incr count] == $lineno} {
            set string ""
            $text insert end "***********************************************\n"
            append string "vvvvv "
            append string ($errmes)
            append string " at "
            append string ($char)
            append string  " vvvvv"
            $text insert end $string\n
            $text insert end "***********************************************\n"
        }
        $text insert end "$line\n"
    }
}

proc SGML::addEmptyTags {taglist} {
    global CW
    foreach tag $taglist {
        set CW(EMPTY,[string toupper $tag]) 1
    }
}

# there are some attributes which are defaulted by the DTD and might not need 
# to be written out to SGML
proc SGML::addSilentAttributes {attlist} {
    global CW
    foreach att $attlist {
        set CW(SILENTATTRIBUTE,[string toupper $att]) 1
    }
}


proc SGML::isEmptyTag {tag} {
    global CW
    set tag [string toupper $tag]
    if {[info exists CW(EMPTY,$tag)]} {
        if {$CW(EMPTY,$tag) == 1} {
            return 1
        }
    }
    return 0
}
        
proc SGML::isSilentAttribute {att} {
    global CW
    set att [string toupper $att]
    if {[info exists CW(SILENTATTRIBUTE,$att)]} {
        if {$CW(SILENTATTRIBUTE,$att) == 1} {
            return 1
        }
    }
    return 0
}

# loads the postprocessing software
proc SGML::loadPostprocessor {dtd} {
    global CW

    set dtd [string toupper $dtd]
    if {[info exists CW(DTD,LOADED,$dtd)]} return
  
# if we *can* load a postprocessor, do so
    set CW(DTD,LOADED,$dtd) 1
    if {[info exists CW(TCLDIR,$dtd)] && \
        [file exists $CW(TCLDIR,$dtd)/postproc.tcl]} {
        puts "Loading postprocessor for: $dtd"
        source $CW(TCLDIR,$dtd)/postproc.tcl
    }
}

#---------------------------TableOfContents-------------------------------

proc Toc::getMaxOccurrences {} {
    return 500
}

proc Toc::autoRedisplay {win {address ""}} {
    global CW
    if {$CW(DISPLAY,AUTO)} {
        Toc::display $win $CW(TOC,DEPTH) [Node::getAddress $address]
    }
}

proc Toc::display {win depth {address ""} {title ""}} {
    global CW TOC

    Node::saveCurrent
    set address [Node::getAddress $address]
    selectNode node $address


    if {!$CW(DOCUMENT,DISPLAY)} return
    if {[SGML::getDTDtype] == ""} return
    if {[catch "[query docroot]" msg]} {
        puts stderr "No current document: $msg"
        return
    }

# display of nodes
    set TOC(CONTENT) 0
    if {$CW(DISPLAY,GI) == "TITLE"} {
        set TOC(GI) 0
        set TOC(TITLE) 1
    } else {
        set TOC(GI) 1
        set TOC(TITLE) 0
    }

# too many nodes??
#    GUI::postStatus "Checking number of nodes"
#    set totalNodes [Node::getCount ""]
#    if {$totalNodes > [Toc::getMaxOccurrences]} {
#        set gis [GI::getList]
#        foreach gi $gis {
#            if {[Node::getCount $gi] > [Toc::getMaxOccurrences]} {
#                GUI::postStatus "Hiding Nodes: $gi"
#                GI::setHidden $gi 1 NODE
#            }
#        }
#    }
    GUI::postStatus ""
    set topel [query docroot child el gi]
    SGML::setDTDtype $topel
    .info.dtdval configure -text $topel
    Log::output ">SGML::setDTDtype $topel"

    regsub {\.[^\.]*$} $win ".lab" label
# title
    if {$title == ""} {
        set title [Node::getMeaningfulTitle]
    }
    set label [Widget::makeOrRenew label $label]
    $label configure -text "TOC for: $title"
    pack $label -fill x
    TextWindow::createWithScrollbar $win 80 25
    set level [Node::getLevel $address]
    GUI::postStatus "Displaying Table of Contents"
    
# depth is given incrementally - make it absolute
#    set depth [expr $depth+$level]
    set level 1
    Toc::outputNode $win 1 1
    Toc::recursiveDisplay $win $level $depth
    GUI::postStatus ""
    Toc::output $win "\n"
# disable TOC from writing
    bind $win <KeyPress> "GUI::errorMessage \"Cannot edit text, sorry\""

    Node::restoreCurrent
}

# display all nodes, moderated by DISPLAY and HIDE
# depth not fully worked out...
proc Toc::recursiveDisplay {win level {depth 99}} {

# can print to the screen (win = <widget>) or to output (win == "")

    global CW
    global TOC

    incr level
# return if screen display and hideNode
##    if {$win != "" && [GI::isHidden [query gi]] && $depth != "ALL"} return

    foreachNode child {
        Toc::outputNode $win $depth $level
    }
    incr level -1
}

proc Toc::showAncestry {win address {tagName ""}} {
    set address [Node::getAddress $address]
    set ancestors [Node::getAncestors $address]
    if {$ancestors == ""} {
        GUI::errorMessage "Apparently no TOC on screen!"
        return
    }
    set nodes ""
    set maxlevel [llength $ancestors]
    set level [expr $maxlevel+1]
    foreach ancestor $ancestors {
        set line [Toc::getLineFromAddress $win $ancestor]
        if {$line != ""} {
            break
        }
        set nodes [concat $ancestor $nodes]
        incr level -1
    }
# insert hierarchy just after first visible ancestor (change this later
# since we have to worry about other visible siblings.
    foreach node $nodes {
        withNode node $node {
            set parent [query parent address]
            set prevNode $parent
# go backwards through the elder siblings till we reach a visible one
            set prevs [query* prev address]
            foreach prev $prevs {
                set line [lindex [Toc::getLineFromAddress $win $prev] 0]
                if {$line != ""} {
                    set prevNode $prev
                    break
                }
            }
# go forward through the younger siblings till we reach a visible one
            set ysibLine ""
            set ysib ""
            set ysibs [query* ysib address]
            foreach ysib $ysibs {
                set ysibLine [lindex [Toc::getLineFromAddress $win $ysib] 0]
                if {$ysibLine != ""} break
            }

# no prev? use parent
            if {$line == ""} {
                set line [lindex [Toc::getLineFromAddress $win $parent] 0]
            }
# now drop until we find a line which is NOT a descendant of previous node
# or is the younger sibling
            while {1} {
                set tag [Toc::getTagFromLine $win $line]
                if {$tag == "" || ![Node::hasAncestor $tag $prevNode]} {
                    break
                }
                incr line
                if {$line == $ysibLine} break
            }
# puts "prevNode: $line ($prevNode); ysib: $ysibLine ($ysib)"

            $win mark set insert $line.0
            set outtag ""
            if {$level == $maxlevel} {
                set outtag $tagName
            }
            Toc::outputNode $win $level $level $outtag
        }
        incr level
    }
}
        

proc Toc::getIndent {level} {
    global TOC CW
    set TOC(INDENT) ""
    set spaces ""
    incr level -1
    if {$CW(DISPLAY,INDENT)} {
        set TOC(INDENT) "      "
        for {set i 0} {$i < $level} {incr i} {
            append spaces $TOC(INDENT)
        }
    }
    return $spaces
}

# assumes that the currently selected node is the right one...
# still messy til we get the priority for display sorted out
proc Toc::outputNode {win depth level {tagName ""}} {
    global TOC CW

    if {$depth < $level} return
    set spaces [Toc::getIndent $level]
    set gi [query gi]
    set nodetype [query nodetype]

# this is very messy - really needs rewriting in OO form
    if {$win == "" || \
        (![GI::isHidden [query gi]] && \
        ($nodetype != "PEL" || $CW(DISPLAY,PEL)))} {

        if {$TOC(TITLE)} {
            set title [Node::getMeaningfulTitle]
        } else {
            set title ""
        }
#            if {$nodetype == "PEL"} {set title *P}
#            if {$nodetype == "CDATA"} {set title *C}

        if {$CW(DISPLAY,ADDRESS)} {
            set add "([query address])"
        } else {
            set add ""
        }
        if {$CW(DISPLAY,NODETYPE)} {
            set node "([query nodetype])"
        } else {
            set node ""
        }
        if {$TOC(GI)} {
            set gitext "<[query gi]>"
        } else {
            set gitext ""
        }
        
        if {$nodetype == "CDATA"} {
            if {$win != ""} {
                Toc::output $win "$spaces$TOC(INDENT)"
                Toc::output $win [content]
                set TOC(CONTENT) 1
            } else {
                Markup::output [content]
            }
        } else {
            if {$win != ""} {
                if {$nodetype != "PEL"} {
                    if {$nodetype != "RE"} {
                        if {$TOC(CONTENT)} {Toc::output $win "\n"}
                        Toc::output $win "$spaces"
                        Toc::output $win \
                           "$title$add$node$gitext" $gi [query address] $tagName
                    }
                    Toc::output $win "\n"
                    set TOC(CONTENT) 0
                }
            } else {
                Markup::output $gi START
            }
        }

        if {$win == "" || ![GI::isHidden $gi CHILDREN]} {
            Toc::recursiveDisplay $win $level $depth
        }
        if {$win == ""} {
            if {$nodetype != "CDATA"} {
                Markup::output [query gi] CLOSE
            }
        } else {
            if {$TOC(GI) && $gi != ""} {
                Toc::output $win "$spaces</$gi>\n"
            }
        }
    }
}

# output text.  If address is given tag the text with this address
proc Toc::output {win text {gi ""} {address ""} {tagName ""}} {
    global CW

# insert the text; always insert at the insert cursor (end by default)
    $win insert insert $text $tagName


# only tag it if an explicit address is given...
    if {$address == ""} return

    $win tag remove $address 1.0 end

# find out where the text lies
    set lt [string length [string trim $text]]
# don't tag RE's
    if {$lt == 0} return
    incr lt
    set re 0
# count newlines
    if {[string first \n $text] != -1} {
        incr lt
        incr re
    } 
    set end [$win index insert-${re}c]
    set start [$win index insert-${lt}c]
    $win tag add $address $start $end

    if {$gi == "" || $address == ""} return
    $win tag configure $address -foreground [GI::getColor $gi] \
        -background white -border 1
    $win tag bind $address <Enter> "$win tag configure $address -relief raised"
    $win tag bind $address <Leave> "$win tag configure $address -relief flat"
    $win tag bind $address <Button-1> "Toc::giSelectAction $win $address"
# add attributes
#    set att $address.att
#    $win insert end (A) $att
#    $win tag bind $att <Enter> "$win tag configure $att -relief raised"
#    $win tag bind $att <Leave> "$win tag configure $att -relief flat"
#    $win tag bind $att <Button-1> "Node::displayAttributes $win $address"

# expand and contract nodes if there are children
    set childtype [query node $address child nodetype]
    
    if {$childtype == "EL"} {
        set plus $address.plus
        $win insert insert + $plus
        $win tag configure $plus -background white
        $win tag bind $plus <Button-1> "Toc::displayChildren $win $address"
    
        set minus $address.minus
        $win insert insert - $minus
        $win tag configure $minus -background white
        $win tag bind $minus <Button-1> "Toc::undisplayChildren $win $address"
    }
# display content if present
    if {$CW(DISPLAY,CONTENT)} {
        set content [Node::getMixedContent $address]
        if {$content != ""} {
            Toc::output $win "   [string range $content 0 30]..."
        }
    }
}

proc Toc::getLineFromAddress {win address} {
    set line ""
    set class [winfo class $win]
    if {$class != "Text"} {
        GUI::errorMessage "Toc::getLineFromAddress called with non-text: $win"
        return
    }
    regexp {(.*)\..*} [lindex [$win tag ranges $address] 0] j line
    return $line
}

proc Toc::displayChildren {win address} {
    GUI::postStatus "displaying children"
# get the level
    withNode node $address {
        set level [countq ancestor]
    }

# undisplay the children first...
    Toc::undisplayChildren $win $address
    set startLine [lindex [Toc::getLineFromAddress $win $address] 0]
    set line [expr $startLine+1]
    $win mark set insert $line.0
    foreachNode node $address child {
        Toc::outputNode $win $level $level
    }
    GUI::postStatus ""
}

proc Toc::getTagFromLine {win line} {
# get only tags of form d:d - not plus or minus
    set index [$win search -regexp -nocase {[^ ]} $line.0]
    set tags [$win tag names $index]
    set tag [lindex $tags 0]
    if {[string first : $tag] != -1} {
        return $tag
    }
}

# deletes all descendants of a node
proc Toc::undisplayChildren {win address} {

    GUI::postStatus "hiding children"
    set firstLine ""
    set startLine [lindex [Toc::getLineFromAddress $win $address] 0]
    regsub {\..*} $startLine "" startLine
    set line $startLine
    while {1} {
        incr line
        set lineAddress [Toc::getTagFromLine $win $line]
        if {![Node::hasAncestor $lineAddress $address]} break
    }
    incr startLine
    $win delete $startLine.0 $line.0
    GUI::postStatus ""
}

# mark addresses on TOC
proc Toc::newTags {frame addresses color tagName} {
    $frame tag delete $tagName
    foreach address $addresses {
        Node::addTextTag $frame $tagName $address
    }
    $frame tag configure $tagName -background $color
}

#--------------------------------  NODES  ----------------------------------

# take action when a node is clicked in the text widget
proc Toc::giSelectAction {win address} {
    global CW
    
    if {$CW(NODEACTION) == "SUBTREE"} {
        Node::display $win $address
    } elseif {$CW(NODEACTION) == "PANED"} {
        Node::display $win $address
    } elseif {$CW(NODEACTION) == "SELECT"} {
        Toc::selectNode $win $address 1
    } elseif {$CW(NODEACTION) == "UNSELECT"} {
        Toc::selectNode $win $address 0
    } elseif {$CW(NODEACTION) == "EXPAND"} {
        set toplevel \
     [TopLevel::createFromParent $win .text[Node::getOKID $address] $win $address]
        set text $toplevel.text
        TextWindow::createWithScrollbar $text 80 25
        withNode node $address {
            Toc::recursiveDisplay $text 0 ALL
        }
    } elseif {$CW(NODEACTION) == "ATTRIBS"} {
        Toc::selectNode $win $address SELECT
    } elseif {$CW(NODEACTION) == "ADD"} {
    } elseif {$CW(NODEACTION) == "DELETE"} {
    } elseif {$CW(NODEACTION) == "EDIT"} {
    }
}

proc Toc::selectNode {win address control} {
    global CW
# make a list of the selected nodes - I ought to be able to do this
# with text tags, but I can't manage the syntax yet
    set color(0) white
    set color(1) lightgrey
    if {![info exists CW(NODE,SELECTEDLIST)]} {
        set CW(NODE,SELECTEDLIST) ""
    }
    if {![info exists CW(NODE,SELECTED,$address)]} {
        set CW(NODE,SELECTED,$address) -1
        lappend CW(NODE,SELECTEDLIST) $CW(NODE,SELECTED,$address)
    }
    set CW(NODE,SELECTED,$address) $control
    $win tag configure $address -background $color($control)
}
#---------------------------------MARKUP----------------------------------

proc Markup::output {text {control CDATA}} {

    if {$text == ""} return
    if {$control == "CDATA"} {
        Output::write $text
    } elseif {$control == "START"} {
        if {[Output::getLanguage] == "HTML" && \
            [info proc $text::formatAttributes] != ""} {
            set string "<BR>"
            append string [Tcl::evalQuote $text::formatAttributes [query address]]
        } else {
            set string "<$text"
            set tuples [Node::getTuples]
            foreach tuple $tuples {
                set name [string toupper [lindex $tuple 0]]
                set val [lindex $tuple 1]
                if {$val != "" && ![SGML::isSilentAttribute $name]} {
                    append string " $name=\"$val\""
                }
            }
            append string ">"
        }
        Output::write $string
    } elseif {$control == "CLOSE"} {
        if {[Output::getLanguage] == "HTML" && \
            [info proc $text::formatAttributes] != ""} {
        } else {
            if {![SGML::isEmptyTag $text]} {
                Output::write "</$text>"
            }
        }
    } else {
        GUI::errorMessage "Markup::output: Bad control: $control"
    }
}


#--------------------------------GI---------------------------------------

proc GI::setHidden {gi onOff {control "NODE"}} {
    global CW
    set CW(HIDE,$control,$gi) $onOff
}

proc GI::isHidden {gi {control "NODE"}} {
    global CW
    if {[info exists CW(HIDE,$control,$gi)]} {
        return $CW(HIDE,$control,$gi)
    }
    return 0
}

proc GI::getHiddenList {{control "NODE"}} {
    global CW
    set tags ""
    foreach name [array names CW] {
        if {[regexp "HIDE,$control,(.*)" $name junk tag]} {
            lappend tags $tag
        }
    }
    return $tags
}

proc GI::makeHiddenMenu {{control "NODE"}} {

    global CW

    if {$control == "NODE"} {
        set label HideNode
        set filename hidenode.html
        set win .bar.hidenode
        set text "Hide Node"
    } elseif {$control == "CHILDREN"} {
        set label HideChild
        set filename hidechild.html
        set win .bar.hidechild
        set text "Hide Children"
    } else {
        GUI::errorMessage "GI::makeHiddenMenu: bad arg : $control"
        return
    }
    catch "destroy $win"
        
    set buttons ""
    lappend buttons "separator"
    lappend buttons "label $label"
    lappend buttons "separator"
    lappend buttons "command Help \"HTML::display $CW(DOCDIR)/$filename\""
    foreach gi [GI::getHiddenList $control] {
        lappend buttons \
            "checkbutton $gi CW(HIDE,$control,$gi) $CW(HIDE,$control,$gi) \
                 \"-foreground [GI::getColor $gi]\""
    }
    lappend buttons {command Redisplay "Toc::display $CW(MAINTOC) $CW(TOC,DEPTH) 1:1"}
    GUI::makeMenu $win "$text" "" "-side left" $buttons
    pack $win -after .bar.view
}


# makes a list of all GIs in the tree in order
proc GI::makeList {} {
    global CW
# clear existing list
    if {[info exists CW(GILIST)]} {
        foreach gi $CW(GILIST) {
            unset CW(GI,$gi)
        }
    }
    set CW(GILIST) ""
    foreachNode doctree {
        set gi [query gi]
        if {$gi != "" && ![info exists CW(GI,$gi)]} {
            set CW(GI,$gi) 1
            lappend CW(GILIST) $gi
        }
    }
}

proc GI::getList {} {
    global CW
    if {![info exists CW(GILIST)]} {
        GI::makeList
    }
    return $CW(GILIST)
}

#------------------------------- Node -----------------------------------

# returns nodeCount ("" gives all, else gives it for $gi)
proc Node::getCount {gi} {
    if {$gi != ""} {
        return [countq doctree withgi $gi]
    } else {
        return [countq doctree]
    }
}

proc Node::getGI {{address ""}} {
    if {$address != ""} {
        withNode node $address {
            return [query gi]
        }
    } else {
        return [query gi]
    }
}

# gets level in hierarchy (docroot = 0)
proc Node::getLevel {{address ""}} {
   
    if {$address != ""} {
        set address [Node::getAddress]
    }
    return [expr [countq node $address ancestor] -1]
}
proc Node::getAncestors {{address ""}} {
    set ancestors [query* node [Node::getAddress $address] ancestor address]
    return $ancestors
}

# is node2 an ancestor of node1 (node is ancestor of self...)
proc Node::hasAncestor {node1 node2} {
    foreachNode node $node1 ancestor {
        if {[query address] == $node2} {return 1}
    }
    return 0
}

# is node2 a child of node1
proc Node::isChildOf {node1 node2} {
    withNode node $node2 parent {
        if {[query address] == $node1} {return 1}
    }
    return 0
}

# save current node (does NOT stack)
proc Node::saveCurrent {} {
    global NODE
    set NODE(CURRENT) [Node::getAddress]
}

proc Node::restoreCurrent {} {
    global NODE
    selectNode node $NODE(CURRENT)
}

proc Node::getParentAddress {{address ""}} {
    set newaddress ""
    withNode node [Node::getAddress $address] parent {
        set newaddress [query address]
    }
    return $newaddress
}

# gets address of current node (or returns address directly)
# (it caters for the case where address may or may not be "")
proc Node::getAddress {{address ""}} {
    if {$address == ""} {
        set address [query address]
    }
    return $address
}

# try to get some useful info about a node
proc Node::getMeaningfulTitle {{address ""}} {

    set address [Node::getAddress $address]
# this must be DTD dependent for most things
    set title [Node::userGetMeaningfulTitle $address]
# else return the GI
    if {$title == ""} {
        set title [Node::getGI $address]
    }
    return $title
}

proc Node::getPropval {propname address} {
    set result ""
    withNode node $address {
        set result [query propval $propname]
    }
    return $result
}

proc Node::setPropval {propname propval {address ""}} {
    set address [Node::getAddress $address]
    withNode node $address {
        setprop $propname $propval
    }
}

proc Node::getAttributeValue {attname {address ""}} {
    set address [Node::getAddress $address]
    set attval ""
    withNode node $address {
        set attval [query attval $attname]
    }
    return $attval
}

proc Node::getTuples {{address ""}} {
    set address [Node::getAddress $address]
    set tuples ""
    withNode node $address {
        foreachNode attlist {
            lappend tuples [list [query attname] [content]]
        }
    }
    return $tuples
}

# get attributes as a string of form: Name1="val1" NAME2="val2" ...
# Omit attributes with null values if $nullVakue==OMIT
proc Node::getAttributeString {{address ""} {nullValue OMIT}} {
    set address [Node::getAddress $address]
    set string ""
    foreach tuple [Node::getTuples $address] {
        set value [lindex $tuple 1]
        if {$nullValue != "OMIT" || $value != ""} {
            append string " [string toupper \
                [lindex $tuple 0]]=\"$value\""
        }
    }
    return $string
}


proc Node::getAttnames {{address ""}} {
    set names ""
    set tuples [Node::getTuples $address]
    foreach tuple $tuples {
        lappend names [lindex $tuple 0]
    }
    return $names
}
    
# substitute the : in the address string as it can cause parsing problems
proc Node::getOKID {address} {
    regsub {[:]} $address z temp
    return $temp
}

# reverses the above
proc Node::ungetOKID {win} {
    if {[regexp {([^0-9])([0-9]*)z([0-9]*)} $win junk junk one two]} {
        return $one:$two
    }
}
# this adds [pid] to OKId to ensure uniqueness of filenames
proc Node::getUniqueOKID {address} {
    return [Node::getOKID $address][pid]
}

# returns a list of all the immediate children of a node
proc Node::getAddressesOfChildren {address} {
    set addresses ""
    foreachNode node $address child {
        lappend addresses [query address]
    }
    return $addresses
}

# gets the raw content of a node  (at $address or current ($address = ""))
proc Node::getContents {{address ""}} {
    withNode node [Node::getAddress $address] {
        return [content]
    }
}

# this will get the content ONLY from PEL nodes and will return an
# event stream for the whole content (e.g. "the <B>strong</B> arm"
# else ""
proc Node::getMixedContent {{address ""}} {
    withNode node [Node::getAddress $address] child PEL {
        set content [Node::getEventStream $address]
        return $content
    }
}

proc Node::getEventStream {address} {
    set content ""
    foreachNode node $address child {
        set newAddress [query address]
        set nodetype [query nodetype]
        if {$nodetype == "EL"} {
            set gi [query gi]
            set attstring [Node::getAttributeString $newAddress]
            append content "<$gi$attstring>"
        } elseif {$nodetype == "RE"} {
            append content "\n"
        } elseif {$nodetype == "CDATA"} {
            append content [content]
        }
        append content [Node::getEventStream $newAddress]
        if {$nodetype == "EL"} {
            append content "</$gi>"
        }
    }
    return $content
}

proc Node::getUnescapedContentAsLines {} {
    regsub -all "\\&lt\;" [content] "<" content
    return [split $content \n]
}

# is a node in a particular context? (PARENT, ANCESTOR, CHILD, DESCENDANT)
# e.g. Node::hasContext ANCESTOR CML returns 1 if CML is an anecstor of
# the node with given address ("" = current), else returns 0

# Unlike CoST a node is NOT an ancestor of itself
proc Node::hasRelative {role gi {address ""}} {
    set retval 0
    set role [string trim [string tolower $role]]
    set address [Node::getAddress $address]
# the next version of CoST will have 'breakNode' - until then go round loop
    foreachNode node [Node::getAddress $address] $role withgi $gi {
        if {[query address] != $address} {
            set retval 1
        }
    }
    return $retval
}

proc Node::addTextTag {win tagName {address ""}} {
    set address [Node::getAddress $address]
    set range [$win tag ranges $address]
    if {$range == ""} {
        Toc::showAncestry $win $address $tagName
    } else {
        set start [lindex $range 0]
        set end [lindex $range 1]
        $win tag add $tagName $start $end
    }
}

proc Search::displayResults {parentWin frame addresses} {
    global CW

    if {$addresses == ""} {
        Widget::makeDialog "Sorry, no hits in search" CONTINUE "" NO
        return
    }
    set toplevel [TopLevel::createFromParent $parentWin $frame "Addresses"]
    Widget::makeSeparator $toplevel .sep
    set text $toplevel.text
    TextWindow::createWithScrollbar $text 80 20
    pack $text -side top
    foreach address $addresses {
        set okid [Node::getOKID $address] 
        set button [Widget::makeOrRenew button $toplevel.$okid]
        $text window create end -window $button
        $button configure -text [Node::getMeaningfulTitle $address] \
            -command "Node::display $parentWin $address"
    }
    
    Widget::makeSeparator $toplevel .sep1

    set user [Widget::makeOrRenew frame $toplevel.bar]
    pack $user -side top
    set l1 [Widget::makeOrRenew label $user.l1]
    $l1 configure -text "UserClass: "
    pack $l1 -side left 
    set e1 [Widget::makeOrRenew entry $user.e1]
    $e1 configure -textvariable CW(USERCLASS,CLASS)
    pack $e1 -side left 
    set l2 [Widget::makeOrRenew label $user.l2]
    $l2 configure -text "Description: "
    pack $l2 -side left 
    set e2 [Widget::makeOrRenew entry $user.e2]
    $e2 configure -textvariable CW(USERCLASS,DESC)
    pack $e2 -side left
    set add [Widget::makeOrRenew button $user.add]
    $add configure -text "Save" \
        -command "Search::addUserClass \"$addresses\"; destroy $toplevel"
    pack $add -side left 

    Widget::makeSeparator $toplevel .sep2

# return if we don't want concatenated hypertree
    return

# NON-ACTIVE CODE
    set toplevel ""
    set win .html$okid
    TopLevel::createFromParent $toplevel $win "Search results"
    foreach address $addresses {
        Node::display $toplevel $address $win APPEND
    }
}

# gets a list of the nodes with GI == $gi.  If control == FIRST, gets
# first one.  If there are none, returns ""
proc Node::searchGI {gi {control ""}} {
    set list ""
    foreachNode doctree withgi $gi {
        if {$control == "FIRST"} {
            return [query address]
        }
        lappend list [query address]
    }
    return $list
}
 
# gets a list of the nodes with VALUE($att) == $value. If control == FIRST,
# gets first one.  If there are none, returns ""
proc Node::searchAttval {att value {control ""}} {
    set list ""
    foreachNode doctree withattval $att $value {
        if {$control == "FIRST"} {
            return [query address]
        }
        lappend list [query address]
    }
    return $list
}

# gets a list of the nodes with non-implied attribute $att.
# If control == FIRST, gets first one.  If there are none, returns ""
proc Node::searchAttname {att {control ""}} {
    set list ""
    foreachNode doctree withatt $att {
        if {$control == "FIRST"} {
            return [query address]
        }
        lappend list [query address]
    }
    return $list
}


    
#-------------------------------User class------------------------------
# the user may set their own class on a node, perhaps as a result of a
# search or manual tagging.  A node may have any number of userClasses,
# with value 0/1 (unset/set).  Only EL nodes can have userClasses.
# (The routines use CoST to keep track of which nodes are set)
# The internal representation is U:<classname> and this is reserved.

# sets the class $class on node with address $address ("" = current node)
proc UserClass::setNode {class {address ""}} {
    set class [string tolower $class]
    if {$address == ""} {
        if {![query? el]} return
        setprop U:$class 1
        return
    }
    withNode node [Node::getAddress $address] {
        if {![query? el]} return
        setprop U:$class 1
    }
}

# unsets the class $class on node with address $address ("" = current node)
proc UserClass::unsetNode {class {address ""}} {
    set class [string tolower $class]
    if {$address == ""} {
        setprop U:$class 0
        return
    }
    withNode node [Node::getAddress $address] {
        setprop U:$class 0
    }
}

# gets the addresses for all nodes with $class set
proc UserClass::getNodes {class} {
    set class [string tolower $class]
    set addresses ""
    foreachNode doctree hasprop U:$class {
        if {[query propval U:$class] == 1} {
            lappend addresses [query address]
        }
    }
    return $addresses
}

# unsets any nodes already set for $class
proc UserClass::unsetAllNodes {class} {
    set class [string tolower $class]
    foreachNode doctree hasprop U:$class {
        if {[query propval U:$class] == 1} {
            setprop U:$class 0
        }
    }
}

proc UserClass::getDescription {class} {
    global CW
    set class [string tolower $class]
    if {[info exists CW(USERCLASS,$class,DESC)]} {
        return $CW(USERCLASS,$class,DESC)
    }
}

proc UserClass::setDescription {class desc} {
    global CW
    set class [string tolower $class]
    if {[info exists CW(USERCLASS,$class)]} {
        set CW(USERCLASS,$class,DESC) $desc
    }
}

proc UserClass::rename {class newclass} {
    global CW
# class must exist
    set class [string tolower $class]
    if {![info exists CW(USERCLASS,$class)]} {
        return
    }
# cannot rename if already present
    set newclass [string tolower $newclass]
    if {[info exists CW(USERCLASS,$newclass)]} {
        return 0
    }

    UserClass::create $newclass [UserClass::getNodes $class] \
        [UserClass::getDescription $class]
    UserClass::delete $class
}

# creates a new UserClass and adds the nodes.  $desc is a text description
# if the class already exists, returns with failure (0).  In this case
# the existing class must be deleted deliberately in advance.

proc UserClass::create {class addresses {desc ""}} {
    global CW
    set class [string tolower $class]
# cannot create if already present
    if {[info exists CW(USERCLASS,$class)]} {
        return 0
    }
# classname can only have alphanumeric and underscore
    if {![regexp -nocase {[A-Z][A-Z0-9_]} $class] || $class == ""} {
        GUI::errorMessage "Bad userClass name: $class"
        return 0
    }
    set CW(USERCLASS,$class,DESC) $desc
    set CW(USERCLASS,$class) 1
    lappend CW(USERCLASS,LIST) $class
    foreach address $addresses {
        UserClass::setNode $class $address
    }
    return 1
}

# deletes a class (since I don't think a propval can be removed, exisiting
# ones are set to 0
proc UserClass::delete {class {confirm ""}} {
    global CW
    set class [string tolower $class]
    if {![info exists CW(USERCLASS,$class)]} {
        return
    }
    if {$confirm == "CONFIRM"} {
        if {[Widget::makeDialog "Really delete $class?" YES NO] == "NO"} {
            return
        }
    }
    UserClass::unsetAllNodes $class
    unset CW(USERCLASS,$class,DESC)
    set CW(USERCLASS,LIST) [List::NOT $CW(USERCLASS,LIST) $class]
}
 
proc UserClass::getList {} {
    global CW
    if {![info exists CW(USERCLASS,LIST)]} {
        return
    }
    set classes ""
    foreach class $CW(USERCLASS,LIST) {
        lappend classes $class
    }
    return $classes
}

proc UserClass::display {class {parentWin dummy}} {
    global CW
    set addresses [UserClass::getNodes $class]
    Search::displayResults $parentWin .$class $addresses
# add results to TableOfContents
    Toc::newTags $CW(MAINTOC) $addresses [UserClass::getTextTagColour] userClass
}

proc UserClass::getTextTagColour {} {
    global CW
    if {![info exists CW(USERCLASS,TEXTTAG,COLOUR)]} {
        set CW(USERCLASS,TEXTTAG,COLOUR) cyan
    }
    return $CW(USERCLASS,TEXTTAG,COLOUR)
}

# get first unused class of type B_result<num>
proc UserClass::getResultName {} {
    for {set i 0} {$i < 99} {incr i} {
        if {[UserClass::getNodes b_result$i] == ""} {
            return b_result$i
        }
    }
    return b_result99
}

# gets simple Boolean query from widget
# creates new userclass for results
proc UserClass::submitQuery {entry name} {
    set query [$entry get]
    set name [$name get]
    if {$name == "" || [UserClass::getNodes $name] != ""} {
        GUI::errorMessage "Null or existing class"
        return
    }
    set first [lindex $query 0]
    set op [lindex $query 1]
    set last [lindex $query 2]
    set command "List::[lindex $query 1] \
        \"[UserClass::getNodes [lindex $query 0]]\" \
        \"[UserClass::getNodes [lindex $query 2]]\""
    set result [eval $command]
    if {$result == ""} {
        GUI::errorMessage "Query generates no nodes"
        return
    }
    UserClass::create $name $result "$first $op $last"
}

proc UserClass::manipulate {{parentWin dummy} {control ""}} {

    if {$control == "REDISPLAY"} {
        destroy $parentWin.userclass
    }
    set userclass [TopLevel::createFromParent $parentWin .userclass "Addresses"]
    Widget::makeSeparator $userclass .separator 

# description box
    set bottomframe [Widget::makeOrRenew frame $userclass.bottomframe]
    pack $bottomframe -side bottom -fill x
    set bottomlabel [Widget::makeOrRenew label $bottomframe.label]
    $bottomlabel configure -text "" 
    pack $bottomlabel -side left

# command line
    set commframe [Widget::makeOrRenew frame $userclass.comm]
    pack $commframe -side bottom
    set commlab [Widget::makeOrRenew label $commframe.lab]
    $commlab configure -text "Query: "
    pack $commlab -side left
    set commentry [Widget::makeOrRenew entry $commframe.ent]
    $commentry configure -text ""
    pack $commentry -side left
    set commnamelab [Widget::makeOrRenew label $commframe.namelab]
    $commnamelab configure -text "Name: "
    pack $commnamelab -side left
# ... default name is B_result<num>
    set resultName [UserClass::getResultName]
    set commname [Widget::makeOrRenew entry $commframe.name]
    $commname configure  -width 10
    $commname insert end $resultName
    pack $commname -side left
    set commsubmit [Widget::makeOrRenew button $commframe.sub]
    $commsubmit configure -text Submit \
        -command "UserClass::submitQuery $commentry $commname; \
        UserClass::manipulate \"\" REDISPLAY"
    pack $commsubmit -side left

# rename frame 
    set renameframe [Widget::makeOrRenew frame $userclass.rename]
    pack $renameframe -side bottom
    set renamelab [Widget::makeOrRenew label $renameframe.lab]
    $renamelab configure -text "Rename to : "
    pack $renamelab -side left
    set renameentry [Widget::makeOrRenew entry $renameframe.ent]
    $renameentry configure -textvariable CW(USERCLASS,NEWNAME)
    pack $renameentry -side left
    bind $renameentry <Return> "UserClass::rename \$CW(USERCLASS,OLDNAME) \
        \[$renameentry get\]; UserClass::manipulate \"\" REDISPLAY"
    pack unpack $renameframe

# middle frames
    set classframe [Widget::makeOrRenew frame $userclass.class]
    pack $classframe -side left
    set descframe [Widget::makeOrRenew frame $userclass.desc]
    pack $descframe -side left
    set delframe [Widget::makeOrRenew frame $userclass.del]
    pack $delframe -side left
    set renframe [Widget::makeOrRenew frame $userclass.ren]
    pack $renframe -side left
    set queryframe [Widget::makeOrRenew frame $userclass.query]
    pack $queryframe -side left

    set classes [UserClass::getList]
    foreach class $classes {
        set classlab [Widget::makeOrRenew button $classframe.$class]
        $classlab configure -text $class -command "UserClass::display $class"
        pack $classlab -side top -fill x
        set desclab [Widget::makeOrRenew button $descframe.$class]
        $desclab configure -text "I" \
            -command "UserClass::showDescription $bottomlabel $class"
        pack $desclab -side top -fill x
        set dellab [Widget::makeOrRenew button $delframe.$class]
        $dellab configure -text "Del" \
            -command "UserClass::delete $class CONFIRM"
        pack $dellab -side top -fill x
        set renlab [Widget::makeOrRenew button $renframe.$class]
        $renlab configure -text "Ren" \
            -command "set CW(USERCLASS,OLDNAME) $class; \
            pack $renameframe -before $commframe"
        pack $renlab -side top -fill x
        set querylab [Widget::makeOrRenew button $queryframe.$class]
        $querylab configure -text "Q" \
            -command "$commentry insert end \" $class\""
        pack $querylab -side top -fill x
    }

# Booleans
    Widget::makeSeparator $userclass .separator1 HORIZ 20
    set boolframe [Widget::makeOrRenew frame $userclass.bool]
    pack $boolframe -side left -fill y -expand yes
    foreach op {AND NOT OR} {
        set oplab [Widget::makeOrRenew button $boolframe.[string tolower $op]]
        $oplab configure -text $op \
            -command "$commentry insert end \" $op\""
        pack $oplab -side top -fill x
    }
}

proc UserClass::showDescription {win class} {
    
    $win configure -text [UserClass::getDescription $class] -bg white
}
    
    
#--------------------------Search routines-------------------------------

# graphical search for GIs
proc Search::topWindow {} {
    global CW

# remember if it was a fresh window
    set new 0
    if {![winfo exists .search]} {
        set new 1
    }
    set toplevel [Widget::makeOrRenew toplevel .search]

# message
    set msg [Widget::makeOrRenew message $toplevel.msg]
    $msg configure \
        -text "SEARCHES\n Search on GI, attribute name, attribute value or node content.  Any combination can be used - if a field is non-blank it is ANDed with the others - if blank it is ignored.  EXPERIMENTAL!"\
     -background white \
     -width 500
    pack $msg -side top

# gi bar
    set gibar [Widget::makeOrRenew frame $toplevel.gibar]
    pack $gibar -side top -fill x

    set label [Widget::makeOrRenew label $gibar.lab]
    $label configure -text "Target: "
    pack $label -side left
# menu for Target GI or UC 
    Search::selectGIorUCs GI $gibar TARGET GI gisearch.html
    set giuc [Widget::makeOrRenew frame $gibar.giuc]
    set l1 "GI gi \"Search::toggleGIUC $gibar target\""
    set l2 "UC uc \"Search::toggleGIUC $gibar target\""
    set l12 [list $l1 $l2]
    Widget::makeRadio $giuc $l12 CW(SEARCH,TARGET,GIUC) gi left
    pack $giuc -side left
    Search::selectGIorUCs UC $gibar TARGET UserClass ucsearch.html
    if {$new} {
        Widget::toggleButtonState $gibar.targetuc
    }

    Widget::makeSeparator $gibar .separator1 HORIZ 15

# choice of context (default "none")
    set label [Widget::makeOrRenew label $gibar.lab1]
    $label configure -text "Context: "
    pack $label -side left

    Widget::makeRadioMenu $gibar.context "Context" CW(SEARCH,CONTEXT) {\
        {"None" none} \
        {"Parent" "parent"} \
        {"Ancestor" "ancestor"} \
        {"Left" "left"} \
        {"Right" "right"} \
        {"Prev" "prev"} \
        {"Next" "next"} \
        {"Esib" "esib"} \
        {"Ysib" "ysib"} \
        {"Child" "child"} \
        {"Descendant" "descendant"} \
        } none

    Widget::makeSeparator $gibar .separator2 HORIZ 5

# menu for Context GI or UC
    Search::selectGIorUCs GI $gibar CONTEXT "GI" gisearch.html
    set giucx [Widget::makeOrRenew frame $gibar.giucx]
    set l1 "GI gi \"Search::toggleGIUC $gibar context\""
    set l2 "UC uc \"Search::toggleGIUC $gibar context\""
    set l12 [list $l1 $l2]
    Widget::makeRadio $giucx $l12 CW(SEARCH,CONTEXT,GIUC) gi left
    pack $giucx -side left
    Search::selectGIorUCs UC $gibar CONTEXT "UserClass" ucsearch.html
    if {$new} {
        Widget::toggleButtonState $gibar.contextuc
    }

    Widget::makeSeparator $toplevel .separator

# attname
    set frame [Widget::makeOrRenew frame $toplevel.name]
    pack $frame -side top -fill x
    Search::makeEntryWidget $frame .name "Att Name" SEARCH,ATTRIB,NAME 

# attval
    set frame [Widget::makeOrRenew frame $toplevel.val]
    pack $frame -side top -fill x
    Search::makeEntryWidget $frame .val "Att value" SEARCH,ATTRIB,VALUE 
    Widget::makeInlineCheckbutton \
        $frame.case Case CW(SEARCH,ATTRIB,VALUE,CASE) 0
    Widget::makeCombo $frame.match {exact glob regex} \
        Match CW(SEARCH,ATTRIB,VALUE,MATCH) exact \
        {{listbox.height 4} {listbox.width 6}}
    pack $frame.match -side left

# content
    set frame [Widget::makeOrRenew frame $toplevel.cont]
    pack $frame -side top -fill x
    Search::makeEntryWidget $frame .cont "Content " SEARCH,CONTENT,VALUE 
    Widget::makeInlineCheckbutton \
        $frame.case Case CW(SEARCH,CONTENT,VALUE,CASE) 0
    Widget::makeCombo $frame.match {exact glob regex range} \
        Match CW(SEARCH,CONTENT,VALUE,MATCH) exact \
        {{listbox.height 4} {listbox.width 6}}
    pack $frame.match -side left

# command line
    Widget::makeSeparator $toplevel .separator3
    set frame [Widget::makeOrRenew frame $toplevel.cmd]
    pack $frame -side top -fill x
    set label [Widget::makeOrRenew label $frame.lab]
    $label configure -text "Command: "
    pack $label -side left
    set entry [Widget::makeOrRenew entry $frame.ent]
    $entry configure -textvariable CW(SEARCH,COMMANDLINE) -width 60
# there is a bug here ...
    $entry configure -state disabled
    pack $entry -side left -fill x
    set button [Widget::makeOrRenew button $frame.clear]
    $button configure -text Clear \
        -command "set CW(SEARCH,COMMANDLINE) \"\""
    pack $button -side left
    Widget::makeSeparator $toplevel .separator4

    set buttons [list \
        "Search \"Search::executeButton $toplevel\"" \
        "Quit \"destroy $toplevel\"" \
        "Help \"HTML::display $CW(DOCDIR)/search.html\""]
    Widget::makeButtonBox $toplevel.box $buttons
}

proc Search::toggleGIUC {gibar targcont} {
    Widget::toggleButtonState $gibar.${targcont}uc
    Widget::toggleButtonState $gibar.${targcont}gi

}

# selection of GI(s) or UserClass by menu (role is PARENT, ANCESTOR, etc
# giuc = GI or UC; targcont = TARGET or CONTEXT
proc Search::selectGIorUCs {giuc frame targcont label helpfile} {

    global CW
    set targcont [string toupper $targcont]
    set buttons ""
    lappend buttons "label \"$label\""
    lappend buttons "command Help \"HTML::display $helpfile\""
    if {$giuc == "GI"} {
        set list [GI::getList]
    } elseif {$giuc == "UC"} {
        set list [UserClass::getList]
    }
    foreach item $list {
        if {![info exists CW(SEARCH,$giuc,$item,$targcont)]} {
            set CW(SEARCH,$giuc,$item,$targcont) 0
        }
        lappend buttons \
            "checkbutton $item CW(SEARCH,$giuc,$item,$targcont) \
            $CW(SEARCH,$giuc,$item,$targcont)"
    }
    set win [string tolower $frame.$targcont$giuc]
    if {![winfo exists $win]} {
        GUI::makeMenu $win $label "" "-side left" $buttons
    }
}

proc Search::executeButton {parentWin} {
    global CW
    set phrase(ANCESTOR) " ancestor "
# target or context?
    foreach targcont {TARGET CONTEXT} {
        set GIUCS($targcont) ""
# GI or UC?
        set giuc [string toupper $CW(SEARCH,$targcont,GIUC)]
        if {$giuc ==  "GI"} {
            set list [GI::getList]
        } elseif {$giuc ==  "UC"} {
            set list [UserClass::getList]
        }
        foreach item $list {
            if {$CW(SEARCH,$giuc,$item,$targcont) == 1} {
                append GIUCS($targcont) "$item "
            }
        }
    }

    set addresses [Search::execute \
        $GIUCS(TARGET) $CW(SEARCH,TARGET,GIUC) \
        $CW(SEARCH,CONTEXT) \
        $GIUCS(CONTEXT) $CW(SEARCH,CONTEXT,GIUC) \
        $CW(SEARCH,ATTRIB,NAME) \
        [list $CW(SEARCH,ATTRIB,VALUE) \
            $CW(SEARCH,ATTRIB,VALUE,CASE) \
            $CW(SEARCH,ATTRIB,VALUE,MATCH)] \
        [list $CW(SEARCH,CONTENT,VALUE) \
            $CW(SEARCH,CONTENT,VALUE,CASE) \
            $CW(SEARCH,CONTENT,VALUE,MATCH)] \
        $CW(SEARCH,COMMANDLINE) \
        ]


# add results to TableOfContents
    global CW

# display results
    Search::displayResults $parentWin .searchResults $addresses 
    Toc::newTags $CW(MAINTOC) $addresses pink searchResults
}

proc Search::makeEntryWidget {frame name text var} {
    global CW

    set bar [Widget::makeOrRenew frame $frame$name]
    pack $bar -side left
    
    set label [Widget::makeOrRenew label $bar.l]
    $label configure -text $text
    pack $label -side left
    set entry [Widget::makeOrRenew entry $bar.e]
    $entry configure -textvariable CW($var)
    pack $entry -side left
}

# returns all nodes satisfying cost query (omit foreachNode and brackets)
# e.g. Search::CoSTForeachNode "withGI X" ==>
#    foreachNode doctree withGI X {lappend addresses [query address]}

proc Search::CoSTForeachNode {root command} {
    set addresses ""
    eval foreachNode $root $command \{lappend addresses \[query address\]\}
    set newaddresses ""
    foreach address $addresses {
        if {$address != ""} {
            lappend newaddresses $address
        }
    }
    return $newaddresses
}

proc Search::execute {targlist targgiuc context contlist contgiuc \
    attname attvallist contentlist commandline} {

    set addresses ""
    set targgiuc [string toupper $targgiuc]
    set contgiuc [string toupper $contgiuc]

    GUI::postStatus "Searching... "

# if commandline is non-null, execute and return
    if {$commandline != ""} {
        set addresses [Search::CoSTForeachNode doctree $commandline]
        return $addresses
    }

# get all EL addresses in the tree
    set alltree ""
    foreachNode doctree {
        if {[query? el]} {
            lappend alltree [query address]
        }
    }
    
# this bit will be easier if Joe adds a withpropval command ...

# first search for GI stuff.  If $gilist is blank, use the whole tree 
    set command ""
    set done 0
    if {$targlist != ""} {
        if {$targgiuc == "GI"} {
            if {$context == "none"} {
                set search " elements \"$targlist\" "
                set done 1
            } else {
                if {$contgiuc == "GI"} {
                    set search \
                        "elements \"$targlist\" $context elements \"$contlist\""
                    set done 1
                } elseif {$contgiuc == "UC"} {
                    set search " elements \"$targlist\" "
                    set done 0
                }
            }
            if {$search != ""} {
                set addresses [Search::CoSTForeachNode doctree $search]
            }
        } elseif {$targgiuc == "UC"} {
            foreach class $targlist {
                set newaddresses \
                    [Search::CoSTForeachNode doctree "hasprop U:$class"]
                foreach address $newaddresses {
                    withNode node $address {
                        if {[query propval U:$class] == 1} {
                            lappend addresses [query address]
                        }
                    }
                }
            }
        }
        if {$addresses == ""} {
            GUI::postStatus ""
            return
        }
    }

# context search? ANDed with overall search
    if {$context != "none" && !$done} {
        set newaddresses ""
# context is GI list
        if {$contgiuc == "GI"} {
# no previous addresses
            if {$addresses == ""} {
                set newaddresses [Search::CoSTForeachNode \
                    doctree "$context elements $contlist"]
            } else {
                foreach address $addresses {
                    set newadds [Search::CoSTForeachNode \
                        "node $address" "$context elements $contlist"]
                    if {$newadds != ""} {
                        set newaddresses [concat $newaddresses $newadds]
                    }
                }
            }
# user class in context - always use list of addresses
        } elseif {$contgiuc == "UC"} {
            if {$addresses == ""} {
                set addresses $alltree
            }
# go through context UCs
            foreach class $contlist {
                foreach address $addresses {
                    set adds [Search::CoSTForeachNode \
                        " node $address " "hasprop U:$class"]
                    foreach add $adds {
                        withNode node $add {
                            if {[query propval U:$class] == 1} {
                                lappend newaddresses [query address]
                            }
                        }
                    }
                }
            }

        }
        set addresses $newaddresses
        if {$addresses == ""} {
            GUI::postStatus ""
            return
        }
    }

    if {$addresses == ""} {
        set addresses $alltree
    }

# attributes
    set attname [string tolower $attname]
    set attval [lindex $attvallist 0]
    set attvalcase [string tolower [lindex $attvallist 1]]
    set attvalmatch [string tolower [lindex $attvallist 2]]
    set doAttSearch 0
    if {$attname != "" || $attval != ""} {set doAttSearch 1}

# attribute names and values
# values are case-insensitive in CoST, so some kludging required below
# for fuzzy matches
# if $attname is given then attval has to pertain to it.  

# EXACT search 
    if {$doAttSearch && $attvalmatch == "exact"} {
        set newaddresses ""
# check case-insenstive attval
        if {$attval != ""} {
            foreachNode doctree withattval $attname $attval {
                lappend newaddresses [query address]
            }
# no attval, just check attnames
        } else {
            foreachNode doctree hasatt $attname {
                lappend newaddresses [query address]
            }
        }
# merge the two lists
        if {$addresses == ""} {
            set addresses $newaddresses
        } else {
            set addresses [List::AND $newaddresses $addresses]
        }
    }

# FUZZY comparisons, but attname is given
    if {$doAttSearch && $attvalmatch != "exact" && $attname != ""} {
        set newaddresses ""
        foreachNode doctree hasatt $attname {
            set val [query attval $attname]
            if {[String::match $val $attval $attvalcase $attvalmatch]} {
                lappend newaddresses [query address]
            }
        }
# merge the two lists
        if {$addresses == ""} {
            set addresses $newaddresses
        } else {
            set addresses [List::AND $newaddresses $addresses]
        }
    }

# FUZZY comparisons, but no attname (slow)
    if {$doAttSearch && $attvalmatch != "exact" && $attname == ""} {
        set newaddresses ""
# if no addresses set, use whole tree (slow)
        if {$addresses == ""} {set addresses $alltree}
# loop through all addresses
        foreach address $addresses {
# get all the attvals for each node
            withNode node $address {
                set tuples [Node::getTuples]
                foreach tuple $tuples {
                    set name [string tolower [lindex $tuple 0]]
                    set val [lindex $tuple 1]
                    if {[String::match $val $attval \
                        $attvalcase $attvalmatch]} {
                        lappend newaddresses [query address]
                    }
                }
            }
        }
        set addresses $newaddresses
        if {$addresses == ""} {
            GUI::postStatus ""
            return
        }
    }

# CONTENT ... 
    set contentval [lindex $contentlist 0]
    set contentcase [string tolower [lindex $contentlist 1]]
    set contentmatch [string tolower [lindex $contentlist 2]]
# get content (EL nodes only)
    if {$contentval != ""} {
        set newaddresses ""
# if no addresses set, use whole tree (slow)
        if {$addresses == ""} {set addresses $alltree}
# loop through all addresses
        foreach address $addresses {
            withNode node $address {
                set content [string trim [content]]
                if {$contentmatch == "exact"} {
                    if {$contentval == $content} {
                        lappend newaddresses [query address]
                    }
                } elseif {$contentmatch == "range"} {
                    if {[String::compareFloat $content $contentval]} {
                        lappend newaddresses [query address]
                    }
                } elseif {[String::match $content $contentval \
                    $contentcase $contentmatch]} {
                    lappend newaddresses [query address]
                }
            }
        }
        set addresses $newaddresses
    }
    GUI::postStatus ""
    return $addresses
}

proc Search::addUserClass {addresses} {
    global CW

    if {$CW(USERCLASS,CLASS) == ""} return
    UserClass::create $CW(USERCLASS,CLASS) $addresses \
        $CW(USERCLASS,DESC)
}

#--------------------------INPUT OUTPUT----------------------------------

# if filename is given, tries to open file, else  brings up file browser
# then opens file and returns {<file> <filehandle>} or null

proc Output::openFile {{outfile ""} {defaults ""}} {
    global CW

    GUI::postStatus "Bringing up file browser"
# was this;>    set outfile [Tix::getFilename . {{{*.esis} {ESIS}}}]

    if {$outfile == ""} {
        set outfile [Tix::getFilename . ""]
    }
    GUI::postStatus ""
    if {[file exists $outfile]} {
        set reply [Widget::makeDialog "Overwrite existing file?" YES NO]
        if {$reply == "NO"} {
            return 0
        }
    }
    if {[catch "set of [open $outfile w]" msg]} {
        GUI::errorMessage "cannot open output file: $outfile ($msg)"
        return ""
    }
# set global variables to remember file
    set CW(OUTFILE) $outfile
    set CW(OUTFILEHANDLE) $of

    return "$outfile $of"
}
    
# writes to currently open output file
proc Output::write {string} {
    global CW
    if {![info exists CW(OUTFILEHANDLE)] || $CW(OUTFILEHANDLE) == ""} {
        puts stdout $string
    } else {
        puts $CW(OUTFILEHANDLE) $string
    }
}

# closes currently open output file if any
proc Output::closeFile {} {
    global CW
    if {$CW(OUTFILEHANDLE) != ""} {
        close $CW(OUTFILEHANDLE)
        set CW(OUTFILEHANDLE) ""
    }
}
        
proc Output::setLanguage {language} {
    global CW
    set CW(OUTPUT,LANGUAGE) $language
}

proc Output::getLanguage {} {
    global CW
    if {[info exists CW(OUTPUT,LANGUAGE)]} {
        return $CW(OUTPUT,LANGUAGE)
    }
}

# Forms output string.  So far deals with HTML, LaTeX
# args:
#  For tags without attributes:
#	1 	tag to be applied to string
#	2	string to be output
#	3	end tag only used if 1 is ""

# Examples:
#	Output::format "" foo		# output "foo"
#	Output::format B foo		# output <B>foo</B>, {\bf foo}, etc
#	Output::format B    		# output <B>, {\bf , etc
#	Output::format HR			# output <HR>, \hrule, etc
#	Output::format "" "" B		# output </B>, }, etc

#  For tags with attributes:
#       1       tag to be applied to string
#	2	attribute string
#       3       value
#       4       end tag only used if 1 is ""

# Examples:
#	Output::format A "name=fred"     	# output <A name=fred>
#	Output::format A "" val		# output <A>val</A>
#	Output::format "" "" "" A		# output </A>

#	Output::format INITIALISE		# intialise routine

proc Output::format {arg1 {arg2 ""} {arg3 ""} {arg4 ""}} {
    global CW
    if {$arg1 == "INITIALISE"} {

        set CW(TAG,HTML,A)          {"<A"           "</A>"         4}
        set CW(TAG,HTML,ADDRESS)    {"<ADDRESS>"    "</ADDRESS>"   3}
        set CW(TAG,HTML,B)          {"<B>"          "</B>"         3}
        set CW(TAG,HTML,BASE)       {"\n<BASE"      ">"            3}
        set CW(TAG,HTML,BLOCKQUOTE) {"<BLOCKQUOTE>" "</BLOCKQUOTE>"3}
        set CW(TAG,HTML,BODY)       {"\n<BODY>\n"   "\n</BODY>\n"  3}
        set CW(TAG,HTML,BR)         {"<BR>\n"       ""             3}
        set CW(TAG,HTML,CITE)       {"<CITE>"       "</CITE>"      3}
        set CW(TAG,HTML,CODE)       {"\n<CODE>\n"   "\n</CODE>\n"  3}
        set CW(TAG,HTML,CR)         {"\n"           ""             3}
        set CW(TAG,HTML,DD)         {"<DD>"         "</DD>"        3}
        set CW(TAG,HTML,DL)         {"<DL>"         "</DL>"        3}
        set CW(TAG,HTML,DT)         {"<DT>"         "</DT>"        3}
        set CW(TAG,HTML,DOC)        {"<HTML>\n"     "\n</HTML>\n"  3}
        set CW(TAG,HTML,EM)         {"<EM>"         "</EM>"        3}
        set CW(TAG,HTML,H1)         {"\n<H1>"       "</H1>\n"      3}
        set CW(TAG,HTML,H2)         {"\n<H2>"       "</H2>\n"      3}
        set CW(TAG,HTML,H3)         {"\n<H3>"       "</H3>\n"      3}
        set CW(TAG,HTML,H4)         {"\n<H4>"       "</H4>\n"      3}
        set CW(TAG,HTML,H5)         {"\n<H5>"       "</H5>\n"      3}
        set CW(TAG,HTML,H6)         {"\n<H6>"       "</H6>\n"      3}
        set CW(TAG,HTML,HEAD)       {"<HEAD>\n"     "\n<HEAD>"     3}
        set CW(TAG,HTML,HR)         {"\n<HR>"       ""             3}
        set CW(TAG,HTML,I)          {"<I>"          "</I>"         3}
        set CW(TAG,HTML,IMG)        {"<IMG"         ""             4}
        set CW(TAG,HTML,KBD)        {"<KBD>"        "</KBD>"       3}
        set CW(TAG,HTML,LI)         {"\n<LI>"       ""             3}
        set CW(TAG,HTML,LINK)       {"\n<LINK"      ">"            3}
        set CW(TAG,HTML,META)       {"<META>\n"     "\n<META>"     3}
        set CW(TAG,HTML,OL)         {"<OL>\n"       "\n</OL>\n"    3}
        set CW(TAG,HTML,P)          {"\n<P>"        ""             3}
        set CW(TAG,HTML,PRE)        {"<PRE>"        "</PRE>"       3}
        set CW(TAG,HTML,SAMP)       {"<SAMP>"       "</SAMP>"      3}
        set CW(TAG,HTML,STRONG)     {"<STRONG>"     "</STRONG>"    3}
        set CW(TAG,HTML,TITLE)      {"<TITLE>\n"    "\n</TITLE>"   3}
        set CW(TAG,HTML,TT)         {"<TT>"         "</TT>"        3}
        set CW(TAG,HTML,UL)         {"<UL>\n"       "\n</UL>\n"    3}
        set CW(TAG,HTML,VAR)        {"<VAR>"        "</VAR>"       3}

        set CW(TAG,LATEX,B)         {"\{\\bf " "\}"}
        set CW(TAG,LATEX,BR)        {"\\\\\n" ""}
        set CW(TAG,LATEX,CR)        {"\n" ""}
        set CW(TAG,LATEX,DOC)       \
   {"\\documentstyle\[html,epsf,12pt,fancyhea\]\{book\}\n\\begin\{document\}" \
                                         "\\end\{document\}"}
        set CW(TAG,LATEX,DD)        {"\\indent" ""}
        set CW(TAG,LATEX,DL)        {"\\begin\{itemize\}\n" \
                                      "\\end\{itemize\}\n"}
        set CW(TAG,LATEX,DT)        {"\\item"       ""}

        set CW(TAG,LATEX,EM)        {"\{\\em " "\}"}
        set CW(TAG,LATEX,H1)        {"\\paragraph\{\\bf \{\\Huge " \
                                     "\}\}\\n"}
        set CW(TAG,LATEX,H2)        {"\\paragraph\{\\bf \{\\Large " \
                                     "\}\}\\n"}
        set CW(TAG,LATEX,H3)        {"\\paragraph\{\\bf \{\\large " \
                                     "\}\}\\n"}
        set CW(TAG,LATEX,H4)        {"\\paragraph\{\\bf \{\\normalsize " \
                                     "\}\}\\n"}
        set CW(TAG,LATEX,H5)        {"\\paragraph\{\\bf \{\\normalsize " \
                                     "\}\}\\n"}
        set CW(TAG,LATEX,H6)        {"\\paragraph\{\\bf \{\\normalsize " \
                                     "\}\}\\n"}
        set CW(TAG,LATEX,HEAD)      {"\{\\null " "\}"}
        set CW(TAG,LATEX,HR)        {"\n\\hrule\n" ""}
        set CW(TAG,LATEX,I)         {"\{\\it " "\}"}
        set CW(TAG,LATEX,LABEL)     {"\{\\label " "\}"}
        set CW(TAG,LATEX,LI)        {"\\item"       ""}
        set CW(TAG,LATEX,LINK)      {"\{\\null " "\}"}
        set CW(TAG,LATEX,META)      {"\{\\null " "\}"}
        set CW(TAG,LATEX,OL)        {"\\begin\{enumerate\}\n" \
                                      "\\end\{enumerate\}\n"}
        set CW(TAG,LATEX,P)         {"\n\n" ""}
        set CW(TAG,LATEX,PRE)       {"\{begin verbatim\}" "\{end verbatim\}"}
        set CW(TAG,LATEX,REL)       {"\{\\null " "\}"}
        set CW(TAG,LATEX,REF)       {"\{\\ref " "\}\{\\sf " "\}"}
        set CW(TAG,LATEX,TITLE)     {"\{\\title " "\}"}
        set CW(TAG,LATEX,TT)        {"\{\\tt " "\}"}
        set CW(TAG,LATEX,UL)        {"\\begin\{itemize\}\n" \
                                      "\\end\{itemize\}\n"}

        return
    }

    set lang $CW(OUTPUT,LANGUAGE)
    if {$lang == ""} {
        set lang HTML
    }

    set prefix ""
    set suffix ""
    set tail ""

    set tag $arg1
    if {$tag == ""} {
        set tag $arg3
    }
    if {$tag == ""} {
        set tag $arg4
    }
    if {$tag == ""} {
        return $arg2
    }
    if {![info exists CW(TAG,$lang,$tag)]} {
        GUI::errorMessage "NO TAG: $tag"
        return
    }
    set prefix [lindex $CW(TAG,$lang,$tag) 0]
    set suffix [lindex $CW(TAG,$lang,$tag) 1]
    set narg [lindex $CW(TAG,$lang,$tag) 2]
    set string ""
    if {$narg == 4} {
        if {$arg1 == ""} {
            set string $arg3$suffix
        } else {
            set string "$prefix $arg2>"
            if {$arg3 != ""} {
                append string "$arg3$suffix"
            }
        }
    } else {
        if {$arg1 == ""} {
            set string "$arg2$suffix"
        } else {
            set string $prefix
            if {$arg2 != ""} {
                append string "$arg2$suffix"
            }
        }
    }

    return $string
}

# capture output in a buffer.  Each call flushes the buffer and returns
# the previous buffer contents - if any.

proc Output::initialise {} {
    global CW
    set CW(OUTPUT,BUFFER) ""
}

proc Output::capture {} {
    global CW
    set temp $CW(OUTPUT,BUFFER)
    set CW(OUTPUT,BUFFER) ""
    return $temp
}

# append string to the captured output
proc Output::append {string} {
    global CW
    append CW(OUTPUT,BUFFER) $string
}

#-------------------------GENERAL PARSING ----------------------------------
proc Tuples::getValue {tuples name} {
    foreach tuple $tuples {
        if {[lindex $tuple 0] == $name} {
            return [lindex $tuple 1]
        }
    }
}
#-----------------------------GENERAL UTILITIES-------------------------

#-----------------------------Lists-------------------------------------
# returns those items common to both lists (result in arbitrary order)
proc List::AND {list1 list2} {
    foreach item $list1 {
        set TEMP($item) 1
    }
    set result ""
    foreach item $list2 {
        if {[info exists TEMP($item)]} {
            lappend result $item
        }
    }
    return $result
}

# returns those items on list1 but NOT on list2 (result in arbitrary order)
proc List::NOT {list1 list2} {
    foreach item $list2 {
        set TEMP($item) 1
    }
    set result ""
    foreach item $list1 {
        if {![info exists TEMP($item)]} {
            lappend result $item
        }
    }
    return $result
}

# returns those items on list1 OR on list2 (result in arbitrary order)
proc List::OR {list1 list2} {
    foreach item $list1 {
        set TEMP($item) 1
    }
    set result $list1
    foreach item $list2 {
        if {![info exists TEMP($item)]} {
            lappend result $item
        }
    }
    return $result
}


#----------------------------------Strings-------------------------------
# compare two strings using either globbing or regexp
# case is 0 (caes-insensitive)  or 1 , matchtype is glob/regexp
# case only applies to regex
proc String::match {target query case matchtype} {
    
    if {$case == 0} {
        set query [string tolower $query]
        set target [string tolower $target]
    }
    if {$matchtype == "glob"} {
        set result [string match $query $target]
    } elseif {$matchtype == "regex"} {
        set result [regexp $query $target]
    } else {
        GUI::errorMessage "Bad String::match arg: $matchtype"
        return 0
    }
    return $result
}

# compare numbers.  The pattern can be:
#	number number 
#	number+-number
# 	> number
# 	< number

# compare numbers.  The calling routine must provide a trimmed string
# with 0 or 1 space
proc String::compareFloat {value pattern} {
    set bit1 [lindex $pattern 0]
    set bit2 [lindex $pattern 1]
    if {$bit1 == "<"} {
        if {$value < $bit2} {
            return 1
        }
        return 0
    } elseif {$bit1 == ">"} {
        if {$value > $bit2} {
            return 1
        }
        return 0
    } elseif {[regexp {(.*)[+][-](.*)} $pattern junk number delta]} {
        set bit1 [expr $number-$delta]
        set bit2 [expr $number+$delta]
    } elseif {$bit2 < $bit1} {
        set temp $bit1
        set bit1 $bit2
        set bit2 $temp
    }
    if {$bit1 < $value && $bit2 > $value} {
        return 1
    }
    return 0
}
    
#------------------------------global-------------------------------------
# global only applies to a scalar or the root of an array, so return
# that if required.
proc Global::root {var} {
    regexp {(.*)[\(]} $var junk var
    return $var
}

#------------------------------Tcl----------------------------------------
# this avoids quotes and collisions with quotes in data
proc Tcl::evalQuote {proc arg} {
    return [eval $proc $arg]

}

# clears arrays of type XYZ(<index>).  If index == "", clears all elements
# of XYZ, else compares <pattern> with index using [string match ...]
proc Tcl::clearArrays {array {pattern ""}} {
    set elems [array names $array]
    if {$pattern == ""} {
        foreach elem $elems {
            unset $array($elem)
        }
    } else {
        foreach elem $elems {
            if {string match $pattern $elem]} {
                unset $array($elem)
            }
        }
    }
}

# optimised (a bit!) for large lists
proc List::reverse {list} {
    set l 0
    foreach elem $list {
        set temp($l) $elem
        incr l
    }
    set list ""
    incr l -1
    
    for {set i $l} {$i >= 0} {incr i -1} {
        lappend list $temp($i)
    }
    return $list
}

#--------------------------------Stacks-----------------------------------

# push on stack (and create a new one if none exists)
proc Tcl::stackPush {name value} {
    global _STACK
    if {![info exists _STACK($name,COUNT)]} {
        set _STACK($name,COUNT) 0
    }
    set _STACK($name,VALUE,$_STACK($name,COUNT)) $value
    incr _STACK($name,COUNT)
}

# retrieves item off stack.  if stack is empty returns ""
# (If the user wishes to store "" on stack they will have to
# use a convention *e.g. \"\"
proc Tcl::stackPop {name} {
    global _STACK
    set retval ""
    if {![info exists _STACK($name,COUNT)]} {
        GUI::errorMessage "Stack ($name) does not exist"
    } elseif {$_STACK($name,COUNT) <= 0} {
    } else {
        incr _STACK($name,COUNT) -1
        set retval $_STACK($name,VALUE,$_STACK($name,COUNT))
    }
    return $retval
}

proc Tcl::stackDelete {name} {
    global _STACK
    set elems [array names _STACK]
    foreach elem $elems {
        if {$name == $elem || [string first $name, $elem] == 0} {
            unset _STACK($elem)
        }
    }
}
    
# sets all values in an array to a given value (simple incremental subscripts)
proc Tcl::setArray {arrayName size value} {
    upvar $arrayName ARR
    for {set i 0} {$i < $size} {incr i} {
        set $ARR($i) $value
    }
}

puts "... finshed cwlib.tcl"
