#----------------------------------------------------------------------
# class PanedWindow
#----------------------------------------------------------------------

DOC "class PanedWindow (Tk)" {
 NAME
	PanedWindow
		- A geometry manager with movable sashes between
		  its managed windows.
 SYNOPSIS
	Creation:
		PanedWindow <window> <options>
		PanedWindow new <window> <options>

	Configuration:
		<window> manage <win> ?win? ?win?..
		<window> slaves
		<window> fractions
		<window> configure <options>

 DESCRIPTION
	Manage one or more windows with movable sashes between.
	Configuration can be altered dynamically.  The case with no
	windows begin managed is handled gracefully as well.

 METHODS
	manage win ?win? ?win?..
		Arrange for the specified windows to be managed by the
		PanedWindow object.  The windows are managed in the specified
		order from top to bottom, or left to right (for horizontal
		layout).  Repeatedly using this method will unmanage all
		currently managed windows, before managing the specified ones.
		Specifying the null list, {}, will unmanage all windows.

	slaves	Returns a list of all currently managed windows.

	fractions
		Returns a list of fractions reflecting the current placement
		of the sashes.  This list can then be used with the
		-fraction option to re-set a specific layout.

	configure <option>
		Configure the PanedWindow object.  See OPTIONS below
		for possible alternatives.

 OPTIONS
	-manage <list>
		Specifies a list of slaves to manage.  Equivalent to using
		the `manage' method, but can be specified at object creation-
		time.
	-orientation [x|y]
		For windows stacked on top of each other specify "y", for
		horizontal layout specify "x".
	-width	Initial width of the paned window.
	-height Initial height of the paned window.
	-min	Minimum fraction of the entire window to which a pane can
		shrink (default 0.1)
	-fraction <list>
		List of initial fractions each pane is to occupy.
		The list should be cumulative (each new position is
		counted from the top/left side).  Example: ".3 .7" would
		give three panes equal space.  If less panes exist than
		specified fractions, the last pane gets the space from the
		"missing" windows.
	-handleSide [begin|end]
		Specifies on which side of the sash the handle should be
		placed.  "begin" means top or left, "end" means bottom
		or right.  Default is "end"
	-handleOffs <pixels>
		Specify how many pixels from the near edge the sash handle
		should be placed.  Measured from the handle's centre.

 EXAMPLE

	# Create a PanedWindow object:

	PanedWindow .pw -width 300 -height 300 -min .15 \
		-manage {.foo .bar .baz}

	# Add window .new_one first:

	eval .pw manage .new_one [.priv.pw slaves]

	# Save current sash settings for later

	set save [.pw fractions]

	# Restore sash settings

	.pw configure -fraction $save

 ACKNOWLEDGEMENTS

     James Noble
	1993 - unknown
     Jay Schmidgall
	1994 - base logic posted to comp.lang.tcl
     Mark L. Ulferts <mulferts@spd.dsccc.com>
	1994 - added additional features and made it a class.
     Joe Hidebrand <hildjj@fuentez.com>
	07/25/94 - posted first multipane version to comp.lang.tcl
	07/28/94 - added support for vertical panes

 AUTHOR

     Patrik Floding <patrik@dynas.se>
	23/10/95 - Adapted from itcl to obTcl.  Changes to allow handling
	of non-child windows.  Moving towards a general geometry
	manager approach.  Changes to handle reconfiguration (managing
	new windows, etc).
}

class PanedWindow
PanedWindow inherit Widget

PanedWindow option {-orientation} y verify {
	switch -exact -- $orientation {
	y {}
	x {}
	default {error "orientation must be 'x' or 'y'"}
	}
}

PanedWindow option {-width} 100 configure {
       	$self.priv.pw config -width $width
}

PanedWindow option {-height} 100 configure {
       	$self.priv.pw config -height $height
}

PanedWindow option {-min} 0.1 verify {
      	if {[expr $min < 0.0 || $min > 1.0]} {
        	error "min size must be 0.0-1.0"
      	}
}

PanedWindow option {-fraction} {} verify {
	instvar frac number

	if { [info exists number] && $number > 0 } {
		$self setNumber $number
	}
} configure {
      	$self replace
}

PanedWindow option {-handleOffs} {-25} verify {
	instvar handlePos hOffs
	
	if { $handleOffs >= 0 } {
		set hOffs $handleOffs
	}
} configure {
	instvar handlePos
	$self configure -handleSide $handlePos
}

PanedWindow option {-handleSide} {end} verify {
	instvar handlePos hOffs handleOffs

      	if { $handleSide == "begin" || $handleSide == 0 } {
	 	set handlePos 0
		set handleOffs $hOffs
      	} elseif { $handleSide == "end" || $handleSide == 1 } {
	 	set handlePos 1
		set handleOffs -$hOffs
      	} else {
         	error "handleSide must be begin or end"
      	}
} configure {
	$self replace
}

PanedWindow option {-manage} {} init {
	eval $self manage $manage
} configure {
	eval $self manage $manage
}

PanedWindow method setNumber { num } {
	instvar frac fraction number

	set number $num
	if {![info exists fraction] || [llength $fraction] < $number-1} {
		set frac(0) 0
         	set part [expr 1.0 / $number.0]
         	for {set i 1} {$i <= $number} {incr i} {
            		set frac($i) [expr $i * $part]
         	}
		while {[info exists frac($i)] == 1} {
			unset frac($i)
			incr i
		}
      	} else {
		$self setFracVector $fraction
	}
}

PanedWindow method setDefaults {} {
	instvar min orientation handlePos hOffs handleOffs frac

	set frac(0) 0
	set handlePos 1
	set hOffs 20
}


# ------------------------------------------------------------------
#  METHOD:  destroy
# ------------------------------------------------------------------
PanedWindow method destroy { args } {
	next $args
}

# ------------------------------------------------------------------
#  METHOD:  init - Called upon object creation.  Creates main window.
# ------------------------------------------------------------------
PanedWindow method init { args } {
	instvar handlePos number orientation inInit pendingManage \
		height width

	$self setDefaults

	set inInit 1

	eval {$self conf_verify} $args
	next -width $width -height $height

	lower $self
	frame $self.priv -borderwidth 0
      	frame $self.priv.pw

	unset inInit
	if [info exists pendingManage] {
		eval $self manage $pendingManage
	}

	eval {$self conf_init} $args
}

PanedWindow method init_unknown { args } {
	eval $self-cmd configure $args
}

# ------------------------------------------------------------------
#  METHOD:  makePanes - internal method
# ------------------------------------------------------------------
PanedWindow method makePanes { no } {
	instvar number orientation handlePos hOffs handleOffs frac \
		fraction

	catch {eval destroy [info commands $self.priv.pane*]}
	catch {eval destroy [info commands $self.priv.sep*]}
	catch {eval destroy [info commands $self.priv.handle*]}
	catch {eval destroy [info commands $self.priv.*marg]}
      	#
      	# Make the windows
      	#

      	for {set i 0} {$i < $no} {incr i} {

	    frame $self.priv.pane$i -borderwidth 0

            if {$i != 0} {

	    	frame $self.priv.pane$i.tmarg -height 5 -width 5
		if { $orientation == "y" } {
		    	pack $self.priv.pane$i.tmarg -side top
		} else {
		    	pack $self.priv.pane$i.tmarg -side left
		}

		#
		# Make the separator
		#
		frame $self.priv.sep$i -height 2 -width 2 -borderwidth 1 \
        		-relief sunken

            	#
            	# Make the sash button
            	#
            	frame $self.priv.handle$i -width 10 -height 10 -borderwidth 2 \
                	-relief raised  \
                	-cursor crosshair

            	if {$orientation == "y"} {
               		bind $self.priv.handle$i <Button-1> \
				"$self start-grip %y $i"
               		bind $self.priv.handle$i <B1-Motion> \
				"$self handle-grip %y $i"
               		bind $self.priv.handle$i <B1-ButtonRelease-1> \
				"$self end-grip %y $i"
            	} else {
               		bind $self.priv.handle$i <Button-1> \
				"$self start-grip %x $i"
               		bind $self.priv.handle$i <B1-Motion> \
				"$self handle-grip %x $i"
               		bind $self.priv.handle$i <B1-ButtonRelease-1> \
				"$self end-grip %x $i"
	    	}

	    }
	    if { $i < $no - 1 } {
		    frame $self.priv.pane$i.bmarg -height 5 -width 5
		    if { $orientation == "y" } {
			    pack $self.priv.pane$i.bmarg -side bottom
		    } else {
			    pack $self.priv.pane$i.bmarg -side right
		    }
	    }
      	}

      	pack $self.priv.pw -fill both -expand yes -anchor w
      	pack $self.priv -fill both -expand yes -anchor w
      	$self replace
}


# ------------------------------------------------------------------
#  METHOD slaves - Returns list of currently managed windows
# ------------------------------------------------------------------
PanedWindow method slaves {} {
	instvar slaves
	return $slaves
}

# ------------------------------------------------------------------
#  METHOD manage - Arranges for a list of windows to be managed
#  		   by PanedWindow.
# ------------------------------------------------------------------
PanedWindow method manage { args } {
	instvar slaves frac fraction number inInit pendingManage

	set slaves {}
	if [info exists inInit] {
		set pendingManage $args
		return
	}
	$self setNumber [llength $args]
	$self makePanes $number
	set j 0
	foreach i $args {
		if { $i == "" } { continue }
		if [catch {pack $i -fill both -expand yes\
			-in $self.priv.pane$j} msg] {
			puts stderr "$msg"
		} else {
			lappend slaves $i
			raise $i $self.priv.pane$j
		}
		incr j
	}
}

# ------------------------------------------------------------------
#  METHOD setFracVector - fills in the frac-array from the fraction var.
# ------------------------------------------------------------------
PanedWindow method setFracVector { fraction } {
	instvar frac number

      	set i 0
      	set frac(0) 0
      	foreach f $fraction {
		if { $i >= $number - 1 } {
			break
		}
        	incr i
         	set frac($i) $f
	 	if {$frac($i) <= $frac([expr $i-1])} {
	    		error "fractions must be in ascending order"
	 	}
      	}
      	incr i
      	set frac($i) 1
      	incr i
	while {[info exists frac($i)]} {
		unset frac($i)
		incr i
	}
}

# ------------------------------------------------------------------
#  METHOD calc-fraction - determines the fraction for the sash
# ------------------------------------------------------------------
PanedWindow method calc-fraction {where num} {
	instvar orientation frac min drag_start frac_start

      	if {$orientation == "y"} {
	 	set frac($num) \
	     		[expr (($where.0 - $drag_start.0) / \
			[winfo height $self]) + $frac_start]
      	} else {
		set frac($num) \
	     		[expr (($where.0 - $drag_start.0) / \
			[winfo width $self]) + $frac_start]
      	}

      	if {$frac($num) < [expr $frac([expr $num - 1]) + $min]} {
         	set frac($num) [expr $frac([expr $num - 1]) + $min]
      	}
      	if {$frac($num) > [expr $frac([expr $num + 1]) - $min]} {
         	set frac($num) [expr $frac([expr $num + 1]) - $min]
      	}
}

# ------------------------------------------------------------------
#  METHOD start-grip - Starts the sash drag and drop operation
# ------------------------------------------------------------------
PanedWindow method start-grip {where num} {
	instvar drag_start drag_win frac_start frac orientation

	set w [winfo toplevel $self]
	if { "$w" == "." } {
		set drag_win ._priv_handle
	} else {
		set drag_win $w._priv_handle
	}

	catch {destroy $drag_win}
	frame $drag_win -relief sunken -borderwidth 1 -width 2 -height 2

	if { $orientation == "y" } {
		place $drag_win -in $self.priv.pw -relx 0 -relwidth 1 \
			-rely $frac($num) -anchor w
	} else {
		place $drag_win -in $self.priv.pw -rely 0 -relheight 1 \
			-relx $frac($num) -anchor n
	}
      	set drag_start $where
	set frac_start $frac($num)
}

# ------------------------------------------------------------------
#  METHOD end-grip - Ends the sash drag and drop operation
# ------------------------------------------------------------------
PanedWindow method end-grip {where num} {
	instvar drag_win

	$self calc-fraction $where $num
	destroy $drag_win
      	$self replace
}

# ------------------------------------------------------------------
#  METHOD handle-grip - Motion action for sash
# ------------------------------------------------------------------
PanedWindow method handle-grip {where num} {
	instvar frac orientation handlePos drag_win

      	$self calc-fraction $where $num

      	if {$orientation == "y"} {
	 	place $drag_win -in $self.priv.pw -relx 0 -relwidth 1 \
	     		-rely $frac($num) -anchor w
      	} else {
	 	place $drag_win -in $self.priv.pw -rely 0 -relheight 1 \
	     		-relx $frac($num) -anchor n
      	}
}

# ------------------------------------------------------------------
#  METHOD replace - Resets the panes of the window following
#                   movement of the sash or reconfiguration
# ------------------------------------------------------------------
PanedWindow method replace {} {
	instvar orientation frac handlePos handleOffs number
 
	if ![winfo exists $self.priv.pane0] { return }

	if {$orientation == "y"} {
	    	place $self.priv.pane0 -in $self.priv.pw -x 0 -rely 0 \
				-relwidth 1 -relheight $frac(1)

	    	for {set i 1} {$i < $number} {incr i} {
	       		place $self.priv.sep$i -in $self.priv.pw -relx 0 -relwidth 1 \
		   		-rely $frac($i) -anchor w
	       		place $self.priv.handle$i -in $self.priv.pw \
				-relx $handlePos -x $handleOffs \
		   		-rely $frac($i) \
		   		-anchor center
	       		place $self.priv.pane$i -in $self.priv.pw -x 0 -rely $frac($i) \
		   		-relwidth 1 \
		   		-relheight \
					[expr $frac([expr $i + 1]) - $frac($i)]
	    	}
	} else {

	    	place $self.priv.pane0 -in $self.priv.pw -y 0 -relx 0 \
				-relheight 1 -relwidth $frac(1)

	    	for {set i 1} {$i < $number} {incr i} {
	       		place $self.priv.sep$i -in $self.priv.pw -rely 0 -relheight 1 \
		   		-relx $frac($i) -anchor n
	       		place $self.priv.handle$i -in $self.priv.pw \
				-rely $handlePos -y $handleOffs \
		   		-relx $frac($i) \
		   		-anchor center
	       		place $self.priv.pane$i -in $self.priv.pw -y 0 -relx $frac($i) \
		   		-relheight 1 \
		   		-relwidth \
					[expr $frac([expr $i + 1]) - $frac($i)]
	    	}
	}
}

# ------------------------------------------------------------------
#  METHOD fractions - Return the current list of fractions
# ------------------------------------------------------------------
PanedWindow method fractions {} {
	instvar number frac

      	set fracs ""
      	for {set i 1} {$i < $number} {incr i} {
         	lappend fracs $frac($i)
      	}
      	return $fracs
}
