
#========================================================================
# TclVSdb-1.1 -- Tcl Very Simple Database
#
# Copyright (c) 1995, Steven B. Wahl
# 
# See the file "license.terms" for information on usage and
# redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# History:
#   04/21/95 - sbw - original code (tclvsdb-1.0)
#   10/28/95 - sbw - upgrade to TclVSdb-1.1
#========================================================================

#----------------
# dbCreate -- create, define, and open a new table for a database
proc dbCreate {path var tablename fieldnames} {
    global $var
	# -- puts stderr "dbCreate: path=\[$path\] var=\[$var\] tablename=\[$tablename\] fieldnames=\[$fieldnames\]"
    # -- basic sanity checking stuff
    if {![file isdirectory $path]} {
        error "dbCreate: database directory \"$path\" does not exist."
    }
    if {![file writable $path]} {
        error "dbCreate: database directory \"$path\" is not writeable by you."
    }
    if {[file exists $path/$tablename.idx]} {
        error "dbCreate: \"$path/$tablename\" already exists...canceled."
    }
    # -- seed the index file and the table file
    exec /bin/echo "00000000 0" > $path/$tablename.idx
    set fldnames [concat seqno $fieldnames DUMMY]
    exec /bin/echo $fldnames > $path/$tablename.tbl
    # -- if the database variable is that of a real open database,
    # -- close it and use the dbOpen to initialize everything nicely
    if {[info exists ${var}(lockhitproc)]} {
        dbClose $var
    }
	set ${var}(isopen) 0
    dbOpen $path $var
}

#----------------
# dbOpen -- open up a database based upon path to database directory
#           set all interesting information in global variable var
proc dbOpen {path var} {
    global $var
	# -- puts stderr "dbOpen: path=\[$path\] var=\[$var\]"
	if {![info exists ${var}(isopen)]} {set ${var}(isopen) 0}
	if {[set ${var}(isopen)]} {return}
    # -- basic sanity checking stuff
    if {![file isdirectory $path]} {
        error "dbOpen: database directory \"$path\" does not exist."
    }
    if {![file writable $path]} {
        error "dbOpen: database directory \"$path\" is not writeable by you."
    }
    if {[catch {glob $path/*.tbl} tablepaths]} {
        error "dbOpen: no database exists at \"$path\".  Use dbCreate first."
    }
    # -- initialize the database global array, open table and
    # -- index files
    set ${var}(path) $path
    set ${var}(tablenames) {}
    set ${var}(lockhitproc) {}
    set ${var}(modnowriteproc) {}
    set ${var}(tablenames) {}
    foreach tablepath $tablepaths {
        set tablename [file tail [file rootname $tablepath]]
        lappend ${var}(tablenames) $tablename
        # -- initialize per table information
        set ${var}($tablename,indexhandle) {}
        set ${var}($tablename,tablehandle) {}
        set ${var}($tablename,fieldnames) {}
        set ${var}($tablename,tablefilepath) {}
        set ${var}($tablename,indexfilepath) {}
        set ${var}($tablename,lastseqno) 0
        set ${var}($tablename,nexttblpos) 0
        set ${var}($tablename,curseqno) 0
        set ${var}($tablename,currowlen) 0
        set ${var}($tablename,currowpos) 0
        set ${var}($tablename,rowdirty) 0
        set ${var}($tablename,tablelock) 0
        set ${var}($tablename,eofflag) 1
        set ${var}($tablename,getcount) 0
        set ${var}($tablename,delcount) 0
        set ${var}($tablename,newcount) 0
        set ${var}($tablename,modcount) 0
        set ${var}($tablename,lckcount) 0
        set ${var}($tablename,stscount) 0
        set ${var}($tablename,stsmax) 0
        set ${var}($tablename,dascount) 0
        set ${var}($tablename,dasmax) 0
        set ${var}($tablename,nuscount) 0
        set ${var}($tablename,nusmax) 0
        # -- now start filling in per table values
        set ${var}($tablename,tablefilepath) "[set ${var}(path)]/$tablename.tbl"
        set ${var}($tablename,indexfilepath) "[set ${var}(path)]/$tablename.idx"
        set ${var}($tablename,tablehandle) [open [set ${var}($tablename,tablefilepath)] r+]
        set ${var}($tablename,indexhandle) [open [set ${var}($tablename,indexfilepath)] r+]
        seek [set ${var}($tablename,tablehandle)] 0 start
        set ${var}($tablename,fieldnames) [gets [set ${var}($tablename,tablehandle)]]
        foreach fieldname [set ${var}($tablename,fieldnames)] {
            set ${var}($tablename,$fieldname) {}
        }
        set indexfilesize [file size [set ${var}($tablename,indexfilepath)]]
        set ${var}($tablename,lastseqno) [expr ($indexfilesize / 11) - 1]
        set ${var}($tablename,nexttblpos) \
            [expr [file size [set ${var}($tablename,tablefilepath)]] - 0]
        dbiRowTraceOff $var $tablename
    }
    if {![info exists ${var}(modnowriteproc)]} {set ${var}(modnowriteproc) {}}
    if {![info exists ${var}(lockhitproc)]} {set ${var}(lockhitproc) {}}
	set ${var}(isopen) 1
}
        
#----------------
# dbCleanup -- compress voids and resequence table files
proc dbCleanup {var {tpath ""}}  {
    global $var
	# -- puts stderr "dbCleanup: var=\[$var\] tpath=\[$tpath\]"
    # -- create a temporary  database variable
    set tvar "${var}_tmp"
    global $tvar 
    # -- define location of cleanup database (default is original location)
    set path [set ${var}(path)]
    if {$tpath == {}} {
        set ${tvar}(path) $path
        set tpath $path
    } else {
        set ${tvar}(path) $tpath
    }
    # -- capture the database table names
    set tablenames [set ${var}(tablenames)]
    # -- basic sanity checking stuff
    if {![file isdirectory [set ${tvar}(path)]]} {
        error "dbCleanup: database directory \"$tpath\" does not exist."
    }
    if {![file writable [set ${tvar}(path)]]} {
        error "dbCreate: database directory \"$tpath\" is not writeable by you."
    }
    # -- create a mirror database definition
    foreach tablename $tablenames {
        set tablenamet "${tablename}_tmp"
        # -- build a list of field names for the table (without seqno, DUMMY)
        set last [expr [llength [set ${var}($tablename,fieldnames)]] - 2]
        set fieldnames [lrange [set ${var}($tablename,fieldnames)] 1 $last]
        # -- create the table files
        dbCreate $tpath $tvar $tablenamet $fieldnames
        # -- now walk through the original rows of the table
        # -- and write them to the temporary database (if they exist)
        set seqno [dbFirstRow $var $tablename]
        while {$seqno > 0} {
            dbNewRow $tvar $tablenamet $seqno
            dbGetRow $var $tablename $seqno
             foreach fieldname $fieldnames {
                set ${tvar}($tablenamet,$fieldname) \
                    [set ${var}($tablename,$fieldname)]
            }
            dbPutRow $tvar $tablenamet
            set seqno [dbNextRow $var $tablename]
        }
    }
    dbClose $tvar
    dbClose $var
    # -- copy the temporary table files to the current database
    # -- location and name, then delete the temporary table files
    foreach tablename $tablenames {
        exec /bin/cp $tpath/${tablename}_tmp.idx $path/${tablename}.idx
        exec /bin/cp $tpath/${tablename}_tmp.tbl $path/${tablename}.tbl
        exec /bin/rm $tpath/${tablename}_tmp.idx
        exec /bin/rm $tpath/${tablename}_tmp.tbl
    }
    dbOpen $path $var
}
        
#----------------
# dbClose -- close a database
proc dbClose {var} {
    global $var
	# -- puts stderr "dbClose: var=\[$var\]"
	if {![set ${var}(isopen)]} {return}
    foreach tablename [set ${var}(tablenames)] {
        if {[set ${var}($tablename,rowdirty)]} {
            if {[string length [set ${var}(modnowriteproc)]] > 0} {
                if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
                    dbPutRow $var $tablename
                }
             } else {
                set ${var}($tablename,rowdirty) 0
            }
        }
        close [set ${var}($tablename,tablehandle)]
        close [set ${var}($tablename,indexhandle)]
    }
    if {![info exists DEBUGDB]} {
        unset $var
    }
	set ${var}(isopen) 0
}

#----------------
# dbGetRow -- fetch a row from a table by its sequence number
#          -- return "1" if row exists, "0" otherwise
proc dbGetRow {var tablename seqno} {
    global $var
	# -- puts stderr "dbGetRow: var=\[$var\] tablename=\[$tablename\] seqno=\[$seqno\]"
    set ${var}($tablename,eofflag) 1
    # -- test if we have uncommitted changes to current row
    if {[set ${var}($tablename,rowdirty)]} {
        if {[string length [set ${var}(modnowriteproc)]] > 0} {
            if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
                dbPutRow $var $tablename
            }
        } else {
            set ${var}($tablename,rowdirty) 0
        }
    }
    dbiRowTraceOff $var $tablename
    # -- if row is outside of range, act as if a deleted row
    if {$seqno <= 0} {
        error "dbGetRow:  bad sequence number \"$seqno\""
    }
    if {$seqno > [set ${var}($tablename,lastseqno)]} {
        set ${var}($tablename,currowlen) 0
        set ${var}($tablename,currowpos) 0
        set ${var}($tablename,seqno) $seqno
		set ${var}($tablename,curseqno) $seqno
        # -- clear the contents of the current row fields
        foreach fieldname [set ${var}($tablename,fieldnames)] {
            if {!($fieldname == "seqno" || $fieldname == "DUMMY")} {
                set ${var}($tablename,$fieldname) {}
            }
        }
        set ${var}($tablename,DUMMY) "                         "
		set ${var}($tablename,seqno) $seqno
        return 0
    }
    set tbloffset [dbiRowLockOn $var $tablename $seqno]
    if {$tbloffset != 0} {
        # -- datafull row, get the data
        seek [set ${var}($tablename,tablehandle)] $tbloffset start
        set irow [gets [set ${var}($tablename,tablehandle)]]
        set ${var}($tablename,currowlen) [string length $irow]
        set ${var}($tablename,curseqno) $seqno
        set ${var}($tablename,currowpos) $tbloffset
        # -- convert newline markers to newlines, remove any trailing newlines
        regsub -all {<CR>} $irow \n trow
        set row [string trimright $trow]
		set i 0
		foreach fieldname [set ${var}($tablename,fieldnames)] {
			set db($tablename,$fieldname) [lindex $row $i]
			incr i
		}
    } else {
        # -- this is a previously deleted row, clear some stuff
        set ${var}($tablename,currowlen) 0
        set ${var}($tablename,currowpos) 0
        set ${var}($tablename,seqno) $seqno
		set ${var}($tablename,curseqno) $seqno
        # -- clear the contents of the current row fields
        foreach fieldname [set ${var}($tablename,fieldnames)] {
            if {!($fieldname == "seqno" || $fieldname == "DUMMY")} {
                set ${var}($tablename,$fieldname) {}
            }
        }
        set ${var}($tablename,DUMMY) "                         "
    }
    set ${var}($tablename,eofflag) 0
    set ${var}($tablename,rowdirty) 0
    dbiRowLockOff $var $tablename $seqno $tbloffset
    dbiRowTraceOn $var $tablename
    incr ${var}($tablename,getcount)
    if {$tbloffset != 0} {
        return 1
    }
    return 0
}

#----------------
# dbPutRow -- write current row into database
proc dbPutRow {var tablename} {
    global $var
	# -- puts stderr "dbPutRow: var=\[$var\] tablename=\[$tablename\]"
    dbiRowTraceOff $var $tablename 
    set ${var}($tablename,eofflag) 0
    # -- if no changes to the fields in the row, simply return
    if {0 == [set ${var}($tablename,rowdirty)]} {
        dbiRowTraceOn $var $tablename
        return
    }
    # -- create output record of all fields appended into a list
    set row ""
    set rowlen 0
    foreach fieldname [set ${var}($tablename,fieldnames)] {
		if {$fieldname != "seqno" && $fieldname != "DUMMY"} { 
			set slen [string length [set ${var}($tablename,$fieldname)]]
			incr slen -1
			set field [string trimright [set ${var}($tablename,$fieldname)]]
			regsub -all \n $field <CR> ofld
			lappend row $ofld
			incr rowlen [string length $ofld]
		} else {
			set slen [string length [set ${var}($tablename,$fieldname)]]
			incr slen -1
			lappend row [set ${var}($tablename,$fieldname)]
		}
    }
    # -- if the row was empty, and is empty now, return
    if {$rowlen == 0 && [set ${var}($tablename,currowlen)] == 0} {
        dbiRowTraceOn $var $tablename
        return 
    }
    # -- if the row in the database was non-empty, but is now empty, 
    # -- delete the row and return
    if {$rowlen == 0 && [set ${var}($tablename,currowlen)] > 0} {
        dbiRowTraceOn $var $tablename
        dbDelRow $var $tablename
        return
    }
    # -- get info on row length, and DUMMY field padding
    set dummylen [string length [set ${var}($tablename,DUMMY)]]
    set rowlen [string length $row]
    set rowdiff [expr $rowlen - [set ${var}($tablename,currowlen)]]
    # -- if the current row is beyond the end of the file, create index
    # -- entries through this row (artifact of dbiRowLockOff)
    dbiRowLockOff $var $tablename
    # -- get a lock on the row and its offset in the table file
    set tbloffset [dbiRowLockOn $var $tablename]
    if {$tbloffset == 0} {
        set tbloffset [set ${var}($tablename,nexttblpos)]
        incr ${var}($tablename,nexttblpos) $rowlen
    }
    set ${var}($tablename,currowpos) $tbloffset
    # -- handle the special case that this is a new row or a
    # -- previously deleted row
    if {[set ${var}($tablename,currowpos)] == 0} {
        # -- get a table lock
        dbiTableLockOn $var $tablename
        set ${var}($tablename,currowpos) [set ${var}($tablename,nexttblpos)]
        set ${var}($tablename,currowlen) $rowlen
        incr ${var}($tablename,nexttblpos) [expr 1 + $rowlen] 
        # -- setup to fall through to actual table write of row
        set rowdiff 0
    } elseif {[expr [set ${var}($tablename,currowpos)] + \
                [set ${var}($tablename,currowlen)] + 1] == \
          [set ${var}($tablename,nexttblpos)]} {
        # -- special case that this is the last physical row in
        # -- table, always expand/contract in place.
        # -- index entry is ok at this point.
        # -- if contracting, expand dummy field to size.
        if {$rowdiff < 0} {
            set ${var}($tablename,DUMMY) \
                [format "%[expr $dummylen + abs($rowdiff)]s" " "]
            set lastelem [expr [llength $row] - 1]
            set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
            set rowlen [string length $row]
        } elseif {$rowdiff > 0} {
            # -- get a table lock
            dbiTableLockOn $var $tablename
            # -- ensure that padding field is maximum size
            set ${var}($tablename,DUMMY) [format "%25s" " "]
            set dummylen [string length [set ${var}($tablename,DUMMY)]]
            set lastelem [expr [llength $row] - 1]
            set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
            set rowlen [string length $row]
        }
        # -- set up to fall through to actual table write
        set rowdiff 0
    }
    if {$rowdiff < 0} {
        # -- row will fit in place with adjustment to dummy pad area
        set ${var}($tablename,DUMMY) [format "%[expr $dummylen + abs($rowdiff)]s" " "]
        set lastelem [expr [llength $row] - 1]
        set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
        set rowlen [string length $row]
    } elseif {$rowdiff > 0} {
        # -- see if the row will fit in place by reducing DUMMY
        if {$rowdiff < $dummylen} {
            set ${var}($tablename,DUMMY) [format "%[expr $dummylen - $rowdiff]s" " "]
            set lastelem [expr [llength $row] - 1]
            set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
            set rowlen [string length $row]
        } else {
            # -- not enough space in current table position, move row
            # -- to end of table file
            # -- first, clear the current entry in the table file.
            set blank [format "%[expr [set ${var}($tablename,currowlen)] + 1]s" " "]
            seek [set ${var}($tablename,tablehandle)] [set ${var}($tablename,currowpos)] start
            puts -nonewline [set ${var}($tablename,tablehandle)] $blank
            # -- reexpand DUMMY field, if necessary
            if {$dummylen < 25} {
                set ${var}($tablename,DUMMY) \
                    "                         "
                set lastelem [expr [llength $row] - 1]
                set row [lreplace $row $lastelem $lastelem [set ${var}($tablename,DUMMY)]]
                set rowlen [string length $row]
            }
            # -- set a lock on the table
            dbiTableLockOn $var $tablename
            # -- now update the index file row offset (row still locked)
            set ${var}($tablename,currowpos) [set ${var}($tablename,nexttblpos)]
            # -- now setup to write row to end of table file
            set ${var}($tablename,currowlen) $rowlen
            set ${var}($tablename,currowpos) [set ${var}($tablename,nexttblpos)]
            incr ${var}($tablename,nexttblpos) [expr 1 + $rowlen]
        }
    }
    # -- write the row to the tablespace file
    seek [set ${var}($tablename,tablehandle)] [set ${var}($tablename,currowpos)] start
    puts [set ${var}($tablename,tablehandle)] $row
    flush [set ${var}($tablename,tablehandle)]
    # -- if the table was locked, free it
    if {[set ${var}($tablename,tablelock)]} {
        incr ${var}($tablename,nexttblpos) [expr $rowlen - [set ${var}($tablename,currowlen)]]
        dbiTableLockOff $var $tablename
    }
    set ${var}($tablename,currowlen) $rowlen
    # -- update the index and free the row lock
    dbiRowLockOff $var $tablename [set ${var}($tablename,curseqno)] \
            [set ${var}($tablename,currowpos)]
    # -- reset row dirty flag and turn field mod trace back on
    set ${var}($tablename,rowdirty) 0
    dbiRowTraceOn $var $tablename
    # -- increment the row write counter for session
    incr ${var}($tablename,modcount)
}

#----------------
# dbClearRow -- clear the contents of the current row, don't modify database
proc dbClearRow {var tablename} {
    global $var
	# -- puts stderr "dbClearRow: var=\[$var\] tablename=\[$tablename\]"
    # -- test if we have uncommitted changes to current row
    dbiRowTraceOff $var $tablename
    if {[set ${var}($tablename,rowdirty)]} {
        if {[string length [set ${var}(modnowriteproc)]] > 0} {
            if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
                dbPutRow $var $tablename
            }
        } else {
            set ${var}($tablename,rowdirty) 0
        }
    }
    foreach fieldname [set ${var}($tablename,fieldnames)] {
        set ${var}($tablename,$fieldname) {}
    }
    set ${var}($tablename,DUMMY) "                         "
    set ${var}($tablename,seqno) [set ${var}($tablename,curseqno)]
    dbiRowTraceOn $var $tablename
}

#----------------
# dbNewRow -- create a new row in a table
#             return the seqno of the newly created row
proc dbNewRow {var tablename} {
    global $var
	# -- puts stderr "dbNewRow: var=\[$var\] tablename=\[$tablename\]"
    # -- test if we have uncommitted changes to current row
    if {[set ${var}($tablename,rowdirty)]} {
        if {[string length [set ${var}(modnowriteproc)]] > 0} {
            if {[eval [set ${var}(modnowriteproc)] $var $tablename]} {
                dbPutRow $var $tablename
            }
        } else {
            set ${var}($tablename,rowdirty) 0
        }
    }
    dbiRowTraceOff $var $tablename
    dbiTableLockOn $var $tablename
    incr ${var}($tablename,lastseqno)
    set ${var}($tablename,curseqno) [set ${var}($tablename,lastseqno)]
    dbClearRow $var $tablename    
    set ${var}($tablename,currowpos) 0
    set ${var}($tablename,seqno) [set ${var}($tablename,curseqno)]
    # -- update the index for the new record via a row unlock
    dbiRowLockOff $var $tablename [set ${var}($tablename,curseqno)] 0
    dbiTableLockOff $var $tablename
    set ${var}($tablename,rowdirty) 0
    dbiRowTraceOn $var $tablename
    incr ${var}($tablename,newcount)
    set ${var}($tablename,eofflag) 0
    return [set ${var}($tablename,curseqno)]
}

#----------------
# dbDelRow -- delete current row from a table
proc dbDelRow {var tablename} {
    global $var
	# -- puts stderr "dbDelRow: var=\[$var\] tablename=\[$tablename\]"
    set seqno [set ${var}($tablename,curseqno)]
    set pos [set ${var}($tablename,currowpos)]
    set ${var}($tablename,eofflag) 0
    # -- test if row has already been deleted
    if {$pos == 0} {return}
    dbiRowTraceOff $var $tablename
    set tbloffset [dbiRowLockOn $var $tablename]
    # -- if there was data in table file, then
    # -- overwrite the row data with an empty line of equal size
    if {$tbloffset != 0} {
        set len [expr 1 + [set ${var}($tablename,currowlen)]]
        set blanks [format "%${len}s" " "]
        seek [set ${var}($tablename,tablehandle)] $pos start
        puts -nownewline [set ${var}($tablename,tablehandle)] $blanks
        flush [set ${var}($tablename,tablehandle)]
    }
    # -- update the index file for this row to show no row data offset
    dbiRowLockOff $var $tablename $seqno 0
    dbClearRow $var $tablename
    set ${var}($tablename,rowdirty) 0
    incr ${var}($tablename,delcount)
}

#----------------
# dbFirstRow -- position to first row from table
#               return seqno if valid row found, "0" otherwise
proc dbFirstRow {var tablename} {
    global $var
    set lastseqno [set ${var}($tablename,lastseqno)]
    set ${var}($tablename,eofflag) 1
    if {$lastseqno == 0} {return 0}
    # -- test if we have uncommitted changes to current row
    if {[set ${var}($tablename,rowdirty)]} {
        if {[string length [set ${var}(modnowriteproc)]] > 0} {
            if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
                dbPutRow $var $tablename
            }
        } else {
            set ${var}($tablename,rowdirty) 0
        }
    }
    set seqno 1
    while {0 == [dbGetRow $var $tablename $seqno]} {
        incr seqno
        if {$seqno > $lastseqno} {return 0}
    }
    set ${var}($tablename,eofflag) 0
    return $seqno
}

#----------------
# dbLastRow -- position to last row in table
#               return seqno if valid row found, "0" otherwise
proc dbLastRow {var tablename} {
    global $var 
    set lastseqno [set ${var}($tablename,lastseqno)]
    set ${var}($tablename,eofflag) 1
    if {$lastseqno == 0} {return 0}
    # -- test if we have uncommitted changes to current row
    if {[set ${var}($tablename,rowdirty)]} {
        if {[string length [set ${var}(modnowriteproc)]] > 0} {
            if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
                dbPutRow $var $tablename
            }
        } else {
            set ${var}($tablename,rowdirty) 0
        }
    }
    set seqno [set ${var}($tablename,lastseqno)]
    while {0 == [dbGetRow $var $tablename $seqno]} {
        incr seqno -1
        if {$seqno <= 0} {return 0}
    }
    set ${var}($tablename,eofflag) 0
    return $seqno
}

#----------------
# dbNextRow -- position to next row in table
#              return seqno if valid row found, "0" otherwise
proc dbNextRow {var tablename} {
    global $var
    set lastseqno [set ${var}($tablename,lastseqno)]
    set ${var}($tablename,eofflag) 1
    if {$lastseqno == 0} {return 0}
    # -- test if we have uncommitted changes to current row
    if {[set ${var}($tablename,rowdirty)]} {
        if {[string length [set ${var}(modnowriteproc)]] > 0} {
            if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
                dbPutRow $var $tablename
            }
        } else {
            set ${var}($tablename,rowdirty) 0
        }
    }
    set seqno [expr [set ${var}($tablename,curseqno)] + 1]
    while {0 == [dbGetRow $var $tablename $seqno]} {
        incr seqno
        if {$seqno > $lastseqno} {return 0}
    }
    set ${var}($tablename,eofflag) 0
    return $seqno
}

#----------------
# dbPrevRow -- position to previous row in table
#              return "1" if valid row found, "0" otherwise
proc dbPrevRow {var tablename} {
    global $var 
    set lastseqno [set ${var}($tablename,lastseqno)]
    set ${var}($tablename,eofflag) 1
    if {$lastseqno == 0} {return 0}
    # -- test if we have uncommitted changes to current row
    if {[set ${var}($tablename,rowdirty)]} {
        if {[string length [set ${var}(modnowriteproc)]] > 0} {
            if {[eval [set ${var}(modnowriteproc)] $var $tablename ""]} {
                dbPutRow $var $tablename
            }
        } else {
            set ${var}($tablename,rowdirty) 0
        }
    }
    set seqno [expr [set ${var}($tablename,curseqno)] - 1]
    while {0 == [dbGetRow $var $tablename $seqno]} {
        incr seqno -1
        if {$seqno <= 0} {return 0}
    }
    if {$seqno <= 0} {return 0}
    set ${var}($tablename,eofflag) 0
    return $seqno
}

#---------------------
# dbSetProc -- register procedure to call for following conditions:
#             1) if the row desired is locked
#                USE:  dbSetProc var lock procname
#                The "procname" procedure needs to have the argument list:
#                  proc procname var tablename
#                and return a "0" if the lock is to be freed, or "1" if
#                the lock is to be retained.
#             2) if the current row has been modified and not written
#                to the database.
#                USE:  dbSetProc var mod procname
#                The "procname" procedure need to have the argument list:
#                  proc procname var tablename
#                and return a "1" if the current row should be written
#                to the database, or a "0" if the new row contents discarded.
#
proc dbSetProc {var function procname} {
    global $var
    switch -glob -- [string tolower $function] {
        loc* {set ${var}(lockhitproc) $procname}
        mod* {set ${var}(modnowriteproc) $procname}
    }
}

#---------------------
# dbEOF -- return 1 if beyond the limits of a table, 0 otherwise
proc dbEOF {var tablename} {
	# -- puts stderr "dbEOF: var=\[$var\] tablename=\[$tablename\]"
    global $var
    return [set ${var}($tablename,eofflag)]
}

#====================== INTERNAL PROCEDURES ===============================
#------------------
# dbiMarkRow -- mark the current row as modified
proc dbiMarkRow {var element unused} {
    global $var
	# -- puts stderr "dbiMarkRow: var=\[$var\] element=\[$element\] unused=\[$unused\]"
    set tablename [lindex [split $element ","] 0]
    set ${var}($tablename,rowdirty) 1
}

#------------------
# dbiRowTraceOn -- set a trace on the current row for any modifications
proc dbiRowTraceOn {var tablename} {
    global $var
	# -- puts stderr "dbiRowTraceOn: var=\[$var\] tablename=\[$tablename\]"
    foreach fieldname [set ${var}($tablename,fieldnames)] {
        set t "${var}($tablename,$fieldname)"
        trace variable $t w dbiMarkRow
    }
}

#------------------
# dbiRowTraceOff -- free a trace previously placed on the current row
#                   for any modifications made to it.
proc dbiRowTraceOff {var tablename} {
    global $var
	# -- puts stderr "dbiRowTraceOff: var=\[$var\] tablename=\[$tablename\]"
    foreach fieldname [set ${var}($tablename,fieldnames)] {
        set t "${var}($tablename,$fieldname)"
        trace vdelete $t w dbiMarkRow
    }
}

#----------------------
# dbiRowLockOn -- set an exclusive lock on (current) row
proc dbiRowLockOn {var tablename {seqno ""}} {
    global $var
	# -- puts stderr "dbiRowLockOn: var=\[$var\] tablename=\[$tablename\] seqno=\[$seqno\]"
    if {$seqno == ""} {
        set seqno [set ${var}($tablename,curseqno)]
    }
    set ioffset [expr $seqno * 11]
    set ifd [set ${var}($tablename,indexhandle)]
    seek $ifd $ioffset start
    scan [gets $ifd] "%x %s" tbloffset lock
    while {$lock} {
        # -- give it one second to free up on its own
        incr ${var}($tablename,lckcount)
        exec /bin/sleep 1
        seek $ifd $ioffset start
        scan [gets $ifd] "%x %s" tbloffset lock
        if {$lock} {
            # -- still locked, call user's proc or handle default way
            # -- which is to spin on the lock 5 times, then clear it
            if {[set ${var}(lockhitproc)] != ""} {
                set lock [eval [set ${var}(lockhitproc)] $var $tablename]
            } else {
                set spin 0
                while {$lock && $spin < 5} {
                    exec /bin/sleep 1
                    seek $ifd $ioffset start
                    scan [gets $ifd] "%x %s" tbloffset lock
                    incr spin
                    if {$lock} {
                        puts stderr "dbiRowLockOn: freeing preexisting persistent row lock in $tablename at row $seqno"
                        set lock 0
                    }
                }
            }
        }
    }
    # -- set the row lock
    seek $ifd $ioffset start
    puts $ifd [format "%08x 1" $tbloffset]
    flush $ifd
    return $tbloffset
}

#----------------------
# dbiRowLockOff -- free an exclusive lock on (current) row
#                  can be used to set the data offset position for a row
proc dbiRowLockOff {var tablename {seqno ""} {rowpos ""}} {
    global $var
	# -- puts stderr "dbiRowLockOff: var=\[$var\] tablename=\[$tablename\] seqno=\[$seqno\] rowpos=\[$rowpos\]"
    if {$seqno == ""} {
        set seqno [set ${var}($tablename,curseqno)]
    }
#    set ${var}($tablename,curseqno) $seqno
    if {$seqno > [set ${var}($tablename,lastseqno)]} {
        dbiGenIndex $var $tablename
    }
    seek [set ${var}($tablename,indexhandle)] [expr $seqno * 11] start
    if {$rowpos == ""} {
        scan [gets [set ${var}($tablename,indexhandle)]] "%x %s" rowpos lock
        seek [set ${var}($tablename,indexhandle)] [expr $seqno * 11] start
    }
    puts [set ${var}($tablename,indexhandle)] [format "%08x 0" $rowpos]
    flush [set ${var}($tablename,indexhandle)]
}

#----------------------
# dbiTableLockOn -- obtain a lock on a table to grow it
proc dbiTableLockOn {var tablename} {
    global $var
	# -- puts stderr "dbiTableLockOn: var=\[$var\] tablename=\[$tablename\]"
    # -- lock the table by locking seqno 0
    set x [dbiRowLockOn $var $tablename 0]
    set ${var}($tablename,tablelock) 1
    # -- recalibrate the table variables
    set ${var}($tablename,nexttblpos) \
            [expr [file size [set ${var}($tablename,tablefilepath)]] - 0]
    set idxsize [file size [set ${var}($tablename,indexfilepath)]]
    set ${var}($tablename,lastseqno) [expr ($idxsize / 11) - 1]
    set ${var}($tablename,nexttblpos) [file size [set ${var}($tablename,tablefilepath)]]
}

#----------------------
# dbiTableLockOff -- free a lock on a table
proc dbiTableLockOff {var tablename} {
    global $var
	# -- puts stderr "dbiTableLockOff: var=\[$var\] tablename=\[$tablename\]"
    dbiRowLockOff $var $tablename 0
    set ${var}($tablename,tablelock) 0
}

#----------------------
# dbiGenIndex -- create empty index entries from end of table to current
#                sequence number
proc dbiGenIndex {var tablename} {
    global $var
	# -- puts stderr "dbiGenIndex: var=\[$var\] tablename=\[$tablename\] curseqno=[set ${var}($tablename,curseqno)]"

    dbiTableLockOn $var $tablename
    set seqno [set ${var}($tablename,curseqno)]
    set endseqno [set ${var}($tablename,lastseqno)]
    set nonewentries [expr $seqno - $lastseqno]
    if {$nonewentries > 20} {
        puts stderr "dbiGenIndex: warning - creating $nonewentries new rows"
    }
    set idxpos [expr ($lastseqno + 1) * 11]
    set ifd [set ${var}($tablename,indexhandle)]
    while {$nonewentries > 0} {
        seek $ifd $idxpos start
        puts $ifd "00000000 0"
        incr nonewentries -1
    }
    flush $ifd
    set ${var}($tablename,lastseqno) $curseqno
}


#=============================================================================
# Database Utilities

#------------------
# dbuDateFromTs -- return date string from timestamp value
proc dbuDateFromTs {ts} {
	set modays(norm) "0 31 59 90 120 151 181 212 243 273 304 334 365"
	set modays(leap) "0 31 60 91 121 152 182 213 244 274 305 335 366"
	set monthnames "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec"
	set year [expr 1970 + ($ts / 31557600)]
	set yrtype norm
	if {[expr $year % 4] == 0} {set yrtype leap}
	set ts [expr $ts % 31557600]
	set day [expr 1 + ($ts / 86400)]
	for {set i 0} {$i < 13} {incr i} {
		if {$day < [lindex $modays($yrtype) $i]} {
			set month [lindex $monthnames [expr $i - 1]]
			set mostart [lindex $modays($yrtype) [expr $i - 1]]
			set day [expr $day - $mostart]	
			break
		}
	}
	return "$month $day, $year"
}

#------------------
# dbuDateToTs -- return timestamp value from a date string
proc dbuDateToTs {date} {
	if {[catch {scan $date "%3s %d, %d" mon day yr}]} {return 0}
	set ts [expr ($yr - 1970) * 31557600]
	set yrtype "norm"
	if {[catch {expr "$day + 0"}]} {return 0}
	if {[catch {expr "$yr + 0"}]} {return 0}
	if {$day < 1 || $day > 31} {return 0}
	if {$yr < 1970 || $yr > 2100} {return 0}
	if {[expr $yr%4] == 0} {set yrtype leap}
	set modays(norm) "0 31 59 90 120 151 181 212 243 273 304 334"
	set modays(leap) "0 31 60 91 121 152 182 213 244 274 305 335"
	set monthnames "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec"
	set moidx [lsearch $monthnames $mon]
	if {$moidx == -1} {return 0}
	set yrday [expr [lindex $modays($yrtype) $moidx] + $day - 1]
	set yrday [expr $yrday * 86400]
	return [expr $ts + $yrday + 3600]
}

#------------------
# dbuDateToday -- return today's date as a string
proc dbuDateToday {} {
	scan [exec /bin/date "+%h %d, 19%y"] "%3s %d, %4s" mo da yr
	return "$mo $da, $yr"
}


#----------------
# return a list of seqno from a table who have a field matching glob
# search string

proc dbuSearchString {var tablename fieldname searchstring} {
	global $var
	if {[set ${var}($tablename,lastseqno)] == 0} {
		incr ${var}($tablename,stscount)
		return {}
	}
	if {[string length $searchstring] == 0} {return {}}
	set searchstring [string tolower "*${searchstring}*"]
	set result ""
	set fieldindex -1
	set i 0
	foreach field [set ${var}($tablename,fieldnames)] {
		if {$field == $fieldname} {
			set fieldindex $i
			break
		}
		incr i
	}
	if {$fieldindex == -1} {
		error "dbuSearchString: field name $fieldname does not \
exist in table $tablename."
	}
	for {set seqno 1} {$seqno <= [set ${var}($tablename,lastseqno)]} {incr seqno} {
		# -- get the row position in table file and lock row
		set rowpos [dbiLockRow $var $tablename $seqno]
		# -- if an empty or deleted row, free lock and go on to next row
		if {$rowpos != 0} {
			# -- get the row from the table and check for string match
			seek [set ${var}($tablename,tablehandle)] $rowpos
			set field [lindex [gets [set ${var}($tablename,tablehandle)]] $fieldindex]
			if {[string match $searchstring [string tolower $field]]} {
				lappend result $seqno
			}
		}
		# -- free the lock on the row
		dbfreerowlock $var $tablename $seqno $rowpos
	}
	incr ${var}($tablename,stscount)
	set resultsize [llength $result]
	if {$resultsize > [set ${var}($tablename,stsmax)]} {
		set ${var}($tablename,stsmax) $resultsize
	}
	return $result
}

#----------------
# return a list of seqnos from a table who have a date field between
# or matching a date in the date range

proc dbuSearchDate {var tablename fieldname date1 date2} {
	global $var
	if {[set ${var}($tablename,lastseqno)] == 0} {
		incr ${var}($tablename,dascount)
		return {}
	}
	if {[dbuDateToTs $date1] == 0} {return {}}
	if {[dbuDateToTs $date2] == 0} {return {}}
	set result {}
	set fieldindex -1
	set i 0
	foreach field [set ${var}($tablename,fieldnames)] {
		if {$field == $fieldname} {
			set fieldindex $i
			break
		}
		incr i
	}
	if {$fieldindex == -1} {
		error "dbuSearchString: field name $fieldname does not \
exist in table $tablename."
	}
	set ts1 [dbuDateToTs $date1]
	set ts2 [dbuDateToTs $date2]
	for {set seqno 1} {$seqno <= [set ${var}($tablename,lastseqno)]} {incr seqno} {
		# -- get the row position in table file and lock row
		set rowpos [dbiLockRow $var $tablename $seqno]
		# -- if an empty or deleted row, free lock and go on to next row
		if {$rowpos != 0} {
			# -- get the row from the table and check for string match
			seek [set ${var}($tablename,tablehandle)] $rowpos
			set field [dbuDateToTs [lindex \
				[gets [set ${var}($tablename,tablehandle)]] $fieldindex]]
			if {$field == 0} {return {}}
			if {$ts1 <= $field && $ts2 >= $field} {
				lappend result $seqno
			}
		}
		# -- free the lock on the row
		dbfreerowlock $var $tablename $seqno $rowpos
	}
	incr ${var}($tablename,dascount)
	set resultsize [llength $result]
	if {$resultsize > [set ${var}($tablename,dasmax)]} {
		set ${var}($tablename,dasmax) $resultsize
	}
	return $result
}

#----------------
# return a list of seqnos from a table who have a numeric value
# between or matching values in a numeric range

proc dbuSearchValue {var tablename fieldname num1 num2} {
	global $var
	if {[set ${var}($tablename,lastseqno)] == 0} {
		incr ${var}($tablename,nuscount)
		return {}
	}
	if {[catch {expr "$num1 + $num2"}]} {
#		# -- puts "dbuSearchValue $num1 or $num2 are not valid numbers."
		return {}
	}
	set result {}
	set fieldindex -1
	set i 0
	foreach field [set ${var}($tablename,fieldnames)] {
		if {$field == $fieldname} {
			set fieldindex $i
			break
		}
		incr i
	}
	if {$fieldindex == -1} {
		error "dbuSearchString: field name $fieldname does not \
exist in table $tablename."
	}
	for {set seqno 1} {$seqno <= [set ${var}($tablename,lastseqno)]} {incr seqno} {
		# -- get the row position in table file and lock row
		set rowpos [dbiLockRow $var $tablename $seqno]
		# -- if an empty or deleted row, free lock and go on to next row
		if {$rowpos != 0} {
			# -- get the row from the table and check for string match
			seek [set ${var}($tablename,tablehandle)] $rowpos
			set field [lindex [gets [set ${var}($tablename,tablehandle)]] $fieldindex]
			if {[catch {expr "$field + 0"}]} {
#				# -- puts stderr "dbuSearchValue: seqno=$seqno, field=$field, invalid numeric value"
				return {}
			}
			if {$num1 <= $field && $num2 >= $field} {
				lappend result $seqno
			}
		}
		# -- free the lock on the row
		dbfreerowlock $var $tablename $seqno $rowpos
	}
	incr ${var}($tablename,nuscount)
	set resultsize [llength $result]
	if {$resultsize > [set ${var}($tablename,nusmax)]} {
		set ${var}($tablename,nusmax) $resultsize
	}
	return $result
}

#
#-------------------
# END OF FILE
#



