/* $Id: stooop.c,v 1.22 1995/12/23 00:27:18 jfontain Exp $ */

/*
    # how to create a dynamically loadable library on different platforms (please send me commands for your platform if not listed):

    # SunOS
    cc -O2 -pic -I/home/sources/tcl7.5a2/generic -c stooop.c                                                   # with stock compiler
    gcc -O2 -fPIC -Wall -I/home/sources/tcl7.5a2/generic -c stooop.c                                                      # with gcc
    ld -assert pure-text -o libstooop.so.2.1 stooop.o

    # AT&T SVR4.0
    cc -O -KPIC -Xa -v -I/home/sources/tcl7.5a2/generic -c stooop.c                                            # with stock compiler
    gcc -O2 -fPIC -Wall -I/home/sources/tcl7.5a2/generic -c stooop.c                                                      # with gcc
    ld -G -o libstooop.so.2.1 -h libstooop.so.2.1 stooop.o

    # (the Tcl generic source code directory is needed for tclInt.h, its location may be different on your machine)

    # to use it in your Tcl code, just load in stooop.tcl, as in:
        load libstooop.so.2.1 stooop
*/

#include <stdlib.h>
#include <tcl.h>
#include <tclInt.h>

static struct Tcl_HashTable hIdentifiers;
static struct Tcl_HashTable hClasses;
static int iNewId;

static unsigned uAddIdentifier (nClass, i)              /* returns a boolean, true if identifier was added to classes associative */
    char *nClass;                                        /* array, which should always be the case since the identifier is unique */
    int i;
{
    struct Tcl_HashEntry *ph;
    char *pc;
    int iCreated;

    ph = Tcl_CreateHashEntry(&hClasses, nClass, &iCreated);                                    /* first eventually register class */
    if(iCreated){                                                                                                    /* new class */
        pc = malloc(strlen(nClass) + 1);           /* class entries are never freed for there should not be a huge number of them */
        strcpy(pc, nClass);
        Tcl_SetHashValue(ph, pc);
    } else
        pc = (char *)Tcl_GetHashValue(ph);

    Tcl_SetHashValue(Tcl_CreateHashEntry(&hIdentifiers, (ClientData)i, &iCreated), pc);                /* now register identifier */
    return iCreated;
}

static int iCopyObject(phInterpreter, iObject, nObject)
    Tcl_Interp *phInterpreter;
    int iObject;                                                                                             /* object identifier */
    char *nObject;                                                                                          /* ditto in text form */
{
    struct Tcl_HashEntry *ph;
    struct Tcl_DString h;
    char *nClass;
    int iIdentifier;                                  /* use local variable for identifier because new can be invoked recursively */

    ph = Tcl_FindHashEntry(&hIdentifiers, (ClientData)iObject);                                              /* find object class */
    if(ph == 0){
        sprintf(phInterpreter->result, "invalid object identifier \"%s\"", nObject);
        return TCL_ERROR;
    }
    nClass = (char *)Tcl_GetHashValue(ph);
    iIdentifier = ++iNewId;                                                                                 /* set new identifier */
    Tcl_DStringInit(&h);
    Tcl_DStringAppend(&h, nClass, -1);                                             /* build class copy constructor procedure call */
    Tcl_DStringAppend(&h, "::_copy ", 8);
    sprintf(phInterpreter->result, "%d", iIdentifier);                                                     /* with new identifier */
    Tcl_DStringAppend(&h, phInterpreter->result, -1);
    Tcl_DStringAppend(&h, " ", 1);
    Tcl_DStringAppend(&h, nObject, -1);                                                                 /* and sibling identifier */
    if(Tcl_Eval(phInterpreter, Tcl_DStringValue(&h)) != TCL_OK)
        return TCL_ERROR;
    Tcl_DStringFree(&h);
    if(!uAddIdentifier(nClass, iIdentifier)){
        sprintf(phInterpreter->result, "fatal error: could not register identifier for new object of class %s", nClass);
        return TCL_ERROR;
    }
    Tcl_ResetResult(phInterpreter);                            /* reset to safe static result buffer for eval probably changed it */
    sprintf(phInterpreter->result, "%d", iIdentifier);                                                   /* return new identifier */
    return TCL_OK;
}

int iNew(zClientData, phInterpreter, iNumberOfArguments, anArguments)
    ClientData zClientData;
    Tcl_Interp *phInterpreter;
    int iNumberOfArguments;
    char *anArguments[];
{
    struct Tcl_DString h;
    int iIdentifier;                                  /* use local variable for identifier because new can be invoked recursively */
    unsigned u;
    char *nClass;

    if(iNumberOfArguments < 2){
        sprintf(phInterpreter->result, "wrong number of arguments, should be: \"%s class ?arg arg ...?\"", anArguments[0]);
        return TCL_ERROR;
    }

    if((iIdentifier = atoi(anArguments[1])) != 0)
        return iCopyObject(phInterpreter, iIdentifier, anArguments[1]);    /* first argument is an object identifier: copy object */

    nClass = anArguments[1];
    iIdentifier = ++iNewId;                                                   /* set new identifier (arrange for 0 to be invalid) */
    Tcl_DStringInit(&h);
    Tcl_DStringAppend(&h, nClass, -1);                                                                  /* call class constructor */
    Tcl_DStringAppend(&h, "::", 2);
    Tcl_DStringAppend(&h, nClass, -1);
    Tcl_DStringAppend(&h, " ", 1);
    sprintf(phInterpreter->result, "%d", iIdentifier);                                                     /* with new identifier */
    Tcl_DStringAppend(&h, phInterpreter->result, -1);
    for(u = 2; u < iNumberOfArguments; u++)                                                          /* and constructor arguments */
        Tcl_DStringAppendElement(&h, anArguments[u]);
    if(Tcl_Eval(phInterpreter, Tcl_DStringValue(&h)) != TCL_OK)
        return TCL_ERROR;
    Tcl_DStringFree(&h);
    if(!uAddIdentifier(nClass, iIdentifier)){
        sprintf(phInterpreter->result, "fatal error: could not register identifier for new object of class %s", nClass);
        return TCL_ERROR;
    }
    Tcl_ResetResult(phInterpreter);                            /* reset to safe static result buffer for eval probably changed it */
    sprintf(phInterpreter->result, "%d", iIdentifier);                                                   /* return new identifier */
    return TCL_OK;
}

int iClassOf(zClientData, phInterpreter, iNumberOfArguments, anArguments)
    ClientData zClientData;
    Tcl_Interp *phInterpreter;
    int iNumberOfArguments;
    char *anArguments[];
{
    struct Tcl_HashEntry *ph;

    if(iNumberOfArguments != 2){
        sprintf(phInterpreter->result, "wrong number of arguments, should be: \"%s identifier\"", anArguments[0]);
        return TCL_ERROR;
    }
    ph = Tcl_FindHashEntry(&hIdentifiers, (ClientData)atoi(anArguments[1]));
                          /* works because atoi returns 0 if string is not a valid integer and 0 is not a valid object identifier */
    if(ph == 0){
        sprintf(phInterpreter->result, "invalid object identifier \"%s\"", anArguments[1]);
        return TCL_ERROR;
    }
    phInterpreter->result = (char *)Tcl_GetHashValue(ph);
    return TCL_OK;
}

int iDeleteObject(zClientData, phInterpreter, iNumberOfArguments, anArguments)
    ClientData zClientData;
    Tcl_Interp *phInterpreter;
    int iNumberOfArguments;
    char *anArguments[];
{                                                                        /* warning: for internal use only, no arguments checking */
    struct Tcl_DString h;
    char *nClass;
    char *nIdentifier;
    int i;
    Tcl_HashEntry *ph;
    Tcl_HashSearch hSearch;
    Var *phArray;
    char *n;

    nClass = anArguments[1];
    nIdentifier = anArguments[2];

    Tcl_DStringInit(&h);
    Tcl_DStringAppend(&h, nClass, -1);                                                                   /* call class destructor */
    Tcl_DStringAppend(&h, "::~", 3);
    Tcl_DStringAppend(&h, nClass, -1);
    Tcl_DStringAppend(&h, " ", 1);
    Tcl_DStringAppend(&h, nIdentifier, -1);                                                                    /* with identifier */
    i = Tcl_Eval(phInterpreter, Tcl_DStringValue(&h));
    Tcl_DStringFree(&h);
    if(i != TCL_OK)
        return TCL_ERROR;

    ph = Tcl_FindHashEntry(&((Interp *)phInterpreter)->globalTable, nClass);                /* class array is at the global level */
    if(ph != 0){                                                                                               /* if array exists */
        Tcl_DStringAppend(&h, nIdentifier, -1);                                       /* build array index prefix with identifier */
        Tcl_DStringAppend(&h, ",", 1);                                                                               /* and comma */
        i = strlen(Tcl_DStringValue(&h));
        phArray = (Var *)Tcl_GetHashValue(ph);
        for(ph = Tcl_FirstHashEntry(phArray->value.tablePtr, &hSearch); ph != 0; ph = Tcl_NextHashEntry(&hSearch)){
            n = Tcl_GetHashKey(phArray->value.tablePtr, ph);
            if(strncmp(n, Tcl_DStringValue(&h), i) == 0)                                         /* unset all object data members */
                Tcl_UnsetVar2(phInterpreter, nClass, n, TCL_GLOBAL_ONLY);
        }
        Tcl_DStringFree(&h);
    }

    Tcl_ResetResult(phInterpreter);                                                                             /* return nothing */
    return TCL_OK;
}

int iDelete(zClientData, phInterpreter, iNumberOfArguments, anArguments)
    ClientData zClientData;
    Tcl_Interp *phInterpreter;
    int iNumberOfArguments;
    char *anArguments[];
{
    unsigned u;
    struct Tcl_HashEntry *ph;
    char *an[3];

    an[0] = "_delete";                                                                  /* invoke the internal delete Tcl command */
    for(u = 1; u < iNumberOfArguments; u++){
        ph = Tcl_FindHashEntry(&hIdentifiers, (ClientData)atoi(anArguments[u]));
        if(ph == 0){
            sprintf(phInterpreter->result, "invalid object identifier \"%s\"", anArguments[u]);
            return TCL_ERROR;
        }
        an[1] = (char *)Tcl_GetHashValue(ph);
        an[2] = anArguments[u];
        if(iDeleteObject(zClientData, phInterpreter, 3, an) != TCL_OK)
            return TCL_ERROR;
        Tcl_DeleteHashEntry(ph);                                                /* remove identifier from class associative array */
    }
    return TCL_OK;
}

static int iCopyArrayMembers(phInterpreter, nClass, nFrom, nTo)
    Tcl_Interp *phInterpreter;
    char *nClass;
    char *nFrom;
    char *nTo;
{
    struct Tcl_DString hFrom;
    struct Tcl_DString hTo;
    Tcl_HashEntry *ph;
    Tcl_HashSearch hSearch;
    Var *phArray;
    char *nArray;
    char *n;
    int iFrom;

    Tcl_DStringInit(&hFrom);
    Tcl_DStringInit(&hTo);

    Tcl_DStringAppend(&hFrom, nClass, -1);                          /* build global array name prefix by concatenating class name */
    Tcl_DStringAppend(&hFrom, nFrom, -1);                                                                /* and object identifier */
    iFrom = strlen(Tcl_DStringValue(&hFrom));

    for(ph = Tcl_FirstHashEntry(&((Interp *)phInterpreter)->globalTable, &hSearch); ph != 0; ph = Tcl_NextHashEntry(&hSearch)){
        nArray = Tcl_GetHashKey(&((Interp *)phInterpreter)->globalTable, ph);
        if(strncmp(nArray, Tcl_DStringValue(&hFrom), iFrom) != 0)
            continue;                                           /* filter out variables that are not array members for this class */
        Tcl_DStringAppend(&hTo, nClass, -1);                                      /* build destination array name with class name */
        Tcl_DStringAppend(&hTo, nTo, -1);                                                        /* destination object identifier */
        Tcl_DStringAppend(&hTo, nArray + iFrom, -1);                                                     /* and member array name */
        phArray = (Var *)Tcl_GetHashValue(ph);
        for(ph = Tcl_FirstHashEntry(phArray->value.tablePtr, &hSearch); ph != 0; ph = Tcl_NextHashEntry(&hSearch)){
            n = Tcl_GetHashKey(phArray->value.tablePtr, ph);
            Tcl_SetVar2(                                                                        /* arrays are at the global level */
                phInterpreter, Tcl_DStringValue(&hTo), n, Tcl_GetVar2(phInterpreter, nArray, n, TCL_GLOBAL_ONLY), TCL_GLOBAL_ONLY
            );
        }
        Tcl_DStringFree(&hTo);
    }

    Tcl_DStringFree(&hFrom);

    return TCL_OK;
}

int iCopy(zClientData, phInterpreter, iNumberOfArguments, anArguments)
    ClientData zClientData;
    Tcl_Interp *phInterpreter;
    int iNumberOfArguments;
    char *anArguments[];
{                                                                        /* warning: for internal use only, no arguments checking */
    struct Tcl_DString h;
    struct Tcl_DString hFrom;
    struct Tcl_DString hTo;
    char *nClass;
    char *nFrom;
    char *nTo;
    int iFrom;
    int iTo;
    Tcl_HashEntry *ph;
    Tcl_HashSearch hSearch;
    Var *phArray;
    char *n;

    nClass = anArguments[1];
    nFrom = anArguments[2];
    nTo = anArguments[3];

    Tcl_DStringInit(&h);
    Tcl_DStringInit(&hFrom);
    Tcl_DStringInit(&hTo);

    ph = Tcl_FindHashEntry(&((Interp *)phInterpreter)->globalTable, nClass);                /* class array is at the global level */
    if(ph != 0){                                                                                               /* if array exists */
        Tcl_DStringAppend(&hFrom, nFrom, -1);                                  /* build array index prefix with source identifier */
        Tcl_DStringAppend(&hFrom, ",", 1);                                                                           /* and comma */
        iFrom = strlen(Tcl_DStringValue(&hFrom));
        Tcl_DStringAppend(&hTo, nTo, -1);                                                       /* build target identifier prefix */
        Tcl_DStringAppend(&hTo, ",", 1);
        iTo = strlen(Tcl_DStringValue(&hTo));
        phArray = (Var *)Tcl_GetHashValue(ph);
        for(ph = Tcl_FirstHashEntry(phArray->value.tablePtr, &hSearch); ph != 0; ph = Tcl_NextHashEntry(&hSearch)){
            n = Tcl_GetHashKey(phArray->value.tablePtr, ph);
            if(strncmp(n, Tcl_DStringValue(&hFrom), iFrom) != 0)
                continue;
            Tcl_DStringAppend(&h, Tcl_DStringValue(&hTo), iTo);                            /* copy all normal object data members */
            Tcl_DStringAppend(&h, n + iFrom, -1);                                                           /* append member name */
            Tcl_SetVar2(
                phInterpreter, nClass, Tcl_DStringValue(&h), Tcl_GetVar2(phInterpreter, nClass, n, TCL_GLOBAL_ONLY), TCL_GLOBAL_ONLY
            );
            Tcl_DStringFree(&h);
        }
        Tcl_DStringFree(&hFrom);
        Tcl_DStringFree(&hTo);
    }

    if(iCopyArrayMembers(phInterpreter, nClass, nFrom, nTo) != TCL_OK)
        return TCL_ERROR;

    Tcl_ResetResult(phInterpreter);                                                                             /* return nothing */
    return TCL_OK;
}

int Stooop_Init(phInterpreter)
    Tcl_Interp *phInterpreter;
{
    Tcl_InitHashTable(&hIdentifiers, TCL_ONE_WORD_KEYS);
    Tcl_InitHashTable(&hClasses, TCL_STRING_KEYS);
    iNewId = 0;

    Tcl_CreateCommand(phInterpreter, "new", iNew, 0, 0);
    Tcl_CreateCommand(phInterpreter, "classof", iClassOf, 0, 0);
    Tcl_CreateCommand(phInterpreter, "delete", iDelete, 0, 0);
    Tcl_CreateCommand(phInterpreter, "_delete", iDeleteObject, 0, 0);
    Tcl_CreateCommand(phInterpreter, "_copy", iCopy, 0, 0);

    return TCL_OK;
}

int Stooop_SafeInit(phInterpreter)
    Tcl_Interp *phInterpreter;
{
    return Stooop_Init(phInterpreter);                                                                  /* is stooop really safe? */
}
