/* -*- C++ -*-
 * 
 *      Developed by The Johns Hopkins University Applied Physics Laboratory
 *
 *   Copyright 1994-96, The Johns Hopkins University Applied Physics Laboratory
 *
 * Project:		Distributed Object Messaging Environment (DOME)
 *
 * Module Name:         tclIR.cc
 *	
 * Description:         TCL extension to send CORBA requests
 *
 * See Also:            tclIR.h, tclDome.[cc|h]
 *
 * Programmers:		George M. Scott
 *
 * Organization:	The Johns Hopkins University Applied Physics Lab
 *	
 * Language:		C++
 *
 */

#include <iostream.h>
#include <strstream.h>
#include <tcl.h>
#include <IR.h>
#include "tclIR.h"

static void *getValue(Tcl_Interp *interp, char *str, const CORBA::typeCode &tc,
                      int &length);
static char *getString(Tcl_Interp *interp, const CORBA::typeCode &tc,
                       void *value);

// NOTE:  This is Orbix specific and will most likely not work with other
// ORBS.
struct _IDL_SEQUENCE_buffer {
    unsigned int _max;
    unsigned int _len;
    void        *_buf;
    _IDL_SEQUENCE_buffer(unsigned int m, unsigned int l, void *b)
	: _max(m), _len(l), _buf(b) {}
};

//
// NOTE: These procedures convert a string to a binary value for a particular
// corba type code.  They can potentially be recursively called to handle
// arbitrarily complex data types.
//
static void *
getSequenceValue(Tcl_Interp *interp, char *str, const CORBA::typeCode &tc,
		 int &length)
{
    ostrstream buffer;
    int argc;
    char **argv;
    void *value;
    int len;

    CORBA::typeCode elementType =
	CORBA::typeCode((char *)tc.parameter(0)._value);

    if (Tcl_SplitList(interp, str, &argc, &argv) != TCL_OK) {
	Tcl_SetResult(interp, "Expected a sequence", TCL_STATIC);
	return 0;
    }
    
    for (int i = 0; i < argc; i++) {
	value = getValue(interp, argv[i], elementType, len);
	if (value == 0)
	    return 0;
	buffer.write((char *)value, len);
    }

    // return a new sequence
    length = sizeof(_IDL_SEQUENCE_buffer);
    return new _IDL_SEQUENCE_buffer(argc, argc, buffer.str());

}

static void *
getStructValue(Tcl_Interp *interp, char *str, const CORBA::typeCode &tc,
	       int &length)
{
    int argc;
    char **argv;
    void *value;
    int len, i;
    // dynamic buffer
    ostrstream buffer;

    int align;
    int structAlign = 1;
    int offset = 0;

    if (Tcl_SplitList(interp, str, &argc, &argv) != TCL_OK) {
	Tcl_SetResult(interp, "Expected a structure", TCL_STATIC);
	return 0;
    }

    // do we have the correct number of parameters
    if (argc != (tc.param_count() - 1)) {
	Tcl_SetResult(interp, "Wrong number of members in structure",
		      TCL_STATIC);
	return 0;
    }
    
    for (i = 1; i < tc.param_count(); i++) {
	// first make sure the stuctures members are the same.
	if (strcmp(*(char **)tc.parameter(i)._value, argv[i - 1])) {
	    Tcl_AppendResult(interp, "Expected structure member",
			     *(char **)tc.parameter(i)._value,
			     " found ", argv[i - 1], 0);
	    return 0;
	}

	// go to the parameter.
	i++;
	
	CORBA::typeCode memberTC((char *)tc.parameter(i)._value);

	value = getValue(interp, argv[i - 1], memberTC, len);

	if (value == 0)
	    return 0;

	switch (memberTC.kind()) {
	  case CORBA::tk_short:
          case CORBA::tk_ushort:
	      align = sizeof(short);
              break;
          case CORBA::tk_long:
          case CORBA::tk_enum:
          case CORBA::tk_ulong:
	      align = sizeof(int);
              break;
          case CORBA::tk_float:
	      align = sizeof(float);
              break;
          case CORBA::tk_double:
	      align = sizeof(double);
              break;
          case CORBA::tk_char:
	  case CORBA::tk_boolean:
	      align = sizeof(char);
              break;
          case CORBA::tk_string :
	      align = sizeof(void *);
              break;
	  case CORBA::tk_struct:
	      // we need to align the structure according to its size
	      align = len % 8;
          default:
	      Tcl_SetResult(interp, "Structure contains unkown CORBA typecode",
			    TCL_STATIC);
	      return 0;
	}

        // overall structure alignment is determined by the size of
        // the larget member.
        if (align > structAlign)
            structAlign = align;


	// we are not properly aligned
	if (offset % align != 0) {
	    int pad = align - (offset % align);
	    // this should be one call. consult ostrstream manual
	    for (i = 0; i < pad; i++) {
		buffer.put((char)0);
	    }
	    offset += pad;
	}

	// copy the data into the stream
	buffer.write((char *)value, len);
	// free the value
	delete value;

	offset += align;
    }

    length = offset;
    if (length % structAlign != 0) {
	int pad = structAlign - (length % structAlign);
	// this should be one call. consult ostrstream manual
	for (i = 0; i < pad; i++) {
	    buffer.put((char)0);
	}
	length += pad;
    }

    // freeze the buffer and return the pointer to the data.
    return buffer.str();
}

static void *
getValue(Tcl_Interp *interp, char *str, const CORBA::typeCode &tc, int &length)
{
    int intVal, i;
    double doubleVal;
    char **cp;
    CORBA::ObjectRef *refPtr;

    // everything is processed differently based on the type information.
    switch (tc.kind()) {
      case CORBA::tk_short:
	  if (Tcl_GetInt(interp, str, &intVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected short, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(short);
	  return new short(intVal);
      case CORBA::tk_long:
	  // treat longs like ints, not longs because of problems on
	  // dec alpha.
	  if (Tcl_GetInt(interp, str, &intVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected long, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(int);
	  return new int(intVal);
      case CORBA::tk_ushort:
	  if (Tcl_GetInt(interp, str, &intVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected unsigned short, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(unsigned short);
	  return new unsigned short(intVal);
      case CORBA::tk_ulong:
	  if (Tcl_GetInt(interp, str, &intVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected unsigned long, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(unsigned int);
	  return new unsigned int(intVal);
      case CORBA::tk_float:
	  if (Tcl_GetDouble(interp, str, &doubleVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected float, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(float);
	  return new float(doubleVal);
      case CORBA::tk_double:
	  if (Tcl_GetDouble(interp, str, &doubleVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected double, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(double);
	  return new double(doubleVal);
      case CORBA::tk_boolean:
	  if (Tcl_GetBoolean(interp, str, &intVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected boolean, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(unsigned char);
	  return new unsigned char(intVal);
      case CORBA::tk_char:
	  if (Tcl_GetInt(interp, str, &intVal) != TCL_OK) {
	      Tcl_AppendResult(interp, "expected char, found: ",
				   str, 0);
	      return 0;
	  }
	  length = sizeof(char);
	  return new char(intVal);
      case CORBA::tk_enum:
	  length = sizeof(int);
	  for (i = 1; i < tc.param_count(); i++) {
	      if (!strcmp(str, *(char **)tc.parameter(i)._value)) {
		  return new int(i-1);
	      }
	  }
	  Tcl_AppendResult(interp, "Invalid enumeration ", str, " for type ",
			   *(char **)tc.parameter(0)._value, 0);
	  return 0;
      case CORBA::tk_string:
	  length = sizeof(char *);
	  cp = new char *;
	  *cp = strdup(str);
	  return cp;
      case CORBA::tk_struct:
	  return getStructValue(interp, str, tc, length);
      case CORBA::tk_sequence:
	  return getSequenceValue(interp, str, tc, length);
      case CORBA::tk_objref:
	  refPtr = new CORBA::ObjectRef;
          TRY {
              *refPtr = CORBA::Orbix.string_to_object(str, IT_X);
          } CATCHANY {
              Tcl_AppendResult(interp, "Invalid object reference", 0);
          } ENDTRY;
	  length = sizeof(CORBA::ObjectRef *);
	  return refPtr;
      default:
	  Tcl_AppendResult(interp, "Unknown CORBA type code", 0);
	  return 0;
    }

}


// 
// These procedures perform the reverse of above procedures.  They convert
// a binary image into a string representation for a particular data type.
// They can also potentially be recursively called to handle complex data
// types.
//

static char *
getStructString(Tcl_Interp *interp, const CORBA::typeCode &/*tc*/,
                void */*value*/)
{
    Tcl_AppendResult(interp,
                     "returning structure types currently unsupported", 0);

    return 0;
}

static char *
getSequenceString(Tcl_Interp *interp, const CORBA::typeCode &/*tc*/,
                  void */*value*/)
{
    Tcl_AppendResult(interp,
                     "returning sequence types currently unsupported", 0);

    return 0;
}

static char *
getString(Tcl_Interp *interp, const CORBA::typeCode &tc, void *value)
{
    CORBA::ObjectRef obj;
    ostrstream buf;

    switch (tc.kind()) {
        case CORBA::tk_short:
            buf << (short)*(short *)value;
            break;
        case CORBA::tk_long:
            buf << (int)*(int *)value;
            break;
        case CORBA::tk_ushort:
            buf << (unsigned short)*(unsigned short *)value;
            break;
        case CORBA::tk_ulong:
            buf << (unsigned int)*(unsigned int*)value;
            break;
        case CORBA::tk_float:
            buf << (float)*(float *)value;
            break;
        case CORBA::tk_double:
            buf << (double)*(double *)value;
            break;
        case CORBA::tk_boolean:
            if ((*(unsigned char *)value) == 0) {
                buf << "false";
            } else {
                buf << "true";
            }
            break;
        case CORBA::tk_char:
        case CORBA::tk_enum:
        case CORBA::tk_string:
            buf << (char *)*(char **)value;
            break;
        case CORBA::tk_struct:
            return getStructString(interp, tc, value);
        case CORBA::tk_sequence:
            return getSequenceString(interp, tc, value);
        case CORBA::tk_objref:
            obj = *(CORBA::ObjectRef *)value;
            return strdup(obj->_object_to_string());
        case CORBA::tk_void:
        case CORBA::tk_null:
                // we need to dynamically allocate this to be consistent
                // with other return values which are dynamically allocated.
            return strdup("");
        default:
            Tcl_AppendResult(interp, "Unknown CORBA type code: ", 0);
            return 0;
    }
    
        // null terminate the string
    buf.put((char)0);
    return buf.str();
}

//
// This is the function the TCL interpeter will call to issues requests
// to the ORB.
//
// This function works as follows:
// 
// if client data is non-null:
//     client data assumed to contain a pointer to a stringified object ref.
//     argv[1] is the method or attribute name
//     all other args, if any, are interperted as defined below
//
// client data is null:
//     argv[1] contains a stringified object reference
//     argv[2] is the method or attribute name
//     all other args, if any, are interpeted as defined below
//
// Arguments
// ---------
//
// If this is an operation on an attribute:
//
// If there are no additional arguments then a get is performed on the 
// attribute and the value is return in the TCL result.
//
// If there is one argument then this is taken to be the value of the 
// attribute and it is converted to the appropriate format and sent
//
//
// If the operation is on a method (operation in CORBA-speak):
//
// Then the additional arguments specify the arguments to the method
// where each parameter name must be given followed by its value.
// All parameters must be given in order.

int
TclIRorbRequestCmd(ClientData data, Tcl_Interp *interp,
		     int argc, char *argv[])
{
    CORBA::Context c;
    CORBA::Request *request;
    CORBA::NamedValue returnValue;
    CORBA::Object *obj = 0;
    char *methodName = 0;
    int argp = 0;
    int extraArg;


    if (data == 0) {
	argp++;
    }

    extraArg = argc - argp;

    // this should catch most of the common arg errors
    if (extraArg < 2) {
	if (data == 0) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " InstanceName MethodName ?Name Value ...?\"", 0);
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " MethodName ?Name Value ...?\"", 0);
	}
	return TCL_ERROR;
    }

    argp++;
    if (data == 0) {
	obj = CORBA::Orbix.string_to_object(argv[1]);
    } else {
	obj = CORBA::Orbix.string_to_object((char *)data);
    }

    methodName = argv[argp++];

    InterfaceDef *interface;
    TRY {
	interface = obj->_get_interface(IT_X);
    } CATCHANY {
	Tcl_AppendResult(interp, "Could not find interface for object", 0);
	return TCL_ERROR;
    } ENDTRY;

    _IDL_SEQUENCE_Contained opAttrList;
    OperationDef *opDef = 0;
    AttributeDef *attrDef = 0;

    TRY {
	opAttrList = interface->lookup_name(methodName, 1,
					    "OperationDef", IT_X);
	if (IT_X || opAttrList._length != 1) {
	    opAttrList = interface->lookup_name(methodName, 1,
						"AttributeDef", IT_X);
	    
	    if (IT_X || opAttrList._length != 1) {
                Tcl_AppendResult(interp, "Invalid attribute or method: ",
                                 methodName, 0);
		return TCL_ERROR;
	    } else {
		attrDef = AttributeDef::_narrow(opAttrList[0], IT_X);
	    }
	} else {
	    opDef = OperationDef::_narrow(opAttrList[0], IT_X);
	}
    } CATCHANY {
        Tcl_SetResult(interp, "Could not get interface information",
                      TCL_STATIC);
	return TCL_ERROR;
    } ENDTRY;

    if (opDef != 0) {
	CORBA::NVList *nvList;

	TRY {
	    CORBA::Orbix.create_operation_list(opDef, nvList, IT_X);
	} CATCHANY {
            Tcl_SetResult(interp, "Could not create nvlist!", TCL_STATIC);
	} ENDTRY;

	// check the arg count.
	if ((argc - argp)/2 != nvList->m_count) {
	    char buf[32];
	    sprintf(buf, "%d", nvList->m_count);
	    Tcl_AppendResult(interp, "Wrong argument count: method ",
			     methodName, " has ", buf, " argument(s).", 0);
	    return TCL_ERROR;
	}

	// pack all of the arguments...
	for (int i = 0; i < nvList->m_count; i++, argp++) {
	    void *value;
	    int length;
	    // args must be in order in arg list.
	    if (strcmp(nvList->m_list[i].name, argv[argp])) {
		char buf[32];
		sprintf(buf, "%d", i + 1);
		Tcl_AppendResult(interp, "Found ", argv[argp],
				 " for parameter ", buf, " expected ",
				 nvList->m_list[i].name, 0);
		return TCL_ERROR;
	    }

	    // we verified the name, let's check the value.
	    argp++;

	    value = getValue(interp, argv[argp],
			     *(nvList->m_list[i].argument._type), length);

	    if (value == 0) {
		return TCL_ERROR;
	    }

	    nvList->m_list[i].argument._value = value;
	}

	TRY {
	    obj->_create_request(c, methodName, nvList,
				 returnValue, request, 0, IT_X);
	} CATCHANY {
            Tcl_SetResult(interp, "Could not create Request!", TCL_STATIC);
            return TCL_ERROR;
	} ENDTRY;
    } else {
	char buf[512];
	int performSet = 0;
	// must be an attribute

	// set opname.  Should look for overflow....
	if (argp == argc) {
	    sprintf(buf, "_get_%s", methodName);
	} else if ((argp + 1) == argc) {
	    performSet = 1;
	    sprintf(buf, "_set_%s", methodName);
	} else {
	    Tcl_SetResult(interp, "Invalid number of arguments for attribute",
			  TCL_STATIC);
	    return TCL_ERROR;
	}

	TRY {
	    obj->_create_request(c, buf, 0, returnValue, request, 0, IT_X);
	} CATCHANY {
            Tcl_SetResult(interp, "Could not create request!", TCL_STATIC);
            return TCL_ERROR;
	} ENDTRY;

	if (performSet) {
	    void *value;
	    int length;

	    value = getValue(interp, argv[argp], attrDef->type(), length);

	    if (value == 0) {
		return TCL_ERROR;
	    }

	    request->add_arg(0, attrDef->type(), value, 0, CORBA::ARG_IN);

	}
    }

    // invoke it...
    TRY {
	request->invoke(IT_X);
    } CATCHANY {
        ostrstream buf;
        buf << IT_X;
            // null terminate the string
        buf.put(char(0));
        Tcl_SetResult(interp, buf.str(), TCL_DYNAMIC);
	return TCL_ERROR;
    } ENDTRY;

    char *result = getString(interp,  *(returnValue.argument._type),
                             returnValue.argument._value);

    if (result == 0) {
        Tcl_AppendResult(interp, "failed to get result", 0);
        return TCL_ERROR;
    }

    Tcl_SetResult(interp, result, TCL_DYNAMIC);
    return TCL_OK;
}

extern "C" int
TclIR_Init(Tcl_Interp *interp)
{
    int rc;

#ifdef USE_ITCL_NAMESPACES
    Itcl_Namespace tclIRns, ns;
#endif

#ifdef USE_ITCL_NAMESPACES
    rc = Itcl_CreateNamesp(interp, "tclIR", 0, 0, &tclIRns);
    if (rc != TCL_OK) {
	return rc;
    }

    if ((ns = Itcl_ActivateNamesp(interp, tclIRns)) == 0) {
	return TCL_ERROR;
    }
#endif
        // These commands are all really the same they just make
        // the Tcl scripts more readable.
    Tcl_CreateCommand(interp, "orbRequest", TclIRorbRequestCmd,
		      (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
    Tcl_CreateCommand(interp, "invokeMethod", TclIRorbRequestCmd,
		      (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
    Tcl_CreateCommand(interp, "setAttribute", TclIRorbRequestCmd,
		      (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
    Tcl_CreateCommand(interp, "getAttribute", TclIRorbRequestCmd,
		      (ClientData) 0, (Tcl_CmdDeleteProc *) 0);

#ifdef USE_ITCL_NAMESPACES
    Itcl_DeactivateNamesp(interp, ns);
#endif

    return TCL_OK;
}

