#
# Name
#
#   tkGrep
#
# Version
#
#   1.0b1
#
# Description
#
#   A Graphical interface to the grep command with optional column
#      formatting.
#
# Notes
#
#   1. Add pattern highlighting to the LOG window.
#   2. Fix xscrollbar in standard library.
#   3. Create online documentation.
#   4. Write an installation script in Tcl.
#
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#
# Modification History
#
# Name              Date         Comment
# ~~~~~~~~~~~~~~~   ~~~~~~~~~~   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Wes Bailey        1996.05.01   Created.
# Wes Bailey        1996.05.02   Finished the command settings and LOG
#                                window. 
# Wes Bailey        1996.05.04   Added the middle panel with clear and 
#                                save options.
# Wes Bailey        1996.05.05   Redid the middle panel with a toolbar
#                                with GIF file images.
# Wes Bailey        1996.05.06   Created the tkGrep_global.tcl file to
#                                store global definitions of the 
#                                image, error, and help definitions.
# Wes Bailey        1996.05.06   Added the directory entry widget to
#                                the command settings so that the 
#                                filename is seperate from the working
#                                directory.
# Wes Bailey        1996.05.06   Added the line number feature.
# Wes Bailey        1996.05.07   Added the Xresources file and
#                                completed the AWK like column
#                                formatting.
# Wes Bailey        1996.05.05   Added error checks to the seperator
#                                and column entry widgets.  Also
#                                added query to the stdlib to when
#                                user wants to overwrite a file.
#

source "standard_library.tcl"
source "tkGrep_xresources.tcl"
source "tkGrep_global.tcl"

# The parent frame for the command settings.
frame .top -width 80 -borderwidth 2 -relief ridge

# Display the larger tkGrep image in a label.
frame .top.left 
label .top.left.img -image $tkGrep(IMAGE_TKGREP) -relief raised
pack .top.left.img -side left

# In the middle there are 3 entry widgets.
frame .top.middle
LabelEntry .top.middle.directory "Directory" 9 "" -width 15 \
	-textvariable tkGrep(WORKDIR)
LabelEntry .top.middle.filename "Filename" 9 "" -width 15 \
	-textvariable tkGrep(FILENAME)
LabelEntry .top.middle.pattern "Pattern" 9 "" -width 15 \
	-textvariable tkGrep(PATTERN)
LabelEntry .top.middle.option "Options" 9 "" -width 15 \
	-textvariable tkGrep(OPTION)
pack .top.middle.directory .top.middle.filename .top.middle.pattern \
	.top.middle.option -side top 

# Create the right side which formats the output of the command.
set tkGrep(FORMAT) "0"
set tkGrep(COUNTER) "0"
frame .top.right 
frame .top.right.1
frame .top.right.2
frame .top.right.3 
frame .top.right.4
frame .top.right.padding -width 10
label .top.right.1.formatlabel -text "Formatted Output" \
	-width 15 -anchor w
radiobutton .top.right.1.formaton -text "On" -value 1 \
	-variable tkGrep(FORMAT)
radiobutton .top.right.1.formatoff -text "Off" -value 0 \
	-variable tkGrep(FORMAT)
pack .top.right.1.formatlabel .top.right.1.formaton \
	.top.right.1.formatoff -side left
label .top.right.2.counterlabel -text "Line Counter" \
	-width 15 -anchor w
radiobutton .top.right.2.counteron -text "On" -value 1 \
	-variable tkGrep(COUNTER)
radiobutton .top.right.2.counteroff -text "Off" -value 0 \
	-variable tkGrep(COUNTER)
pack .top.right.2.counterlabel .top.right.2.counteron \
	.top.right.2.counteroff -side left
LabelEntry .top.right.3.seperator "Seperator" 10 "" -width 10 \
	-textvariable tkGrep(SEPERATOR) 
pack .top.right.3.seperator .top.right.padding -side left 
LabelEntry .top.right.4.column "Columns" 10 "" -width 10 \
	-textvariable tkGrep(COLUMN)
pack .top.right.4.column .top.right.padding -side left
pack .top.right.1 .top.right.2 .top.right.3 .top.right.4 \
	-side top -fill x

pack .top.left .top.middle .top.right -side left \
	-padx 20 -pady 2 -fill x

# Create the toolbar in the middle above the text log.
frame .middle -borderwidth 2 -relief ridge
button .middle.run -image $tkGrep(IMAGE_RUN) \
	-command ExecGrepCommand
button .middle.reset -image $tkGrep(IMAGE_RESET) \
	-command ResetCommandSetting
button .middle.clear -image $tkGrep(IMAGE_CLEAR) \
	-command ClearLogWindow
button .middle.save -image $tkGrep(IMAGE_SAVE) \
	-command {SaveDialog $tkGrep(TEXT_WIDGET_LOG)}
button .middle.help -image $tkGrep(IMAGE_HELP) \
	-command ViewHelpDoc
button .middle.exit -image $tkGrep(IMAGE_EXIT) -command exit
label .middle.helplabel -width 25 -height 3 -text ""
pack .middle.run .middle.reset .middle.clear .middle.save \
	.middle.help .middle.exit .middle.helplabel  -side left 

# Put a scrolled text widget to log the output of the command into.
frame .bottom -borderwidth 2 -relief ridge
ScrolledWidget .bottom.output 2 sunken text .bottom.output.text \
	-font -adobe-helvetica-bold-r-*-*-12-*-*-*-*-*-*-* \
	-width 80 -height 20
pack .bottom.output -side bottom -fill x

# Pack up the parent frame.
pack .top .middle .bottom -side top -fill both -expand true 

# Store the class definition of the LOG window.
set tkGrep(TEXT_WIDGET_LOG) .bottom.output.text

# Set globals from command line options
set tkGrep(WORKDIR) [lindex $argv 0]
set tkGrep(FILENAME) [lindex $argv 1]
set tkGrep(PATTERN) [lindex $argv 2]
set tkGrep(OPTION) [lindex $argv 3]

# Set the starting focus depending on which globals are defined.
if ![string length $tkGrep(WORKDIR)] {
    focus .top.middle.directory.entry
} elseif ![string length $tkGrep(FILENAME)] {
    focus .top.middle.filename.entry
} elseif ![string length $tkGrep(PATTERN)] {
    focus .top.middle.pattern.entry
} elseif ![string length $tkGrep(OPTION)] {
    focus .top.middle.option.entry
}

# Store the class definitions of all entry widgets.
set tkGrep(ENTRY_WIDGET_WORKDIR) .top.middle.directory.entry 
set tkGrep(ENTRY_WIDGET_FILENAME) .top.middle.filename.entry
set tkGrep(ENTRY_WIDGET_PATTERN) .top.middle.pattern.entry
set tkGrep(ENTRY_WIDGET_OPTION)	.top.middle.option.entry
set tkGrep(ENTRY_WIDGET_SEPERATOR) .top.right.3.seperator.entry
set tkGrep(ENTRY_WIDGET_COLUMN) .top.right.4.column.entry

# Bind mouse entry events to appropriate help message in label.
bind .top <Enter> {.middle.helplabel configure -text $tkGrep(HELP%W)}
bind .bottom <Enter> {.middle.helplabel configure -text $tkGrep(HELP%W)}
bind .middle.run <Enter> {.middle.helplabel configure \
	-text $tkGrep(HELP%W)}
bind .middle.reset <Enter> {.middle.helplabel configure \
	-text $tkGrep(HELP%W)}
bind .middle.clear <Enter> {.middle.helplabel configure \
	-text $tkGrep(HELP%W)}
bind .middle.save <Enter> {.middle.helplabel configure \
	-text $tkGrep(HELP%W)}
bind .middle.help <Enter> {.middle.helplabel configure \
	-text $tkGrep(HELP%W)}
bind .middle.exit <Enter> {.middle.helplabel configure \
	-text $tkGrep(HELP%W)}

#<def> ---------------------------------------------------------------
#<def> ResetCommandSetting: Clear the command center at the top of the 
#<def>       application.
#<def> 
#<def> Parameters:
#<def>    None
#<def> 
#<def> Global Variables:
#<def>    tkGrep(WORKDIR)   - The working directory.
#<def>    tkGrep(FILENAME)  - Name of the file to grep.
#<def>    tkGrep(PATTERN)   - Pattern in grep command.
#<def>    tkGrep(OPTION)    - Options to the grep command
#<def>    tkGrep(SEPERATOR) - String to seperate formatted output.
#<def>    tkGrep(COLUMN)    - Columns to display in formatted output.
#<def> 
#<def> Notes:
#<def>    This procedure clears all the entry widgets in the command 
#<def>    center and resets the radio buttons.
#<def> 

proc ResetCommandSetting { } {
    global tkGrep

    # Null entry widget global strings.
    set tkGrep(WORKDIR) ""
    set tkGrep(FILENAME) ""
    set tkGrep(PATTERN) ""
    set tkGrep(OPTION) ""
    set tkGrep(SEPERATOR) ""
    set tkGrep(COLUMN) ""

    # Redefine the format and counter globals for the radio buttons.
    set tkGrep(FORMAT) 0
    set tkGrep(COUNTER) 0

    # Clear all of the top entry widgets.
    ClearEntryWidget $tkGrep(ENTRY_WIDGET_WORKDIR)
    ClearEntryWidget $tkGrep(ENTRY_WIDGET_FILENAME)
    ClearEntryWidget $tkGrep(ENTRY_WIDGET_PATTERN)
    ClearEntryWidget $tkGrep(ENTRY_WIDGET_OPTION)
    ClearEntryWidget $tkGrep(ENTRY_WIDGET_SEPERATOR)
    ClearEntryWidget $tkGrep(ENTRY_WIDGET_COLUMN)
    
    # Set the focus to the WORKDIR widget
    focus $tkGrep(ENTRY_WIDGET_WORKDIR)
}

#<def> ---------------------------------------------------------------
#<def> ClearLogWindow: Clear the LOG window below the ToolBar.
#<def> 
#<def> Parameters:
#<def>    None
#<def> 
#<def> Global Variables:
#<def>    tkGrep(TEXT_WIDGET_LOG) - The log window.
#<def> 
#<def> Notes:
#<def>    This is a very simple procedure.
#<def> 

proc ClearLogWindow { } {
    global tkGrep

    # Clear the log window.
    ClearTextWidget $tkGrep(TEXT_WIDGET_LOG)
}

#<def> ---------------------------------------------------------------
#<def> ExecGrepCommand: Construct, execute, and format the output 
#<def>       of the grep command.
#<def> 
#<def> Parameters:
#<def>    None
#<def> 
#<def> Global Variables:
#<def>    tkGrep(WORKDIR)   - The working directory.
#<def>    tkGrep(FILENAME)  - Name of the file to grep.
#<def>    tkGrep(PATTERN)   - Pattern in grep command.
#<def>    tkGrep(OPTION)    - Options to the grep command
#<def>    tkGrep(SEPERATOR) - String to seperate formatted output.
#<def>    tkGrep(COLUMN)    - Columns to display in formatted output.
#<def> 
#<def> Notes:
#<def>    This procedure relies heavily on EntryWidgetErrorCheck.
#<def> 

proc ExecGrepCommand { } {
    global tkGrep

    # Perform error checks on entry widget global variables.
    EntryWidgetErrorCheck

    # Create the filename to search and the grep command.
    set p_file "$tkGrep(WORKDIR)/$tkGrep(FILENAME)"

    # Create the grep command.
    set p_grep_cmd "|grep $tkGrep(OPTION) \"$tkGrep(PATTERN)\" \
	    $p_file"

    # Create the output buffer.
    if [catch {set p_fileid [open $p_grep_cmd r]} p_error] {
	ErrorDialog $p_error
    } else {
	set p_buffer [read $p_fileid]
	close $p_fileid
    }

    # Display the output buffer in the TEXT_WIDGET_LOG window.
    #   Format output to user specifications.
    set p_line_cntr 0
    foreach p_buffer_line [split $p_buffer \n] {
	if $tkGrep(FORMAT) {
	    if [string length $tkGrep(SEPERATOR)] {
	    set p_buffer_line [ColumnFormat $p_buffer_line \
		    $tkGrep(SEPERATOR) $tkGrep(COLUMN)]
	    }
	    if $tkGrep(COUNTER) {
		incr p_line_cntr
		set p_number [LeftPad $p_line_cntr 4]
		set p_buffer_line "$p_number: $p_buffer_line"
	    }
	}
	append p_buffer_line \n
	$tkGrep(TEXT_WIDGET_LOG) insert end $p_buffer_line
    }    
}
#<def> ---------------------------------------------------------------
#<def> EntryWidgetErrorCheck: Check for user errors on entries.
#<def> 
#<def> Parameters:
#<def>    None
#<def> 
#<def> Global Variables:
#<def>    tkGrep(WORKDIR)   - The working directory.
#<def>    tkGrep(FILENAME)  - Name of the file to grep.
#<def>    tkGrep(PATTERN)   - Pattern in grep command.
#<def>    tkGrep(OPTION)    - Options to the grep command
#<def>    tkGrep(SEPERATOR) - String to seperate formatted output.
#<def>    tkGrep(COLUMN)    - Columns to display in formatted output.
#<def> 
#<def> Notes:
#<def>    Beginning/Ending single and double quotes are stripped off
#<def>    each global variable.  The follow error checks are also 
#<def>    performed:
#<def> 
#<def>    1. If the FILENAME contains the directory, it is stripped
#<def>       from FILENAME and stored in WORKDIR.
#<def>    2. Tilde expansion on the WORKDIR is done.
#<def>    3. Last slash removed from WORKDIR.
#<def>    4. OPTION list is checked for missing delimiters.
#<def>    5. WORKDIR/FILENAME are check for:
#<def>       a) existence
#<def>       b) isdirectory/isfile
#<def>       c) read permission.
#<def> 

proc EntryWidgetErrorCheck { } {
    global tkGrep

    # Strip off quotes and remove directory name if given in
    #   the FILENAME.
    set tkGrep(FILENAME) [StripQuotes $tkGrep(FILENAME)]
    set p_slash [string last "/" $tkGrep(FILENAME)]
    if {$p_slash != -1} {
	set tkGrep(WORKDIR) [string range $tkGrep(FILENAME) \
		0 $p_slash]
	set p_last [expr [string length $tkGrep(FILENAME)-1]]
	set tkGrep(FILENAME) [string range $tkGrep(FILENAME) \
		[expr $p_slash+1] $p_last]
    }

    # Strip off quotes if any, do tilde expansion and delete last
    #   slash to the working directory.
    set tkGrep(WORKDIR) [StripQuotes $tkGrep(WORKDIR)]
    set p_last [expr [string length $tkGrep(WORKDIR)]-1]
    if {[string index $tkGrep(WORKDIR) 0] == "~"} {
	set p_tail [string range $tkGrep(WORKDIR) 1 $p_last]
	if {[string index $p_tail 0] == "/"} {
	    set tkGrep(WORKDIR) $tkGrep(HOME)$p_tail
	} else {
	    set tkGrep(WORKDIR) $tkGrep(HOME)/$p_tail
	}
    }
    set p_last [expr [string length $tkGrep(WORKDIR)]-1]
    if {[string index $tkGrep(WORKDIR) $p_last] == "/"} {
	set p_prev [expr $p_last-1]
	set tkGrep(WORKDIR) [string range $tkGrep(WORKDIR) 0 $p_prev]
    }

    # Check for invalid grep options and syntax errors.
    set tkGrep(OPTION) [StripQuotes $tkGrep(OPTION)]
    foreach p_option [split $tkGrep(OPTION)] {
	if ![regexp -- {^-[a-zA-Z0-9]*} $p_option] {
	    set p_lastidx [lsearch -exact $tkGrep(OPTION) $p_option]
	    set p_pridx [expr $p_lastidx-1]
	    set p_prflag [lindex $tkGrep(OPTION) $p_pridx]
	    if [regexp -- {^-[ef]} $p_prflag] {
		continue
	    } else {
		focus $tkGrep(ENTRY_WIDGET_OPTION)
		ErrorDialog "$tkGrep(ERROR,SE): missing delimiter"
	    }
	}
    }

    # Strip quotes off the matching pattern.
    set tkGrep(PATTERN) [StripQuotes $tkGrep(PATTERN)]

    # Clean up the remaining global strings for the formatted output.
    set tkGrep(SEPERATOR) [StripQuotes $tkGrep(SEPERATOR)]
    set tkGrep(COLUMN) [StripQuotes $tkGrep(COLUMN)]

    # Create the filename to search and the grep command.
    set p_file "$tkGrep(WORKDIR)/$tkGrep(FILENAME)"

    # Perform error checks on required global variables involving
    #   the process pipeline for grep.
    if {[string length $tkGrep(WORKDIR)] == 1} {
	focus $tkGrep(ENTRY_WIDGET_WORKDIR)
	ErrorDialog "$tkGrep(ERROR,IWD): Null"
    } elseif ![file isdirectory $tkGrep(WORKDIR)] {
	focus $tkGrep(ENTRY_WIDGET_WORKDIR)
	ErrorDialog $tkGrep(ERROR,IWD)
    } elseif ![file readable $tkGrep(WORKDIR)] {
	focus $tkGrep(ENTRY_WIDGET_WORKDIR)
	ErrorDialog $tkGrep(ERROR,RPD)
    } 
    if ![file exists $p_file] {
	focus $tkGrep(ENTRY_WIDGET_FILENAME)
	ErrorDialog $tkGrep(ERROR,FDNE)
    } elseif ![file isfile $p_file] {
	focus $tkGrep(ENTRY_WIDGET_FILENAME)
	ErrorDialog $tkGrep(ERROR,FNF)
    } elseif ![file readable $p_file] {
	focus $tkGrep(ENTRY_WIDGET_FILENAME)
	ErrorDialog $tkGrep(ERROR,RPD)
    }
    if ![string length $tkGrep(PATTERN)] {
	focus $tkGrep(ENTRY_WIDGET_PATTERN)
	ErrorDialog $tkGrep(ERROR,ISP)
    }
    if {[string length $tkGrep(SEPERATOR)] > 1} {
	focus $tkGrep(ENTRY_WIDGET_SEPERATOR)
	ErrorDialog $tkGrep(ERROR,SNSC)
    } elseif {[string length $tkGrep(SEPERATOR)] == 1} {
	if ![string length $tkGrep(COLUMN)] {
	    focus $tkGrep(ENTRY_WIDGET_COLUMN)
	    ErrorDialog $tkGrep(ERROR,CNS)
	}
    } 
    if [string length $tkGrep(COLUMN)] {
	if ![string length $tkGrep(SEPERATOR)] {
	    focus $tkGrep(ENTRY_WIDGET_SEPERATOR)
	    ErrorDialog $tkGrep(ERROR,SNS)
	}
    } 
    if [regexp {[a-zA_Z]} $tkGrep(COLUMN) p_error] {
	focus $tkGrep(ENTRY_WIDGET_COLUMN)
	ErrorDialog "$tkGrep(ERROR,ID): $p_error"
    }
}

#<def> ---------------------------------------------------------------
#<def> ColumnFormat: Print specified columns, delimited by a 
#<def>       seperator, from a string. 
#<def> 
#<def> Parameters:
#<def>    p_string  - String to format into columns
#<def>    p_sepchar - Seperate string into columns delimited by this 
#<def>                character.
#<def>    p_column  - Actual columns to display.
#<def> 
#<def> Global Variables:
#<def>    None
#<def> 
#<def> Notes:
#<def>    This procedure provides simple AWK like column formatting.
#<def> 
#<def> Example:
#<def>    % set p_str {> 1 > 2 >x > 3}
#<def>    % > 1 > 2 >x > 3
#<def>    % ColumnFormat $p_str > "1 2 3"
#<def>    % 1 2 3
#<def> 

proc ColumnFormat { p_string p_sep_char p_col_list } {
    # Split the string by the delimiter.
    set p_split_line [split $p_string $p_sep_char]

    # Store each individual element in splitline into an array.  The
    #   array index is NOT the column index because NULL list elements
    #   represent the sep_char.
    for {set i 0} {$i < [llength $p_split_line]} {incr i} { 
	set p_col_string([expr $i+1]) [lindex $p_split_line $i]
    }

    # Create the format string from columns in p_col_list.
    set p_format_string ""
    for {set i 0} {$i < [llength $p_split_line]} {incr i} {
	if {[lsearch -exact $p_col_list [expr $i+1]] != -1} {
	    append p_format_string $p_col_string([expr $i+1])
	}
    }

    return $p_format_string
}

#<def> ---------------------------------------------------------------
#<def> StripQuotes: Strip off starting and ending quotes from string.
#<def> 
#<def> Parameters:
#<def>    p_string - string to strip the quotes off of.
#<def> 
#<def> Global Variables:
#<def>    None.
#<def> 
#<def> Notes:
#<def>    Escape backslashes (\) prevent stripping.
#<def> 

proc StripQuotes { p_string } {
    # Define single and double quotes
    set p_single \'; set p_double \"

    # Indeces of the last and 2nd to last characters in the string.
    set p_last [expr [string length $p_string]-1]
    set p_prev [expr $p_last-1]

    # Indicator for string starting with a quote.
    set p_start 0		;# False.

    # Strip off the first quote and decrement the indeces.
    if {[string index $p_string 0] == $p_single} {
	set p_string [string range $p_string 1 $p_last]
	set p_start 1; incr p_last -1; incr p_prev -1
    } elseif {[string index $p_string 0] == $p_double} {
	set p_string [string range $p_string 1 $p_last]
	set p_start 1; incr p_last -1; incr p_prev -1
    } 

    # Strip off the last quote unless the previous character is
    #   an escape backslash.
    if $p_start {
	if {[string index $p_string $p_last] == $p_single} {
	    if {[string range $p_string $p_prev $p_prev] != "\\"} {
		return [string range $p_string 0 $p_prev]
	    }
	} elseif {[string index $p_string $p_last] == $p_double} {
	    if {[string range $p_string $p_prev $p_prev] != "\\"} {
		return [string range $p_string 0 $p_prev]
	    }
	}
    }
    return $p_string
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                  tkGrep_global.tcl                                                                                   100600     765     765         4247  6144377426  12527  0                                                                                                    ustar   wesb                            wesb                                                                                                                                                                                                                   #
# Name
#
#   tkGrep_global.tcl
#
# Version
#
#   1.0
#
# Description
#
#   Contains global definitions used in tkGrep
#
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#
# Modification History
#
# Name              Date         Comment
# ~~~~~~~~~~~~~~~   ~~~~~~~~~~   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Wes Bailey        1996.05.06   Created.
#

set tkGrep(HOME) $env(HOME)

set tkGrep(WORKDIR) ""
set tkGrep(FILENAME) ""
set tkGrep(PATTERN) ""
set tkGrep(OPTION) ""
set tkGrep(SEPERATOR) ""
set tkGrep(COLUMN) ""

# Define global image variables.
set tkGrep(IMAGEDIR) "/home/wesb/tkWorld_src/image"
set tkGrep(IMAGE_TKGREP) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_tkgrep.gif]
set tkGrep(IMAGE_RUN) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_run.gif]
set tkGrep(IMAGE_EXIT) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_exit.gif]
set tkGrep(IMAGE_RESET) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_reset.gif]
set tkGrep(IMAGE_CLEAR) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_clear.gif]
set tkGrep(IMAGE_HELP) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_help.gif]
set tkGrep(IMAGE_SAVE) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_save.gif]

# Define help messages which appear when mouse enters the defined
#    widget.
set tkGrep(HELP.top) "Command settings"
set tkGrep(HELP.bottom) "LOG Window"
set tkGrep(HELP.middle.run) "Execute grep command"
set tkGrep(HELP.middle.reset) "Reset command settings"
set tkGrep(HELP.middle.clear) "Clear the LOG window"
set tkGrep(HELP.middle.save) "Save LOG window to file" 
set tkGrep(HELP.middle.help) "View help documentation"
set tkGrep(HELP.middle.exit) "Exit tkGrep"

# Define the error messages.

set tkGrep(ERROR,SE) "Syntax Error"
set tkGrep(ERROR,IWD) "Invalid Working Directory"
set tkGrep(ERROR,FDNE) "Filename Does Not Exist"
set tkGrep(ERROR,FNF) "Filename Not File"
set tkGrep(ERROR,RPD) "Read Permission Denied"
set tkGrep(ERROR,ISP) "Invalid Search Pattern"
set tkGrep(ERROR,SNSC) "Seperator Not Single Character"
set tkGrep(ERROR,SNS) "Seperator Not Specified"
set tkGrep(ERROR,CNS) "Columns Not Specified"
set tkGrep(ERROR,ID) "Invalid Delimeter"                                                                                                                                                                                                                                                                                                                                                         tkGrep_global.tcl~                                                                                  100600     765     765         4253  6144276165  12720  0                                                                                                    ustar   wesb                            wesb                                                                                                                                                                                                                   #
# Name
#
#   tkGrep_global.tcl
#
# Version
#
#   1.0
#
# Description
#
#   Contains global definitions used in tkGrep.tcl
#
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#
# Modification History
#
# Name              Date         Comment
# ~~~~~~~~~~~~~~~   ~~~~~~~~~~   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Wes Bailey        1996.05.06   Created.
#

set tkGrep(HOME) $env(HOME)

set tkGrep(WORKDIR) ""
set tkGrep(FILENAME) ""
set tkGrep(PATTERN) ""
set tkGrep(OPTION) ""
set tkGrep(SEPERATOR) ""
set tkGrep(COLUMN) ""

# Define global image variables.
set tkGrep(IMAGEDIR) "/home/wesb/tkWorld_src/image"
set tkGrep(IMAGE_TKGREP) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_tkgrep.gif]
set tkGrep(IMAGE_RUN) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_run.gif]
set tkGrep(IMAGE_EXIT) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_exit.gif]
set tkGrep(IMAGE_RESET) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_reset.gif]
set tkGrep(IMAGE_CLEAR) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_clear.gif]
set tkGrep(IMAGE_HELP) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_help.gif]
set tkGrep(IMAGE_SAVE) [image create photo \
	-file $tkGrep(IMAGEDIR)/tkWorld_save.gif]

# Define help messages which appear when mouse enters the defined
#    widget.
set tkGrep(HELP.top) "Command settings"
set tkGrep(HELP.bottom) "LOG Window"
set tkGrep(HELP.middle.run) "Execute grep command"
set tkGrep(HELP.middle.reset) "Reset command settings"
set tkGrep(HELP.middle.clear) "Clear the LOG window"
set tkGrep(HELP.middle.save) "Save LOG window to file" 
set tkGrep(HELP.middle.help) "View help documentation"
set tkGrep(HELP.middle.exit) "Exit tkGrep"

# Define the error messages.

set tkGrep(ERROR,SE) "Syntax Error"
set tkGrep(ERROR,IWD) "Invalid Working Directory"
set tkGrep(ERROR,FDNE) "Filename Does Not Exist"
set tkGrep(ERROR,FNF) "Filename Not File"
set tkGrep(ERROR,RPD) "Read Permission Denied"
set tkGrep(ERROR,ISP) "Invalid Search Pattern"
set tkGrep(ERROR,SNSC) "Seperator Not Single Character"
set tkGrep(ERROR,SNS) "Seperator Not Specified"
set tkGrep(ERROR,CNS) "Columns Not Specified"
set tkGrep(ERROR,ID) "Invalid Delimeter"                                                                                                                                                                                                                                                                                                                                                     tkGrep_xresources.tcl                                                                               100600     765     765         1464  6143733703  13460  0                                                                                                    ustar   wesb                            wesb                                                                                                                                                                                                                   #
# Name
#
#   tkGrep_xresources.tcl
#
# Version
#
#   1.0
#
# Description
#
#   Adds Xresources to tkGrep.tcl
#
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#
# Modification History
#
# Name              Date         Comment
# ~~~~~~~~~~~~~~~   ~~~~~~~~~~   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Wes Bailey        1996.05.07   Created.
#

option add *background #8fdea54ecdd2
option add *foreground #000000000000
option add *activeBackground #8fdea54ecdd2
option add *activeForeground #000000000000
option add *selectBackground #000000000000
option add *selectForeground #8fdea54ecdd2
option add *troughColor #8fdea54ecdd2
option add *selectColor #000000000000
option add *entry.highlightThickness 2
option add *highlightForeground #000000000000
option add *highlightBackground #8fdea54ecdd2                                                                                                                                                                                                            tkGrep_xresources.tcl~                                                                              100600     765     765         1324  6143724015  13645  0                                                                                                    ustar   wesb                            wesb                                                                                                                                                                                                                   #
# Name
#
#   tkGrep_xresources.tcl
#
# Version
#
#   1.0
#
# Description
#
#   Adds Xresources to tkGrep.tcl
#
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#
# Modification History
#
# Name              Date         Comment
# ~~~~~~~~~~~~~~~   ~~~~~~~~~~   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Wes Bailey        1996.05.07   Created.
#

option add *background #8fdea54ecdd2
option add *foreground #000000000000
option add *activeBackground #8fdea54ecdd2
option add *activeForeground #000000000000
option add *selectBackground #000000000000
option add *selectForeground #8fdea54ecdd2
option add *troughColor #8fdea54ecdd2
option add *selectColor #000000000000
option add *highlightThickness 0

                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            