#
# Name
#
#   standard_Library.tcl
#
# Version
#
#   1.0
#
# Description
#
#   This file contains reusable program modules.
#
# Notes
#
#   1) Code in this library is completely reusable meaning procedures
#      do NOT contain globally defined variables.
#
#   2) Use the program tkgrep to view procedure definition comments 
#      delimeted by #<DEF> for source code in this file.
#
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#
# Modification History
#
# Name              Date         Comment
# ~~~~~~~~~~~~~~~   ~~~~~~~~~~   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Wes Bailey        1996.04.17   Created the following procedures:
#                                     CloseWindow
#                                     ClearTextWidget
#                                     ClearEntryWidget
#                                     ErrorDialog
#                                     ScrolledTextWidget
#                                     ScrolledCanvasWidget
# Wes Bailey        1996.04.18   Added:
#                                     ScrolledWidget
#                                     DirTreeWidget
#                                     DirTreeDisplay
#                                Deleted:
#                                     ScrolledTextWidget
#                                     ScrolledCanvasWidget
# Wes Bailey        1996.04.29   Added:
#                                     RightPad
#                                     LeftPad
#                                     DirTreeBindButton
#                                     DirTreeSelect
#                                     DirTreeFindFile 
# Wes Bailey        1996.05.02   Added:
#                                     LabelEntry
#                                     SaveDialog
#                                     SaveFile
# Wes Bailey        1996.05.03   Added:
#                                     CheckSystem
# Wes Bailey        1996.05.08   Added:
#                                     OverWriteFileDialog
#


#<def> ---------------------------------------------------------------
#<def> CloseWindow: Destroy the supplied window.
#<def>
#<def> Parameters: 
#<def>    p_window - Window to destroy.
#<def>

proc CloseWindow {p_window} {destroy $p_window}

#<def> ---------------------------------------------------------------
#<def> ClearTextWidget: Clear the supplied text widget.
#<def>
#<def> Parameters: 
#<def>    p_widget - Text widget to clear.
#<def>

proc ClearTextWidget {p_widget} {$p_widget delete 0.0 end}

#<def> ---------------------------------------------------------------
#<def> ClearEntryWidget: Clear the supplied entry widget.
#<def>
#<def> Parameters: 
#<def>    p_widget - Entry widget to clear.
#<def>

proc ClearEntryWidget {p_widget} {$p_widget delete 0 end}

#<def> ---------------------------------------------------------------
#<def> ErrorDialog: Display an error in a standard dialog box.
#<def>
#<def> Parameters: 
#<def>    p_error_msg - Text to be display as the error message.
#<def>

proc ErrorDialog { p_error_msg } {
    set f [toplevel .error]

    # Annoy the user.
    bell

    # Set the window title.
    wm title $f "Error"

    # Make other windows inoperable until this window is closed.
    grab $f

    # Create the top frame and display the error message.
    frame $f.top -borderwidth 2 -relief sunken
    message $f.top.msg -aspect 1000 -text $p_error_msg -justify left
    label $f.top.bitmap -bitmap error
    pack $f.top.bitmap $f.top.msg -side left -padx 4 

    # Create the error okay button and pack it into the error frame.
    button $f.ok -text Ok -command [list CloseWindow $f]
    pack $f.ok -side top -pady 4 -ipadx 12

    # Pack up the error frame.
    pack $f.top $f.ok -side bottom

    # Wait for the error button to be pressed.
    tkwait variable $f.ok

    # Destroy the window from the window manager.
    wm protocol $f WM_DELETE_WINDOW {CloseWindow $f; grab release $f}

    # Release this window.
    grab release $f
}

#<def> ---------------------------------------------------------------
#<def> ScrolledWidget: Put scrollbars around a given widget.
#<def> 
#<def> Parameters: 
#<def>    p_frame        - Parent frame to place the widget in.
#<def>    p_frame_bw     - Parent frame borderwidth.
#<def>    p_frame_relief - Parent frame relief.
#<def>    p_widget       - Type of widget to put in the parent frame.
#<def>    args           - Arguments which characterize the widget.
#<def> 
#<def> Example: 
#<def>    ScrolledWidget .main 10 raised text .main.text \
#<def>            -width 70 -height 20
#<def>                  

proc ScrolledWidget {p_frame p_frame_bw p_frame_relief p_widget \
	p_widget_frame args} {
    frame $p_frame -borderwidth $p_frame_bw -relief $p_frame_relief

    # Create the main widget to add scrollbars too.
    eval $p_widget $p_widget_frame $args \
	    {-xscrollcommand [list $p_frame.xscroll set] \
	    -yscrollcommand [list $p_frame.yscroll set]}
 
    # Create the y scrollbar.
    scrollbar $p_frame.yscroll -orient vertical \
	    -command [list $p_widget_frame yview]

    # Create a frame for the padding and x scrollbar.
    frame $p_frame.bottom
    scrollbar $p_frame.xscroll -orient horizontal \
	    -command [list $p_widget_frame xview]

    # Determine the size of the padding.
    set pad [expr [$p_frame.yscroll cget -width] + \
	    2*([$p_frame.yscroll cget -bd] + \
	       [$p_frame.yscroll cget -highlightthickness])]

    # Create a frame for the padding to be packed in $p_frame.bottom
    frame $p_frame.pad -width $pad -height $pad

    # Pack up the bottom of the window.
    pack $p_frame.bottom -side bottom -fill x -expand true
    pack $p_frame.pad -in $p_frame.bottom -side right
    pack $p_frame.xscroll -in $p_frame.bottom -side bottom -fill x

    pack $p_frame.yscroll -side right -fill y
    pack $p_widget_frame -side left -fill both -expand true
    pack $p_frame -side top -fill both -expand true
}

#<def> ---------------------------------------------------------------
#<def> DirTreeWidget: This widget displays a directory tree.
#<def>
#<def> Parameters: 
#<def>    p_path          - Path for the tree.
#<def>    p_frame         - Parent frame to display tree in.
#<def>    p_frame_bw      - Parent frame borderwidth.
#<def>    p_frame_relief  - Parent frame relief.
#<def>    p_canvas_widget - Dummy argument.
#<def>    p_frame_canvas  - Frame of the canvas
#<def>    args            - Agruments which characterize the canvas.
#<def>
#<def> References:
#<def>    DirTreeDisplay
#<def> 
#<def> Example:
#<def>    DirTreeWidget $env(HOME) .bottom.dirtree 2 raised \
#<def> 	          canvas .bottom.dirtree.canvas \
#<def>            -height 500 -width 250 \
#<def> 	          -borderwidth 2 -relief sunken \
#<def>            -scrollregion {0 0 250 5000}
#<def> 

proc DirTreeWidget {p_path p_frame p_frame_bw p_frame_relief \
	p_canvas_widget p_frame_canvas args} {
    # Create the scrolled canvas to display the directory tree in.
    eval {ScrolledWidget $p_frame $p_frame_bw $p_frame_relief \
	    canvas $p_frame_canvas} $args

    # Put a filled circle at the top of the tree.
    $p_frame_canvas create oval 20 18 30 28 -fill black

    # Call the recursive procedure which display the directory
    #   tree for the given path.  The last 2 arguments indicate
    #   the starting level and depth in the tree.
    DirTreeDisplay $p_path $p_frame_canvas 0 0
}

#<def> ---------------------------------------------------------------
#<def> DirTreeDisplay: Display the directory tree in the canvas.
#<def>
#<def> Parameters:
#<def>    p_path  - Current LOCAL path to display.
#<def>    p_frame - Frame which contains the canvas display.
#<def>    p_level - Current level in the tree.
#<def>    p_depth - Current depth in the tree.
#<def>
#<def> References:
#<def>    Called from DirTreeWidget
#<def> 
#<def> Notes: Think of the level and depth like the row and column
#<def>    in a spreadsheet.  In the following diagram each entry is
#<def>    followed by the label (level,depth)
#<def>
#<def>    o                 (0,0)
#<def>    |
#<def>    +---bin           (1,1)
#<def>    |
#<def>    +---examples      (2,1)
#<def>    |   |
#<def>    |   +---fortran   (3,2)
#<def>    |   |
#<def>    |   +---tcl       (4,2)
#<def>    |
#<def>    +---Mail          (5,1)
#<def>        |
#<def>        +---people    (6,2)
#<def>            |
#<def>            +---Dave  (7,3)
#<def>            |
#<def>            +---chris (8,3)
#<def>

proc DirTreeDisplay { p_path p_frame p_level p_depth} {
    # Set the home position
    set p_home(x) 25
    set p_home(y) 25

    # Set up array elements for the L which is the line between
    #   directory names.  The magnitude of the L lines is given 
    #   by p_L(mag) while the start of the next tree under the text
    #   is displaced by p_L(dmag).
    set p_L(mag) 20		
    set p_L(dmag) 10		
				
    # Scan all files in the path and store all of the directories
    #   in the LOCAL list p_dir_listing.
    set p_file_listing [glob -nocomplain $p_path/*]
    set p_dir_listing ""
    foreach p_file $p_file_listing {
	if [file isdirectory $p_file] {
	    append p_dir_listing "$p_file "
	}
    }

    # Store the number of directories in p_dir_count and set
    #   the current directory index p_curr_dir which tells you
    #   which directory you are at in p_dir_listing.
    set p_dir_count [llength $p_dir_listing]
    set p_curr_dir 0

    # If their are no directories, the exit with the current level.
    #   Otherwise increment the depth to reflect subdirectories
    #   exist.
    if !$p_dir_count {
	return $p_level
    } else {
	incr p_depth +1
    }
    foreach p_dir $p_dir_listing {
	# Increment the counter to tell me where I am at in the
	#   current LOCAL directory listing.
	incr p_curr_dir +1

	# Increment the level to reflect another element in the
	#   directory listing.
	incr p_level +1

	# Trim the /home/path from the directory name.
	set p_dir_start [expr [string length $p_path] + 1]
	set p_dir_end [string length $p_dir]
	set p_dir [string range $p_dir $p_dir_start $p_dir_end]

	# Determine the 3 points which make up the L based on the
	#   current level and depth in the directory tree.
	set p_L(1,x) [expr $p_home(x) + \
		($p_depth-1)*($p_L(mag) + $p_L(dmag))]
	set p_L(1,y) [expr $p_home(y) + \
		($p_level-1)*$p_L(mag) + ($p_depth-1)*$p_L(dmag)]

	set p_L(2,x) $p_L(1,x)
	set p_L(2,y) [expr $p_L(1,y) + $p_L(mag)]

	set p_L(3,x) [expr $p_L(2,x) + $p_L(mag)]
	set p_L(3,y) $p_L(2,y)

	# Store the corner point of the L for the continuing
	#   vertical line that is drawn after the recursive call.
	set p_vert_line(1,x) $p_L(2,x)
	set p_vert_line(1,y) $p_L(2,y)

	# Create the L line coordinates as one string with
	#   trailing space on each individual coordinate.
	foreach i {1 2 3} {
	    append p_L(xylist) "$p_L($i,x) $p_L($i,y) "
	}

	# Create the text coordinates to display the directory name.
	set p_text(xylist) "[expr $p_L(3,x) + $p_L(dmag)] $p_L(3,y)"

	# Display the tree with L line and the directory name.
	eval {$p_frame create line} $p_L(xylist) {-width 2}
	eval {$p_frame create text} $p_text(xylist) \
		{ -text $p_dir -anchor w -tag text}

	# Store the current LOCAL level and recursively call the
	#   function to display tree for the sub-directories.  
	set p_old_level $p_level
	set p_next $p_path/$p_dir
	set p_level [DirTreeDisplay $p_next $p_frame $p_level $p_depth]

	# Update the level if this is the last element in the
	#   local directory listing so that when we return from the
	#   recursive call then previous tree will not start on
	#   top of this one.  If this is not the last element in the
	#   list, then go back and draw the vertical connecting the
	#   previous directory name with the current one.
	if {$p_curr_dir == $p_dir_count} {
	    incr p_level +1
	} else {
	    set p_level_diff [expr $p_level-$p_old_level]
	    set p_vert_line(2,x) $p_vert_line(1,x)
	    set p_vert_line(2,y) [expr $p_vert_line(1,y) + \
		    $p_level_diff*($p_L(mag)+$p_L(dmag))]
	    set p_vert_line(xylist) "$p_vert_line(1,x) \
		    $p_vert_line(1,y) $p_vert_line(2,x) \
		    $p_vert_line(2,y)"
	    eval {$p_frame create line} \
		    $p_vert_line(xylist) {-width 2}
	}

	# Reset the L line and text coordinates.
	set p_L(xylist) ""
	set p_vert_line(xylist) ""
	set p_text(xylist) ""
    }
    return $p_level
}

#<def> ---------------------------------------------------------------
#<def> DirTreeBindButton: Bind the button in the given widget to the 
#<def>                    supplied command.
#<def>
#<def> Parameters:
#<def>    p_button - Tk Button to bind [without < and >].
#<def>    p_from   - The canvas which the user will click on.
#<def>    p_cmd    - Command which gets executed when $p_button is 
#<def>               pressed. 
#<def>
#<def> Notes: This module uses DirTreeSelect and assumes you want to
#<def>    process the selection ($p_sel) in some manner.  Supplying 
#<def>    the command allows freedom to do whatever you want with the
#<def>    the selection.
#<def>
#<def> Example: 
#<def>   set p_cmd ".listbox insert end \$p_sel"
#<def>   DirTreeBindButton Button-1 .canvas $p_cmd
#<def>

proc DirTreeBindButton { p_button p_from p_cmd } {
    bind $p_from <$p_button> {
	focus %W
	if {[%W find overlapping [expr %x-2] [expr %y-2] \
		[expr %x+2] [expr %y+2]] == {}} {
	    %W focus {}
	}
    }
    $p_from bind text <$p_button> [list DirTreeSelect $p_from $p_cmd]
}

#<def> ---------------------------------------------------------------
#<def> DirTreeSelect:  Evaluates the supplied command.
#<def> 
#<def> Parameters:
#<def>   p_from - Canvas the selection comes from.
#<def>   p_cmd  - Supplied command which gets executed.
#<def>   p_sel  - The returned selection which the command may use.
#<def> 
#<def> Example:
#<def>   See example for DirTreeButtonBind.
#<def> 

proc DirTreeSelect { p_from p_cmd } {
    $p_from select from current 0
    $p_from select to current end
    if ![catch {selection get} p_sel] { 
	eval $p_cmd
    }
}

#<def> ---------------------------------------------------------------
#<def> DirTreeFindFile: An ad hoc file finding routine.
#<def>   
#<def> Parameters:
#<def>   p_path - Defaults to home directory.
#<def>   p_file - File to search for.
#<def> 
#<def> Notes:
#<def>   Right now this routine executes a system call.
#<def> 

proc DirTreeFindFile {{p_path $env(HOME)} p_file} {
    # Search for given filename.
    set p_cmd [list exec find $p_path -name $p_file -print]
    if [catch {set p_output [eval $p_cmd]} p_error] {
	return 0
    }
    foreach p_file $p_output {
	if [file isdirectory $p_file] {
	    return $p_file
	}
    }
    return 0
}

#<def> ---------------------------------------------------------------
#<def> RightPad: Makes a string a certain length by adding a certain
#<def>           set of characters (default is space) to the right.
#<def> 
#<def> Parameters:
#<def>   p_string  - String to make certain length.
#<def>   p_length  - Length to make the string.
#<def>   p_padding - Set of characters to add to the string.
#<def> 

proc RightPad {p_string p_pad_length {p_padding " "}} {
    # Get the current length of string.
    set p_curr_length [string length $p_string]

    # return p_string if smaller than p_pad_length.
    if {$p_curr_length > $p_pad_length} {
	return $p_string
    }

    # Create the padding string.
    set tmp_string ""
    for {set i $p_curr_length} {$i < $p_pad_length} {incr i +1} {
        append tmp_string $p_padding
    }
    return $p_string$tmp_string
}

#<def> ---------------------------------------------------------------
#<def> LeftPad: Makes a string a certain length by adding a certain
#<def>          set of characters (default is space) to the left.
#<def> 
#<def> Parameters:
#<def>   p_string  - String to make certain length.
#<def>   p_length  - Length to make the string.
#<def>   p_padding - Set of characters to add to the string.
#<def> 

proc LeftPad {p_string p_pad_length {p_padding " "}} {
    # Get the current length of string.
    set p_curr_length [string length $p_string]

    # return p_string if smaller than p_pad_length.
    if {$p_curr_length > $p_pad_length} {
	return $p_string
    }

    # Create the padding string.
    set tmp_string ""
    for {set i $p_curr_length} {$i < $p_pad_length} {incr i +1} {
        append tmp_string $p_padding
    }
    return $tmp_string$p_string
}

#<def> ---------------------------------------------------------------
#<def> LabelEntry: Create a label and entry widget together.
#<def> 
#<def> Parameters:
#<def>    p_frame     - Frame to put the combination in.
#<def>    p_label     - Text to put in the label widget.
#<def>    p_lbl_width - Label width.
#<def>    p_command   - Command bound to <Return> for entry widget.
#<def>    args        - Arguments for the entry widget.
#<def> 
#<def> Example:
#<def>    LabelEntry .top 10 Filename SaveFile -width 30
#<def> 

proc LabelEntry { p_frame p_label p_lbl_width p_command args } {
    # Create the frame to contain the label and the entry widgets.
    frame $p_frame

    # Create the label and entry widgets.
    label $p_frame.label -text $p_label -width $p_lbl_width -anchor w
    eval {entry $p_frame.entry -relief sunken} $args
    
    # Pack up the current frame.
    pack $p_frame.label -side left
    pack $p_frame.entry -side right -fill x -expand true

    # Bind the pressing of return to the supplied command.
    bind $p_frame.entry <Return> $p_command

    return $p_frame.entry
}

#<def> ---------------------------------------------------------------
#<def> SaveDialog: Create a SaveFile dialog box.
#<def> 
#<def> Parameters:
#<def>    p_widget - The name of the widget to save contents into a 
#<def>               filename.
#<def> 
#<def> Notes:
#<def>    Uses SaveFile procedure to actually save the contents into
#<def>    a file.
#<def> 

proc SaveDialog { p_widget } {
    global p_savefile
    toplevel .savedlg

    # Set the window name
    wm title .savedlg "SaveFile"

    # Grab the save dialog so the user has to interact with it.
    grab .savedlg

    # Initialize the global variable for the SaveFile proc.
    set p_savefile(WIDGET) $p_widget
    set p_savefile(FILENAME) ""

    # Create the label and entry widgets.
    LabelEntry .savedlg.top "Filename:" 10 \
	    {SaveFile ; CloseWindow .savedlg} \
	    -textvariable p_savefile(FILENAME)

    # Frame for the command buttons.
    frame .savedlg.bottom

    # Create the Ok and Cancel buttons.
    button .savedlg.bottom.ok -text Ok \
	    -command {SaveFile ; CloseWindow .savedlg}
    button .savedlg.bottom.cancel -text Cancel \
	    -command {CloseWindow .savedlg}

    # Pack up the bottom frame.
    pack .savedlg.bottom.ok -ipadx 12
    pack .savedlg.bottom.ok .savedlg.bottom.cancel -side left \
	    -padx 4 

    # Pack up the dialog frame.
    pack .savedlg.top .savedlg.bottom -side top -pady 4 -padx 10
    
    # Set the focus on the entry widget.
    focus .savedlg.top.entry

    # Close the window on exit.
    wm protocol .savedlg WM_DELETE_WINDOW {CloseWindow .savedlg ; \
	grab release .savedlg}
}

#<def> ---------------------------------------------------------------
#<def> SaveFile: Save the contents of a TEXT widget into a file.
#<def> 
#<def> Parameters:
#<def>    None - Uses the global variable p_savefile.
#<def> 
#<def> Global Variables: Needed so that calls from entry widgets can
#<def>    be made without having filename variables undefined.
#<def> 
#<def>    p_savefile(WIDGET)   - Widget name to get contents from.
#<def>    p_savefile(FILENAME) - Name of file to store the contents.
#<def> 
#<def> Notes:
#<def>    Automatically checks if the path has been supplied.
#<def> 

proc SaveFile { } {
    global p_savefile ; global env

    set p_file [file tail $p_savefile(FILENAME)]
    set p_workdir [file dirname $p_savefile(FILENAME)]
    set p_widget $p_savefile(WIDGET)

    # Do tilde expansion or place the home path in the working
    #   directory.
    if {[string index $p_workdir 0] == "."} {
	set p_workdir $env(HOME)
    } elseif {[string index $p_workdir 0] == "~"} {
	set p_workdir $env(HOME)/[string range $p_workdir 1 end]
    }

    # Strip off the last slash from the working directory if present.
    set p_lastidx [expr [string length $p_workdir]-1]
    if {[string index $p_workdir $p_lastidx] == "/"} {
	set p_workdir [string trimright $p_workdir "/"]
    }

    # Perform error checks on the file and working directory.
    if ![file writable $p_workdir] {
	set p_error "Write Permission Denied"
	ErrorDialog $p_error
	return
    } elseif ![file readable $p_workdir] {
	set p_error "Read Permission Denied"
	ErrorDialog $p_error
	return
    } elseif ![string length $p_file] {
	set p_error "No File Specified"
	ErrorDialog $p_error
	return 
    } elseif [file exists "$p_workdir/$p_file"] {
	set p_overwrite [OverWriteFileDialog "$p_workdir/$p_file"]
	if !$p_overwrite {return}
    }

    # Store the contents of the supplied widget into the buffer.
    set p_buffer [$p_widget get 1.0 end] 

    # Open the output file.
    if [catch {set p_fileId [open "$p_workdir/$p_file" w]} p_error] {
	ErrorDialog $p_error
    }

    # Place the contents of the widget into the file.
    foreach p_line [split $p_buffer \n] {
	puts $p_fileId $p_line
    }
    
    # Close the output file
    close $p_fileId
}

#<def> ---------------------------------------------------------------
#<def> OverWriteFileDialog: Query user to overwrite file.
#<def>
#<def> Parameters:
#<def>   None
#<def> 
#<def> GlobalVariables:
#<def>   None
#<def>
#<def> Notes:
#<def>   Returns 1 if user wants to overwrite file.
#<def>

proc OverWriteFileDialog { p_file } {
    global p_response ;# this variable stores the users response.

    # Put this window at the top
    set f [toplevel .overwrite -borderwidth 2]

    # Annoy the user.
    bell

    # Set the window title for the window manager.
    wm title $f "OverWriteFile"

    # Make other windows inoperative by grabbing this one.
    grab $f

    # Create the display message.
    set p_msg "OverWrite File:\n$p_file"

    # Create the top frame and display the user query.
    frame $f.top -borderwidth 2 -relief sunken
    message $f.top.msg -aspect 1000 -text $p_msg -justify left
    label $f.top.bitmap -bitmap questhead
    pack $f.top.bitmap $f.top.msg -side left -padx 4

    # Create the yes and no buttons.
    frame $f.bottom -borderwidth 2
    button $f.bottom.yes -text Yes -command [list set p_response 1]
    button $f.bottom.no -text No -command [list set p_response 0]
    pack $f.bottom.yes $f.bottom.no -side left -padx 4 -ipadx 12

    # Pack up the overwrite widget.
    pack $f.top $f.bottom -side top

    # Wait for a selection.
    tkwait variable p_response

    # Destroy the window if done through the window manager.
    wm protocol $f WM_DELETE_WINDOW {CloseWindow $f ; grab release $f}

    # Release the widget.
    grab release $f
    
    # Close the window and return the selected value.
    CloseWindow $f

    return $p_response
}

#<def> ---------------------------------------------------------------
#<def> CheckSystem: Check if the system has the supplied command.
#<def> 
#<def> Parameters:
#<def>    p_cmd - command you are checking the system for.
#<def> 
#<def> Notes: Returns 1 for true, meaning the system has the command.
#<def> 

proc CheckSystem { p_cmd } {
    if [catch {set p_found [exec which $p_cmd]} p_error] {
	set p_found ""
    }

    if ![string length $p_found] {
	return 0
    } else {
	return 1
    }
}

