# ===================================================================
# edit.tcl : callback routines for mouse button
# 
# ===================================================================

# edit:b1Proc maped w x y
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
#   by refarencing maped(curObj) which set by obj buttons, switch tasks.
proc edit:b1Proc {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    set item {}
    # Unselect recent item.
    edit:unSelectItem $maped $w
    # $data(curObj) = current Object.
    switch $data(curObj) {
	select {
	    # if in select mode.
	    if {$data(utilFlag) != 0} {
		# if editor or viewer radiobutton was checked ...
		set url [canvas:checkPoints $x $y \
			[lindex [set tags [$w gettags current]] \
			[lsearch -glob $tags {url:*}]]] 
		# exec utils.
		if {$url != {url:} && \
			[regexp {url:(.*)} $url dummy url] > 0} {
		    edit:exec $url [option get . $data(utilFlag)Exec {}]
		}
	    } else {
		# else, select current item.
		if {[set item [$w find withtag current]] != ""} {
		    edit:selectItem $maped $w [$w find withtag current]
		}
	    }
	}
	circle {
	    # if in drawing circle mode.
	    set data(recX) $x; set data(recY) $y; # store current x,y
	    # create circle obj and bind drawing routines to it.
	    bind $w <Motion> [list edit:editingItem $maped %W %x %y]
	    bind $w <Button-1> [list edit:finishEditItem $maped %W]
	    msg:set questhead {<Motion> : resize circle. <1> : finish.}
	    # and modify color, width, and tag.
	    $w itemconfigure [set item [canvas:drawCircle $x $y 0]] \
		    -outline [option get $w drawingColor {}] \
		    -width [option get $w drawingWidth {}]
	    $w addtag {Editting} withtag $item
	    mbar:disable
	}
	rectangle {
	    # if in drawing rectangle mode.
	    set data(recX) $x; set data(recY) $y; # store current x,y
	    # create rectangle obj and bind drawing routines to it.
	    bind $w <Motion> [list edit:editingItem $maped %W %x %y]
	    bind $w <Button-1> [list edit:finishEditItem $maped %W]
	    msg:set questhead {<Motion> : resize rectangle. <1> : finish.}
	    # and modify color, width, and tag.
	    $w itemconfigure [set item [canvas:drawRectangle $x $y $x $y]] \
		    -outline [option get $w drawingColor {}] \
		    -width [option get $w drawingWidth {}]
	    $w addtag {Editting} withtag $item
	    mbar:disable
	}
	polygon {
	    # if in drawing polygon mode.
	    set data(recX) $x; set data(recY) $y; # store current x,y
	    # create polygon obj and bind drawing routines to it.
	    # NOTE: firstry create line item insted of proygon.
	    set item [$w create line $x $y $x $y  \
		    -fill [option get $w drawingColor {}] \
		    -width [option get $w drawingWidth {}] -tags {Editting}]
	    bind $w <Motion> [list edit:editingPolygon $maped %W %x %y]
	    bind $w <Button-1> {edit:addVerTexToEditingPolygon %W %x %y}
	    bind $w <Button-3> \
		    [list edit:finishEditPolygon $maped %W %x %y]
	    msg:set questhead {<Motion> : drawing. <3> : finish. <1> : add vertex.}
	    mbar:disable
	}
	point {
	    # if in drawing polygon mode.
	    set data(recX) $x; set data(recY) $y; # store current x,y
	    # create point and select it.
	    edit:selectItem $maped $w \
		    [set item [lindex [canvas:drawPoint $x $y] 0]]
	}
	default {
	    # maped(curObj) is broken ... !!!
	    puts stderr "Fatal Error: unknown object: $data(curObj)"
	}
    }
    return $item
}

# edit:selectItem maped w item
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        item  - ID for object.
# Side effects - 
#  1) change width and color for selected item.
#  2) enable to editing for url and alt fields.
#  3) add handle to selected item.
proc edit:selectItem {maped w item} {
    upvar #0 $maped data
    set outline [option get $w selectColor {}]
    set selectWidth [option get $w selectWidth {}]
    set retVal $item

    set data(curObj) {select}
    $w addtag {Selected} withtag $item

    set url {}; set alt {}
    regexp {url:(.*)} [lindex [set tags [$w gettags $item]] \
	    [lsearch -glob $tags {url:*}]] dummy url
    regexp {alt:(.*)} [lindex $tags [lsearch -glob $tags {alt:*}]] dummy alt
    if {[set type [$w type {Selected}]] == {image}} {
	if {$data(curFormat) != {Client Side Map}} {
	    url:enable; url:set $url
	    msg:set questhead {You can edit default URL attribute..}
	}
	return $item
    } else {
	url:enable; url:set $url
	if {$data(curFormat) == {Client Side Map}} {
	    alt:enable; alt:set $alt
	}
    }

    $w itemconfigure $item -width $selectWidth;  # bold 
    # <1> on the $item, then strat move
    if {$type == {polygon}} {
	# <Control-1> on the prolygon $item, then add vertex to it.
	$w bind {Selected} <Control-Button-1> \
		[list edit:addVertexToPolygon $maped %W %x %y]
    } else {
	$w bind {Selected} <Control-Button-1> {}
    }
    $w bind {Selected} <Shift-Button-1> \
	    [list edit:startMove $maped %W %x %y]
    if {$type == {line}} {
	# $item is line, i.e. $item is POINT, then select other line.
	if {[regexp {id:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {id:*}]] dummy otherItem]} {
	    $w itemconfigure $item -fill $outline
	    $w itemconfigure $otherItem -width $selectWidth \
		    -fill $outline;                  # bold and change color
	    $w addtag {Selected} withtag $otherItem; # item is "Selected"
	    lappend retVal $otherItem
	}
    } else {	
	$w itemconfigure $item -outline $outline \
		-fill $outline -stipple [option get $w selectStipple {}]
	edit:createHandle $maped $w $item;           # add handles
    }
    switch $type {
	{oval} -
	{line} -
	{rectangle} {
	    $w bind {Selected} <Enter> {
		msg:set questhead {<Shift-1> : start on moving.}
	    }
	    $w bind {Selected} <Leave> {msg:del}
	    $w bind {Handle} <Enter> {
		msg:set questhead {<Shift-1> : start on resizing.}
	    }
	    $w bind {Handle} <Leave> {msg:del}
	}
	{polygon} {
	    $w bind {Selected} <Enter> {
		msg:set questhead {<Shift-1> : start on moving. <Ctrl-1> : add vertex.}
	    }
	    $w bind {Selected} <Leave> {msg:del}
	    $w bind {Handle} <Enter> {
		msg:set questhead {<Shift-1> : start on resizing. <Ctrl-1> : del vertex.}
	    }
	    $w bind {Handle} <Leave> {msg:del}
	}
    }
    return $retVal
}

proc edit:cancelItem {maped w} {
    upvar #0 $maped data
    $w delete {Editting}
    mbar:enable
    edit:unSelectItem $maped $w
    switch $data(curObj) {
	{select} {
	    if {$data(utilFlag) == {editor}} {
		msg:set questhead {<1> on item : edit associated URL file.}
	    } elseif {$data(utilFlag) == {view}} {
		msg:set questhead {<1> on item : view associated URL file.}
	    } else {
		msg:set questhead {<1> on item : select item.}
	    }
	}
	{circle} -
	{rectangle} {
	    msg:set questhead {<1> : start on editting.}
	}
	{polygon} {
	    msg:set questhead {<1> : start. }
	}
	{point} {
	    msg:set questhead {<1> : edit point.}
	}
    }
}

# edit:unSelectItem w
# Args - w - canvas widget name
# Side effect -
#  1) unselect current selected item.
proc edit:unSelectItem {maped w} {
    upvar #0 $maped data
    set itemColor [option get $w itemColor {}]
    set itemWidth [option get $w itemWidth {}]
    set stipple   [option get $w stipple {}]
    set retVal {}
    foreach i [$w find withtag {Selected}] {
	# fill mode
	if {[set type [$w type $i]] == {line}} {
	    $w itemconfigure $i -width $itemWidth -fill $itemColor
	} elseif {$type != {image}} {
	    if {$data(fillFlag)} {
		$w itemconfigure $i -width $itemWidth -outline $itemColor \
			-fill $itemColor -stipple $stipple
	    } else {
		$w itemconfigure $i -width $itemWidth -outline $itemColor \
			-fill {} -stipple {}
	    }
	}
	# update url & alt attrib
	set tags [lreplace [set tags [$w gettags $i]] \
		[set idx [lsearch -glob $tags {url:*}]] \
		$idx "url:[set url [url:get]]"]
	if {$data(curFormat) == {Client Side Map}} {
	    set tags [lreplace $tags [set idx [lsearch -glob $tags {alt:*}]] \
		    $idx "alt:[alt:get]"]
	}
	$w dtag $i; $w itemconfigure $i -tags $tags
	lappend retVal $i
	if {[$w type $i] ==  {image}} {
	    set data(defaultURL) $url
	}
    }
    url:disable; alt:disable
    $w bind Selected <Enter> {}
    $w bind Selected <Leave> {}
    $w bind Selected <Control-Button-1> {}
    $w bind Selected <Shift-Button-1> {}
    $w bind Selected <Button-1> {}
    $w dtag Selected Selected
    $w bind Habdle <Enter> {}
    $w bind Habdle <Leave> {}
    $w bind Habdle <Control-Button-1> {}
    $w bind Habdle <Shift-Button-1> {}
    $w bind Handle <Double-Button-1> {}
    $w delete Handle
    bind $w <Motion> {}
    msg:del
    return $retVal
}

# edit:createHandle maped w item
# Args - maped - global array data
#        w - canvas widget name
#        item - ID which associated with Handles
# Side effects -
#  1) delete existing Handles
#  2) create Handles on item's each vertex.
#  3) bind some procs to Handles
proc edit:createHandle {maped w item} {
    set retVal {}
    set selectWidth [option get $w selectWidth {}]; # half of handle width
    set handleColor [option get $w handleColor {}]; # color for handle
    $w delete {Handle}; # delete existing Handle
    set len [llength [set coords [$w coords $item]]]
    if {[$w type $item] == "polygon"} {incr len -2}
    for {set i 0} {$i < $len} {incr i 2} {
	lappend retVal [$w create rectangle \
		[expr [set x [lindex $coords $i]] - $selectWidth] \
		[expr [set y [lindex $coords [expr $i+1]]]-$selectWidth] \
		[expr $x + $selectWidth] [expr $y + $selectWidth] \
		-fill $handleColor -outline $handleColor \
		-tags [list Handle item:$item pos:$i]]
    }
    $w bind Handle <Shift-Button-1> [list edit:startResize $maped %W %x %y]
    if {[$w type $item] == {polygon}} {
	$w bind Handle <Control-Button-1> \
		[list edit:delVertexFromPolygon $maped %W]
    } else {
	$w bind Handle <Control-Button-1> {}
    }
    return $retVal
}
# edit:editingItem maped w x y 
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
#  1) change coords for editing item.
# NOTE:
#  This is for circle and rectangle, not for point, polygon.
proc edit:editingItem {maped w x y} {
    upvar #0 $maped data
    if {[set coords [$w coords {Editting}]] != {}} {
	set x [$w canvasx $x]; set y [$w canvasy $y]
	if {[$w type {Editting}] == {oval}} {
	    set ux [expr [lindex $coords 0] - $x + [lindex $coords 2]]
	    set uy [expr [lindex $coords 1] - $y + [lindex $coords 3]]
	} else {
	    set ux [lindex $coords 0]; set uy [lindex $coords 1]
	}
	$w coords {Editting} $ux $uy $x $y
    }    
}

# edit:finishEditItem maped w x y 
# Args - maped - global data (Array maped)
#        w     - canvas window name
# Side effects - 
#  1) fix coords for editing item.
#  2) select editted item.
# NOTE:
#  This is for circle and rectangle, not for point, polygon.
proc edit:finishEditItem {maped w} {
    if {[$w type [set item [$w find withtag {Editting}]]] == {oval}} {
	set ux [lindex [set coords [$w coords $item]] 0]
	set uy [lindex $coords 1]
	set x [lindex $coords 2]; set y [lindex $coords 3]
	if {[set rx [expr $x - $ux]] > [set ry [expr $y - $uy]]} {
	    $w coords $item [expr $x - $rx] [expr $y - $rx] $x $y
	} elseif {$rx < $ry} {
	    $w coords $item [expr $x - $ry] [expr $y - $ry] $x $y
	}
    }
    $w dtag {Editting} {Editting}; # delete "Editting" tag
    mbar:enable
    edit:selectItem $maped $w $item; # select item
    canvas:initBinds
}

# edit:editingPolygon w x y 
# Args - w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
#  1) edge of line follow  mouse pointer.
# NOTE:
#  This is only for porygon item.
proc edit:editingPolygon {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    eval $w coords {Editting} [concat [lrange [set coords \
	    [$w coords {Editting}]] 0 [expr [llength $coords] - 3]] $x $y]
}

# edit:addVerTexToEditingPolygon w x y
# Args - w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
# 1) add vertex to editing polygon.
proc edit:addVerTexToEditingPolygon {w x y} {
    set x [$w canvasx $x]; set y [$w canvasy $y]
    eval $w coords {Editting} [concat [$w coords {Editting}] $x $y]
}

# edit:finishEditPolygon w x y
# Args - w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
# 1) fix coords for editing polygon
# 2) create "true" polygon item using coors of "line" item
# 3) select polygon
proc edit:finishEditPolygon {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    if {[set len [llength [set coords [$w coords {Editting}]]]] == 4} {
	set coords [concat $coords $x $y]; # if coords is sort, then add to it
    }
    if {$len > 6 && \
	    [lindex $coords [expr $len-4]]==[lindex $coords [expr $len-2]] && \
	    [lindex $coords [expr $len-3]]==[lindex $coords [expr $len-1]]} {
	set coords [lrange $coords 0 [expr $len - 3]]; # check duplicated tail
    }
    $w delete {Editting}
    set item [eval $w create polygon $coords]
    if {$data(fillFlag)} {
	$w itemconfigure $item -fill [option get $w fillColor {}] \
		-stipple [option get $w stipple {}] \
		-outline [option get $w itemColor {}] \
		-width [option get $w itemWidth {}] \
		-tags [list {url:} {alt:}]
    } else {
	$w itemconfigure $item -outline [option get $w itemColor {}] \
		-width [option get $w itemWidth {}] \
		-tags [list {url:} {alt:}] -fill {}
    }
    mbar:enable
    edit:selectItem $maped $w $item; # select item
    canvas:initBinds
    return $item
}

# edit:startMove maped w x y 
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
#  1) store current mouse pointer x y
#  2) bind for moving.
proc edit:startMove {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    set data(recX) $x; set data(recY) $y
    bind $w <Motion> [list edit:movingItem $maped %W %x %y]
    $w bind {Selected} <Shift-Button-1> [list edit:stopMoving $maped %W %x %y]
    $w bind {Handle} <Enter> {}
    $w bind {Handle} <Leave> {}
    $w bind {Selected} <Enter> {}
    $w bind {Selected} <Leave> {}
    bind $w <Button-1> {}
    msg:set questhead {<Shift-1> : finish.}
    return
}
# edit:movingItem maped w x y 
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
#  1) store current mouse pointer x y
#  2) calcutate amount of mouse track.
#  3) move item.
proc edit:movingItem {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    set dx [expr $x - $data(recX)]; set dy [expr $y - $data(recY)]
    $w move {Selected} $dx $dy
    $w move {Handle} $dx $dy
    set data(recX) $x; set data(recY) $y
    return [list $dx $dy]
}

# edit:stopMoving maped w
# Args - maped - global data (Array maped)
#        w     - canvas window name
# Side effects - 
# 1) remove bind for moving.
proc edit:stopMoving {maped w x y} {
    edit:movingItem $maped $w $x $y
    set item [$w find withtag {Selected}]
    edit:unSelectItem $maped $w
    canvas:initBinds
    edit:selectItem $maped $w [lindex $item 0]
}

# edit:addVertexToPolygon maped w x y 
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
#  1) add vertex to existing polygon item
# NOTE: Huuummmm... I think this routine is too heavy.
# Does any one know more excerent one ?
proc edit:addVertexToPolygon {maped w x y} {
    set x [$w canvasx $x]; set y [$w canvasy $y]
    set max [expr [llength [set coords [$w coords {Selected}]]] - 4]
    set ret {}
    for {set i 0} {$max >= $i} {incr i 2} {
	set x1 [lindex $coords $i]
	set y1 [lindex $coords [expr $i + 1]]
	set x2 [lindex $coords [expr $i + 2]]
	set y2 [lindex $coords [expr $i + 3]]
	set a [expr $x2 - $x1]; set b [expr $x - $x1]
	set c [expr $y2 - $y1]; set d [expr $y - $y1]
	if {(abs($a) >= 1 && abs($b) >= 1 && \
		abs($c / $a - $d / $b) < 0.5 && \
		(($x1 < $x && $x < $x2) || ($x1 > $x && $x > $x2))) || \
		(abs($a) < 1 && abs($b) < 1 && \
		(($y1 < $y && $y < $y2) || ($y1 > $y && $y > $y2))) || \
		(abs($c) < 1 && abs($d) < 1 && \
		(($x1 < $x && $x < $x2) || ($x1 > $x && $x > $x2)))} {
	    eval $w coords {Selected} [set ret [concat \
		    [lrange $coords 0 [expr $i + 1]] \
		    $x $y [lrange $coords [expr $i + 2] end]]]
	    edit:createHandle $maped $w [$w find withtag {Selected}]
	    set i $max ; # break this loop.
	}
    }	
    return $ret
}

# edit:delVertexFromPolygon maped w x y 
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects - 
#  1) delete vertex from existing polygon item
proc edit:delVertexFromPolygon {maped w} {
    set item {}; set pos {}
    if {[regexp {item:(.*)} [lindex [set tags [$w gettags current]] \
	    [lsearch -glob $tags {item:*}]] dummy item] > 0 && \
	    [scan [lindex $tags [lsearch -glob $tags {pos:*}]] {pos:%d} pos] > 0 &&  \
	    [set len [llength [set coords [$w coords \
	    [$w find withtag {Selected}]]]]] > 6} {
	eval $w coords $item [concat [lrange $coords 0 [expr $pos-1]] \
		[lrange $coords [expr $pos+2] end]]
	edit:createHandle $maped $w $item
    }
    return $item
}

# edit:exec url execstr
# Args - url - URL
#        execstr - string for exec
# Side efect -
#  1) perform subst %F and %U in execstr
#  2) eval execstr
# NOTE: In execstr, we can use following substitution
#    %F - relpace to "File Name"
#    %U - replace to "URL"
proc edit:exec {url execstr} {
    # %U substitution : replace with linked URL
    if {[string match {*%U*} $execstr]} {
	if {![string match {http:*} $url]} {
	    set url "file:[pwd]/$url"
	}
	regsub -all {%U} $execstr $url execstr
    } elseif {[string match {*%F*} $execstr]} {
	# %F substitution : replace with actual file name. 
	set url [split $url {/}]; # break $url to each path items.
	# if url is protocol:/site/path... form,
	if {[string match {*:} [lindex $url 0]]} {
	    # then add serverTopDir
	    if {[set topdir [option get . serverTopDir {}]] != {}} {
		set url [concat [split $topdir {/}] [lrange $url 2 end]]
	    } else {
		set url [lrange $url 2 end]
	    }
	}
	# if $url contain `~' ...  
	if {[set idx [lsearch -glob $url {~*}]] >= 0} {
	    # then treat it to ~username/publicHtmlDir/...
	    set url [join [concat [lindex $url $idx] \
		    [option get . publicHtmlDir {}] \
		    [lrange $url [expr $idx + 1] end]] {/}]
	}
	# if $url is directory, then add welcomeHtml.
	if {[file isdirectory $url]} {
	    if {![string match {*/} $url]} {
		set url "$url/"
	    }
	    set url "$url[option get . welcomeHtml {}]"
	}
	regexp {(.*)#.*} $url dummy url; # strip label part.
	# replace %F with url.
	regsub -all {%F} $execstr $url execstr
    }
    if {[catch [list eval $execstr] msg]} {
	msg:set error $msg; # if error occur then print it.
    }
}

# edit:resizingItem maped w x y
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects -
# 1) move current Handle
# 2) resize/reshape current item
proc edit:resizingItem {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    set selectWidth [option get $w selectWidth {}]; # half of handle width
    $w coords $data(curHandle) \
	    [expr $x - $selectWidth] [expr $y - $selectWidth] \
	    [expr $x + $selectWidth] [expr $y + $selectWidth]
    set coords [$w coords {Selected}]
    if {$data(curPos) != 0} {
	eval $w coords {Selected} [concat \
		[lrange $coords 0 [expr $data(curPos) - 1]] \
		$x $y \
		[lrange $coords [expr $data(curPos) + 2] end]]
    } else {
	if {[$w type {Selected}] == "polygon"} {
	    eval $w coords {Selected} [concat $x $y \
		    [lrange $coords 2 [expr [llength $coords] - 3]] $x $y]
	} else {
	    eval $w coords {Selected} [concat $x $y [lrange $coords 2 end]]
	}
    }
    return
}

# edit:startResize maped w x y
# Args - maped - global data (Array maped)
#        w     - canvas window name
#        x     - x coords for mause event
#        y     - y coords for mause event
# Side effects -
#  1) bind resizing procs to current Handle
#  2) delete binds on the other Handles.
proc edit:startResize {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    set data(curHandle) [$w find withtag current]
    scan [lindex [set tags [$w gettags {current}]] \
	    [lsearch -glob $tags {pos:*}]] {pos:%d} data(curPos)
    $w bind $data(curHandle) <Shift-Button-1> \
	    [list edit:stopResizing $maped %W]
    $w bind {Handle} <Shift-Button-1> {}
    $w bind {Handle} <Enter> {}
    $w bind {Handle} <Leave> {}
    $w bind {Selected} <Enter> {}
    $w bind {Selected} <Leave> {}
    bind $w <Motion> [list edit:resizingItem $maped %W %x %y]
    bind $w <Button-1> {}
    msg:set questhead \
	    "<Motion> : resize/rehape. <Shift-1> : finish."
    return
}

# edit:stopResizing maped w
# Args - maped - global data (Array maped)
#        w     - canvas window name
# Side effects -
# 1) initialize bindings for Handles
proc edit:stopResizing {maped w} {
    upvar #0 $maped data
    if {[$w type [set item [$w find withtag {Selected}]]] == {oval}} {
	set ux [lindex [set coords [$w coords $item]] 0]
	set uy [lindex $coords 1]
	set x [lindex $coords 2]; set y [lindex $coords 3]
	if {[set rx [expr $x - $ux]] > [set ry [expr $y - $uy]]} {
	    $w coords $item [expr $x - $rx] [expr $y - $rx] $x $y
	} elseif {$rx < $ry} {
	    $w coords $item [expr $x - $ry] [expr $y - $ry] $x $y
	}
    }
    $w bind $data(curHandle) <Shift-Button-1> {}
    canvas:initBinds
    edit:unSelectItem $maped $w
    edit:selectItem $maped $w $item
    set data(curHandle) {}; set data(curPos) {}
    msg:del
    return
}

proc edit:doPaste {maped w x y} {
    upvar #0 $maped data
    set x [$w canvasx $x]; set y [$w canvasy $y]
    set url {}; set alt {}
    foreach i [lindex $data(cutBuf) 2] {
	if {[string match {url:*} $i]} {
	    regexp {url:(.*)} $i dummy url
	} elseif {[string match {alt:*} $i]} {
	    regexp {alt:(.*)} $i dummy alt
	}
    }
    if {[set type [lindex $data(cutBuf) 0]] == {oval} || \
	    $type == {rectangle} || $type == {polygon}} {
	set len [llength [set coords [lindex $data(cutBuf) 1]]]
	set xs [lindex $coords 0]; set ys [lindex $coords 1]
	set newcoords [list $x $y]
	for {set i 2} {$i < $len} {incr i 2} {
	    lappend newcoords [expr $x + [lindex $coords $i] - $xs] \
		    [expr $y + [lindex $coords [expr $i + 1]] - $ys]
	}
	set item [eval $w create $type $newcoords]
	$w itemconfigure $item -tags [list "url:$url" "alt:$alt"]
    } else {
	set item [canvas:drawPoint $x $y $url $alt]
    }
    edit:selectItem $maped $w [lindex $item 0]
    canvas:initBinds
    return $item
}    
