# ============================================================================
# canvas.tcl
# procs when exec startp.
# ============================================================================
# 

# canvas:create maped w --
# Args:
#  maped - global data structure (array "maped")
#  w - widget name. defaults is .text
# SideEffects:
#  create radiobuttons and canvas widget.
# Methods:

# ##
proc canvas:create {maped {w .body}} {
    upvar #0 $maped data
    option add *[string trimleft $w*borderWidth .] 0 widgetDefault
    option add *[string trimleft $w*relief .] flat widgetDefault
    option add *[string trimleft $w*highlightThickness .] 0 widgetDefault
    frame $w.bbar
    frame $w.canvas
    pack $w.bbar -side left -fill y
    pack $w.canvas -side right -fill both -expand true

    set iconPath [option get . defaultIconPath {}]
    option add *[string trimleft $w.bbar.select.pixmap .] \
	    "$iconPath/select.ppm" widgetDefault
    option add *[string trimleft $w.bbar.circle.pixmap .] \
	    "$iconPath/circle.ppm" widgetDefault
    option add *[string trimleft $w.bbar.rectangle.pixmap .] \
	    "$iconPath/rectangl.ppm" widgetDefault
    option add *[string trimleft $w.bbar.polygon.pixmap .] \
	    "$iconPath/polygon.ppm" widgetDefault
    option add *[string trimleft $w.bbar.point.pixmap .] \
	    "$iconPath/point.ppm" widgetDefault
    option add *[string trimleft $w.bbar.Radiobutton.borderWidth .] \
	    2 widgetDefault
    option add *[string trimleft $w.bbar.Radiobutton.highlightThickness .] \
	    2 widgetDefault
    option add *[string trimleft $w.bbar.Radiobutton.relief .] \
	    raised widgetDefault
    foreach i {select circle rectangle polygon point} {
	radiobutton $w.bbar.$i -variable [subst $maped](curObj) -value $i \
		-command [list edit:cancelItem $maped $w.canvas.canvas]
	$w.bbar.$i configure -image \
		[image create photo -file [option get $w.bbar.$i pixmap {}]]
	pack $w.bbar.$i -side top
    }
    set edit:mode select

    option add *[string trimleft $w.canvas.canvas.borderWidth .] \
	    2 widgetDefault
    option add *[string trimleft $w.canvas.canvas.relief .] \
	    sunken widgetDefault
    option add *[string trimleft $w.canvas.canvas.pointX .] \
	    4 widgetDefault
    option add *[string trimleft $w.canvas.canvas.pointY .] \
	    4 widgetDefault
    option add *[string trimleft $w.canvas*Scrollbar.borderWidth .] \
	    2 widgetDefault
    option add *[string trimleft $w.canvas*Scrollbar.highlightThickness .] \
	    2 widgetDefault
    option add *[string trimleft $w.canvas*Scrollbar.relief .] \
	    flat widgetDefault
    set canvas:canvasName [canvas $w.canvas.canvas \
	    -xscrollcommand [list $w.canvas.b.x set] \
	    -yscrollcommand [list $w.canvas.y set]]
    frame $w.canvas.b
    scrollbar $w.canvas.b.x -orient horizontal \
	    -command [list $w.canvas.canvas xview]
    scrollbar $w.canvas.y -orient vertical \
	    -command [list $w.canvas.canvas yview]
    set pad [expr [$w.canvas.y cget -width] + 2 * ([$w.canvas.y cget -bd] + \
	    [$w.canvas.y cget -highlightthickness])]
    frame $w.canvas.b.pad -width $pad -height $pad
    pack $w.canvas.b -side bottom -fill x
    pack $w.canvas.b.pad -side right
    pack $w.canvas.b.x -side bottom -fill x
    pack $w.canvas.y -side right -fill y
    pack $w.canvas.canvas -fill both -expand true

    # define usefull (class) methods ^^;---

    # canvas:new -
    # side effects : clear global variable and clear canvas.
    proc canvas:new [list [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	set data(defaultURL) {}
	set data(curName) {}
	set data(curFileName) {}
	edit:unSelectItem $maped $w
	eval $w delete [$w find all]
	canvas:initBinds
    }
    proc canvas:delItems [list [list w $w.canvas.canvas] [list maped $maped]] {
	edit:unSelectItem $maped $w
	foreach i [$w find all] {
	    if {[$w type $i] != {image}} {
		$w delete $i
	    }
	}
	canvas:initBinds
    }	
    # canvas:initBinds -
    # side efects : initialize all binds for canvas window.
    proc canvas:initBinds [list [list w $w.canvas.canvas] \
	    [list maped $maped]] {
	bind $w <Button-1> [list edit:b1Proc $maped %W %x %y]
	bind $w <Shift-Button-1> {# No operation! }
	bind $w <Control-Button-1> {# No operation! }
	bind $w <Double-Button-1> {# No operation! }
	bind $w <Button-3> {# No operation! }
    }
    # canvas:setDefaultURL url
    # args - url - url for default
    # side effects : set url to default attrib.
    proc canvas:setDefaultURL [list url \
	    [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	set data(defaultURL) $url
	if {[set tags [$w gettags {Image}]] != {}} {
	    $w dtag {Image} [lindex $tags [lsearch -glob $tags {url:*}]]
	    $w addtag "url:$url" withtag {Image}
	}
    }
    proc canvas:loadImage [list file type [list w $w.canvas.canvas]] {
	$w create image 0 0 -anchor nw -tags [list url: alt: Image] \
		-image [image create $type -file $file]
	$w lower Image
    }
    proc canvas:saveImage [list file type [list w $w.canvas.canvas]] {
	switch $type {
	    {gif} {
		if {[catch [list [$w itemcget {Image} -image] write $file] msg]} {
		    msg:set error $msg		
		}
	    }
	    {ppm} {
		if {[catch [list [$w itemcget {Image} -image] \
			write $file -format $type] msg]} {
		    msg:set error $msg		
		}
	    }
	}
    }
    proc canvas:adjScrRegion [list [list w $w.canvas.canvas]] {
	$w configure -scrollregion [$w bbox all]
    }
    proc canvas:drawCircle [list x y r {url {}} {alt {}} \
	    [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	set color [option get $w itemColor {}]
	set width [option get $w itemWidth {}]
	set item [$w create oval [expr $x - $r] [expr $y - $r] \
		[expr $x + $r] [expr $y + $r] -outline $color -width $width \
		-tags [list url:$url alt:$alt]]
	if $data(fillFlag) {
	    $w itemconfigure $item -fill [option get $w fillColor {}] \
		    -stipple [option get $w stipple {}]
	}
	return $item
    }
    proc canvas:drawRectangle [list x1 y1 x2 y2 {url {}} {alt {}} \
	    [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	set color [option get $w itemColor {}]
	set width [option get $w itemWidth {}]

	set item [$w create rectangle $x1 $y1 $x2 $y2 -outline $color \
		-width $width -tags [list url:$url alt:$alt]] 
	if $data(fillFlag) {
	    $w itemconfigure $item -fill [option get $w fillColor {}] \
		    -stipple [option get $w stipple {}]
	}
	return $item
    }
    proc canvas:drawPolygon [list argList {url {}} {alt {}} \
	    [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	set color [option get $w itemColor {}]
	set width [option get $w itemWidth {}]

	lappend argList -fill {} -outline $color -width $width \
		-tags [list url:$url alt:$alt]
	set item [eval $w create polygon $argList]
	if $data(fillFlag) {
	    $w itemconfigure $item -fill [option get $w fillColor {}] \
		    -stipple [option get $w stipple {}]
	}
	return $item
    }
    proc canvas:drawPoint [list x y {url {}} {alt {}} \
	    [list w $w.canvas.canvas]] {
	set color [option get $w itemColor {}]
	set width [option get $w itemWidth {}]
	set xd [option get $w pointX {}]
	set yd [option get $w pointY {}]

	set id1 [$w create line [expr $x - $xd] [expr $y - $yd] \
		[expr $x + $xd] [expr $y + $yd] -fill $color -width $width]
	set id2 [$w create line [expr $x + $xd] [expr $y - $yd] \
		[expr $x - $xd] [expr $y + $yd] -fill $color -width $width]
	$w itemconfigure $id1 -tags \
		[list url:$url alt:$alt id:$id2 Point]
	$w itemconfigure $id2 -tags \
		[list url:$url alt:$alt id:$id1]
	return [list $id1 $id2]
    }
    proc canvas:writeFileBody [list maped fp \
	    [list w $w.canvas.canvas]] {
	upvar #0 $maped data
	if {$data(DEBUG)} {puts stdout "canvas:writeFileBody"}
	if {$data(curFormat) != {Client Side Map}} {
	    if {$data(DEBUG)} {puts stdout [canvas:writeDefault $maped]}
	    puts $fp [canvas:writeDefault $maped]
	}
	for {set i [expr [llength [set all [$w find all]]] - 1]} \
		{$i > -1} {incr i -1} {
	    set item [lindex $all $i]
	    if {[$w type $item] != {image} && \
		    ![regexp {Editting} [set tags [$w gettags $item]] tags] && \
		    ![regexp {Handle} $tags tags]} {
		set str {}
		switch [$w type $item] {
		    {oval} {set str [canvas:writeCircle $maped $item]}
		    {rectangle} {set str [canvas:writeRectangle $maped $item]}
		    {polygon} {set str [canvas:writePolygon $maped $item]}
		    {line} {
			if {[lsearch $tags Point] != -1} {
			    set str [canvas:writePoint $maped $item]
			}
		    }
		}
		if {$str != {}} {
		    if {$data(curFormat) == {Client Side Map}} {
			if {$data(DEBUG)} {puts stdout "write : <[join $str]>"}
			puts $fp "<[join $str]>"
		    } else {
			if {$data(DEBUG)} {puts stdout "write : [join $str]"}
			puts $fp [join $str]
		    }
		}
	    }
	}
    }
    proc canvas:writeCircle [list maped item \
	    [list w $w.canvas.canvas]] {
	upvar #0 $maped data
	set writeStr {}; set url \{\}; set alt \{\}
	set coords [$w coords $item]
	set cx [expr int(([lindex $coords 2] + [lindex $coords 0]) / 2)]
	set cy [expr int(([lindex $coords 3] + [lindex $coords 1]) / 2)]
	set r [expr int($cx - [lindex $coords 0])]
	regexp {url:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {url:*}]] dummy url
	regexp {alt:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {alt:*}]] dummy alt
	switch $data(curFormat) {
	    {CERN Httpd} {
		set writeStr [list circle "($cx,$cy)" $r $url]
	    }
	    {NCSA Httpd} {
		set writeStr [list circle $url "$cx,$cy" $r]
	    }
	    {NetSite} {
		set writeStr [list circle $url $cx $cy [expr $cx + $r] $cy]
	    }
	    {Client Side Map} {
		set writeStr [list {AREA} {SHAPE=CIRCLE} \
			"COORDS=\"$cx,$cy,$r\""]
		if {$url != {}} {
		    lappend writeStr "HREF=\"$url\""
		} else {
		    lappend writeStr NOHREF
		}
		if {$alt != {}} {
		    lappend writeStr "ALT=\"$alt\""
		}
	    }
	}
	return $writeStr
    }
    proc canvas:writeRectangle [list maped item \
	    [list w $w.canvas.canvas]] {
	upvar #0 $maped data
	set writeStr {}; set url \{\}; set alt \{\}
	set x1 [expr int([lindex [set coords [$w coords $item]] 0])]
	set y1 [expr int([lindex $coords 1])]
	set x2 [expr int([lindex $coords 2])]
	set y2 [expr int([lindex $coords 3])]
	regexp {url:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {url:*}]] dummy url
	regexp {alt:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {alt:*}]] dummy alt
	switch -glob $data(curFormat) {
	    {CERN Httpd} {
		set writeStr [list rectangle "($x1,$y1)" "($x2,$y2)" $url]
	    }
	    {NCSA Httpd} {
		set writeStr [list rect $url "$x1,$y1" "$x2,$y2"]
	    }
	    {NetSite} {
		set writeStr [list rect $url $x1 $y1 $x2 $y2]
	    }
	    {Client Side Map} {
		set writeStr [list {AREA} {SHAPE=RECT} \
			"COORDS=\"$x1,$y1,$x2,$y2\""]
		if {$url != {}} {
		    lappend writeStr "HREF=\"$url\""
		} else {
		    lappend writeStr NOHREF
		}
		if {$alt != {}} {
		    lappend writeStr "ALT=\"$alt\""
		}
	    }
	}
	return $writeStr
    }
    proc canvas:writePolygon [list maped item \
	    [list w $w.canvas.canvas]] {
	upvar #0 $maped data
	set writeStr {}; set url \{\}; set alt \{\}
	foreach i [lrange [set coords [$w coords $item]] 0 \
		[set len [expr [llength $coords] - 2]]] {
	    lappend tmp [expr int($i)]
	}
	set coords $tmp
	regexp {url:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {url:*}]] dummy url
	regexp {alt:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {alt:*}]] dummy alt
	switch $data(curFormat) {
	    {CERN Httpd} {
		set writeStr {polygon}
		for {set i 0} {$i < $len} {incr i 2} {
		    lappend writeStr "([lindex $coords $i],[lindex $coords [expr $i + 1]])"
		}
		lappend writeStr $url
	    }
	    {NCSA Httpd} {
		set writeStr [list poly $url]
		for {set i 0} {$i < $len} {incr i 2} {
		    lappend writeStr "[lindex $coords $i],[lindex $coords [expr $i + 1]]"
		}
	    }
	    {NetSite} {
		set writeStr [concat [list poly $url] \
			[lrange $coords 0 [expr $len - 1]]]
	    }
	    {Client Side Map} {
		set writeStr [list {AREA} {SHAPE=POLYGON} \
			"COORDS=\"[join [lrange $coords 0 [expr $len - 1]] ,]\""]
		if {$url != {}} {
		    lappend writeStr "HREF=\"$url\""
		} else {
		    lappend writeStr NOHREF
		}
		if {$alt != {}} {
		    lappend writeStr "ALT=\"$alt\""
		}
	    }
	}
	return $writeStr
    }
    proc canvas:writePoint [list maped item \
	    [list w $w.canvas.canvas]] {
	upvar #0 $maped data
	set writeStr {}; set url \{\}; set alt \{\}
	set x [expr int(([lindex [set coords [$w coords $item]] 0] + \
		[lindex $coords 2]) / 2)]
	set y [expr int(([lindex $coords 1] + [lindex $coords 3]) / 2)]
	regexp {url:(.*)} [lindex [set tags [$w gettags $item]] \
		[lsearch -glob $tags {url:*}]] dummy url
	switch $data(curFormat) {
	    {CERN Httpd} {
		set writeStr [list point "($x,$y)" $url]
	    }
	    {NCSA Httpd} {
		set writeStr [list point $url "$x,$y"]
	    }
	}
	return $writeStr
    }
    proc canvas:writeDefault [list maped \
	    [list w $w.canvas.canvas]] {
	upvar #0 $maped data
	set writeStr {}
	switch $data(curFormat) {
	    {CERN Httpd} -
	    {NCSA Httpd} -
	    {NetSite} {
		set writeStr [list default $data(defaultURL)]
	    }
	}
	return $writeStr
    }
    proc canvas:delete [list [list w $w.canvas.canvas] [list maped $maped]] {
	if {[set item [$w find withtag {Selected}]] != {}} {
	    edit:unSelectItem $maped $w; # un-select selected item. 
	    if {[$w type $item] == {image}} {
		catch [list image delete [$w itemcget $item -image]]
	    }
	    foreach i $item {
		$w delete $item; # delete item
	    }
	}
	return $item
    }	
    proc canvas:cut [list [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	set item [canvas:copy]
	edit:unSelectItem $maped $w; # un-select selected item. 
	foreach i $item {
	    $w delete $i
	}
	return $item
    }	    
    proc canvas:copy [list [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	if {[set item [$w find withtag {Selected}]] != {} && \
		[set type [$w type $item]] != {image}} {
	    foreach i $item {
		set tags [$w gettags $i]
		if {($type == {line} && [lsearch $tags {Point}] != -1) || \
			$type != {line}} {
		    set newtags {}
		    foreach j $tags {
			if {[string match {url:*} $j] || \
				[string match {alt:*} $j]} {
			    lappend newtags $j
			}
		    }
		}
	    }
	    set data(cutBuf) [list [$w type [lindex $item 0]] \
		    [$w coords [lindex $item 0]] $newtags]
	    return $item
	} else {
	    return {}
	}
    }	    
    proc canvas:paste [list [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	if {$data(cutBuf) != {}} {
	    edit:unSelectItem $maped $w
	    msg:set questhead {<1> : paste item.}
	    bind $w <Button-1> [list edit:doPaste $maped %W %x %y]
	}
    }
    proc canvas:fore [list [list w $w.canvas.canvas]] {
	$w raise {Selected}
	$w raise {Handle}
	$w lower {Image}
    }
    proc canvas:back [list [list w $w.canvas.canvas]] {
	$w lower {Handle}
	$w lower {Selected}
	$w lower {Image}
    }
    proc canvas:raise [list [list w $w.canvas.canvas] [list maped $maped]] {
	edit:unSelectItem $maped $w; # for safty ^^;
	$w raise {Image}
    }
    proc canvas:lower [list [list TagOrID Image] [list w $w.canvas.canvas]] {
	$w lower $TagOrID
    }
    proc canvas:fillItems [list [list w $w.canvas.canvas]] {
	set color [option get $w fillColor {}]
	set selcol [option get $w selectColor {}]
	set stipple [option get $w stipple {}]
	set selstip [option get $w selectStipple {}]
	foreach i [$w find all] {
	    if {[lsearch [set tags [$w gettags $i]] {Handle}] == -1 && \
		    ([set type [$w type $i]] == {oval} || \
		    $type == {rectangle} || $type == {polygon})} {
		if {[lsearch $tags {Selected}] >= 0} {
		    $w itemconfigure $i -fill $selcol -stipple $selstip
		} else {
		    $w itemconfigure $i -fill $color -stipple $stipple
		}
	    }
	}
    }
    proc canvas:unFillItems [list [list w $w.canvas.canvas]] {
	foreach i [$w find all] {
	    if {[set type [$w type $i]] == {oval} || \
		    $type == {rectangle} || $type == {polygon}} {
		$w itemconfigure $i -fill {} -stipple {}
	    }
	}
    }
    proc canvas:checkPoints [list mx my url \
	    [list w $w.canvas.canvas]] {
	if {[set points [$w find withtag Point]] != ""} {
	    set coords [$w coords [lindex $points 0]]
	    set x [expr ([lindex $coords 0] + [lindex $coords 2]) / 2]
	    set y [expr ([lindex $coords 1] + [lindex $coords 3]) / 2]
	    set dist [expr hypot($x - $mx, $y - $my)]
	    set url [lindex [set tags [$w gettags [lindex $points 0]]] \
		    [lsearch -glob $tags {url:*}]]
	    foreach i [lrange $points 1 end] {
		set coords [$w coords $i]
		set x [expr ([lindex $coords 0] + [lindex $coords 2]) / 2]
		set y [expr ([lindex $coords 1] + [lindex $coords 3]) / 2]
		set tmpdist [expr hypot($x - $mx, $y - $my)]
		if {$tmpdist < $dist} {
		    set dist $tmpdist
		    set url [lindex [set tags [$w gettags $i]] \
			    [lsearch -glob $tags {url:*}]]
		}
	    }
	}
	return $url
    }
    proc canvas:doPrint [list [list w $w.canvas.canvas] [list maped $maped]] {
	upvar #0 $maped data
	option add *[string trimleft $w*ovalURLAnchor .] nw widgetDefault
	option add *[string trimleft $w*rectangleURLAnchor .] nw widgetDefault
	option add *[string trimleft $w*polygonURLAnchor .] nw widgetDefault
	option add *[string trimleft $w*pointURLAnchor .] n widgetDefault
	option add *[string trimleft $w*defaultURLAnchor .] nw widgetDefault

	foreach i [$w find all] {
	    if {[set url [lindex [set tags [$w gettag $i]] \
		    [lsearch -glob $tags {url:*}]]] != {}} {
		if {[regexp {url:(.*)} $url dummy url] != 1} {set url {}}
		set coords [$w coords $i]
		switch -regexp [set t [$w type $i]] {
		    {oval|rectangle|polygon} {
			$w create text [lindex $coords 0] [lindex $coords 1] \
				-anchor [option get $w [set t]URLAnchor {}] \
				-text $url -tags {TEXT}
		    }
		    {line} {
			if {[lsearch [$w gettags $i] Point] != -1} {
			    $w create text $x $y -text $url -anchor \
				    [option get $w pointURLAnchor {}] \
				    -tags {TEXT}
			}
		    }
		    {image} {
			$w create text 0 0 -text $url -anchor \
				[option get $w defaultURLAnchor {}] \
				-tags {TEXT}
		    }
		}
	    }
	}

	set cmdArgs [list -colormode $data(PS.Color) \
		-width [lindex [set coords [$w bbox all]] 2] \
		-height [lindex $coords 3]]
	if {$data(PS.Orientation) == {landscape}} {
	    set cmdArgs [concat $cmdArgs -rotate true]
	}
	switch $data(PS.Size) {
	    {letter} {
		set cmdArgs [concat $cmdArgs -pageheight 11i -pagewidth 8.5i]
	    }
	    {legal} {
		set cmdArgs [concat $cmdArgs -pageheight 14i -pagewidth 8.5i]
	    }
	    {a4} {
		set cmdArgs [concat $cmdArgs -pageheight 297m -pagewidth 210m]
	    }
	}
	if {$data(PS.Distination) == {printer}} {
	    if {[catch [list open "|$data(PS.cmdStr)" w] fp]} {
		msg:set error $fp
	    }
	    if {[catch [list puts $fp [eval $w postscript $cmdArgs]] msg]} {
		msg:set error $msg
	    }
	    if {[catch [list close $fp] msg]} {
		msg:set error $msg
	    }
	} else {
	    if {$data(DEBUG)} {puts stdout "$w postscript \
		    -file $data(PS.PostScript) $cmdArgs"}
	    if {[catch [eval $w postscript -file $data(PS.PostScript) \
		    $cmdArgs] msg]} {
		msg:set error $msg
	    }
	}
	$w delete {TEXT}
    }
    canvas:initBinds; # initialize canvas bindings.
    return ${canvas:canvasName}
}