/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

#define MAINPROGRAM

/* #include <stdlib.h> */
#include <stdio.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <sys/un.h>

#include <errno.h>
#include <signal.h>
#include <tcl.h>
#include <tclInt.h>

#include "ot.h"
#include "otInt.h"

#define	mask(s)	(1 << ((s) - 1))

typedef struct {
    OTTemplate    *otc_base;	/* base template - contains field, list info */
    long	   otc_top;	/* highest CR number			     */
    OTTemplate    *otc_arrayp;	/* pointer to actual CRs                     */
    int            otc_initBrk; /* initial 'break' value		     */
    int	           otc_subsBrk; /* break value after initial CR load	     */
} OTArray;

int daemon();
void cleanup(), reapchild(), otKwikKwery(), otFreeKwikTemplate();
OTErr otBuildArray(), otLoadArray(), otAllocHeaderHist(), otFreeArray(),
    otKwikService(), otSlowService();

OTArray *arrayp;
int	 selectedCount;
char   **selectedKwd;
int     *numlist;
int stdoutFd, stderrFd;


bool slowPix = FALSE;		/* if TRUE, run as a filter, not as a server */

char socketName[MAXPATHLEN];
int funix;

/*
 * otKwikPix -p <projName>
 */
int main(argc,argv)
int argc;
char *argv[];
{
    OTErr err;
    OTProject *prjp;
    char logFile[NAMELEN];
    char *cp;
    uid_t uid;
    int omask;
    long crCount;
    struct sockaddr_un sockUnix;

    if ( cp = strstr(argv[0], "otSlowPix") )
	slowPix = TRUE;

    if ( err = otInitialInit(&argc, argv) )
	exit(1);

    sprintf(logFile, "%s/serverV3.log", BASE);
    sprintf(otCB->cb_pcb->pcb_logPid, "%ld", (long) getpid());
    if ((otCB->cb_pcb->pcb_efp = fopen(logFile, "a")) == NULL) {
	fprintf(stderr, "can't append to log file (%s)\n", logFile);
	exit(1);
    }
    chmod(logFile, 0666);

    otCB->cb_pcb->pcb_imTheServer = TRUE;
    if ( err = otReadProject( otCB->cb_project ) ) {
	logWarn("otKwikPix, in otReadProject(): %s", otGetPostedMessage());
	exit(1);
    }

    chdir(otCB->cb_pcb->pcb_project->proj_dir);
    /*
     * Parse -p argument.
     */
    if (err = otParseQryArgsCLI(argc,argv)) {
	logWarn("otKwikPix, in otParseArgsCLI(): %s", otGetPostedMessage());
	exit(1);
    }

    prjp = otCB->cb_pcb->pcb_project;

    if ( err = otReadTopNumber(prjp, &crCount) ) {
	logWarn("otKwikPix, in otReadTopNumber(): %s", otGetPostedMessage());
	exit(1);
    } else
	otCB->cb_pcb->pcb_topNumber = crCount;

    if ( slowPix ) {

        crCount = 1;
	if ( err = otBuildArray( &arrayp, crCount ) ) {
	    logWarn("otKwikPix, in otBuildArray(): %s", otGetPostedMessage());
	    exit(1);
	}

	otSlowService();

    } else {
	/*
	 * This is an invocation of otKwikPix as storing server.
	 * otKwikPix must be running with root user id.
         */
	if ( uid = geteuid() ) {
	    logWarn("otKwikPix, must be superuser: currently %ld", uid);
	    exit(1);
	}

	stdoutFd = dup(1);
	stderrFd = dup(2);
 
	daemon(0,1);

	if ( err = otBuildArray( &arrayp, crCount ) ) {
	    logWarn("otKwikPix, in otBuildArray(): %s", otGetPostedMessage());
	    exit(1);
	}
	if ( err = otLoadArray( arrayp ) ) {
	    logWarn("otKwikPix, error in loading: %s", otGetPostedMessage());
	    exit(1);
	}

	signal(SIGCHLD, reapchild);
	if ( (funix = socket(AF_UNIX, SOCK_STREAM, 0)) < 0 ) {
	    logWarn("otKwikPix, error in creating socket: %s\n",
		strerror(errno));
	    exit(1);
	}
	omask = sigblock(mask(SIGHUP) | mask(SIGINT) | mask(SIGQUIT) |
	    mask(SIGTERM));
	signal(SIGHUP, cleanup);
	signal(SIGINT, cleanup);
	signal(SIGQUIT, cleanup);
	signal(SIGTERM, cleanup);

	sprintf(sockUnix.sun_path, "%s/%s/socket", prjp->proj_dir, prjp->name);
	strcpy(socketName, sockUnix.sun_path);
	sockUnix.sun_family = AF_UNIX;

	if ( bind(funix, (struct sockaddr *)&sockUnix,
	    strlen(sockUnix.sun_path) + sizeof(sockUnix.sun_family)) < 0 ) {

	    logWarn("otKwikPix, error in bind: %s\n", strerror(errno));
	    exit(1);
	}

	sigsetmask(omask);
	otKwikService();
    }

}



/*
 * otKwikService() reads commands from a UNIX socket and returns data to the
 * same socket.
 *
 * Commands recognized by the otKwikPix server:
 *
 * Commands processed by parent:
 *   - delta  n
 *	  
 *   - delete n
 *	  
 * Commands processed by child processes:
 *   - ot_bugs argList
 *	   where argList is packaged as a TCL list!
 *
 */
OTErr 
otKwikService()
{
    OTErr err = OT_SUCCESS;
    OTTemplate *templates, *tp;
    long crNum;
    char *cp;
    char *headline, *histline;
    OTProject *prjp;
    OTNotedata *ndp;
    int defreadfds, readfds, nfds, domain, s, fromlen, n;
    struct sockaddr_un fromunix;
    char buf[100000];

    templates = arrayp->otc_arrayp;

    defreadfds = 1 << funix;
    if ( listen(funix, 5) < 0 ) {
        logWarn("otKwikPix: %s\n", strerror(errno));
	exit(1);
    }

    for (;;) {

	s = -1;
	readfds = defreadfds;
	nfds = select(20, &readfds, 0, 0, 0);
	if (nfds <= 0)
	    continue;
	if ( readfds & (1 << funix) ) {
	    domain = AF_UNIX;
	    fromlen = sizeof(fromunix);
	    do {
		errno = 0;
		s = accept(funix, (struct sockaddr *)&fromunix, &fromlen);
	    } while ( s < 0 || errno == EINTR );
	}
	if (s < 0)
	    continue;

	if ( (n = read(s, buf, sizeof(buf)) ) == 0 )
	    break;
	cp = buf;
	buf[n] = 0;

	/*
	 * Acknowledge and terminate connection.
	 */

	/*
	 * The commands handled by the parent otKwikPix are
	 *
	 * - delta\nheaderline\nhistline
	 *    update the CR with the info in headerline and histline
	 * - exit
	 *    brings down parent
	 */

	if        ( !strncmp(cp, "delta",  5) ) {
	    write(s, "\n", 1);
	    (void) close(s);

	    for ( cp += 5; *cp && *cp == ' '; cp++);
	    sscanf(cp, "%ld", &crNum);

	    while ( *cp && *cp != '\n' )
		cp++;
	    cp++;
	    while ( *cp && isdigit(*cp) )
		cp++;
	    while ( *cp && isspace(*cp) )
		cp++;

	    headline = cp;
	    while ( *cp != '\n' )
	        cp++;
	    *cp = 0;
	    cp++;

	    while ( *cp && isdigit(*cp) )
		cp++;
	    while ( *cp && isspace(*cp) )
		cp++;
	    
	    histline = cp;
	    while ( *cp && *cp != '\n' )
		cp++;
	    *cp = 0;

	    
	    if ( err = otAllocHeaderHist(arrayp, crNum, headline, 0) ) {
	        logWarn("Error in allocating header and notedata info for cr %d (|%s|%s|): %s\n",
		    crNum, headline, histline, otGetPostedMessage());
		exit(1);
	    }

	    tp = &(templates[crNum]);
	    if ( err = otAppendHeaderFieldToTemplateFromKwikString(tp,
		headline) ) {
		logWarn("Error in reading kwik string (%s) - %s\n", headline,
		    otGetPostedMessage());
		exit(1);
	    }
	    continue;
	    
	} else if ( !strncmp(cp, "exit",   4) ) {

	    (void) close(s);
	    exit(0);

#ifdef MULTIPLE_KWIKS
	} else if ( ! fork() ) {
	    otKwikKwery(s, cp);
	} else
	    (void) close(s);
#else
	} else 
	    otKwikKwery(s, cp);
#endif

    }


}


/*
 */
OTErr otBuildArray(arrayp, crCount)
OTArray **arrayp;
long crCount;
{
    OTErr err;
    unsigned long templateBytes, headerFieldBytes;
    OTPrivateCB *pvp   = otCB->cb_pcb;
    OTEnterCB   *enp   = otCB->cb_ecb;
    OTProject   *prjp  = pvp->pcb_project;
    unsigned long headerBytes, totalBytes;
    register OTTemplate *tmptp;
    register OTHeaderField *tmphfp, *baseHfp, *fieldHfp;
    char hdrDbfile[PATHLEN];
    struct stat statbuf;
#ifndef NORLIMIT
    struct rlimit rl;
#endif
    int status;
    register int i;

    if ( !(*arrayp = calloc(1, sizeof(OTArray))) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "otBuildArray()");
	return OT_MALLOC_LOCATION;
    }

    DBUG_MED((stderr, "%X is sbrk() before template alloc\n", sbrk(0)));
    (*arrayp)->otc_initBrk = (int)sbrk(0);
    (*arrayp)->otc_top = crCount;

    if ( err = otInitTemplate() )
	return err;
    (*arrayp)->otc_base = enp->ecb_blankStruct;
    enp->ecb_blankStruct = 0;

    DBUG_MED((stderr, "%d is size of OT Template\n", sizeof(OTTemplate)));
    DBUG_MED((stderr, "%d is size of OTHeaderField\n", sizeof(OTHeaderField)));
    DBUG_MED((stderr, "%d is crCount, %d is MAXTEMPLATE\n", crCount, MAXTEMPLATE));

    if ( !slowPix ) {
        if ( crCount < 1000 )
	    crCount = 1000;	
	/*
	 * Set the memory limits for this process.
	 */
	templateBytes = crCount * sizeof(OTTemplate);
	headerFieldBytes = crCount * (unsigned long)MAXTEMPLATE *
	    sizeof(OTHeaderField);

	sprintf(hdrDbfile, "%s/%s/%s", prjp->proj_dir, otCB->cb_project,
	    HEADERDB);
	if ( stat(hdrDbfile, &statbuf) ) {
	    headerBytes = 0;
	    /*
	    otPutPostedMessage(OT_STAT, hdrDbfile);
	    return OT_STAT;
	     */
	} else
	    headerBytes = statbuf.st_size;

	totalBytes = templateBytes + headerFieldBytes + headerBytes;
	logWarn("temp %ld hFld %ld hdrs %ld tota %ld\n",
	    templateBytes, headerFieldBytes, headerBytes, totalBytes);

#ifndef NORLIMIT
	if ( status = getrlimit(RLIMIT_DATA, &rl) ) {
	    otPutPostedMessage(OT_SETRLIMIT);
	    return OT_SETRLIMIT;
	}

	logWarn("limit DATA - cur: %ld - max: %ld\n", rl.rlim_cur,
	    rl.rlim_max);
	if ( rl.rlim_cur < (2 * totalBytes) ) {
	    logWarn("rl.rlim_cur < totalBytes\n");
	    rl.rlim_cur = (totalBytes * 2) > rl.rlim_max ? 
		rl.rlim_max : (totalBytes * 2);
	    if ( (status = setrlimit(RLIMIT_DATA, &rl)) < 0 ) {
		otPutPostedMessage(OT_SETRLIMIT);
		return OT_SETRLIMIT;
	    }
	}
#endif

    }

    /*
     * Set up arrays of templates, header fields, and associate the two.
     */
    if ( !((*arrayp)->otc_arrayp = calloc(crCount + 1, sizeof(OTTemplate))) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "otBuildArray()");
	return OT_MALLOC_LOCATION;
    }
    return err;

}


OTErr otFreeArray(arrayp)
OTArray *arrayp;
{
    OTTemplate *templates;
    OTHeaderField *hfp;
    int i;

    otFreeTemplate( arrayp->otc_base );
    templates = arrayp->otc_arrayp;
    for( i = arrayp->otc_top; i >= 0; i-- ) {
	for (hfp = templates[i].tr; hfp && hfp->field && *hfp->field; hfp++) {
	    if (hfp->value)
		free(hfp->value);
	}
	if (templates[i].tr)
	    free(templates[i].tr);
    }
    free(arrayp->otc_arrayp);
    otFreeTemplate(arrayp->otc_base);

    return OT_SUCCESS;
}

void cleanup()
{
  unlink(socketName);
  exit(0);
}

void reapchild()
{
  /* union wait status; */	/* Not needed any more */
  int stat_loc;
  pid_t i;

  while ((i = waitpid(-1, &stat_loc, WNOHANG)) > 0);

}

int daemon(nochdir, noclose)
int nochdir, noclose;
{
    int cpid;

    if ((cpid = fork()) == -1)
	return (-1);
    if (cpid)
	exit(0);
    (void) setsid();
    if (!nochdir)
	(void) chdir(BASE);
    
    if (!noclose) {
	int devnull = open("/dev/null", O_RDWR, 0);

	if (devnull != -1) {
	    (void) dup2(devnull, 0);
	    (void) dup2(devnull, 1);
	    (void) dup2(devnull, 2);
	    if (devnull > 2)
		(void) close(devnull);
	}
    }
    return (0);

}


/*
 * This is only called by otKwikPix (not by otSlowPix).
 */
OTErr 
otLoadArray(arrayp)
OTArray *arrayp;
{
    OTTemplate *templates;
    OTEnterCB *enp = otCB->cb_ecb;
    OTPrivateCB *pvp = otCB->cb_pcb;
    OTProject *prjp = pvp->pcb_project;
    char hdrDbfile[PATHLEN], histDbfile[PATHLEN];
    FILE *hdrfp;
    char *hdrKwikp, *tmphdr;
    OTErr err, hdrErr, histErr;
    char *termChar = "\n";
    long hdrCrIdx;
    Tcl_Interp *interp = otCB->cb_pcb->pcb_interp;
    int result, i;

    hdrfp = 0;

    /*
     * Load array.
     */
    templates = arrayp->otc_arrayp;
    if ( !*prjp->name ) {
	logWarn("otKwikPix: Project without name\n");
	exit(1);
    }

    sprintf(hdrDbfile, "%s/%s/%s", prjp->proj_dir, prjp->name, HEADERDB);
    if ( (hdrfp = fopen(hdrDbfile, "r")) == NULL ) {
	logWarn("Error opening %s\n", hdrDbfile);
	return;
	/* exit(1); */
    }

    otCB->cb_operation = QUERY;
    pvp->pcb_template = 0;
    enp->ecb_blankStruct = 0;

    while ( 1 ) {

        hdrKwikp = 0;
	if ( (hdrErr = otReadStringFromFileHandle(&hdrKwikp, hdrfp, termChar,
	    NULL, FALSE)) == OT_TRANSMISSION_CONCLUDED )
	    break;
	sscanf(hdrKwikp, "%ld", &hdrCrIdx);

	tmphdr = hdrKwikp;
	tmphdr += strspn(tmphdr, "0123456789");
	tmphdr += strspn(tmphdr, " ");

	if ( err = otAllocHeaderHist(arrayp, hdrCrIdx, tmphdr, 0) ) {
	    logWarn("Error in allocating header and notedata info for cr %d (%s): %s\n",
		hdrCrIdx, tmphdr, otGetPostedMessage());
	    exit(1);
	}

	if ( (hdrCrIdx % 200) == 0 )
	    logWarn("otKwikPix loading %s CR %ld\n", prjp->name, hdrCrIdx);

	if ( tmphdr && *tmphdr ) {
	    err = otAppendHeaderFieldToTemplateFromKwikString(
		&(templates[hdrCrIdx]), tmphdr);
	}

	if ( hdrKwikp )
	    free( hdrKwikp );
	hdrKwikp = 0;
	if ( err )
	    break;
    }

    if (err) {
	logWarn("otKwikPix, error in loading: %s\n", otGetPostedMessage());
	exit(1);
    }

    arrayp->otc_subsBrk = (int)sbrk(0);
    DBUG_MED((stderr, "%X is sbrk() after template alloc\n", sbrk(0)));

    if (hdrfp)
	fclose(hdrfp);

    logWarn("Change in break during load = %ld\n",
	arrayp->otc_subsBrk - arrayp->otc_initBrk);
    return err;

}


/*
 */
OTErr otAllocHeaderHist(arrayp, hdrCrIdx, tmphdr, tmphist)
OTArray *arrayp;
long hdrCrIdx;
char *tmphdr;
char *tmphist;
{
    OTTemplate *templates, *tp, *basetp;
    register OTMetaTemplate *mtp;
    register OTHeaderField *hfp, *basehfp;
    register OTNotedata *tndp;
    int mtFldCount, hdrFldCount, histFldCount, i, len;
    register char *cp;
    OTErr err = OT_SUCCESS;

    hdrFldCount = histFldCount = 0;
    basetp = arrayp->otc_base;
    /*
     * If there are header fields in templates[hdrCrIdx] free the memory.
     * If there are notedata in templates[hdrCrIdx] free the memory.
     * Count the number of header fields and allocate num + 1 (null-filled).
     * Count the number of notedata fields and allocate num + 1 (null-filled).
     */
    
    templates = arrayp->otc_arrayp;


    if ( hdrCrIdx > arrayp->otc_top ) {

	/*
	 * Extend the template array in the case of an enter operation.
	 */
	if (!(templates = realloc(templates,
	    (hdrCrIdx + 1) * sizeof(OTTemplate)))) {
	    otPutPostedMessage( OT_MALLOC_LOCATION,
		"otAllocHeaderHist() - enter" );
	    return OT_MALLOC_LOCATION;
	}

	for ( i = arrayp->otc_top + 1; i <= hdrCrIdx; i++ )
	    memset( &(templates[i]), 0, sizeof(OTTemplate));

	arrayp->otc_top = hdrCrIdx;
    }

    if ( templates[hdrCrIdx].tr )
	otFreeKwikTemplate( &(templates[hdrCrIdx]) );

    tp = &(templates[hdrCrIdx]);

    hdrFldCount = histFldCount = 0;
    for( cp = tmphdr; cp && *cp && *cp != OTHEADERSEP; cp++ )
        if ( *cp == OTFIELDHEADERSEP )
	    hdrFldCount++; 	

    /*
     * Added check in case the metatemplate is changed and the headdb,
     * histdb files are not updated.
     */
    mtp = otCB->cb_pcb->pcb_project->otmp;
    mtFldCount = 0;
    for ( ; *mtp->field ; mtp++ ) 
	if ( *mtp->field != '!' )
	    mtFldCount++;

    if ( hdrFldCount && (hdrFldCount != mtFldCount) ) {
	otPutPostedMessage(OT_INTERNAL_ERROR, "db files (out of sync with metatemplate)");
	return OT_INTERNAL_ERROR;
    }

    for( ; cp && *cp && *cp != '\n'; cp++ )
	if ( *cp == OTNEWLINE )
	    hdrFldCount++;

    if ( slowPix ) {
        /*
	 * The otSlowPix utility reads in all the history information.
	 */
	for( cp = tmphist; cp && *cp && *cp != '\n'; cp++ )
	    if ( *cp == OTHEADERSEP )
		histFldCount++;
    } 

    if (hdrFldCount) {
	hdrFldCount++;

	DBUG_MED((stderr, "%ld: hdrFldCount %d\n", hdrCrIdx, hdrFldCount));
	DBUG_MED((stderr, "%ld: histFldCount %d\n", hdrCrIdx, histFldCount));

	if ( !(tp->tr = calloc( hdrFldCount, sizeof(OTHeaderField) )) ) {
	    logWarn("Header malloc failure on startup: %d - %d\n",
		hdrCrIdx, hdrFldCount);
	    otPutPostedMessage( OT_MALLOC_LOCATION, "otAllocHeaderHist()" );
	    return OT_MALLOC_LOCATION;
	} else {
	    basehfp = basetp->tr;
	    hfp = tp->tr;

	    for ( ; *basehfp->field; basehfp++, hfp++ ) {
		hfp->field = basehfp->field;
		hfp->list = basehfp->list;
		hfp->type = basehfp->type;
	    }
	}
    }

    if (histFldCount) {
	histFldCount++;

	if ( !(tp->ndp = calloc( histFldCount, sizeof(OTNotedata) )) ) {
	    logWarn("History malloc failure on startup: %d - %d\n",
		hdrCrIdx, histFldCount);
	    otPutPostedMessage( OT_MALLOC_LOCATION, "otAllocHeaderHist()" );
	    return OT_MALLOC_LOCATION;
	}
    }

    return err;
}



void otKwikKwery(s, qry)
int s;
char *qry;
{
    OTErr err;
    OTPrivateCB *pvp = otCB->cb_pcb;
    OTQueryCB *qcb = otCB->cb_qcb;
    OTProject *prjp = pvp->pcb_project;
    Tcl_Interp *interp = pvp->pcb_interp;
    OTTemplate *templates = arrayp->otc_arrayp;
    char *cp, *crString, *tclStr;
    int tclRes;
    bool statMatch;
    int i, k, first, j, t, result, ix;
    char tmpbuf[LONGVALUE];
    FILE *fp;
 
    /*
     * Standard in, out and err were redirected to/from /dev/null by
     * the daemon() routine (called only for otKwik, not otSlowPix).  Now
     * redirect them so 'puts stdout' statements in TCL procedures will
     * work properly.
     */

#ifdef notdef
    dup2(stdoutFd, 1);
    dup2(stderrFd, 2);
#endif

    close(1);
    close(2);
    dup(s);
    dup(s);

    if ( (qcb->qcb_outputFile = fp = fdopen(s, "w")) == NULL ) {
	logWarn("fdopen failure");
	exit(1);
    }

    /*
     * The child must emulate the behavior of the ot_bugs command.
     */
    signal(SIGCHLD, SIG_IGN);
    signal(SIGHUP, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    signal(SIGQUIT, SIG_IGN);
    signal(SIGTERM, SIG_IGN);

#ifdef MULTIPLE_KWIKS
    (void) close(funix);
    close(1), close(2);
    dup(s);
    dup(s);
    close(s);
#endif

    if ( qcb->qcb_sortField ) {
	free(qcb->qcb_sortField);
	qcb->qcb_sortField = 0;
    }
    if ( qcb->qcb_filter ) {
	free(qcb->qcb_filter);
	qcb->qcb_filter = 0;
    }

#ifdef SUPPORT_FOR_USER_TCL
    if ( interp ) {
	Tcl_DeleteInterp(interp);
	if ( (err = otTclInitInterp()) || (err = otTclInitCmds()) ) {
	    logWarn("otKwikPix: TCL interpreter failure: %s\n",
		otGetPostedMessage());
	}
    }
#endif

    if ( (err = otInitQuery()) || (err = otSetupQuery()) ) {
	logWarn("%s", otGetPostedMessage());
    } else {
	/*
	 * This evaluates the 'control' command.
	 */
	if ( tclRes = Tcl_Eval(interp, qry) ) {
	    fprintf(fp, "TCL error: %s\n", interp->result);
	    err = OT_TCL;
	}
    }

    if ( !err ) {
	if (!(numlist = malloc((pvp->pcb_topNumber+1) * sizeof(long))))
	    logWarn("otKwikPix: error in allocating numlist array\n");
    }

#ifdef SUPPORT_FOR_USER_TCL
    if ( !err ) {
	if ( otCB->cb_tclOtrc ) {
	    tclStr = strdup(otCB->cb_tclOtrc);
	    err = otTclEval(tclStr);
	    free(tclStr);
	}
    }
    if ( !err ) {
	if ( otCB->cb_tclFile ) {
	    tclStr = strdup(otCB->cb_tclFile);
	    err = otTclEval(tclStr);
	    free(tclStr);
	}
    }
    if ( !err ) {
	if ( prjp->projTclCode ) {
	    tclStr = strdup(prjp->projTclCode);
	    err = otTclEval(tclStr);
	    free(tclStr);
	}
    }
#endif

    if (err) {
        if ( cp = otGetPostedMessage() )
	    fprintf(stdout, "%s\n", cp);
	fflush(stdout);
	fflush(fp);
	write( s, "\000", 1);
	close(s);
	return;
    }

    i = arrayp->otc_top + 1;
    j = t = 0;

    if (otCB->cb_tclString) {
	pvp->pcb_template = 0;
	Tcl_ExprBoolean(pvp->pcb_interp, otCB->cb_tclString, &result);
    } else
	otSetCBMember("tclString", "1");
    pvp->pcb_tclBegin = FALSE;

    for (ix = 0; ix < MAXNUMVAL; ix++) {
	pvp->pcb_sumPrior[ix] = 0;
	pvp->pcb_sumSever[ix] = 0;
	pvp->pcb_sumStat[ix] = 0;
    }
    pvp->pcb_bugCount = 0;

    while ( --i > 0 ) {
        if ( !templates[i].tr )
	    continue;
	if ( templates[i].tr[0].field == 0 )
	    break;

	pvp->pcb_CRmatch = 0;
	pvp->pcb_template = &(templates[i]);

	if ( otProcessTcl( &(templates[i]) ) ) {
	    fprintf(fp, "%s\n", otGetPostedMessage());
	    break;
        }

	if ( pvp->pcb_CRmatch ) {
	    numlist[j++] = i;
	    pvp->pcb_bugCount++;

	    if (qcb->qcb_summary)    {
	        cp = otGetHeaderFieldValue("Priority", &(templates[i]));
		if (cp && (*cp >= '0') && (*cp <= '4'))
		    pvp->pcb_sumPrior[*cp - '0']++;
		cp = otGetHeaderFieldValue("Severity", &(templates[i]));
		if (cp && (*cp >= 'A') && (*cp <= 'F'))
		    pvp->pcb_sumSever[*cp - 'A']++;

		if ( cp = otGetHeaderFieldValue("Status", &(templates[i])) ) {
		    statMatch = FALSE;
		    for (k=0; (pvp->pcb_statName[k][0]) && !statMatch ; k++) {
			if (!strcmp(cp, pvp->pcb_statName[k])) {
			    pvp->pcb_sumStat[k]++;
			    statMatch = TRUE;
			}
		    }
		    if (!statMatch) {
			for (k=0; pvp->pcb_statName[k][0]; k++) {
			    if (!strcmp(pvp->pcb_statName[k], "other"))
				pvp->pcb_sumStat[k]++;
			}
		    }
		}
	    }
	}

	numlist[j] = 0;
    }

    if (otCB->cb_tclString && strcmp(otCB->cb_tclString, "1")) {
	pvp->pcb_tclEnd = TRUE;
        pvp->pcb_template = 0;
	Tcl_ExprBoolean(pvp->pcb_interp, otCB->cb_tclString, &tclRes);
    }

    if	      (qcb->qcb_summary || qcb->qcb_count) {

	otPrintCountSummary();

    } else if (!qcb->qcb_fullText && !qcb->qcb_keyInText && !pvp->pcb_quiet) {

        if ( ! j ) {
	    /*
	     * otPutPostedMessage(OT_NO_OBJECT_MATCH, prjp->object_name);
	     * fprintf(fp, "%s\n", otGetPostedMessage());
	     */
	} else {
	    for ( t = 0; t < j; t++ ) {
		pvp->pcb_template = &(templates[numlist[t]]);
		if (err = otPrintOneLiner(&(templates[numlist[t]]), prjp->otmp,
		    FALSE) )
		    fprintf(fp, "%s\n", otGetPostedMessage());
	    }
	}

    } else if ( !pvp->pcb_quiet  ) {

	/*
	 * Otherwise, this is a request for a series of numbers.
	 */
	first = 1;
	if ( !(crString = malloc( 8 * pvp->pcb_topNumber )) ) {
	    logWarn("otKwikPix: memory allocation error in otKwikKwery()");
	    exit(1);
	} else
	    crString[0] = 0;

	for(j=0; numlist[j]; j++)
	    if ( first ) {
		sprintf(tmpbuf, "%d", numlist[j]);
		strcat(crString, tmpbuf);
		first = 0;
	    } else {
		sprintf(tmpbuf, " %d", numlist[j]);
		strcat(crString, tmpbuf);
	    }

	logWarn("crString = |%s|\n", crString);
	fprintf(fp, "%s\n", crString);
	free(crString);

    }

    if ( numlist )
	free(numlist);
    fflush(stdout);
    fflush(fp);
    write( s, "\000", 1);
    fclose(fp);

#ifdef MULTIPLE_KWIKS
    exit(0);
#endif

}

void
otFreeKwikTemplate(tp)
OTTemplate *tp;
{
    register OTHeaderField *hfp;
    register OTNotedata *tndp;

    if ( !tp )
        return;
    for ( hfp = tp->tr; hfp && hfp->field; hfp++ )
	if (hfp->value)
	    free( hfp->value );
    free(tp->tr);
    tp->tr = 0;

    for ( tndp = tp->ndp; tndp && tndp->nd_flags; tndp++ ) {
	if ( tndp->nd_kwd )
	    free( tndp->nd_kwd );
	if ( tndp->nd_agent )
	    free( tndp->nd_agent );
	if ( tndp->nd_field )
	    free( tndp->nd_field );
	if ( tndp->nd_previous )
	    free( tndp->nd_previous );
	if ( tndp->nd_current )
	    free( tndp->nd_current );
    }		
    free(tp->ndp);
    tp->ndp = 0;

}
    


OTErr
otSlowService()
{
    OTErr err, hdrErr, histErr;
    OTPrivateCB *pvp = otCB->cb_pcb;
    OTProject   *prjp = pvp->pcb_project;
    OTQueryCB   *qcb;
    OTTemplate *templates = arrayp->otc_arrayp;
    OTTemplate *tp;
    char hdrDbfile[PATHLEN], histDbfile[PATHLEN];
    FILE *hdrfp, *histfp;
    char *hdrKwikp, *histKwikp, *tmphdr, *tmphist;
    Tcl_Interp  *interp = pvp->pcb_interp;
    char buf[100000], tmpbuf[LONGVALUE];
    char *termChar = "\n";
    char *cp, *crString, *tclStr;
    long hdrCrIdx, histCrIdx;
    int i, j, k, t, n, tclRes, first;
    bool statMatch;

    hdrfp = histfp = 0;
    j = t = 0;

    /*
     * NB buf needs to be dynamic now we are sending otrc etc.
     */
    memset(buf, 0, sizeof(buf));
    n = read(0, buf, sizeof(buf));

    if ( (err = otInitQuery()) || (err = otSetupQuery()) ) {
	logWarn("%s", otGetPostedMessage());
	exit(1);
    } 

    qcb = otCB->cb_qcb;
    /*
     * This evaluates the 'control' command.
     */
    if ( tclRes = Tcl_Eval(interp, buf) ) {
	fprintf(stdout, "TCL error: %s\n", interp->result);
	exit(1);
    }

    if (!(numlist = malloc( (pvp->pcb_topNumber+1) * sizeof(long) ))) {
	logWarn("otSlowPix: error in allocating numlist array\n");
	exit(1);
    }

#ifdef SUPPORT_FOR_USER_TCL
    if ( !err ) {
	if ( otCB->cb_tclOtrc ) {
	    tclStr = strdup(otCB->cb_tclOtrc);
	    err = otTclEval(tclStr);
	    free(tclStr);
	}
    }
    if ( !err ) {
	if ( otCB->cb_tclFile ) {
	    tclStr = strdup(otCB->cb_tclFile);
	    err = otTclEval(tclStr);
	    free(tclStr);
	}
    }
    if ( !err ) {
	if ( prjp->projTclCode ) {
	    tclStr = strdup(prjp->projTclCode);
	    err = otTclEval(tclStr);
	    free(tclStr);
	}
    }
#endif

    if ( err ) {
	fprintf(stdout, "TCL error: %s\n", otGetPostedMessage());
	exit(1);
    }

    if (otCB->cb_tclString) {
	pvp->pcb_template = 0;
	Tcl_ExprBoolean(pvp->pcb_interp, otCB->cb_tclString, &tclRes);
    } else
	otSetCBMember("tclString", "1");   /* i.e. return all CR nums */
    pvp->pcb_tclBegin = FALSE;

    sprintf(hdrDbfile, "%s/%s/%s", prjp->proj_dir, prjp->name, HEADERDB);
    if ( (hdrfp = fopen(hdrDbfile, "r")) == NULL ) {
	logWarn("Error opening %s\n", hdrDbfile);
	exit(1);
    }

    if ( qcb->qcb_historyLines ) {
	sprintf(histDbfile, "%s/%s/%s", prjp->proj_dir, prjp->name, HISTORYDB);
	if ( (histfp = fopen(histDbfile, "r")) == NULL ) {
	    logWarn("Error opening %s\n", histDbfile);
	    exit(1);
	}
    }

    otCB->cb_operation = QUERY;
    pvp->pcb_template = 0;
    hdrKwikp = histKwikp = 0;

    while ( 1 ) {

	tmphdr = tmphist = 0;
	if ( (hdrErr = otReadStringFromFileHandle(&hdrKwikp, hdrfp, termChar,
	    NULL, FALSE)) == OT_TRANSMISSION_CONCLUDED )
	    break;
	sscanf(hdrKwikp, "%ld", &hdrCrIdx);
	tmphdr = hdrKwikp;
	tmphdr += strspn(tmphdr, "0123456789");
	tmphdr += strspn(tmphdr, " ");

	if ( qcb->qcb_historyLines ) {
	    if ( (histErr = otReadStringFromFileHandle(&histKwikp, histfp,
		termChar, NULL, FALSE)) == OT_TRANSMISSION_CONCLUDED )
		break;
	    sscanf(histKwikp, "%ld", &histCrIdx);
	    tmphist = histKwikp;
	    tmphist += strspn(tmphist, "0123456789");
	    tmphist += strspn(tmphist, " ");

	    if ( hdrCrIdx != histCrIdx ) {
		logWarn("Error in order of header/history info at CR %ld\n",
		    hdrCrIdx);
		exit(1);
	    }
	}

	if ( err = otAllocHeaderHist(arrayp, 0, tmphdr, tmphist) ) {
	    logWarn("Error in allocating header and notedata info for cr %d (%s%s): %s\n",
		hdrCrIdx, tmphdr, tmphist, otGetPostedMessage());
	    exit(1);
	}

	if ( tmphdr && *tmphdr ) {
	    err = otAppendHeaderFieldToTemplateFromKwikString(&(templates[0]),
		tmphdr);
	    if ( !err && tmphist && *tmphist )
		err = otAppendNotedataToTemplateFromKwikString(&(templates[0]),
		    tmphist);
	}

	if ( hdrKwikp )
	    free( hdrKwikp );
	if ( histKwikp )
	    free( histKwikp );
	hdrKwikp = histKwikp = 0;
	if ( err )
	    break;

        if ( !templates[0].tr )
	    continue;
	if ( templates[0].tr[0].field == 0 )
	    break;

	pvp->pcb_CRmatch = 0;
	pvp->pcb_template = &(templates[0]);

	if ( otProcessTcl( &(templates[0]) ) ) {
	    fprintf(stdout, "%s\n", otGetPostedMessage());
	    exit(1);
	}

	if ( pvp->pcb_CRmatch ) {
	    numlist[j++] = hdrCrIdx;
	    pvp->pcb_bugCount++;

	    if (qcb->qcb_summary)    {
	        cp = otGetHeaderFieldValue("Priority", &(templates[0]));
		if (cp && (*cp >= '0') && (*cp <= '4'))
		    pvp->pcb_sumPrior[*cp - '0']++;
		cp = otGetHeaderFieldValue("Severity", &(templates[0]));
		if (cp && (*cp >= 'A') && (*cp <= 'F'))
		    pvp->pcb_sumSever[*cp - 'A']++;

		if ( cp = otGetHeaderFieldValue("Status", &(templates[0])) ) {
		    statMatch = FALSE;
		    for (k=0; (pvp->pcb_statName[k][0]) && !statMatch ; k++) {
			if (!strcmp(cp, pvp->pcb_statName[k])) {
			    pvp->pcb_sumStat[k]++;
			    statMatch = TRUE;
			}
		    }
		    if (!statMatch) {
			for (k=0; pvp->pcb_statName[k][0]; k++)
			    if (!strcmp(pvp->pcb_statName[k], "other"))
				pvp->pcb_sumStat[k]++;
		    }
		}
	    } /* qcb_summary */ 
	    if (!qcb->qcb_fullText && !qcb->qcb_summary && !qcb->qcb_count &&
		!qcb->qcb_keyInText && !pvp->pcb_quiet) {
		pvp->pcb_template = &(templates[0]);
		if (err = otPrintOneLiner(&(templates[0]), prjp->otmp,
		    FALSE) ) {
		    logWarn("%s\n", otGetPostedMessage());
		}
	    }
	} /* pcb_CRmatch */

	numlist[j] = (long)0;

    }

    if (otCB->cb_tclString && strcmp(otCB->cb_tclString, "1")) {
	pvp->pcb_tclEnd = TRUE;
        pvp->pcb_template = 0;
	Tcl_ExprBoolean(pvp->pcb_interp, otCB->cb_tclString, &tclRes);
    }

    if	      (qcb->qcb_summary || qcb->qcb_count) {

	otPrintCountSummary();

    } else if (!qcb->qcb_fullText && !qcb->qcb_keyInText && !pvp->pcb_quiet) {

        if ( ! j ) {
	    /* 
	     * otPutPostedMessage(OT_NO_OBJECT_MATCH, prjp->object_name);
	     * fprintf(stdout, "%s\n", otGetPostedMessage());
	     */
	}

    } else if (!pvp->pcb_quiet  ) {

	/*
	 * Otherwise, this is a request for a series of numbers.
	 */
	first = 1;
	if ( !(crString = malloc( 8 * pvp->pcb_topNumber )) ) {

	    logWarn("otKwikPix: memory allocation error in otKwikKwery()");
	    exit(1);
	} else
	    crString[0] = 0;

	for(j=0; numlist[j]; j++)
	    if ( first ) {
		sprintf(tmpbuf, "%ld", numlist[j]);
		strcat(crString, tmpbuf);
		first = 0;
	    } else {
		sprintf(tmpbuf, " %ld", numlist[j]);
		strcat(crString, tmpbuf);
	    }

	fprintf(stdout, "%s\n", crString);
	free(crString);

    }

    if (hdrfp)
	fclose(hdrfp);
    if (histfp)
	fclose(histfp);

    fflush(stdout);
    exit(0);

}
