# hypetool.tcl,v 1.1 1995/11/17 00:42:04 steve Exp
#
#	PASTIME Project
#	Cooperative Research Centre for Advanced Computational Systems
#	COPYRIGHT NOTICE AND DISCLAIMER.
#
#	Copyright (c) 1995 ANU and CSIRO
#	on behalf of the participants in
#	the CRC for Advanced Computational Systems (ACSys)
#
# This software and all associated data and documentation ("Software")
# was developed for research purposes and ACSys does not warrant that 
# it is error free or fit for any purpose.  ACSys disclaims all liability
# for all claims, expenses, losses, damages and costs any user may incur 
# as a result of using, copying or modifying the Software.
#
# You may make copies of the Software but you must include all of this
# notice on any copy.

### This module provides support for interacting with hypertools.
### Several classes of tools are defined, each one having its own
### protocol.  The protocol may specify which parts are to be exposed
### to applets (if any).  Hypertools communicate by means of the Tk send command.
### Note that SurfIt! itself is a hypertool.

# All procedures in this module are prefixed by "Hypertool"
# All global variables are prefixed by "HT"

# The Continuous Media Hypertool manager prefixes all procedures and global
# variables with CM

# A program which implements a particular class of hypertool
# may register itself or may be pre-registered.  We probably need
# some sort of hypertool broker.

# The array "HThypertools" holds the registration of hypertools,

proc Hypertool_register {class type tool program} {
    global HThypertools HTtypes

    set HThypertools($class) [list $tool $program]
    foreach t $type {
	set HTtypes($t) $class
    }
}

proc Hypertool {class} {
    global HThypertools

    if {[info exists HThypertools($class)]} {
	if {[lsearch [winfo interps] [lindex $HThypertools($class) 0]] == -1} {
	    # We need to startup the tool
	    exec [lindex $HThypertools($class) 1] &
	    # Wait for the application to start
	    after idle Hypertool_check [lindex $HThypertools($class) 0]
	    tkwait variable HT[lindex $hypertools($class) 0]
	}
	return [lindex $HThypertools($class) 0]
    } else {error "hypertool \"$class\" class is not defined"}
}

proc Hypertool_check {tool} {
    global HTwait
    upvar #0 HT$tool var

    # Is the application responding?
    if {[catch {send $tool info tclversion}]} {
	if {![info exists HTwait($tool)]} {
	    set HTwait($tool) 9	;# wait up to 2 seconds
	}
	incr HTwait($tool) -1
	if {$HTwait($tool)} {
	    after 250 Hypertool_check $tool
	} else {
	    puts stderr "unable to contact hypertool for $class application"
	    catch {unset HTwait($tool)}
	    set var 0	;# terminates pending tkwait
	}
    } else {
	catch {unset HTwait($tool)}
	set var 1	;# terminates pending tkwait
    }
}

# Wrapper to send commands to hypertools

proc Hypertool_send {class cmd} {
    global HThypertools

    # Try straight-forward send first,
    # if unsuccessful may need to (re-)start tool
    if {[catch {send $HThypertools($class) $cmd}]} {
	if {[catch {send [Hypertool $class] $cmd} err]} {
	    puts "send to hypertool failed: \"$err\""
	}
    }
}

###
### Continuous Media hypertool
###

# Here we define a class of hypertools for handling continuous media.
# The Berkeley Plateau Multimedia Research Group's Continuous Media
# Toolkit contains an application, cmplayer, which is representative
# of this class.

Hypertool_register CM \
	{video/mpeg video/mjpg audio/basic audio/audiofile application/x-cmt-script movie/cmt} \
	cmplayer cmplayer

# First, we define content-type handlers for continuous media document
# types.  We also include a content-type handler for CMT scripts.
# Then, various procedures are defined for communicating with the
# hypertool (the protocol).

# Register content-type handlers.  These handle files that have been
# copied to the local cache (similar to images) by passing them to
# the hypertool.  Unfortunately we have the latency of copying the file 
# first :-(
# Future: 1) handle data as it arrives from the network
#	  2) Display inline video (shouldn't be too hard)

# Of course, what you really want to do is use CMT scripts.

PRregister_type video/mpeg .mpg CM_handler {
    {PRdata {[error "-data option not yet implemented]}}
    {PRfile {[CM_file $read_handler_state $data]}}
    {PRfd {[$read_handler $read_handler_state]}}
    {PRimage {[error "incompatible data source"]}}
} {} CM_close

PRregister_type video/mjpg .mjpg CM_handler {
    {PRdata {[error "-data option not yet implemented]}}
    {PRfile {[CM_file $read_handler_state $data]}}
    {PRfd {[$read_handler $read_handler_state]}}
    {PRimage {[error "incompatible data source"]}}
} {} CM_close

PRregister_type audio/basic .au CM_handler {
    {PRdata {[error "-data option not yet implemented]}}
    {PRfile {[CM_file $read_handler_state $data]}}
    {PRfd {[$read_handler $read_handler_state]}}
    {PRimage {[error "incompatible data source"]}}
} {} CM_close

PRregister_type audio/audiofile .acf CM_handler {
    {PRdata {[error "-data option not yet implemented]}}
    {PRfile {[CM_file $read_handler_state $data]}}
    {PRfd {[$read_handler $read_handler_state]}}
    {PRimage {[error "incompatible data source"]}}
} {} CM_close

PRregister_type application/x-cmt-script .script CM_handler {
    {PRdata $data}
    {PRfile {[$read_handler $read_handler_state]}}
    {PRfd {[$read_handler $read_handler_state]}}
    {PRimage {[error "incompatible data source"]}}
} {} CM_close

proc CM_file {state filename} {
    upvar #0 $state var

    set var(eof) 1
    return $filename
}

# Register cmtp: network protocol handler.  Of course, we'll just pass
# the URL to cmplayer.  cmtp: URLs don't behave quite like normal URLs;
# they are, in a way, documents themselves.

PRregister_type movie/cmt .cmt CM_cmtp_handler {
    {PRdata $data}
    {PRfile {[error "incompatible data type"]}}
    {PRfd {[error "incompatible data type"]}}
    {PRimage {[error "incompatible data type"]}}
} {} {}

PRregister_protocol cmtp CM_protocol

proc CM_protocol {state data} {
    upvar #0 $state var
    upvar 2 $data data_return


    set var(read_handler) CM_read
    set data_return {}
    set var(HDRcontent-length) 0
    set var(HDRcontent-type) {movie/cmt}
    set var(eof) 1
    return PRdata
}

proc CM_read {state} {
    upvar #0 $state var
    set var(eof) 1
    return $var(url)
}

proc CM_cmtp_handler {data win} {
    set win [string trim $win]
    upvar #0 PR$win var

    puts "Hypertool_send CM \"Extern_NewFile $var(url)\; play\""
    Hypertool_send CM "Extern_NewFile $var(url)\; play"
}

# If this is a CMT script then store the text
# so that CM_close can send it to cmplayer
# Otherwise ignore it since we'll use the cached file
proc CM_handler {data win} {
    set win [string trim $win]
    upvar #0 PR$win var

    if {$var(HDRcontent-type) == "application/x-cmt-script"} {
	upvar #0 CM$win script
	append script($var(id)) $data
    }
    return {}
}

proc CM_close {win} {
    upvar #0 PR$win var

    if {!$var(eof)} return;	# Wait until all data is loaded

    if {($var(HDRcontent-type) == "application/x-cmt-script")} {
	upvar #0 CM$win script
	puts "Hypertool_send CM \"Extern_NewScript \{$script($var(id))\}\; play\""
	Hypertool_send CM "Extern_NewScript \{$script($var(id))\}\; play"
	unset script($var(id))
    } else {
	# Find cache entry
	upvar #0 PR$win.dummy dummy
	set dummy(url) $var(url)
	set data [Image_close_get_cache PR${win}.dummy filename]
	if {$data != "PRfile"} return;	# Oops!
	# Play it
	puts "Hypertool_send CM \"Extern_NewFile $filename\; play\""
	Hypertool_send CM "Extern_NewFile $filename\; play"
    }
}
