#
# handlers.tcl,v 1.4 1995/11/17 00:42:02 steve Exp
#
#	PASTIME Project
#	Cooperative Research Centre for Advanced Computational Systems
#	COPYRIGHT NOTICE AND DISCLAIMER.
#
#	Copyright (c) 1995 ANU and CSIRO
#	on behalf of the participants in
#	the CRC for Advanced Computational Systems (ACSys)
#
# This software and all associated data and documentation ("Software")
# was developed for research purposes and ACSys does not warrant that 
# it is error free or fit for any purpose.  ACSys disclaims all liability
# for all claims, expenses, losses, damages and costs any user may incur 
# as a result of using, copying or modifying the Software.
#
# You may make copies of the Software but you must include all of this
# notice on any copy.
###
# Content-type handlers for sundry types, plus renderers for Text widget.

# The Text handling procedures are all prepended by "Text"
# The image handling procedures are all prepended by "Image"
# The Helper application procedures are all prepended by "Helper"

proc Text_plain {txt {cmd Text_test} {start {}}} {
    eval $cmd {$txt}
}

proc Text_test {win txt} {
    puts "text ==> $txt"
}

proc Text_render {win txt} {
    $win tag configure Tnormal -font [HMx_font courier 14 medium r]
    $win insert end $txt Tnormal
}

# Register this handler for text/plain, text/text and text/*

foreach t {text/text text/plain text/*} {
    catch {
	PRregister_type $t .txt Text_plain {
	    {PRdata $data}
	    {PRfile {[$read_handler $read_handler_state]}}
	    {PRfd {[$read_handler $read_handler_state]}}
	    {PRimage {[error "incompatible data"]}}
	} Text_render
    }
}

#
# Image type handlers - image/x-xbitmap, image/gif, image/x-portable-pixmap
# (and later, image/jpeg)
# These can't actually do anything until the data has been read into the
# cache file.  At that time, load the image data into the image type using
# the -file switch.
#

proc Image_handler {data {cmd Image_test} {start {}}} {
    eval $cmd $data
}

proc Image_test {win img} {
    puts "image ==> $win $img"
}

proc Image_render {win {img {}}} {
    if {$img == {}} return; # Image download was delayed
    if {[winfo class $win] == "Text"} {
	set l [label $win.img -image $img]
	$win window create end -window $l
    } elseif {[winfo class $win] == "Label" || \
	      [winfo class $win] == "Button"} {
	if {[winfo name $win] == "label"} {
	    pack propagate [winfo parent $win] 1
	}
	$win configure -image $img
    }
}

proc Image_close {win} {
    upvar #0 PR$win var

    if {!$var(eof)} return; # Wait until all data is loaded

    # Find cache entry
    upvar #0 PR$win.dummy dummy
    set dummy(url) $var(url)
    set data [Image_close_get_cache PR${win}.dummy filename]
    switch $data {
	PRfile {
	    set img [Image_create_from_file PR$win $filename]
	}
	PRimage {
	    set img $filename
	}
	default {return}
    }

    Image_render $win $img
}

proc Image_close_get_cache {state filename} {
    Cache_get_document $state $filename
}

proc Image_create_from_file {state filename} {
    upvar #0 $state var

    set var(eof) 1
    # Create an image and read the cache file contents into it
    switch $var(HDRcontent-type) {
	image/x-xbitmap {set img [image create bitmap]}
	image/x-portable-pixmap {set img [image create photo -format ppm]}
	image/gif {set img [image create photo -format gif]}
	default {return}
    }
    $img configure -file $filename
    # Replace the old file reference with an image reference
    Cache_add_image $var(url) $img $var(HDRcontent-type)
    return $img
}

catch {
    PRregister_type image/x-xbitmap .xbm Image_handler {
	{PRdata {[error "-data option not yet implemented"]}}
	{PRfile {[Image_create_from_file $read_handler_state $data]}}
	{PRfd {[$read_handler $read_handler_state]}}
	{PRimage $data}
    } Image_render Image_close
}

catch {
    PRregister_type image/x-portable-pixmap {.ppm .pnm .pbm} Image_handler {
	{PRdata {[error "-data option not yet implemented"]}}
	{PRfile {[Image_create_from_file $read_handler_state $data]}}
	{PRfd {[$read_handler $read_handler_state]}}
	{PRimage $data}
    } Image_render Image_close
}

catch {
    PRregister_type image/gif .gif Image_handler {
	{PRdata {[error "-data option not yet implemented"]}}
	{PRfile {[Image_create_from_file $read_handler_state $data]}}
	{PRfd {[$read_handler $read_handler_state]}}
	{PRimage $data}
    } Image_render Image_close
}

# Handler to pass data to "helper applications"
# It will be easy to implement passing data as it comes
# off the network, but for the moment we'll just point the
# helper app at the cache file

proc Helper_register {type ext cmd} {
    global Helpers

    set Helpers($type) $cmd
    catch {
	PRregister_type $type $ext Helper_handler {
	    {PRdata {}}
	    {PRfile {}}
	    {PRfd {[$read_handler $read_handler_state]}}
	    {PRimage {}}
	} {} Helper_close
    }
}

# Nothing to do (for now).  A future version might pipe the
# data to the help app.

proc Helper_handler {data {cmd {}} {start {}}} {}

proc Helper_close {win} {
    global Helpers
    upvar #0 PR$win var

    if {!$var(eof)} return; # Wait until all data is loaded

    # Find the file that the data has been written to
    # by the cache, and then start the helper app
    eval exec [format $Helpers($var(HDRcontent-type)) [Cache_get_file $var(url)]] &
}

#
# Register helper apps specified in the user's .mailcap
#

if {![catch {open $env(HOME)/.mailcap r} mailcapfd]} {
    set mailcap [read -nonewline $mailcapfd]
    foreach line [split $mailcap \n] {
	if {[regexp {([^;]*);[ 	]*(.*)} $line all type cmd]} {
	    Helper_register $type {} $cmd
	}
    }
    close $mailcapfd
    unset mailcapfd
    unset mailcap
}
