# LabelEntry.tcl - Itcl widget for displaying a labeled entry
#
# Copyright (C) 1994 Allan Brighton (allan@piano.sta.sub.org)


itcl_class LabelEntry {
    inherit FrameWidget

    # constructor: create a new LabelEntry widget

    constructor {config} {
	FrameWidget::constructor

	pack [label $this.label] \
	    -side $side -ipadx 1m

	if {"$valueWidth" == ""} {
	    set expand 1
	    set fill x
	} else {
	    set expand 0
	    set fill none
	}
	pack [entry $this.entry -relief $relief] \
	    -side $side -expand $expand -fill $fill -padx 1m -ipadx 1m

	set initialized_ 1

	#  Explicitly handle config's that may have been ignored earlier
	foreach attr $config {
	    config -$attr [set $attr]
	}
    }


    #  Get the value in the entry

    method get {} {
	return [set value [$this.entry get]]
    }


    #  Get the value in the entry (without setting variable)

    method get_ {} {
	return [$this.entry get]
    }

    
    #  return true if the value in the entry has changed since it was 
    # last configured or read

    method changed {} {
	return [expr {"[$this.entry get]" != "$value"}]
    }

   
    # called for traces on textvariable

    method trace_ {args} {
	if {!$notrace_} {
	    _action_proc $changecmd
	}
    }


    #  called for for return or keypress in entry, calls action proc with new value

    method _action_proc {cmd} {
	lappend cmd [$this.entry get]
	eval $cmd
    }

    
    # -- public variables --


    # set the text entry's label
    public text {} {
	set label $text
	if {$initialized_} {
	    $this.label config -text $text
	}
    }

    # synonym for -text
    public label {} {
	if {$initialized_} {
	    config -text $label
	}
    }

    # set the value in the entry
    public value {} {
	if {$initialized_} {
	    set prev_state [cget $this.entry  -state]
	    $this.entry config -state normal
	    set notrace_ 1
	    $this.entry delete 0 end
	    $this.entry insert 0 $value
	    if {"$justify" == "right"} {
		$this.entry icursor end
		$this.entry xview moveto 1
	    }
	    $this.entry config -state $prev_state
	    set notrace_ 0
	}
    }


    # set the width for  displaying the label
    public labelWidth {} {
	if {$initialized_} {
	    $this.label config -width $labelWidth
	}
    }
    
    # set the width for displaying the value
    public valueWidth {} {
	if {$initialized_} {
	    $this.entry config -width $valueWidth
	}
    }
    
    # set the font for  displaying the label
    public labelFont {} {
	if {$initialized_} {
	    $this.label config -font $labelFont
	}
    }
    
    # set the font for displaying the value
    public valueFont {} {
	if {$initialized_} {
	    $this.entry config -font $valueFont
	}
    }
    
    # set the entry's state (normal, disabled)
    public state {normal} {
	if {$initialized_} {
	    $this.entry config -state $state
	}
    }
    
    # set the background color
    public bg {} {
	if {$initialized_} {
	    $this.entry config -bg $bg
	    $this.label config -bg $bg
	}
    }
    
    # set the entry relief 
    public relief {sunken} {
	if {$initialized_} {
	    $this.entry config -relief $relief
	}
    }
    
    
    # set the label anchor (the entry follows)
    public anchor {} {
	if {$initialized_} {
	    $this.label config -anchor $anchor
	}
    }
    
    # the action for <Return> in the entry, called with the new value
    public action {} {
	if {$initialized_} {
	    bind $this.entry <Return> [list $this _action_proc $action]
	}
    }

    # set to "right" to make sure the end of the entry is visible
    public justify {left}

    # pack option: use left for horizontal orient, top for vert.
    public side {left}

    
    # commands to evaluate whenever the entry value changes
    public changecmd {} {
	if {$initialized_} {
	    global $this
	    $this.entry config -textvariable $this
	    trace variable $this w "$this trace_"
	}
    }


    # -- protected vars --

    # this flag is set to 1 to avoid tracing when changing the entry's value
    protected notrace_ 0

    # set to 1 in constructor
    protected initialized_ 0
}



