#include <stdio.h>
#include <stdlib.h>

#ifndef NO_STRING_H
#   include <string.h>
#else
#   include <strings.h>
#endif

#include <pwd.h>
#include "tkpvmInt.h"

/*
 * The following structure is used for queueing pvm events on the
 * Tcl event queue.
 */

typedef struct pvmEvent {
    Tcl_Event header;		/* Standard information for all events. */
    int info;			/* PVM buffer identity. */
} pvmEvent;

/*
 * For each Pvm-event registered in a call to Pvm_CreatePvmHandler,
 * there is one record of the following type.  All of these records
 * are chained together into a single list.
 */

typedef struct pvmHandler {
    int tid;			/* task identifier (or -1 if none) */
    int msgtag;			/* message identifier (or -1 if none) */
    Pvm_EventProc *proc;	/* procedure to call if pvm packet received */
    ClientData clientData;	/* Argument to pass to proc. */
    struct pvmHandler *nextPtr; /* Next in list */
} pvmHandler;
static pvmHandler *firstPvmHandlerPtr = NULL;  /* list of all Pvm handlers */

typedef struct TkpvmTidInfo {
    int tid;			/* task identifier */
    Tcl_Interp *interp;		/* interpreter that spawned the tid
				   (NULL if unknown) */
    int msgtag;			/* msgtag for output redirect */
    Tcl_Channel channel;		/* channel for output redirect */
    struct TkpvmTidInfo *nextPtr; /* next in list */
} TkpvmTidInfo;
static TkpvmTidInfo *firstPvmTidPtr = NULL;


Tcl_Interp *PvmInterp = NULL; /* interpreter that calls Pvm_Init()
					 for the first time */
int PvmMytid = 0;

/*
 * Prototypes for procedures in this file:
 */

static void		PvmHandlerCheckProc _ANSI_ARGS_((
			    ClientData clientData, int flags));
static int		PvmHandlerEventProc _ANSI_ARGS_((Tcl_Event *evPtr,
			    int flags));
static void		PvmHandlerExitProc _ANSI_ARGS_((ClientData data));
static void		PvmHandlerSetupProc _ANSI_ARGS_((
			    ClientData clientData, int flags));
static void		PvmTclCleanup _ANSI_ARGS_((ClientData clientData));

/*
 * This function adds "tid" to the process list, so we can keep track
 * of deletion of this process from now on.
 */

int PvmNotifyTid(interp, tid, msgtag, channel)
Tcl_Interp *interp;
int tid;
int msgtag;
Tcl_Channel channel;
{
    int pvmerror;
    TkpvmTidInfo *infoPtr = firstPvmTidPtr;
    while(infoPtr && infoPtr->tid !=tid) {
	infoPtr = infoPtr->nextPtr;
    }
    if (!infoPtr) {
	if ((pvmerror=pvm_notify(PvmTaskExit,KILLTID_MSGTAG, 1, &tid))<0) {
	    return Pvm_ReturnError(interp,pvmerror);
	}
	infoPtr = (TkpvmTidInfo *) ckalloc(sizeof(TkpvmTidInfo));
	infoPtr->nextPtr = firstPvmTidPtr;
	firstPvmTidPtr = infoPtr;
    }
    infoPtr->interp	= interp;
    infoPtr->tid	= tid;
    infoPtr->msgtag	= msgtag;
    infoPtr->channel	= channel;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Pvm_StartDaemon --
 *	Start pvm daemon.
 *
 * Results:
 *	0 if failed.
 *	tid of current process if succeeded.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int
Pvm_StartDaemon(interp)
    Tcl_Interp *interp;
{
    char *p;
    int argc, code, i, j;
    char **argv = NULL;
    Tcl_DString buffer;
    char *args[2];
    char *hostfile = NULL;
    char buf[30];

    if (PvmMytid) return PvmMytid;

    p = Tcl_GetVar2(interp, "argv", (char *) NULL, TCL_GLOBAL_ONLY);
    if (p != NULL) {
	if (Tcl_SplitList(interp, p, &argc, &argv) != TCL_OK) {
	    argError:
	    Tcl_AddErrorInfo(interp,
		    "\n    (processing arguments in argv variable)");
	    return TCL_ERROR;
	}
	code = 1;
	for (i=0,j=0; i<argc;) {
	    if (code && (*argv[1]=='-')) {
		if (!strcmp(argv[i],"-hostfile")) {
		    i++; hostfile = argv[i++];
		} else {
		    if (!strcmp(argv[i],"--")) {
			code = 0;
		    }
		    argv[j++] = argv[i++];
		}
	    } else {
		argv[j++] = argv[i++];
	    }
	}
	argv[j]=NULL;
	p = Tcl_Merge(j, argv);
	Tcl_SetVar2(interp, "argv", (char *) NULL, p, TCL_GLOBAL_ONLY);
	sprintf(buf, "%d", j);
	Tcl_SetVar2(interp, "argc", (char *) NULL, buf, TCL_GLOBAL_ONLY);
	ckfree(p);
    }

    if (hostfile == NULL) {
	hostfile = "~/.Pvm_hosts";
    }

    args[0] = Tcl_TranslateFileName(interp, hostfile, &buffer);
    args[1] = (char *)NULL;

    if (argv != NULL) {
	ckfree((char *) argv);
    }

    pvm_setopt(PvmAutoErr, 0);

    code = pvm_start_pvmd(args[0]?1:0, args, 1);

    Tcl_DStringFree(&buffer);

    if (code < 0) {
	if (code != PvmDupHost) {
	    Tcl_AppendResult(interp,"Sorry, cannot start PVM daemon\n",(char *)NULL);
	    pvm_exit();
	    return 0;
	}
    }
    Tcl_CreateEventSource(PvmHandlerSetupProc, PvmHandlerCheckProc,
	    (ClientData) NULL);
    Tcl_CreateExitHandler(PvmHandlerExitProc, (ClientData) NULL);
    return PvmMytid = pvm_mytid();
}

/*
 *--------------------------------------------------------------
 *
 * Pvm_ReturnError --
 *
 *	This procedure changes pvm errors into Tcl errors.
 *
 *
 * Results:
 *	TCL_OK if there is no error.
 *
 *	TCL_ERROR if there is an error. Then interp will
 *	contain the PVM error message.
 *
 * Side effects:
 *	None
 *
 *
 *--------------------------------------------------------------
 */

extern int pvm_nerr;	    /* number of errors in the list */
extern char *pvm_errlist[]; /* make use of the error-list that already
			     * exists in the pvm library. */

int
Pvm_ReturnError(interp, error)
    Tcl_Interp *interp;		/* Current interpreter. */
    int error;			/* PVM error-number */
{
    char storage[4];

    if (error>=0) return TCL_OK;
    error = -error;
    if (error>1) {
	Tcl_AppendResult(interp,"PVM error: ",(char *)NULL);
    } else {
	return TCL_ERROR;
    }
    if (error<pvm_nerr) {
	Tcl_AppendResult(interp, pvm_errlist[error],(char *)NULL);
    } else {
	sprintf(storage,"-%d",error);
	Tcl_AppendResult(interp, "unknown error: ",storage,(char *)NULL);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_TasksCmd --
 *
 *	This procedure is invoked to process the "tasks" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_TasksCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int tid, pvmerror;
    char storage[16];
    struct pvmhostinfo *hostp;
    struct pvmtaskinfo *taskp;
    int i,j,k,nhost,narch,ntask;
    char *host = NULL, *arch = NULL, *name = NULL, *group = NULL;

    if (!PvmMytid) {
	if (!Pvm_StartDaemon(interp))
	    return TCL_ERROR;
    }
    argc-=1;argv+=1;
    while(argc>0) {
	if (argc==1) {
	    Tcl_AppendResult(interp, "no value for option \"", argv[0],
		    "\"",(char *)NULL);
	    return TCL_ERROR;		
	}
	if (!strcmp(argv[0],"-host")) {
	    host = argv[1];
	} else if (!strcmp(argv[0],"-arch")) {
	    arch = argv[1];
	} else if (!strcmp(argv[0],"-group")) {
	    group = argv[1];
	} else if (!strcmp(argv[0],"-name")) {
	    name = argv[1];
	} else {
	    Tcl_AppendResult(interp, "invalid option \"", argv[0],
		    "\": expected -arch, -group, -host or -name",(char *)NULL);
	    return TCL_ERROR;
	}
	argc-=2; argv+=2;
    }
    if ((pvmerror = pvm_config(&nhost, &narch, &hostp))<0) return Pvm_ReturnError(interp,pvmerror);
    if ((pvmerror = pvm_tasks(0, &ntask, &taskp))<0) return Pvm_ReturnError(interp,pvmerror);
    if (group==NULL) {
	for(i=0; i<ntask; i++) {
	    if (name!=NULL && !Tcl_StringMatch(taskp[i].ti_a_out,name)) {
		continue;
	    }
	    if (host != NULL) {
		j=0; while(j<nhost && (taskp[i].ti_host!=hostp[j].hi_tid ||
			    strcmp(host,hostp[j].hi_name))) j++;
		if (j>=nhost) continue;
	    }
	    if (arch != NULL) {
		j=0; while(j<nhost && (taskp[i].ti_host!=hostp[j].hi_tid ||
			strcmp(arch,hostp[j].hi_arch))) j++;
		if (j>=nhost) continue;
	    }
	    sprintf(storage,"0x%x",taskp[i].ti_tid);
	    Tcl_AppendElement(interp,storage);
	}
    } else {
	int gsize = pvm_gsize(group);
	for(k=0; k<gsize; k++) {
	    i=0; while(i<ntask && (taskp[i].ti_tid!=pvm_gettid(group,k))) i++;
	    if (i>=ntask || (name!=NULL && !Tcl_StringMatch(taskp[i].ti_a_out,name))) {
		continue;
	    }
	    if (host != NULL) {
		j=0; while(j<nhost && (taskp[i].ti_host!=hostp[j].hi_tid ||
			strcmp(host,hostp[j].hi_name))) j++;
		if (j>=nhost) continue;
	    }
	    if (arch != NULL) {
		j=0; while(j<nhost && (taskp[i].ti_host!=hostp[j].hi_tid ||
			strcmp(arch,hostp[j].hi_arch))) j++;
		if (j>=nhost) continue;
	    }
	    sprintf(storage,"0x%x",taskp[i].ti_tid);
	    Tcl_AppendElement(interp,storage);
	}
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Pvm_CreateEventHandler --
 *
 *	Arrange for a given procedure to be invoked whenever
 *	a given Pvm packet arrives.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	From now on, whenever a PVM packet described by tid
 *	and msgtag arrives, proc will be invoked.
 *	If tid and msgtag are already registered then the old
 *      mask and proc and clientData values will be replaced
 *      with new ones
 *
 *--------------------------------------------------------------
 */

void
Pvm_CreateEventHandler(tid, msgtag, proc, clientData)
    int tid;			/* task descriptor */
    int msgtag;			/* message task */
    Pvm_EventProc *proc;	/* Procedure to call for each
				 * selected event. */
    ClientData clientData;	/* Arbitrary data to pass to proc. */
{
    register pvmHandler *pvmPtr,*prevPtr,*Ptr;

    /*
     * Make sure tid and msgtag aren't already registered.  Create a
     * new record in the normal case where there's no existing
     * record.
     */

    for (prevPtr = NULL, pvmPtr = firstPvmHandlerPtr; pvmPtr != NULL;
	    prevPtr = pvmPtr,pvmPtr = pvmPtr->nextPtr) {
	if ((unsigned)pvmPtr->tid > (unsigned)tid || (pvmPtr->tid == tid &&
		(unsigned)pvmPtr->msgtag >= (unsigned)msgtag)) {
	    break;
	}
    }
    if (pvmPtr == NULL||pvmPtr->tid!=tid||pvmPtr->msgtag!=msgtag) {
	Ptr = (pvmHandler *) ckalloc(sizeof(pvmHandler));
	Ptr->tid = tid;
	Ptr->msgtag = msgtag;
	Ptr->nextPtr = pvmPtr;
	Ptr->clientData = (ClientData)NULL;
	if (prevPtr ==  NULL) {
	    firstPvmHandlerPtr = Ptr;
	} else {
	    prevPtr->nextPtr = Ptr;
	}
	pvmPtr = Ptr;
    }
    /*
     * The remainder of the initialization below is done
     * regardless of whether or not this is a new record
     * or a modification of an old one.
     */

    pvmPtr->proc = proc;
    pvmPtr->clientData = clientData;

}

/*
 *--------------------------------------------------------------
 *
 * Pvm_DeleteEventHandler --
 *
 *	Cancel a previously-arranged callback arrangement for
 *	a pvm packet.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If a callback was previously registered, remove it.
 *
 *--------------------------------------------------------------
 */

void
Pvm_DeleteEventHandler(tid, msgtag)
    int tid;
    int msgtag;
{
    register pvmHandler *pvmPtr,*prevPtr;

    /*
     * Find the entry for the given packet (and return if there
     * isn't one).
     */

    for (prevPtr = NULL, pvmPtr = firstPvmHandlerPtr; ;
	    prevPtr = pvmPtr, pvmPtr = pvmPtr->nextPtr) {
	if (pvmPtr == NULL) {
	    return;
	}
	if (pvmPtr->tid == tid && pvmPtr->msgtag == msgtag) {
	    break;
	}
    }

    /*
     * Clean up information in the callback record.
     */

    if (prevPtr == NULL) {
	firstPvmHandlerPtr = pvmPtr->nextPtr;
    } else {
	prevPtr->nextPtr = pvmPtr->nextPtr;
    }
    ckfree((char *) pvmPtr);

}

/*
 *----------------------------------------------------------------------
 *
 * PvmHandlerSetupProc --
 *
 *	This procedure is part of the "event source" for pvm.
 *	It is invoked by Tcl_DoOneEvent before it calls select (or
 *	whatever it uses to wait).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Tells the notifier which handles to wait for.
 *
 *----------------------------------------------------------------------
 */

#ifndef TCL_ACTIVE
#define TCL_ACTIVE (1<<4)
#endif

static void
PvmHandlerSetupProc(clientData, flags)
    ClientData clientData;		/* Not used. */
    int flags;				/* Not used. */
{
    int *fdlist, i, n;
    Tcl_Time blockTime;

    if (firstPvmHandlerPtr == NULL) return;

    /*
     * register File handler for all fd's in use for pvm, such
     * that it will be executed when a PVM-package is received.
     */

    n = pvm_getfds(&fdlist);
    for(i=0; i<n;i++) {
	Tcl_WatchFile(Tcl_GetFile((ClientData)fdlist[i],TCL_UNIX_FD),
		TCL_READABLE|TCL_EXCEPTION|TCL_ACTIVE);
    }
    blockTime.sec = 0;
    blockTime.usec = 100000;
    Tcl_SetMaxBlockTime(&blockTime);
}

/*
 *----------------------------------------------------------------------
 *
 * PvmHandlerCheckProc --
 *
 *	This procedure is the second part of the "event source" for
 *	file handlers.  It is invoked by Tcl_DoOneEvent after it calls
 *	select (or whatever it uses to wait for events).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Makes entries on the Tcl event queue for each pvm package that is
 *	now ready.
 *
 *----------------------------------------------------------------------
 */

static void
PvmHandlerCheckProc(clientData, flags)
    ClientData clientData;		/* Not used. */
    int flags;				/* Not used. */
{
    pvmHandler *pvmHandlerPtr;
    pvmEvent *pvmEvPtr;
    int info,number,msgtag,tid;

    while ((info = pvm_nrecv(-1,-1))>0) {
	pvmEvPtr = (pvmEvent *) ckalloc(sizeof(pvmEvent));
	pvmEvPtr->header.proc = PvmHandlerEventProc;
	pvmEvPtr->info = info;
	Tcl_QueueEvent((Tcl_Event *) pvmEvPtr, TCL_QUEUE_TAIL);
        pvm_bufinfo(info,&number,&msgtag,&tid);
	pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PvmHandlerExitProc --
 *
 *	Callback invoked during exit cleanup to destroy the pvm event
 *	source.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Destroys the pvm event source.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static void
PvmHandlerExitProc(clientData)
    ClientData clientData;		/* Not used. */
{
    Tcl_DeleteEventSource(PvmHandlerSetupProc, PvmHandlerCheckProc,
	(ClientData) NULL);
    if (PvmMytid!=0) {
	pvm_exit();
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PvmHandlerEventProc --
 *
 *	This procedure is called by Tcl_DoOneEvent when a pvm event
 *	reaches the front of the event queue.
 *
 * Results:
 *	Returns 1 always.
 *
 * Side effects:
 *	Whatever the pvm handler callback procedures do.
 *
 *----------------------------------------------------------------------
 */

static int
PvmHandlerEventProc(evPtr, flags)
    Tcl_Event *evPtr;		/* Event to service. */
    int flags;			/* not used. */
{
    int buffer, info, tid,msgtag, number;
    register pvmHandler *pvmPtr, *prev_pvmPtr;
    register TkpvmTidInfo *ptdPtr, *prev_ptdPtr;
    char s[100];

    pvmEvent *pvmEvPtr = (pvmEvent *) evPtr;

    buffer = pvmEvPtr->info;

    pvm_setrbuf(buffer);
    pvm_bufinfo(buffer,&number,&msgtag,&tid);
    if (tid==0x80000000) {
	switch(msgtag) {
	case KILLTID_MSGTAG:
	    info = pvm_upkint(&tid,1,1);
	    if (info<0) {
		Tcl_AppendResult(PvmInterp,"Error receiving data from PVM daemon",(char *)NULL);
		sprintf(s,"\n     msgtag:%d, error:%d",msgtag,info);
		Tcl_AddErrorInfo(PvmInterp,s);
		Tcl_BackgroundError(PvmInterp);
		pvm_freebuf(buffer);
		return 1;
	    } else {
		Tcl_DoWhenIdle(PvmTclCleanup, (ClientData) tid);
	    }
	    break;
	case REDIRECT_MSGTAG:
		{
		    char *p;
		    int t,c;
		    TkpvmTidInfo *infoPtr;

		    pvm_upkint(&t,1,1);
		    if (number<9) {
			return 1;
		    }
		    infoPtr = firstPvmTidPtr;
		    while(infoPtr && infoPtr->tid !=t) {
			infoPtr = infoPtr->nextPtr;
		    }
		    if (infoPtr && infoPtr->channel) {
			pvm_upkint(&c,1,1);
			if (c>0) {
			    p = ckalloc(c);
			    pvm_upkbyte(p,c,1);
			    Tcl_Write(infoPtr->channel,p,c);
			    ckfree(p);
			}
		    } else if (infoPtr && infoPtr->msgtag) {
			for (pvmPtr = firstPvmHandlerPtr; ;pvmPtr = pvmPtr->nextPtr) {
			    if (pvmPtr == NULL) {
				Tcl_AppendResult(PvmInterp,"Unknown PVM-packet received",(char *)NULL);
				sprintf(s,"\n\ttid:\t0x%x\n\tmsgtag:\t%d\n\tnumber:\t%d",t,
					infoPtr->msgtag,number-sizeof(int));
				Tcl_AddErrorInfo(PvmInterp,s);
				Tcl_BackgroundError(PvmInterp);
				return 1;
			    }
			    if ((pvmPtr->tid == -1 ||pvmPtr->tid == t) &&
				(pvmPtr->msgtag == -1 ||pvmPtr->msgtag == infoPtr->msgtag)) {
				break;
			    }
			}
			/* execute the procedure */
			pvmPtr->proc(t,infoPtr->msgtag,number-sizeof(int),pvmPtr->clientData);
		    }
		}
		break;
	default:
		Tcl_AppendResult(PvmInterp,"Unknown data received from PVM daemon",(char *)NULL);
		sprintf(s,"\n     msgtag:%d, number: %d",msgtag,number);
		Tcl_AddErrorInfo(PvmInterp,s);
		Tcl_BackgroundError(PvmInterp);
		pvm_freebuf(buffer);
		return 1;
	    }
    } else {
	for (pvmPtr = firstPvmHandlerPtr; ;pvmPtr = pvmPtr->nextPtr) {
	    if (pvmPtr == NULL) {
		Tcl_AppendResult(PvmInterp,"Unknown PVM-packet received",(char *)NULL);
		sprintf(s,"\n\ttid:\t0x%x\n\tmsgtag:\t%d\n\tnumber:\t%d",tid,msgtag,number);
		Tcl_AddErrorInfo(PvmInterp,s);
		Tcl_BackgroundError(PvmInterp);
		pvm_freebuf(buffer);
		return 1;
	    }
	    if ((pvmPtr->tid == -1 ||pvmPtr->tid == tid) &&
		    (pvmPtr->msgtag == -1 ||pvmPtr->msgtag == msgtag)) {
		break;
	    }
	}
	/* execute the procedure */
	pvmPtr->proc(tid,msgtag,number,pvmPtr->clientData);
    }
    pvm_freebuf(buffer);
    return 1;
}

static void PvmTclCleanup(clientData)
    ClientData clientData;
{
    int tid = (int) clientData;
    register pvmHandler *pvmPtr, *prev_pvmPtr;
    register TkpvmTidInfo *ptdPtr, *prev_ptdPtr;

    prev_ptdPtr = NULL; ptdPtr = firstPvmTidPtr;
    while(ptdPtr && ptdPtr->tid !=tid) {
	prev_ptdPtr = ptdPtr;
	ptdPtr = ptdPtr->nextPtr;
    }
    if (ptdPtr) {
	if (prev_ptdPtr) {
	    prev_ptdPtr->nextPtr = ptdPtr->nextPtr;
	} else {
	    firstPvmTidPtr = ptdPtr->nextPtr;
	}
	ckfree(ptdPtr);
    }
    prev_pvmPtr=NULL; pvmPtr = firstPvmHandlerPtr;
    while (pvmPtr) {
	if (pvmPtr->tid==tid) {
	    if (pvmPtr->msgtag == -2) {
		/* execute the 'kill' binding */
		pvmPtr->proc(tid,-2,0,pvmPtr->clientData);
	    }
	    if (prev_pvmPtr) {
		prev_pvmPtr->nextPtr = pvmPtr->nextPtr;
	    } else {
		firstPvmHandlerPtr = pvmPtr->nextPtr;
	    }
	    ckfree(pvmPtr);
	} else {
	    prev_pvmPtr = pvmPtr;
	}
	pvmPtr = pvmPtr->nextPtr;
    }
    PvmCleanup(tid);

}
