# rcsid {$Id: picutil.tcl,v 4.17 1996/05/21 16:15:11 mangin Rel $}
# This package is free software. Redistribution and use of this file
# are permitted without restrictions.
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
#	Frank Mangin
#	Frank.Mangin@ceram.fr  -  Frank.Mangin@sophia.inria.fr

################################################################
####		 Error Messages Procedures                  ####
################################################################

##
## MessageBox
##

proc MessageBox {mess} {
  
  if [winfo exists .confirm] {
    tkwait window .confirm
  }
  toplevel .confirm
  
  # Window manager configurations
  global tkVersion
  wm positionfrom .confirm ""
  wm sizefrom .confirm ""
  wm maxsize .confirm [winfo screenwidth .] [winfo screenheight .]
  wm minsize .confirm 10 10
  wm title .confirm {Picasso}
  
  
  # build widget .confirm.abort
  button .confirm.quit -command "destroy .confirm" \
    -text "Ok"
  
  label .confirm.visu -text $mess
  
  # pack widget .confirm
  pack append .confirm\
  .confirm.visu {top frame center expand fill}\
  .confirm.quit {left frame center expand}
  
  return .confirm
}

##
## ConfirmBox
##

proc ConfirmBox { {message ""} args } {
  global confirmStatus

  set confirmStatus ""
  
  if [winfo exists .confirm] {
    destroy .confirm
  }
  toplevel .confirm
  
  # Window manager configurations
  global tkVersion
  wm positionfrom .confirm ""
  wm sizefrom .confirm ""
  wm maxsize .confirm  [winfo screenwidth .] [winfo screenheight .]
  wm minsize .confirm 10 10
  wm title .confirm {Picasso}

  label .confirm.confirmLabel \
    -text $message
  
  pack append .confirm \
    .confirm.confirmLabel {top frame n expand pady 10}

  while {$args != ""} {
    set label [lvarpop args]
    set action [lvarpop args]

    if {"$action" == ""} {
      set action "global confirmStatus; set confirmStatus $label"
    }
    
    button .confirm.$label \
      -command "destroy .confirm; $action" \
      -text $label

    pack append .confirm \
      .confirm.$label {left frame center expand fill pady 8}
  }
  tkwait window .confirm
  return $confirmStatus    
}

# Procedure: ChangeMode
proc ChangeMode {modeName} {
  global CurrentMode ModeHelp HelpEntries

  grab release .f2.f22.cv
  # ImageMode == false mode! #

  if {$modeName == "ImageMode"} {
    cimage_dialog photo
    return
  } elseif {$modeName == "BitmapMode"} {
    cimage_dialog bitmap
    return
  }
  
  ModeExitHook $CurrentMode
  [SN $CurrentMode] configure -bg [.frame0.modeFr0 cget -bg]
  ModeEntryHook $modeName
  [SN $modeName] configure -bg #c00 
  set CurrentMode $modeName
}

################################################################
####			 Gripe                              ####
################################################################

proc Gripe {} {
  global gripeFile

  if {$gripeFile == ""} { return }

  set content [textBox {Gripe Message:} "" Ok Cancel]
  if {[catch {open $gripeFile "a"} st]} {
    warn "Couldn't open $gripeFile for writing"
    return
  }

  puts $st "[lindex $content 1]\n"
  close $st
}  

##
##  Circulate Entry bindings
##

proc circEntryBind {fwd bwd args} {

  set ids [lindex $args 0]
  set l [llength $ids]
  foreach action $fwd {
    for {set i 0} {$i < $l} {incr i} {
      bind [lindex $ids $i] $action \
	"+ focus [lindex $ids [expr ($i+1)%($l)]]"
    }
  }

  foreach action $bwd {
    for {set i 0} {$i < $l} {incr i} {
      bind [lindex $ids $i] $action \
	"+ focus [lindex $ids [expr ($i-1)%($l)]]"
    }
  }
}

################################################################
####	  Handling of displayed option frames               ####
####     3 option frames are simultaneously displayed.      ####
####  The older one is removed when a new one is requested  ####
################################################################

proc optionViewRaise { w } {
  global optionIsVisible
  ## inserts w on top of packing order,
  ##  then calls optionViewHandle

  ## disable configure binding to avoid nested calls !
  bind .optionFr <Configure> {}

  pack forget $w
  pack $w -before [lindex [pack slaves .optionFr] 1] \
  -fill both
  set optionIsVisible($w) 1
  update
  optionViewHandle
}

proc optionViewHandle {} {
  global optionIsVisible

  update
  ## disable configure binding to avoid nested calls !
  bind .optionFr <Configure> {}

  foreach w [pack slaves .optionFr] {
    if {[winfo height $w] < [winfo reqheight $w]} {
      pack forget $w
      set optionIsVisible($w) 0
    }
  }
  
  ##  restore configure binding  ##
  update	
  bind .optionFr <Configure> { optionViewHandle }
}

proc MenuPopupHandle { xfMenu xfW xfX xfY} {
  ##########
  # Procedure: MenuPopupHandle
  # Description: handle the popup menus
  # Arguments: xfMenu - the menu to attach
  #            xfW - the widget
  #            xfX - the root x coordinate
  #            xfY - the root x coordinate
  # Returns: none
  # Sideeffects: none
  ##########
  
  if {"[info commands $xfMenu]" != "" && [winfo ismapped $xfMenu]} {
    set xfPopMinX [winfo rootx $xfMenu]
    set xfPopMaxX [expr $xfPopMinX+[winfo width $xfMenu]]
    if {($xfX >= $xfPopMinX) &&  ($xfX <= $xfPopMaxX)} {
      $xfMenu activate @[expr $xfY-[winfo rooty $xfMenu]]
    } else {
      $xfMenu activate none
    }
  }
}

################################################################
####			 Utils                              ####
################################################################

proc cv {} {
  return .f2.f22.cv
}

########  lmember  ########

proc lmember {l elt} {
  return [expr "[lsearch -exact "$l" "$elt"] >= 0"]
}

proc getAccel {procName} {
  global theBindings

  foreach spec $theBindings {
    lassign $spec ev proc accel
    if {[string trim "$proc"] == [string trim "$procName"]} {
      return $accel
    }
  }
  return ""
}

proc isfpnum {s} {
  scan $s {%f} value
  if {$value == [string trim $s]} {
    return 1
  } else {
    warn "Floating point value required"
    return 0
  }
}

################################################################
####		  Text font management                      ####
################################################################

####  Get list of available fonts  ####
# set FontList [split [exec xlsfonts] "\n"]
set FontList {"-*-Courier-Medium-R-Normal--*-100-*-*-*-*-*-*"}

proc getFontName {name size weight slant} {
  global FontList

  set slant [string index $slant 0]
  set i [lsearch -glob $FontList "*$name-$weight-$slant-*--$size*"]
  if {$i < 0} {
    return ""
  } else {
    return [lindex $FontList $i]
  }
}

##  Finds a font name as close as possible of current  ##
##  requested field values (name, size, weight, slant)  ##
## Updates currentFont and current field values accordingly ##

proc setCurrentFont {args} {
  global fontAvail
  global currentFName currentFSize currentFWeight currentFSlant
  global currentFont

  set font [getFontName $currentFName $currentFSize $currentFWeight $currentFSlant]

  if {$font != ""} {
    set currentFont $font
    return
  }
  
  ##  Try to modify one of the field. args contains requested fields, so  ##
  ##  don't modify these  ##
  if {![lmember $args "name"] &&
    ($currentFName != "symbol") } {
      #  modify name  #
     foreach name $fontAvail(Name) {
	if {($name == "symbol") ||
	  ($name == $currentFName)} { continue }
	set font [getFontName $name $currentFSize $currentFWeight $currentFSlant]
	if {$font != ""} {
	  set currentFName $name
	  set currentFont $font
	  warn "Had to change font name to $name" 5000
	  return
	}
      }
    }

  if {![lmember $args "size"]} {
    #  modify size, trying nearest sizes first  #
    set ssizes [lsort -command {nearestSize $currentFSize} $fontAvail(Size)]
    
    foreach size $ssizes {
      if {$size == $currentFSize} { continue }
      set font [getFontName $currentFName $size $currentFWeight $currentFSlant]
      if {$font != ""} {
	set currentFSize $size
	set currentFont $font
	warn "Had to change font size to $size" 5000
	return
      }
    }
  }

  if {![lmember $args "weight"]} {
    #  modify weight #
    
    foreach weight $fontAvail(Weight) {
      if {$weight == $currentFWeight} { continue }
      set font [getFontName $currentFName $currentFSize $weight $currentFSlant]
      if {$font != ""} {
	set currentFWeight $weight
	set currentFont $font
	warn "Had to change font weight to $weight" 5000
	return
      }
    }
  }
  
  if {![lmember $args "slant"]} {
    #  modify slant #
    
    foreach slant $fontAvail(Slant) {
      if {$slant == $currentFSlant} { continue }
      set font [getFontName $currentFName $currentFSize $currentFWeight $slant]
      if {$font != ""} {
	set currentFSlant $slant
	set currentFont $font
	warn "Had to change font slant to $slant" 5000
	return
      }
    }
  }

  warn "Couldn't find any appropriate font"
}

proc nearestSize {ref s1 s2} {
  return [expr "abs($ref-$s1) - abs($ref-$s2)"]
}


################################################################
####		     Symbolic Names                         ####
################################################################

if {"[info procs SymbolicName]" == ""} {
  proc SN { {xfName ""}} {
    global symbolicName
    
    if {"$xfName" != ""} {
      set xfArrayName ""
      append xfArrayName symbolicName ( $xfName )
      if {![catch "set \"$xfArrayName\"" xfValue]} {
        return $xfValue
      } else {
        puts stderr "XF error: unknown symbolic name:\n$xfName"
      }
    }
    return ""
  }
}

################################################################
####		      Warning area                          ####
################################################################

set lastMsg {}
set lastWarnMsg {}
set lastWarnBg  {}

proc warn {args} {
  global lastWarnMsg lastMsg

  # warn <text> [<timeout>] #

  # If there's already a warning, ignore this one #
  if {$lastWarnMsg == ""} {
    
    if {[llength $args] == 1} {
      set tm 5000
    } else {
      set tm [lindex $args 1]
    }
    
    set bg [lindex [.frame1 configure -background] 4]
    set lastMsg [lindex [.frame1.warnLb configure -text] 4]
    set lastWarnMsg [lindex $args 0]
    .frame1.warnLb configure \
      -text [lindex $args 0] \
      -background {#8ba3ce}
    
    update
    after $tm ".frame1.warnLb configure -background $bg -text [list $lastMsg]
               set lastWarnMsg \"\"
               update"
  }
}

proc saveMsg {} {
  global lastMsg lastWarnBg
  set lastMsg [lindex [.frame1.warnLb configure -text] 4]
  set lastWarnBg [lindex [.frame1 configure -background] 4]
}

proc restoreMsg {} {
  global lastMsg lastWarnBg

  .frame1.warnLb configure \
    -background $lastWarnBg \
    -text $lastMsg
}

proc msg {msg} {
  .frame1.warnLb configure \
    -text "$msg"
  update
}

################################################################
####		    Popup Text area                         ####
################################################################

set textBoxContent {}
set textBoxButton  {}

proc textBox {{label ""} {content ""} args} {
  global textBoxContent textBoxButton
  # args = list of buttons to create #

  catch {destroy .textBox}
  toplevel .textBox -borderwidth 0
  wm geometry .textBox \
    [expr int(0.6*[winfo screenwidth .])] [expr int(0.4*[winfo screenheight .])]
  wm title .textBox {Text box}
  wm maxsize .textBox [winfo screenwidth .] [winfo screenheight .]
  wm minsize .textBox 100 100

  label .textBox.message1  -anchor c  -relief raised \
    -text $label

  text .textBox.text

  .textBox.text insert 1.0 $content

  frame .textBox.buttonFr -borderwidth 0

  if {$args == 0} {
    set args {ok cancel}
  }

  foreach but $args {
    button .textBox.buttonFr.b${but}Bt \
      -text $but \
      -relief raised \
      -command "
	set textBoxContent  \[.textBox.text get 1.0 end\]
	set textBoxButton $but
        grab release .textBox
	destroy .textBox"

    pack append .textBox.buttonFr \
      .textBox.buttonFr.b${but}Bt {left fill expand}
  }

  pack append .textBox \
    .textBox.message1 {top fillx frame n pady 10} \
    .textBox.text     {top expand fill} \
    .textBox.buttonFr {top fill}

  update idletask
  grab set .textBox
  tkwait window .textBox
  grab release .textBox
  focus .
  return [list $textBoxButton $textBoxContent]
}

##
##  GUI images creation
##    returns a photo image name if .gif found,
##    or a bitmap image name if .xbm found
## 

proc mkimage {file} {
  global picLibDir

  if {[catch {image create photo -file [file join "$picLibDir" "bitmaps" "$file.gif"]} name]} {
    if {[catch {image create bitmap \
      -file [file join "$picLibDir" "bitmaps" "$file.xbm"]} bname]} {
	error $name
      } else {
	return $bname
      }
  } else {
    return $name
  }
}

proc mkbitmap {args} {
  global PicassoLib bitmapPath

  if {$args == ""} { return "" }
  set name [join $args .]

  #  args == <spec> <par1> ... <parn>  #
  # if file <spec>.<par1>. ... .<parn>.xbm exist, return it #
  ## otherwise look for a bitmap file

  set found ""
  foreach dir $bitmapPath {
    if {[file readable [file join "$dir" "$name.xbm"]]} {
      set found [file join "$dir" "$name.xbm"]; break;
    }
    if {[file readable [file join "$dir" "$name"]]} {
      set found [file join "$dir" "$name.xbm"]; break;
    }
  }
  if {"$found" != ""} {
      return "@$found"
  } else {
      MessageBox "WARNING: bitmap file \"$name\" couldn't be found
in the bitmap search path
$bitmapPath"
    return ""
  }
}

##
##  Returns either
##    - an existing bitmap image name matching spec
##    - an image bitmap name loaded from file in bitmapPath
##    - an image bitmap name buit using -data option
##

proc mkbitmapImage {args} {
  global PicassoLib bitmapPath

  #  args == <spec> <par1> ... <parn>  #
  # if file <spec>.<par1>. ... .<parn>.xbm exist, return it #
  #  otherwise build the bitmap  #

  if {$args == ""} { return "" }
  set name [join $args .]
  ## if an existing bitmap image matches spec, return its name
  if {[set i [lsearch -exact [image names] "_$name"]] >= 0} {
    return "_$name"
  }
    
  ## otherwise look for a bitmap file
  set found ""
  foreach dir $bitmapPath {
    if {[file readable [file join "$dir" "$name.xbm"]]} {
      set found [file join "$dir" "$name.xbm"]; break;
    }
    if {[file readable [file join "$dir" "$name"]]} {
      set found [file join "$dir" "$name.xbm"]; break;
    }
  }
  if {"$found" != ""} {
    image create bitmap "_$name" -file "$found"
    return "_$name"
  }
  
  ##  Build the bitmap following specification  ##
  switch [lvarpop args] {

    width {
      set w [lindex $args 0]
      #  16x8 bitmap  #
      # prepare data string
      
      set pixels {}
      set blankline {0x00,0x00}
      set blackline {0xff,0xff}

      for {set nl 0} {$nl < [expr "(8-$w)/2"]} {incr nl} {
        lappend pixels " $blankline"
      }
      for {set i 0} {$i < $w} {incr nl; incr i} {
        lappend pixels "$blackline"
      }
      while {$nl < 8} {
        lappend pixels "$blankline"
        incr nl
      }
      set data "#define b_width 16
#define b_height 8
static char b_bits\[\] = {
[join $pixels ,]
};
"
      # create image bitmap
      image create bitmap "_$name" \
      -data "$data"
      return "_$name"
    }

    default {
      puts stderr \
	"\nWARNING : couldn't find bitmap $name
  Check your PicassoLib environment variable
  and the directory \$PicassoLib/bitmaps\n"
      return "" }
  }      
}
