# rcsid {$Id: picproc.tcl,v 4.18 1996/05/21 16:15:08 mangin Rel $}
# This package is free software. Redistribution and use of this file
# are permitted without restrictions.
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
#	Frank Mangin
#	Frank.Mangin@ceram.fr  -  Frank.Mangin@sophia.inria.fr

if {[info exists env(TEMP)]} {
  set TmpDir $env(TEMP)
} elseif {[info exists env(TMPDIR)]} {
  set TmpDir $env(TMPDIR)
} else {
  set TmpDir $env(HOME)
}
if {[info exists env(WINDOWID)]} {
  set TmpSuff ".$env(WINDOWID)"
} else {
  set TmpSuff ""
}

set TmpNum 0

################################################################
####		    Signal handling                             ####
################################################################

#if {$tcl_platform(platform) == "unix"} {
  #  signal trap {SIGTERM SIGQUIT SIGINT} { cleanExit %S }
  # }

proc cleanExit {args} {
  global TmpDir TmpSuff

  if {$args != ""} {
    #  Signal generated exit  #
    puts stderr "caught signal $args - cleaning up"
    catch {exec rm [sglob "$TmpDir/picasso${TmpSuff}*"]}
    set dumpfile [mkTmpFile Dump]
    if {[catch {SaveDrawingInFile $dumpfile}]} {
      puts stderr "Couldn't dump drawing"
      catch {exec rm $dumpfile}
    } else {
      puts stderr "Drawing dumped to $dumpfile"
    }
    update
    destroy .
    exit 1
  } else {
    catch {exec rm [sglob "$TmpDir/picasso${TmpSuff}*"]}
    destroy .
  }
}

################################################################
####	       Useful general procedures                    ####
################################################################

########  Wait cursor  ########

set savedCursor {}

proc setCursor {name} {
  global savedCursor

  if {[lindex [. configure -cursor] 4] != "watch"} {
    set savedCursor [lindex [. configure -cursor] 4]
  }
  . configure -cursor "$name"
  update
}

proc waitCursor {} {
  global savedCursor

  if {[lindex [. configure -cursor] 4] != "watch"} {
    set savedCursor [lindex [. configure -cursor] 4]
  }
  . configure -cursor watch
  update
}

proc restoreCursor {} {
  global savedCursor
  . configure -cursor $savedCursor
  update
}

########  Temporary file names  ########
proc mkTmpFile {base} {
  global TmpNum TmpDir TmpSuff

  return [file join "$TmpDir" "picasso${TmpSuff}.${base}[incr TmpNum]"]
}

########  Safe glob  ########
proc sglob {path} {
  set new [glob -nocomplain -- "$path"]
  if {$new == {}} {
    set new [glob -nocomplain -- [file dirname "$path"]]
    if {"$new" == ""} {
      return "$path"
    } else {
      return [file join [lindex $new 0] [file tail $path]]
    }
  } else {
    return [lindex $new 0]
  }
}

####  Utils  ####

proc st {} {
  return .optionFr.styleFr.object
}

proc fontCv {} {
  return .optionFr.fontFr.showCv
}

# Procedure Quit

proc Quit {} {
  global ioState
  
  set cstate [getCurrentState]
  if {($ioState(lastSaved) != $cstate) &&
    ($cstate != {{} {} {} {}})} {
      ConfirmBox "Save current drawing before exiting?" \
	{save} { SaveDrawing } \
	{discard} { cleanExit} \
	{cancel}  { return }
    } else {
      cleanExit
    }
}

#  lvartail  #

proc lvartail {varname} {
  upvar 1 $varname var

  set l [expr "[llength $var] - 1"]
  set elt [lindex $var $l]
  incr l -1
  set var [lrange $var 0 $l]
  return $elt
}

#  llast  #

proc llast {l} {
  return [lindex $l [expr "[llength $l]-1"]]
}

################################################################
####		 Geometric calculations                     ####
################################################################

##  Pour remettre les coordonnees ds l'ordre  ##
proc dragMotion {tagOrId x y} {
  global x1fixed y1fixed

  set bb [.f2.f22.cv coords $tagOrId]
  
  if {$bb != ""} {
    if {$x1fixed} {
      set x1 [lindex $bb 0]
      if {$x < $x1} {
	set newx1 $x
	set newx2 $x1
	set x1fixed 0
      } else {
	set newx1 $x1
	set newx2 $x
      }
    } else {
      set x2 [lindex $bb 2]
      if {$x < $x2} {
	set newx1 $x
	set newx2 $x2
      } else {
	set newx1 $x2
	set newx2 $x
	set x1fixed 1
      }
    }
    
    if {$y1fixed} {
      set y1 [lindex $bb 1]
      if {$y < $y1} {
	set newy1 $y
	set newy2 $y1
	set y1fixed 0
      } else {
	set newy1 $y1
	set newy2 $y
      }
    } else {
      set y2 [lindex $bb 3]
      if {$y < $y2} {
	set newy1 $y
	set newy2 $y2
      } else {
	set newy1 $y2
	set newy2 $y
	set y1fixed 1
      }
    }
    
    .f2.f22.cv coords $tagOrId $newx1 $newy1 $newx2 $newy2
  }
}

##  Given a point and a bounding box, decide on which corner
##  or side we are 

proc cornerOrSide {x y bb} {
  set x1 [lindex $bb 0]
  set y1 [lindex $bb 1]
  set x2 [lindex $bb 2]
  set y2 [lindex $bb 3]
  
  set dx1 [expr "($x-$x1)*($x-$x1)"]
  set dy1 [expr "($y-$y1)*($y-$y1)"]
  set dx2 [expr "($x-$x2)*($x-$x2)"]
  set dy2 [expr "($y-$y2)*($y-$y2)"]
  
  set d11 [expr "$dx1 + $dy1"]
  set d12 [expr "$dx1 + $dy2"]
  set d21 [expr "$dx2 + $dy1"]
  set d22 [expr "$dx2 + $dy2"]
  
  set dmin $d11
  set xid 1
  set yid 1
  
  if {$d12 < $dmin} {
    set dmin $d12
    set xid 1
    set yid 2
  }
  if {$d21 < $dmin} {
    set dmin $d21
    set xid 2
    set yid 1
  }
  if {$d22 < $dmin} {
    set dmin $d22
    set xid 2
    set yid 2
  }
  
  if { $xid == 1 } {
    set xside [expr "($x1 != $x2) && ($x - $x1+0.0)/($x2 - $x1) > 0.33"]
  } else {
    set xside [expr "($x1 != $x2) && ($x - $x2+0.0)/($x1 - $x2) > 0.33"]
  }
  
  if { $yid == 1 } {
    set yside [expr "($y1 != $y2) && ($y - $y1+0.0)/($y2 - $y1) > 0.33"]
  } else {
    set yside [expr "($y1 != $y2) && ($y - $y2+0.0)/($y1 - $y2) > 0.33"]
  }
  
  if { $yside && $xside } { return none }
  if { $xside } { return "sx$yid" }
  if { $yside } { return "sy$xid" }
  
  return "c$xid$yid"
}

################################################################
####		 Operating mode                                 ####
################################################################

proc globalBindings {} {
  global theBindings
  bindtags .f2.f22.cv ".f2.f22.cv Canvas"

  foreach spec $theBindings {
    lassign $spec event cmd shortcut
    bind . $event "$cmd"
    # bind abort to the .f2.f22.cv so that it works 
    # while grab .f2.f22.cv is set
    if {[string match {*Abort*} $cmd]} {
      bind .f2.f22.cv $event "$cmd"
    }      
  }
  bind .optionFr <Configure> { optionViewHandle }
  GridSetBindings
}

################################################################
####		     Mode hooks                                 ####
################################################################

proc ModeEntryHook {mode} {
  global ModeFancyNames ModeState

  set info $ModeFancyNames($mode)
  set ModeState 0
  
  switch $mode {
    ArcMode -
    PieMode {
      set info "$info : set center with mouse button 1"
    }
    ReshapeMode {
      global Shaping ShapeId
      set Shaping 0
      set ShapeId {}
      deselect_all
    }
  }

  msg $info
}

proc ModeExitHook {mode} {
  
  switch $mode {
    PieMode -
    ArcMode {
      global ModeState
      set ModeState 0
      .f2.f22.cv delete inCreation
      .f2.f22.cv delete carcl1
      .f2.f22.cv delete carcl2
    }
    ReshapeMode {
      global Shaping ShapeId
      set Shaping 0
      set ShapeId {}
      .f2.f22.cv delete rsHandle
    }
    TextMode { createEnd_hook }
  }
}

###############################
##   Action Bindings       ####
###############################
proc Raise {} {
  ## Prepare undo
  saveDisplayOrder
  ApplyOnSelection raise
  canUndo
}

proc Lower {} {
  ## saveDisplayOrder
  ApplyOnSelection lower
  canUndo
}

proc Delete {} {
  ## Prepare for undo
  saveObjects s
  ApplyOnSelection delete
  canUndo
}

proc ApplyOnSelection {action} {
  global CurrentMode
  
  .f2.f22.cv addtag toApply withtag sOfr
  .f2.f22.cv addtag toApply withtag inCreation
  
  if {[.f2.f22.cv find withtag toApply] == {}} {
    warn "No item selected"
  } else {
    .f2.f22.cv $action toApply
    .f2.f22.cv dtag all toApply
  }
}

###############################
##  Current Style          ####
###############################

##  There is one entry per picasso object type in currentStyle  ##
##  Each entry is the option line to apply to this type of object  ##

set currentStyle(rectangle)  {}
set currentStyle(line)       {}
set currentStyle(curve)      {}
set currentStyle(oval)       {}
set currentStyle(arc)        {}
set currentStyle(pieslice)   {}
set currentStyle(polygon)    {}
set currentStyle(text)       {}
set currentStyle(image)      {}
set currentStyle(bitmap)     {}

##
##  Mapping between picasso styles
##  and canvas item config parameters
##

array set Style2Config {
  rectangle.lineColor 	-outline
  rectangle.fillColor -fill		
  rectangle.lineWidth	-width		
  rectangle.stipple	-stipple	
  line.lineColor 		-fill		
  line.lineWidth		-width		
  line.arrow			-arrow		
  curve.lineColor 	-fill		
  curve.lineWidth		-width		
  curve.arrow			-arrow		
  oval.lineColor 		-outline	
  oval.fillColor 		-fill		
  oval.lineWidth		-width		
  oval.stipple		-stipple	
  arc.lineColor 		-outline	
  arc.lineWidth		-width		
  pieslice.lineColor	-outline	
  pieslice.fillColor 	-fill		
  pieslice.lineWidth	-width		
  pieslice.stipple	-stipple	
  polygon.fillColor 	-fill		
  polygon.lineColor	-outline	
  polygon.stipple		-stipple	
  text.lineColor 		-fill		
  text.font 			-font		
  image.lineColor 	-foreground
  image.fillColor	-background
}

##
## Initial setup:
##   set default values,
##   and create style viewer sample objects
##

proc styleSetup {} {
  global currentFont
  
  [st] create oval 8 8 24 40 \
  -tags {show}
  [st] create line 36 30 46 12 54 35 62 29 \
  -tags {show}

  SetStyle lineColor 	black
  SetStyle fillColor 	black
  SetStyle stipple	{}
  SetStyle lineWidth	1
  SetStyle arrow     	none
  if {"$currentFont" != ""} {
    SetStyle font 		$currentFont
  }
}

##
## Replace value in matching "-opt value" 
##   or append new opt
##
proc styleOverride {type opt value} {
  global currentStyle
  
  set i [lsearch -exact $currentStyle($type) $opt]
  if {$i < 0} {
    lappend currentStyle($type) $opt "$value"
  } else {
    incr i
    set currentStyle($type) [lreplace $currentStyle($type) $i $i "$value"]
  }
}

##
## The procedure invoked when the user changes 
##   a setting in an option frame
##

proc SetStyle {style value} {
  global currentStyle Style2Config
  ##  style is picasso defined; we translate it  ##
  ##  into appropriate canvas item options  ##

  cantUndo
  ##  Apply style on font displayer object  ##
  foreach item [[fontCv] find withtag all] {
    set sty [[fontCv] type $item].$style
    if {[info exists Style2Config($sty)]} {
      [fontCv] itemconfigure $item \
	$Style2Config($sty) $value
      #  recenter object  #
      [fontCv] coords $item \
	[expr "[winfo width [fontCv]]/2"] \
	[expr "[winfo height [fontCv]]/2"]
      [fontCv] itemconfigure $item -anchor center
    }
  }
  
  ##  Apply style on style viewer objects  ##
  foreach item [[st] find withtag show] {
    set sty [[st] type $item].$style
    if {[info exists Style2Config($sty)]} {
      [st] itemconfigure $item \
	$Style2Config($sty) $value
    }
  }
  
  ##  Apply current style on selected objects  ##
  ##  and on objects being created  ##

  set items [concat [.f2.f22.cv find withtag s] [.f2.f22.cv find withtag inCreation]]

  foreach item $items {

    set type [.f2.f22.cv type $item]

    if {($type == "arc") &&
      ([lindex [.f2.f22.cv itemconfigure $item -style] 4] == "pieslice")} {
        set type pieslice
      }

    if {[info exists Style2Config($type.$style)]} {
      if {[catch {.f2.f22.cv itemcget $item -image} im]} {
        .f2.f22.cv itemconfigure $item $Style2Config($type.$style) $value
      } else {
        catch {$im config $Style2Config($type.$style) $value} err
      }
    }
  }

  ##  Register change in currentStyle array  ##
  foreach name [array names Style2Config "*.$style"] {
    set type [lindex [split $name .] 0]
    styleOverride $type $Style2Config($name) $value
  }
  
  redrawFrames
}

###############################
####		       OPERATIONS                           ####
###############################

proc dragStartHook {} {
  grab .f2.f22.cv
  focus .f2.f22.cv
}

proc dragEndHook {} {
  grab release .f2.f22.cv
  focus .
}

###############################
####		       Selection                                ####
###############################

##  TAGS :
##  - s     => is selected and isn't frame  
##  - sOfr 	=> is selected or is a select frame
##  - sFr   => is a select frame
##  - firstS => reference for align

proc notAFrame {item} {
  return [expr "[lsearch -exact [.f2.f22.cv gettags $item] {sFr}] < 0"]
}

## Select frames:
##   Red canvas rectangles drawn around around
##     selected objects to signal them	

proc putFrame {gtag {col red} {w 2}} {
  global sFrId

  lassign [.f2.f22.cv bbox $gtag] x1 y1 x2 y2
  if {$x1 == ""} { return }
  incr x1 -1
  incr y1 -1
  incr x2
  incr y2
  return \
  [set sFrId($gtag) \
  [.f2.f22.cv create rectangle \
  $x1 $y1 $x2 $y2 \
  -width $w \
  -outline $col -width 1 \
  -tags [list sOfr sFr x$gtag ctl]]]
}

# Redraw frames around selected groups

proc redrawFrames {{col red} {w 1} } {
  .f2.f22.cv delete sFr
  foreach gtag [getTopGroupTags s] {
    putFrame $gtag $col $w
  }
}

########  ########

set x1fixed 1
set y1fixed 1

proc deselect_all {} {
  .f2.f22.cv delete   sFr
  .f2.f22.cv dtag all s
  .f2.f22.cv dtag all sOfr
  .f2.f22.cv dtag all firstS
}

proc SelectAll {} {

  deselect_all

  ##  add selection tags and store existing groups  ##

  set gtags {}
  foreach item [.f2.f22.cv find withtag all] {
    .f2.f22.cv addtag s withtag $item
    .f2.f22.cv addtag sOfr withtag $item
    lappend gtags [getTopGroupTags $item]
  }

  ## put a frame around each existing group ##
  foreach gtag [lrmdups $gtags] {
    putFrame $gtag
  }
}

proc iselect {x y} {
  global xdragAnchor ydragAnchor x1fixed y1fixed
  
  deselect_all

  set x [.f2.f22.cv canvasx $x]
  set y [.f2.f22.cv canvasy $y]
  
  if {([set item [.f2.f22.cv find withtag current]] != "") &&
    [notAFrame current]} {
      ##  single item selection  ##
      set gtag [getTopGroupTags $item]
      .f2.f22.cv addtag s withtag $gtag
      .f2.f22.cv addtag sOfr withtag $gtag
      .f2.f22.cv addtag firstS withtag $item
      putFrame $gtag
    } else {
      ##  no item -> start region selection  ##
      dragStartHook
      .f2.f22.cv create rectangle $x $y $x $y \
	-outline red -tags {regionsFr}
      set x1fixed 1
      set y1fixed 1
    }
}

proc iselectRegion {x y} {
  dragMotion regionsFr [.f2.f22.cv canvasx $x] [.f2.f22.cv canvasy $y]
}

proc iselectRegionEnd {} {

  set bb [.f2.f22.cv coords regionsFr]
  .f2.f22.cv delete regionsFr
  
  if {$bb != ""} {
    #  find items enclosed in region  #
    set into [eval ".f2.f22.cv find enclosed $bb"]
    #  find groups existing in these items  #
    set intogtags {}
    foreach item $into {
      lappend intogtags [getTopGroupTags $item]
    }
    #  only keep groups entirely included in region  #
    #  and corresponding items #
    set allintogtags {}
    set allinto {}
    foreach gtag [lrmdups $intogtags] {
      set gcontent [.f2.f22.cv find withtag $gtag]
      set ok 1
      foreach id $gcontent {
      	if {[lsearch -exact $into $id] < 0} {
	  set ok 0
	  break
      	}
      }
      if {$ok} {
     	lappend allintogtags $gtag
     	lappend allinto $gcontent
      }
    }
    foreach gtag $allintogtags {
      .f2.f22.cv addtag s withtag $gtag
      .f2.f22.cv addtag sOfr withtag $gtag
      putFrame $gtag
    }
  }
  dragEndHook
}            

proc iselectAugment {} {
  global sFrId

  set item [.f2.f22.cv find withtag current]
  if {($item != "") && [notAFrame $item]} {
    if {[lsearch [.f2.f22.cv gettags $item] s] < 0} {
      #  select it  #
      set gtag [getTopGroupTags $item]
      .f2.f22.cv addtag s withtag $gtag
      .f2.f22.cv addtag sOfr withtag $gtag
      putFrame $gtag
    } else {
      #  deselect it  #
      set gtag [getTopGroupTags $item]
      .f2.f22.cv delete withtag $sFrId($gtag)
      .f2.f22.cv dtag $gtag s
      .f2.f22.cv dtag $gtag sOfr
    }
  }
}

################################################################
####		    Drag management                         ####
################################################################

set Dragging 0

proc select_for_drag {} {
  global Dragging

  if {[set item [.f2.f22.cv find withtag current]] != ""} {
    set tags [.f2.f22.cv gettags $item]
    if {([lsearch -exact $tags "ctl"] < 0) &&
      [lsearch -exact $tags s] < 0} {
	set gtag [getTopGroupTags $item]
	.f2.f22.cv addtag sForDrag withtag $gtag
	.f2.f22.cv addtag s withtag $gtag
	.f2.f22.cv addtag sOfr withtag $gtag
	.f2.f22.cv addtag forDragFrame withtag [putFrame $gtag]
      }
  }

  set ids [.f2.f22.cv find withtag s]
  if {$ids == ""} {
    set Dragging 0
    return ""
  } else {
    set Dragging 1
    return $ids
  }
}

proc deselect_for_drag {} {
  global Dragging
  
  if {$Dragging} {
    set Dragging 0
    .f2.f22.cv delete forDragFrame
    .f2.f22.cv dtag sForDrag s
    .f2.f22.cv dtag sForDrag sOfr
    .f2.f22.cv dtag all sForDrag
  }
}

################################
##
##  Cut, copy & paste
##
## The killBuf contains the list of commands to execute 
##   to recreate saved objects
##

proc Cut {} {
  Copy
  Delete
}

##
##  Foreach selected object, get type, coords & config
##    Also copy group tags, creatinf new group numbers
##

# Next free "program" group id
set nextPGroupId 0

proc Copy {} {
  global killBuf nextPGroupId
  
  set killBuf ""
  set ids [.f2.f22.cv find withtag s]
  foreach id $ids {
    append killBuf ".f2.f22.cv create [.f2.f22.cv type $id] [.f2.f22.cv coords $id]"
    foreach spec [.f2.f22.cv itemconfigure $id] {
      lassign $spec opt foo foo dflt val
      if {$opt == "-tags"} {
        set newtags "-tags {s sOfr"
        foreach tag $val {
          switch -glob -- $tag {
            s -
            sOfr {}
            Gr* -
            tGr* {
              set groups [split $tag .]
              set newtag [lvarpop groups]
              foreach group $groups {
                if {[catch {append newtag ".$newgids($group)"}]} {
                  set newgids($group) "p[incr nextPGroupId]"
                  append newtag ".$newgids($group)"
                }
              }
              append newtags " $newtag"
            }
          }
        }
        append killBuf " $newtags}"
      } elseif {"$val" != {} && "$val" != "$dflt"} {
        append killBuf " [lindex $spec 0] \"[lindex $spec 4]\""
      }
    }
    append killBuf "\n"
  }
  global undoCommand
  set undoCommand {global killBuf; eval $killBuf}
  canUndo
}


proc Paste {} {
  global killBuf
  
  deselect_all
  eval $killBuf
  redrawFrames
}

########################
##
##  Abort
##

proc Abort {} {
  global CurrentMode
   
  ## Try to find what's going on...
  if {[winfo exists .fsbox]} {
    fsboxAbort
    return
   }
   
  switch -exact -- $CurrentMode {
    SelectMode {}
    ScaleMode  {}
    StretchMode {}
    ReshapeMode {}
    default {
      global ModeState ModeFancyNames
      # Object creation mode: reset 
      .f2.f22.cv delete inCreation
      set ModeState 0
      .f2.f22.cv delete carcl1
      .f2.f22.cv delete carcl2
      grab release .f2.f22.cv
      .f2.f22.cv focus {}
      focus .
      
      msg $ModeFancyNames($CurrentMode)
    }
  }
}   
 
#########################
##
##  Undo
##

set undoCommand {}

proc canUndo  {} {
  .frame1.editBt.m entryconfig {*Undo*} -state normal
  update
}

proc cantUndo {} {
  .frame1.editBt.m entryconfig {*Undo*} -state disabled
  update
}

proc Undo {} {
  global undoCommand

  if {[.frame1.editBt.m entryconfig {*Undo*} -state] == "disabled"} {
    return
  }
  
  .frame1.editBt.m entryconfig {*Undo*} -state disabled
  ## seems easy, isn't it?
  eval "$undoCommand"
  .f2.f22.cv delete rsHandle
  .f2.f22.cv delete opFr
  global ShapeId; set ShapeId {}
  global ModeState; set ModeState 0
  redrawFrames
  update
}

proc saveCoords {tag} {
  global undoCommand

  set undoCommand ""
  foreach id [.f2.f22.cv find withtag $tag] {
    append undoCommand ".f2.f22.cv coords $id [.f2.f22.cv coords $id];";
  }
}

proc saveDisplayOrder {} {
  global undoCommand
  set undoCommand "setDisplayOrder [.f2.f22.cv find all]"
}

proc setDisplayOrder {list} {
  ## Reset display order as specified by list
  ##  simply raise items one by one
  foreach id $list {
    catch {.f2.f22.cv raise $id}
  }
}

proc saveObjects {tag} {
  ## Try to be fast...
  global undoCommand
  set undoCommand "restoreObjects"
  foreach id [.f2.f22.cv find withtag $tag] {
    lappend undoCommand \
      [list [.f2.f22.cv type $id] [.f2.f22.cv coords $id] [.f2.f22.cv itemconfig $id]]
  }
}

proc restoreObjects {args} {
  .f2.f22.cv dtag all s
  .f2.f22.cv dtag all sOfr
  
  foreach spec $args {
    lassign $spec type coords config
    set options ""
    foreach cf $config {
      if {"[lindex $cf 4]" != ""} {
	append options " [lindex $cf 0] \"[lindex $cf 4]\""
      }
    }
    eval ".f2.f22.cv create $type $coords $options"
  }
  redrawFrames
  update
}
